// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include "gts_interface.h"

namespace libk3dgts
{

/////////////////////////////////////////////////////////////////////////////
// poly_sphere_tessellation_implementation

class poly_sphere_tessellation_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	poly_sphere_tessellation_implementation(k3d::idocument& Document) :
		base(Document),
		m_tessellation_level(k3d::init_name("tessellation_level") + k3d::init_description("Columns [integer]") + k3d::init_value(3) + k3d::init_document(Document) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_document(Document) + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_tessellation_level));
		enable_serialization(k3d::persistence::proxy(m_radius));

		register_property(m_tessellation_level);
		register_property(m_radius);

		m_material.changed_signal().connect(SigC::slot(*this, &poly_sphere_tessellation_implementation::on_reset_geometry));
		m_tessellation_level.changed_signal().connect(SigC::slot(*this, &poly_sphere_tessellation_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &poly_sphere_tessellation_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &poly_sphere_tessellation_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Create new GTS surface
		GtsSurface* s = gts_surface_new(gts_surface_class(),
			gts_face_class(), gts_edge_class(), gts_vertex_class());

		// Generate tessellated sphere
		gts_surface_generate_sphere(s, m_tessellation_level.property_value());
		return_val_if_fail(s != NULL, 0);

		// Load sphere into a k3d::mesh
		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());
		return_val_if_fail(copy_surface(s, *mesh.get()), 0);

		// Apply radius
		double radius = m_radius.property_value();
		k3d::mesh::points_t::iterator mesh_end(mesh->points.end());
		for(k3d::mesh::points_t::iterator p = mesh->points.begin(); p != mesh_end; ++p)
			(*p)->position *= radius;

		// Set material
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();
		polyhedron.material = m_material.interface();

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_sphere_tessellation_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x5d228c83, 0xff624faa, 0xa95cb0ee, 0xc2b1613b),
			"PolySphereTessellation",
			"Generates a polygonal sphere by recursive subdivision of an isocahedron",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_tessellation_level;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
};

/////////////////////////////////////////////////////////////////////////////
// poly_sphere_tessellation_factory

k3d::iplugin_factory& poly_sphere_tessellation_factory()
{
	return poly_sphere_tessellation_implementation::get_factory();
}

} // namespace libk3dgts


