#ifndef K3DUI_OBJECT_CHOOSER_H
#define K3DUI_OBJECT_CHOOSER_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares k3d::object_chooser, which provides a user interface for either creating new, or selecting existing, K-3D objects
		\author Tim Shead (tshead@k-3d.com)
*/

#include "dynamic_menu.h"
#include "k3dcontrol.h"

#include <k3dsdk/iobject_collection.h>
#include <k3dsdk/iplugin_factory.h>
#include <k3dsdk/iproperty.h>
#include <k3dsdk/iwritable_property.h>
#include <k3dsdk/result.h>

#ifdef	interface
#undef	interface
#endif	// interface

// Forward declarations
namespace k3d { class idocument; }
namespace k3d { class iobject; }

namespace k3d
{

namespace object_chooser
{

/////////////////////////////////////////////////////////////////////////////
// idata_proxy

/// Abstract interface for an object that proxies a data source for an object_chooser::control (i.e. the "model" in model-view-controller)
class idata_proxy
{
public:
	virtual ~idata_proxy() {}

	/// Called to return the underlying data value
	virtual iobject* object() = 0;
	/// Called to set a new data value
	virtual void set_object(iobject* Object) = 0;
	/// Called to return the owning document
	virtual idocument& document() = 0;
	/// Signal emitted if the underlying data changes
	typedef SigC::Signal0<void> changed_signal_t;
	/// Signal emitted if the underlying data changes
	virtual changed_signal_t& changed_signal() = 0;

protected:
	idata_proxy() {}
	idata_proxy(const idata_proxy& RHS) {}
	idata_proxy& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// iselection_filter

/// Abstract interface for an object that will be queried to filter the set of available choices
class iselection_filter
{
public:
	virtual ~iselection_filter() {}

	/// Return true iff "None" should be available to the user as a selection
	virtual bool allow_none() = 0;
	/// Return true iff the specified plugin type should be available to the user as a "Create XXX" selection
	virtual bool allow(k3d::iplugin_factory& Factory) = 0;
	/// Return true iff the specified iobject should be available to the user for selection
	virtual bool allow(iobject& Object) = 0;

protected:
	iselection_filter() {}
	iselection_filter(const idata_proxy& RHS) {}
	iselection_filter& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// control

/// Provides a UI for choosing objects (either existing, or newly-created) (i.e. the view and the controller from model-view-controller)
class control :
	public k3dControl
{
public:
	control(k3d::iunknown* const CommandNodeParent, const std::string CommandNodeName);
	~control();

	/// Called by the framework when instantiating the button from a GTKML (XML) document
	bool Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element);

	/// Attaches the button to the data it fronts for
	bool attach(std::auto_ptr<iselection_filter> SelectionFilter, std::auto_ptr<idata_proxy> Data, k3d::istate_recorder* const StateRecorder, const std::string StateChangeName);

	const std::string CustomType() const;

	// k3d::icommand_node implementation
	bool execute_command(const std::string& Command, const std::string& Arguments);

private:
	typedef k3dControl base;

	/// Called whenever the the selection state needs to be updated
	void update();

	/// Called when new objects are added to the document
	void on_objects_added(const iobject_collection::objects_t&);
	/// Called when objects are removed from the document
	void on_objects_removed(const iobject_collection::objects_t&);

	/// Called to handle SDPGTK events
	void OnEvent(sdpGtkEvent* Event);
	/// Called when the GTK+ widgets are about to disappear
	void on_destroy();
	/// Called to display the set of available choices
	void on_choose();
	/// Called when the user wants to edit an existing object
	void on_edit_object();
	/// Called when the user decides to select no object
	void on_none();
	/// Called when the user selects an existing object
	void on_select_existing_object(iobject* const Object);
	/// Called when the user wants to create a new object
	void on_create_new_object(k3d::iplugin_factory* const Factory);

	/// Called to modify the underlying data object
	void set_object(iobject* const Object, const std::string UndoName);

	/// Stores a reference to the underlying data object
	std::auto_ptr<idata_proxy> m_data;
	/// Stores a filter object for controlling the set of available choices
	std::auto_ptr<iselection_filter> m_selection_filter;

	class menu_item :
		public dynamic_menu::item
	{
	public:
		menu_item(const std::string& Name, const slot_t& Slot = slot_t()) : dynamic_menu::item(Name, Slot), object(0), factory(0) { }
		menu_item(const std::string& Name, iobject* const Object, const slot_t& Slot = slot_t()) : dynamic_menu::item(Name, Slot), object(Object), factory(0) { }
		menu_item(const std::string& Name, k3d::iplugin_factory* const Factory, const slot_t& Slot = slot_t()) : dynamic_menu::item(Name, Slot), object(0), factory(Factory) { }
	
		friend bool operator==(const menu_item& LHS, const menu_item& RHS)
		{
			return LHS.object == RHS.object && LHS.factory == RHS.factory;
		}
	
		iobject* object;
		k3d::iplugin_factory* factory;
	};
	
	typedef std::vector<menu_item> menu_items;
	
	dynamic_menu::control<menu_items> m_menu;
};

/// Provides an implementation of k3d::object_chooser::idata_proxy that supports any data source that supports the object(), set_object(), object_collection(), and changed_signal() concepts
template<typename data_t>
class data_proxy :
	public idata_proxy
{
public:
	data_proxy(data_t& Data, k3d::idocument& Document) :
		m_data(Data),
		m_document(Document)
	{
	}

	iobject* object()
	{
		return m_data.object();
	}

	void set_object(iobject* Object)
	{
		m_data.set_object(Object);
	}

	idocument& document()
	{
		return m_document;
	}

	changed_signal_t& changed_signal()
	{
		return m_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_data;
	idocument& m_document;
};

/// Specialization of k3d::object_chooser::data_proxy for use with k3d::iproperty objects
template<>
class data_proxy<iproperty> :
	public idata_proxy
{
public:
	typedef iproperty data_t;
	
	data_proxy(data_t& Data, idocument& Document) :
		m_readable_data(Data),
		m_writable_data(dynamic_cast<iwritable_property*>(&Data)),
		m_document(Document)
	{
	}

	iobject* object()
	{
		return boost::any_cast<iobject*>(m_readable_data.value());
	}

	void set_object(iobject* Object)
	{
		return_if_fail(m_writable_data);
		m_writable_data->set_value(Object);
	}

	idocument& document()
	{
		return m_document;
	}

	changed_signal_t& changed_signal()
	{
		return m_readable_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_readable_data;
	iwritable_property* const m_writable_data;
	idocument& m_document;
};

/// Convenience factory function for creating k3d::object_chooser::idata_proxy objects
template<typename data_t>
std::auto_ptr<idata_proxy> proxy(data_t& Data, idocument& Document)
{
	return std::auto_ptr<idata_proxy>(new data_proxy<data_t>(Data, Document));
}

/// Provides an implementation of k3d::object_chooser::iselection_filter that filters based on a specific interface type
template<typename interface_t>
class selection_filter :
	public iselection_filter
{
public:
	selection_filter(const bool AllowNone) :
		m_allow_none(AllowNone)
	{
	}

	bool allow_none()
	{
		return m_allow_none;
	}

	bool allow(k3d::iplugin_factory& Factory)
	{
		return Factory.implements(typeid(interface_t));
	}

	bool allow(iobject& Object)
	{
		return dynamic_cast<interface_t*>(&Object) ? true : false;
	}

private:
	bool m_allow_none;
};

/// Convenience factory function for creating k3d::object_chooser::iselection_filter objects
template<typename interface_t>
std::auto_ptr<iselection_filter> filter(const bool AllowNone)
{
	return std::auto_ptr<iselection_filter>(new selection_filter<interface_t>(AllowNone));
}

/// Convenience factory function for creating k3d::object_chooser::iselection_filter objects for use with k3d::iproperty objects
std::auto_ptr<iselection_filter> filter(iproperty& Data);

} // namespace object_chooser

} // namespace k3d

#endif // K3DUI_OBJECT_CHOOSER_H

