
/*
 * mpg123.h -- written for Juice
 *  Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef MPG123_MPG123_H
#define MPG123_MPG123_H

#include "typedefs.h"							 /* bool */
#include "describe.h"							 /* song_t */

/* Seek modes */
#define SEEK_BEG 1								 /* Seek from beginning */
#define SEEK_REW 2								 /* Seek backwards */
#define SEEK_FWD 3								 /* Seek forwards */

/* g++ is too braindamaged to use a enum nested in a struct so we
 * declare it here as a type */
typedef enum { VERSION, STATUS, STREAM, FRAME, PLAYSTAT, ERROR } type_t;

/* Input structure */
typedef struct
{
   type_t type;
   union
	 {
		/* Startup version message */
		char version[33];						 /* version string */
		/* Status message after loading a song */
		struct
		  {
			 bool has_id3tag;					 /* has ID3 song info? */
			 union
			   {
				  char filename[256];			 /* filename without path and extension */
				  struct
					{
					   char title[31];			 /* title (exactly 30 chars) */
					   char artist[31];			 /* artist (exactly 30 chars) */
					   char album[31];			 /* album (exactly 30 chars) */
					   char year[5];			 /* year (exactly 4 chars) */
					   char comment[31];		 /* comment (exactly 30 chars) */
					   char genre[33];			 /* genre (string) */
					} tag;
			   } info;
		  } status;
		/* Status message after loading a song (stream info) */
		struct
		  {
			 char mpegtype[33];					 /* mpeg type (string) */
			 int layer;							 /* layer (int) */
			 int freq;							 /* sampling frequency (int) */
			 char mode[33];						 /* mode (string) */
			 int mode_ext;						 /* mode extension (int) */
			 int framesize;						 /* framesize (int) */
			 int stereo;						 /* stereo (int) */
			 int copyright;						 /* copyright (int) */
			 int errprot;						 /* error protection (int) */
			 int emphasis;						 /* emphasis (int) */
			 int bitrate;						 /* bitrate (int) */
			 int extension;						 /* extension (int) */
		  } stream;
		/* Status message during playing (frame info) */
		struct
		  {
			 int framecount;					 /* framecount (int) */
			 int framesleft;					 /* frames left this song (int) */
			 float secs;						 /* seconds (float) */
			 float secsleft;					 /* seconds left (float) */
		  } frame;
		/* Playing status */
		int playstat;							 /* 0: playing stopped
												  * 1: playing paused
												  * 2: playing unpaused
												  * 3: end of song reached */
		/* An error occured */
		char errmsg[256];						 /* error message (string) */
	 } data;
} recv_t;

/* Poll structure */
typedef struct
{
   /* Startup version message */
   char version[33];							 /* version string */
   /* Status message after loading a song */
   struct
	 {
		bool has_id3tag;						 /* has ID3 song info? */
		bool has_desc;							 /* has 4dos-like description? */
		union
		  {
			 char filename[256];				 /* filename without path and extension */
			 struct
			   {
				  char title[31];				 /* title (exactly 30 chars) */
				  char artist[31];				 /* artist (exactly 30 chars) */
				  char album[31];				 /* album (exactly 30 chars) */
				  char year[5];					 /* year (exactly 4 chars) */
				  char comment[31];				 /* comment (exactly 30 chars) */
				  char genre[33];				 /* genre (string) */
			   } tag;
		  } info;
		song_t song;							 /* filename description */
	 } status;
   /* Status message after loading a song (stream info) */
   struct
	 {
		char mpegtype[33];						 /* mpeg type (string) */
		int layer;								 /* layer (int) */
		int freq;								 /* sampling frequency (int) */
		char mode[33];							 /* mode (string) */
		int mode_ext;							 /* mode extension (int) */
		int framesize;							 /* framesize (int) */
		int stereo;								 /* stereo (int) */
		int copyright;							 /* copyright (int) */
		int errprot;							 /* error protection (int) */
		int emphasis;							 /* emphasis (int) */
		int bitrate;							 /* bitrate (int) */
		int extension;							 /* extension (int) */
	 } stream;
   /* Playing status */
   int playstat;								 /* 0: playing stopped
												  * 1: playing paused
												  * 2: playing unpaused
												  * 3: end of song reached
												  * 4: busy playing
												  */
   /* Current track */
   char curtrack[256];
   /* Flags to indicate state of structure */
   bool has_version;
   bool has_status;
   bool has_stream;
   bool has_curtrack;
} poll_t;

/* fd to get responses from mpg123 */
extern volatile int mpg123_fdrecv;

/* data received from polling child */
extern volatile poll_t mpg123_poll_status;

/***************************** MPG123 FUNCTIONS *****************************/

/*
 * Fork process, and execute cmd with specified arguments. An unnamed
 * pipe is used to redirect the child's stdin, stdout, and stderr.
 * Returns TRUE if successful, FALSE otherwise.
 */
bool mpg123_start (const char *cmd, ...);

/*
 * Check for input on mpg123_fdrecv. If there is, the input is
 * parsed, the result is stored in input and TRUE is returned.
 * If no input is available or a read error occurs, FALSE is
 * returned.
 */
bool mpg123_receive (recv_t *input);

/*
 * Quit player. It will attempt to do so by first sending it a
 * QUIT sequence. If the player does not respond, a TERM signal
 * will be sent. If the player is still alive, it will be killed.
 * All open pipe filehandles is closed as well.
 */
void mpg123_quit ();

/*
 * Play the given filename.
 */
bool mpg123_load (const char *filename);

/*
 * Stop the player (without exiting).
 */
bool mpg123_stop ();

/*
 * Pause / Unpause the player.
 */
bool mpg123_pause ();

/*
 * Seek nframes. If mode is equal to SEEK_FWD the player
 * will advance the stream by nframes frames, if mode is
 * equal to SEEK_REW, the player will rewind the stream
 * by nframes instead and in the case of SEEK_BEG, the
 * player will start playing the stream at frame nframes.
 */
bool mpg123_seek (unsigned int nframes,int mode);

/***************************** POLLING FUNCTIONS *****************************/

/*
 * Update the input flag of the polling child. If input is TRUE, the child
 * will automatically poll all input from mpg123 and save the relevant
 * information to mpg123_poll_status. If input is FALSE, the child will not
 * poll any data from mpg123.
 */
void mpg123_sendinput (bool input);

/*
 * Update skip flag of the polling child. If skip is TRUE, the child
 * will play the next song from the queue (if available) when a track
 * finish, otherwise it will stop the player and empty the track.
 * queue.
 */
void mpg123_sendskip (bool skip);

/*
 * Update the playing status of the polling child.
 */
void mpg123_sendplaystat ();

/*
 * Add a track to the player queue.
 */
void mpg123_sendtrack (const char *filename);

/*
 * Send poll data to child. This should be done if you've
 * disabled the automatic polling and want to enable it
 * again.
 */
//void mpg123_sendpolldata ();

/*
 * Install poll child. The child automatically polls data received by
 * MPG123 depending on the status of two flags (which can be set with
 * sendinput() and sendskip()). It is also responsible for managing
 * the queue of songs to be played. These songs can be feeded to the
 * child with sendtrack(). You must specify the error handler that
 * can be called when a fatal error occurs. Returns true if successful,
 * FALSE otherwise.
 */
bool mpg123_poll (void (*errhandler) (const char *format, ...));

/*
 * Kill the polling child by sending a TERM signal to it. If the child
 * is still alive, it will be killed. All open pipe filehandles is
 * closed as well.
 */
void mpg123_poll_quit ();

#endif
