
/*
 * vfs.h -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef DIALOGS_VFS_H
#define DIALOGS_VFS_H

#include "typedefs.h"

typedef struct
{
   char *filename;          /* Real name of file, path included */
   char *name;              /* Shortened name */
   int type;                /* Type of file: FT_DIR - directory, FT_FILE - file */
} EntryRec;

typedef struct FifoStackRec
{
   struct FifoStackRec *next;
   char *buffer;
} FifoStackList;

class FifoStack
{
 public:
   /* Initialize FifoStack */
   FifoStack ();
   /* Cleanup FifoStack linked list */
   ~FifoStack ();
   /* Push a new buffer into the queue/stack */
   void Push (const char *buffer);
   /* Pull a buffer from the queue/stack, i.e. get the first buffer in the linked list */
   char *Pull ();
   /* Pop a buffer from the queue/stack, i.e. get the last buffer in the linked list */
   char *Pop ();
   /* Check if queue/stack is empty */
   bool IsEmpty ();
 private:
   /* FIFO linked list */
   FifoStackList *fifostack;
};

#define METHOD_TRUNCATE	1
#define METHOD_SPLIT	2

class VirtualFS
{
 public:
   /* Initialize datastructures */
   VirtualFS ();
   /* Cleanup datastructures */
   ~VirtualFS ();
   /* Create a directory listing of directories in dirname */
   void CreateDirListing (const char *dirname,int maxdirlen);
   /* Create a file listing of the files matching filespec in dirname. For details */
   /* on the type of regular expressions allowed, see regexp.h */
   void CreateFileListing (const char *dirname,const char *filespec,int maxfilelen);
   /* Add a new entry to the list */
   void AddEntry (const EntryRec *entry);
   /* Change an existing entry in the list */
   void ChangeEntry (int entryno,const EntryRec *entry);
   /* Remove an entry from the list */
   void RemoveEntry (int entryno);
   /* Swap entries in list */
   void SwapEntries (int entry1,int entry2);
   /* Shuffle entries in list */
   void ShuffleEntries ();
   /* Sort entries in list */
   void SortEntries (int mode);
   /* Get number of entries in list */
   int NumEntries ();
   /* Remove all entries from list */
   void RemoveEntries ();
   /* Get name stored in given entry */
   char *EntryName (int entryno);
   /* Get type of given entry */
   int EntryType (int entryno);
   /* Get filename of given entry */
   char *EntryFilename (int entryno);
   /* Shortens fullname to a string of maxlen if necessary */
   void shortname (char **newname,const char *fullname,int maxlen);
   /* Set method by which names is shortened */
   int shortenmethod;
 private:
   /* Entries in current directory/file listing */
   EntryRec *entries;
   /* Number of entries in current directory/file listing */
   int nentries;
   /* Put the specified field in field. Fields are seperated by "/" */
   bool cutfield (char *field,const char *pathname,int fieldno);
 protected:
   /* Check if given filename is a directory */
   bool isdir (const char *dirname,const char *filename);
   /* Check if given filename is a directory */
   bool isdir2 (const char *name);
   /* Functions used by SortEntries() */
   int sort_mode;
   const char *getext (const char *str);
   int compare (const char *s1,const char *s2);
   void quicksort (int l,int r);
};

#endif
