
/*
 * juice.C -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef JUICE_C
#define JUICE_C

#include <ncurses.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "dialogs/typedefs.h"
#include "dialogs/dialogs.h"

#include "config/typedefs.h"
#include "config/config.h"

#include "player/child.h"
#include "player/console.h"
#include "player/cmd.h"

#include "mpg123/mpg123.h"
#include "mpg123/describe.h"
#include "mpg123/id3.h"
#include "mpg123/utils.h"

#include "typedefs.h"
#include "playlist.h"
#include "menus.h"
#include "juice.h"

/* global variable */
bool track_paused;

/*
 * File checking routines
 */

/* Check if filename is a regular file and is readable */
bool freadable (const char *filename)
{
   struct stat filestat;
   if (stat (filename,&filestat) != 0) return FALSE;
   if (!S_ISREG (filestat.st_mode)) return FALSE;
   if (filestat.st_uid == getuid ())			 /* Are we the user? */
	 return (S_IRUSR & filestat.st_mode);
   else if (filestat.st_gid == getgid ())		 /* Are we in the group then? */
	 return (S_IRGRP & filestat.st_mode);
   else return (S_IROTH & filestat.st_mode);	 /* We're a lowly other then */
}

/* Check if filename is a regular file and is writable */
bool fwritable (const char *filename)
{
   struct stat filestat;
   if (stat (filename,&filestat) != 0) return FALSE;
   if (!S_ISREG (filestat.st_mode)) return FALSE;
   if (filestat.st_uid == getuid ())				 /* Are we the user? */
	 return (S_IWUSR & filestat.st_mode);
   else if (filestat.st_gid == getgid ())			 /* Are we in the group then? */
	 return (S_IWGRP & filestat.st_mode);
   else return (S_IWOTH & filestat.st_mode);		 /* We're a lowly other then */
}

/* Check if filename is a regular file and is executable */
bool fexecutable (const char *filename)
{
   struct stat filestat;
   if (stat (filename,&filestat) != 0) return FALSE;
   if (!S_ISREG (filestat.st_mode)) return FALSE;
   if (filestat.st_uid == getuid ())				 /* Are we the user? */
	 return (S_IXUSR & filestat.st_mode);
   else if (filestat.st_gid == getgid ())			 /* Are we in the group then? */
	 return (S_IXGRP & filestat.st_mode);
   else return (S_IXOTH & filestat.st_mode);		 /* We're a lowly other then */
}

/* Check if it's a valid directory and that we have read/write/execute permissions */
bool dexist (const char *dirname)
{
   struct stat filestat;
   if (stat (dirname,&filestat) != 0) return FALSE;
   if (!S_ISDIR (filestat.st_mode)) return FALSE;
   if (filestat.st_uid == getuid ())				 /* Are we the user? */
	 return (S_IRWXU & filestat.st_mode);
   else if (filestat.st_gid == getgid ())			 /* Are we in the group then? */
	 return (S_IRWXG & filestat.st_mode);
   else return (S_IRWXO & filestat.st_mode);		 /* We're a lowly other then */
}

/*
 * Hotkey functions
 */

#define F1	0
#define F2	1
#define F3	2
#define F4	3
#define F10	4

#define QUIT		0
#define HELP		1
#define MIXER		2
#define PREVIOUS	3
#define LOAD		4
#define SAVE		5
#define SETUP		6

static const char *fkeystr[] = { "F1", "F2", "F3", "F4", "F10" };
static const char *hotkeystr[] = { "Quit", "Help", "Mixer", "Previous", "Load", "Save", "Setup" };

#ifdef NUM_IS_FUNC
#define ADDHOTKEY(key,dupkey,dlg) \
  {									\
	 Lst.AddHotkey (key,dlg);		\
	 Lst.AddHotkey (dupkey,dlg);	\
  }
#else
#define ADDHOTKEY(key,dupkey,dlg) Lst.AddHotkey (key,dlg)
#endif

#define ADDHOTKEYS1(fnct,boolflag,helpdlg) \
  static void fnct ()											\
  {																\
	 if (boolflag)												\
	   {														\
		  Lst.AddHotkeyStatus (fkeystr[F1],hotkeystr[HELP]);	\
		  ADDHOTKEY (KEY_F(1),'1',&helpdlg);					\
	   }														\
	 Lst.AddHotkeyStatus (fkeystr[F2],hotkeystr[MIXER]);		\
	 ADDHOTKEY (KEY_F(2),'2',&mixer_main);						\
	 Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);		\
	 ADDHOTKEY (KEY_F(10),'0',&quit);							\
  }

#define ADDHOTKEYS2(fnct,boolflag,helpdlg) \
  static void fnct ()											\
  {																\
	 if (boolflag)												\
	   {														\
		  Lst.AddHotkeyStatus (fkeystr[F1],hotkeystr[HELP]);	\
		  ADDHOTKEY (KEY_F(1),'1',&helpdlg);					\
	   }														\
	 Lst.AddHotkeyStatus (fkeystr[F2],hotkeystr[MIXER]);		\
	 ADDHOTKEY (KEY_F(2),'2',&mixer_main);						\
	 Lst.AddHotkeyStatus (fkeystr[F3],hotkeystr[PREVIOUS]);		\
	 ADDHOTKEY (KEY_F(3),'3',&previous);						\
	 Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);		\
	 ADDHOTKEY (KEY_F(10),'0',&quit);							\
  }

static void helphelp_addhotkeys ()
{
   Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);
   ADDHOTKEY (KEY_F(10),'0',&quit);
}

ADDHOTKEYS1 (audiopath_addhotkeys,path_has_help,audiopath_help)
ADDHOTKEYS1 (input_addhotkeys,input_has_help,input_help)
ADDHOTKEYS1 (playlistpath_addhotkeys,path_has_help,playlistpath_help)
ADDHOTKEYS1 (rearrange_addhotkeys,rearrange_has_help,rearrange_help)
ADDHOTKEYS1 (browse_addhotkeys,browse_has_help,browse_help)
ADDHOTKEYS1 (mpg123_addhotkeys,mpg123_has_help,mpg123_help)

static void help_addhotkeys ()
{
   if (help_has_help)
	 {
		Lst.AddHotkeyStatus (fkeystr[F1],hotkeystr[HELP]);
		ADDHOTKEY (KEY_F(1),'1',&help_help);
	 }
   Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);
   ADDHOTKEY (KEY_F(10),'0',&quit);
}

ADDHOTKEYS2 (stopresume_addhotkeys,stopresume_has_help,stopresume_help)
ADDHOTKEYS2 (curtrack_addhotkeys,curtrack_has_help,curtrack_help)
ADDHOTKEYS2 (newtrack_addhotkeys,newtrack_has_help,newtrack_help)
ADDHOTKEYS2 (save_addhotkeys,save_has_help,save_help)
ADDHOTKEYS2 (load_addhotkeys,load_has_help,load_help)
ADDHOTKEYS2 (playlists_addhotkeys,playlists_has_help,playlists_help)
ADDHOTKEYS2 (setup_addhotkeys,setup_has_help,setup_help)

static void edit_addhotkeys ()
{
   if (main_has_help)
	 {
		Lst.AddHotkeyStatus (fkeystr[F1],hotkeystr[HELP]);
		ADDHOTKEY (KEY_F(1),'1',&edit_help);
	 }
   Lst.AddHotkeyStatus (fkeystr[F2],hotkeystr[MIXER]);
   ADDHOTKEY (KEY_F(2),'2',&mixer_main);
   Lst.AddHotkeyStatus (fkeystr[F3],hotkeystr[SETUP]);
   ADDHOTKEY (KEY_F(3),'3',&setup_main);
   Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);
   ADDHOTKEY (KEY_F(10),'0',&quit);
}

static void mixer_addhotkeys ()
{
   if (mixer_has_help)
	 {
		Lst.AddHotkeyStatus (fkeystr[F1],hotkeystr[HELP]);
		ADDHOTKEY (KEY_F(1),'1',&mixer_help);
	 }
   Lst.AddHotkeyStatus (fkeystr[F2],hotkeystr[LOAD]);
   ADDHOTKEY (KEY_F(2),'2',&mixer_load);
   Lst.AddHotkeyStatus (fkeystr[F3],hotkeystr[SAVE]);
   ADDHOTKEY (KEY_F(3),'3',&mixer_save);
   Lst.AddHotkeyStatus (fkeystr[F4],hotkeystr[PREVIOUS]);
   ADDHOTKEY (KEY_F(4),'4',&previous);
   Lst.AddHotkeyStatus (fkeystr[F10],hotkeystr[QUIT]);
   ADDHOTKEY (KEY_F(10),'0',&quit);
}

#undef ADDHOTKEYS1
#undef ADDHOTKEYS2
#undef ADDHOTKEY

#ifdef NUM_IS_FUNC
#define REMOVEHOTKEY(key,dupkey) \
  {									\
	 Lst.RemoveHotkey (key);		\
	 Lst.RemoveHotkey (dupkey);		\
  }
#else
#define REMOVEHOTKEY(key,dupkey) Lst.RemoveHotkey (key)
#endif

#define REMOVEHOTKEYS1(fnct,boolflag) \
  static void fnct ()								\
  {													\
	 if (boolflag)									\
	   {											\
		  Lst.RemoveHotkeyStatus (fkeystr[F1]);		\
		  REMOVEHOTKEY (KEY_F(1),'1');				\
	   }											\
	 Lst.RemoveHotkeyStatus (fkeystr[F2]);			\
	 REMOVEHOTKEY (KEY_F(2),'2');					\
	 Lst.RemoveHotkeyStatus (fkeystr[F10]);			\
	 REMOVEHOTKEY (KEY_F(10),'0');					\
  }

#define REMOVEHOTKEYS2(fnct,boolflag) \
  static void fnct ()								\
  {													\
	 if (boolflag)									\
	   {											\
		  Lst.RemoveHotkeyStatus (fkeystr[F1]);		\
		  REMOVEHOTKEY (KEY_F(1),'1');				\
	   }											\
	 Lst.RemoveHotkeyStatus (fkeystr[F2]);			\
	 REMOVEHOTKEY (KEY_F(2),'2');					\
	 Lst.RemoveHotkeyStatus (fkeystr[F3]);			\
	 REMOVEHOTKEY (KEY_F(3),'3');					\
	 Lst.RemoveHotkeyStatus (fkeystr[F10]);			\
	 REMOVEHOTKEY (KEY_F(10),'0');					\
  }

static void helphelp_removehotkeys ()
{
   Lst.RemoveHotkeyStatus (fkeystr[F10]);
   REMOVEHOTKEY (KEY_F(10),'0');
}

REMOVEHOTKEYS1 (audiopath_removehotkeys,path_has_help)
REMOVEHOTKEYS1 (input_removehotkeys,input_has_help)
REMOVEHOTKEYS1 (playlistpath_removehotkeys,path_has_help)
REMOVEHOTKEYS1 (rearrange_removehotkeys,rearrange_has_help)
REMOVEHOTKEYS1 (browse_removehotkeys,browse_has_help)
REMOVEHOTKEYS1 (mpg123_removehotkeys,mpg123_has_help)

static void help_removehotkeys ()
{
   if (help_has_help)
	 {
		Lst.RemoveHotkeyStatus (fkeystr[F1]);
		REMOVEHOTKEY (KEY_F(1),'1');
	 }
   Lst.RemoveHotkeyStatus (fkeystr[F10]);
   REMOVEHOTKEY (KEY_F(10),'0');
}

REMOVEHOTKEYS2 (save_removehotkeys,save_has_help)
REMOVEHOTKEYS2 (stopresume_removehotkeys,stopresume_has_help)
REMOVEHOTKEYS2 (curtrack_removehotkeys,curtrack_has_help)
REMOVEHOTKEYS2 (newtrack_removehotkeys,newtrack_has_help)
REMOVEHOTKEYS2 (load_removehotkeys,load_has_help)
REMOVEHOTKEYS2 (playlists_removehotkeys,playlists_has_help)
REMOVEHOTKEYS2 (setup_removehotkeys,setup_has_help)
REMOVEHOTKEYS2 (edit_removehotkeys,main_has_help)

static void mixer_removehotkeys ()
{
   if (mixer_has_help)
	 {
		Lst.RemoveHotkeyStatus (fkeystr[F1]);
		REMOVEHOTKEY (KEY_F(1),'1');
	 }
   Lst.RemoveHotkeyStatus (fkeystr[F2]);
   REMOVEHOTKEY (KEY_F(2),'2');
   Lst.RemoveHotkeyStatus (fkeystr[F3]);
   REMOVEHOTKEY (KEY_F(3),'3');
   Lst.RemoveHotkeyStatus (fkeystr[F4]);
   REMOVEHOTKEY (KEY_F(4),'4');
   Lst.RemoveHotkeyStatus (fkeystr[F10]);
   REMOVEHOTKEY (KEY_F(10),'0');
}

#undef REMOVEHOTKEYS1
#undef REMOVEHOTKEYS2
#undef REMOVEHOTKEY

static void addhotkeys ()
{
   switch (Lst.GetMenu ())
	 {
	  case ID_EDIT:			edit_addhotkeys ();			break;
	  case ID_SETUP:		setup_addhotkeys ();		break;
	  case ID_MIXER:		mixer_addhotkeys ();		break;
	  case ID_BROWSE:		browse_addhotkeys ();		break;
	  case ID_REARRANGE:	rearrange_addhotkeys ();	break;
	  case ID_PLAYLISTS:	playlists_addhotkeys ();	break;
	  case ID_SAVE:			save_addhotkeys ();			break;
	  case ID_LOAD:			load_addhotkeys ();			break;
	  case ID_NEWTRACK:		newtrack_addhotkeys ();		break;
	  case ID_STOPRESUME:	stopresume_addhotkeys ();	break;
	  case ID_CURTRACK:		curtrack_addhotkeys ();		break;
	  case ID_HELP:			help_addhotkeys ();			break;
	  case ID_AUDIOPATH:	audiopath_addhotkeys ();	break;
	  case ID_PLAYLISTPATH:	playlistpath_addhotkeys ();	break;
	  case ID_INPUT:		input_addhotkeys ();		break;
	  case ID_HELPHELP:		helphelp_addhotkeys ();		break;
	  case ID_MPG123:		mpg123_addhotkeys ();		break;
	  default:
		/* This shouldn't happen */
		break;
	 }
}

static void removehotkeys ()
{
   switch (Lst.GetMenu ())
	 {
	  case ID_EDIT:				edit_removehotkeys ();			break;
	  case ID_SETUP:			setup_removehotkeys ();			break;
	  case ID_MIXER:			mixer_removehotkeys ();			break;
	  case ID_BROWSE:			browse_removehotkeys ();		break;
	  case ID_REARRANGE:		rearrange_removehotkeys ();		break;
	  case ID_PLAYLISTS:		playlists_removehotkeys ();		break;
	  case ID_LOAD:				load_removehotkeys ();			break;
	  case ID_SAVE:				save_removehotkeys ();			break;
	  case ID_NEWTRACK:			newtrack_removehotkeys ();		break;
	  case ID_CURTRACK:			curtrack_removehotkeys ();		break;
	  case ID_STOPRESUME:  		stopresume_removehotkeys ();	break;
	  case ID_HELP:				help_removehotkeys ();			break;
	  case ID_AUDIOPATH:		audiopath_removehotkeys ();		break;
	  case ID_PLAYLISTPATH:		playlistpath_removehotkeys ();	break;
	  case ID_INPUT:			input_removehotkeys ();			break;
	  case ID_HELPHELP:			helphelp_removehotkeys ();		break;
	  case ID_MPG123:			mpg123_removehotkeys ();		break;
	  default:
		/* This shouldn't happen */
		break;
	 }
}

static void previous ()
{
   Lst.SetAbort (TRUE);
}

static void _strcpy (char *d,const char *s,size_t n)
{
   strncpy (d,s,n);
   d[n - 1] = '\0';
}

static void _strcat (char *d,const char *s,size_t n)
{
   n--;
   while (*d++ != '\0') n--;
   while (*s && n-- > 0) *d++ = *s++;
   *d = '\0';
}

static const char *_filename (const char *s)
{
   static char filename[PATH_MAX];
   char *home = getenv ("HOME");

   if (home == NULL)
	 fatal_error ("HOME environment variable is not set");

   _strcpy (filename,home,PATH_MAX);
   _strcat (filename,"/.juice",PATH_MAX);

   if (mkdir (filename,0755) < 0 && errno != EEXIST)
	 fatal_error ("unable to create user configuration directory: %s",filename);

   _strcat (filename,s,PATH_MAX);

   return (filename);
}

static const char *savedplaylist ()
{
   return (_filename ("/saved.jpl"));
}

#define CONFIG_USER 1
#define CONFIG_SYSTEM 2

static const char *_configfile (int which)
{
   if (which == CONFIG_USER)
	 return (_filename ("/juice.conf"));

   return (configfile);
}

static void quit_handler (int sig)
{
   Lst.Update ();
   if (mpg123frontend)
	 {
		mpg123_quit ();
		mpg123_poll_quit ();
	 }
   else
	 {
		child_flush_stop ();
		child_destroy ();
		console_destroy ();
		destroy_split_args ();
	 }
   if (intersessionmemory) save_playlist (savedplaylist ()); else unlink (savedplaylist ());
   Lst.Close ();
   exit (0);
}

static void quit ()
{
   bool result = TRUE;
   removehotkeys ();
   if (confirmdialogs)
	 result = Lst.Confirm ((COLS - 28) >> 1,LINES / 6,"Quit Program",A_BOLD,WHITE,RED);
   if (result)
	 {
		kill (getpid (),SIGQUIT);
		usleep (DELAY);
	 }
   addhotkeys ();
}

/*
 * This creates the background
 */

static void create_background ()
{
   Lst.CreateBackground ("Juice",PROGRAM_VERSION,"Abraham vd Merwe");
}

/*
 * Help dialogs
 */

#define HELPDLG1(fnct,addf,remf,id,dlg,desc) \
  static void fnct ()		\
  {							\
	 removehotkeys ();		\
	 addf ();				\
	 Lst.AddMenu (id);		\
	 Lst.Help (dlg,desc);	\
	 Lst.RemoveMenu (id);	\
	 remf ();				\
	 addhotkeys ();			\
  }

#define HELPDLG2(fnct,dlg,desc) HELPDLG1 (fnct,help_addhotkeys,help_removehotkeys,ID_HELP,dlg,desc)

HELPDLG1 (help_help,helphelp_addhotkeys,helphelp_removehotkeys,ID_HELPHELP,helphelp,"Help help")

HELPDLG2 (audiopath_help,pathhelp,"Audio searchpath help")
HELPDLG2 (input_help,inputhelp,"Input help")
HELPDLG2 (playlistpath_help,pathhelp,"Playlist searchpath help")
HELPDLG2 (edit_help,mainhelp,"Playlist help")
HELPDLG2 (setup_help,setuphelp,"Setup help")
HELPDLG2 (mixer_help,mixerhelp,"Mixer help")
HELPDLG2 (browse_help,browsehelp,"Browse help")
HELPDLG2 (rearrange_help,rearrangehelp,"Rearrange help")
HELPDLG2 (playlists_help,playlistshelp,"Playlists help")
HELPDLG2 (save_help,savehelp,"Save help")
HELPDLG2 (load_help,loadhelp,"Load help")
HELPDLG2 (newtrack_help,newtrackhelp,"New track help")
HELPDLG2 (curtrack_help,curtrackhelp,"Current track help")
HELPDLG2 (stopresume_help,stopresumehelp,"Stop/Resume help")
HELPDLG2 (mpg123_help,mpg123help,"MPG123 frontend help")

#undef HELPDLG1
#undef HELPDLG2
/*
 * Mixer dialogs
 */

static void load_mixer_settings ()
{
   Config Cfg;
   int chan[CHANNUM];
   Lst.GetSettings (chan[0],chan[1],chan[2],chan[3],chan[4],chan[5],chan[6]);
   if (Cfg.Parse (_configfile (CONFIG_USER)) == CFG_ERROR_NONE || Cfg.Parse (_configfile (CONFIG_SYSTEM)) == CFG_ERROR_NONE)
	 {
		int type,value;
		bool state;
		char txt[CFG_LINE_LENGTH];
		for (int i = 0; i < CHANNUM; i++)
		  {
			 if (Cfg.GetEntry (chanstr[i],type,value,state,txt)) if (type == CFG_TYPE_INTEGER) chan[i] = value;
		  }
	 }
   Lst.SetSettings (chan[0],chan[1],chan[2],chan[3],chan[4],chan[5],chan[6]);
}

static void save_mixer_settings ()
{
   Config Cfg;
   char string[STRNUM][CFG_LINE_LENGTH];
   bool boolean[BOOLNUM];
   int integer[INTNUM],chan[CHANNUM];
   register int i;
   Lst.GetSettings (chan[0],chan[1],chan[2],chan[3],chan[4],chan[5],chan[6]);
   for (i = 0; i < STRNUM; i++) strcpy (string[i],strini[i]);
   for (i = 0; i < BOOLNUM; i++) boolean[i] = boolini[i];
   for (i = 0; i < INTNUM; i++) integer[i] = intini[i];
   if (Cfg.Parse (_configfile (CONFIG_USER)) == CFG_ERROR_NONE || Cfg.Parse (_configfile (CONFIG_SYSTEM)) == CFG_ERROR_NONE)
	 {
		int type,value;
		bool state;
		char txt[CFG_LINE_LENGTH];
		for (i = 0; i < STRNUM; i++)
		  {
			 if (Cfg.GetEntry (strnam[i],type,value,state,txt)) if (type == CFG_TYPE_STRING)
			   strcpy (string[i],txt);
		  }
		for (i = 0; i < BOOLNUM; i++)
		  {
			 if (Cfg.GetEntry (boolnam[i],type,value,state,txt)) if (type == CFG_TYPE_BOOLEAN)
			   boolean[i] = state;
		  }
		for (i = 0; i < INTNUM; i++)
		  {
			 if (Cfg.GetEntry (intnam[i],type,value,state,txt)) if (type == CFG_TYPE_INTEGER)
			   integer[i] = value;
		  }
		Cfg.RemoveAllEntries ();
	 }
   Cfg.RemoveAllEntries ();
   for (i = 0; i < STRNUM; i++) Cfg.AddString (strnam[i],string[i]);
   for (i = 0; i < BOOLNUM; i++) Cfg.AddBoolean (boolnam[i],boolean[i]);
   for (i = 0; i < INTNUM; i++) Cfg.AddInteger (intnam[i],integer[i]);
   for (i = 0; i < CHANNUM; i++) Cfg.AddInteger (chanstr[i],chan[i]);
   if (Cfg.Save (_configfile (CONFIG_USER)) != CFG_ERROR_NONE)
	 {
		Cfg.RemoveAllEntries ();
		for (i = 0; i < STRNUM; i++) Cfg.AddString (strnam[i],strini[i]);
		for (i = 0; i < BOOLNUM; i++) Cfg.AddBoolean (boolnam[i],boolini[i]);
		for (i = 0; i < INTNUM; i++) Cfg.AddInteger (intnam[i],intini[i]);
		for (i = 0; i < CHANNUM; i++) Cfg.AddInteger (chanstr[i],chan[i]);
		Cfg.Create (_configfile (CONFIG_USER),"Juice %s Configuration file",PROGRAM_VERSION);
	 }
}

static void mixer_load ()
{
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (Lst.Question ((COLS - 17) >> 1,LINES >> 1,"Load settings",A_BOLD,WHITE,RED))
		  {
			 Lst.Reset ();
			 load_mixer_settings ();
		  }
	 }
   else
	 {
		Lst.Reset ();
		load_mixer_settings ();
		Lst.Info ((COLS - 19) >> 1,LINES >> 1,"Settings loaded",A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void mixer_save ()
{
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (Lst.Question ((COLS - 17) >> 1,LINES >> 1,"Save settings",A_BOLD,WHITE,RED))
		  save_mixer_settings ();
	 }
   else
	 {
		save_mixer_settings ();
		Lst.Info ((COLS - 18) >> 1,LINES >> 1,"Settings saved",A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void mixer_main ()
{
   removehotkeys ();
   mixer_addhotkeys ();
   Lst.AddMenu (ID_MIXER);
   Lst.ShowMixer ();
   Lst.RemoveMenu (ID_MIXER);
   mixer_removehotkeys ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

/*
 * Setup dialogs
 */

static void showstr (WINDOW *win,int y,const char *format, ...)
{
   char tmp[51];
   va_list ap;
   va_start (ap,format);
   vsnprintf (tmp,50,format,ap);
   va_end (ap);
   mvwaddstr (win,y,3,tmp);
}

static void setup_show ()
{
   static char *boolstr[2] = { "Disabled", "Enabled" };
   WINDOW *curwin;
   removehotkeys ();
   chtype ch;
   Lst.Add (50,16,(COLS - 50)>> 1,(LINES - 16) >> 1,"Current settings",A_BOLD,WHITE,MAGENTA);
   curwin = Lst.GetWindow ();
   showstr (curwin,2,"Audio search path = %s",audiopath);
   showstr (curwin,3,"Playlist search path = %s",playlistpath);
   showstr (curwin,4,"Confirmation dialogs = %s",confirmdialogs ? boolstr[1] : boolstr[0]);
   showstr (curwin,5,"Player name = %s",playername);
   showstr (curwin,6,"Audio type = %s",audiotype);
   showstr (curwin,7,"Playlist type = %s",playlisttype);
   showstr (curwin,8,"Fading time = %d ms",fadingtime);
   showstr (curwin,9,"MPG123 frontend = %s",mpg123frontend ? boolstr[1] : boolstr[0]);
   showstr (curwin,10,"MPG123 command = %s",mpg123name);
   showstr (curwin,11,"Intersession memory = %s",intersessionmemory ? boolstr[1] : boolstr[0]);
   switch (displaymode)
	 {
	  case 0:
		showstr (curwin,12,"Display mode = No information");
		break;
	  case 1:
		showstr (curwin,12,"Display mode = Description file");
		break;
	  case 2:
		showstr (curwin,12,"Display mode = ID3 tags");
		break;
	  case 3:
		showstr (curwin,12,"Display mode = Description file / ID3 tags");
		break;
	  case 4:
		showstr (curwin,12,"Display mode = ID3 tags / Description file");
		break;
	  default:
		showstr (curwin,12,"Display mode = BUG: %s: line %d",__FILE__,__LINE__);
	 }
   showstr (curwin,13,"Help dialogs = %s",helpdialogs ? boolstr[1] : boolstr[0]);
   switch (displayformat)
	 {
	  case 0:
		showstr (curwin,14,"Display format = Title - Artist (Album)");
		break;
	  case 1:
		showstr (curwin,14,"Display format = Title - Artist");
		break;
	  case 2:
		showstr (curwin,14,"Display format = Title (Album)");
		break;
	  case 3:
		showstr (curwin,14,"Display format = Title");
		break;
	  default:
		showstr (curwin,14,"Display format = BUG: %s: line %d",__FILE__,__LINE__);
	 }
   switch (shortenmethod)
	 {
	  case 0:
		showstr (curwin,15,"Shorten method = Split and shorten");
		break;
	  case 1:
		showstr (curwin,15,"Shorten method = Truncate");
		break;
	  default:
		showstr (curwin,15,"Shorten method = BUG: %s: line %d",__FILE__,__LINE__);
	 }
   Lst.Update ();
   ch = getch ();
   Lst.Remove ();
   flushinp ();
   addhotkeys ();
}

#undef SHOWSTR

static void setup_setaudiopath ()
{
   char tmp[PATHLENGTH];
   removehotkeys ();
   audiopath_addhotkeys ();
   strcpy (tmp,audiopath);
   Lst.AddMenu (ID_AUDIOPATH);
   Lst.SelectDirectory (tmp,PATHLENGTH,"Select audio search path");
   Lst.RemoveMenu (ID_AUDIOPATH);
   audiopath = (char *) realloc (audiopath,strlen (tmp) + 1);
   strcpy (audiopath,tmp);
   audiopath_removehotkeys ();
   addhotkeys ();
}

static void setup_setplaylistpath ()
{
   char tmp[PATHLENGTH];
   removehotkeys ();
   playlistpath_addhotkeys ();
   strcpy (tmp,playlistpath);
   Lst.AddMenu (ID_PLAYLISTPATH);
   Lst.SelectDirectory (tmp,PATHLENGTH,"Select playlist search path");
   Lst.RemoveMenu (ID_PLAYLISTPATH);
   playlistpath = (char *) realloc (playlistpath,strlen (tmp) + 1);
   strcpy (playlistpath,tmp);
   playlistpath_removehotkeys ();
   addhotkeys ();
}

static void setup_setplayer ()
{
   removehotkeys ();
   input_addhotkeys ();
   Lst.AddMenu (ID_INPUT);
   Lst.Input (playername,48,(COLS - 50) >> 1,LINES >> 1,"Set player command",A_BOLD,WHITE,YELLOW);
   Lst.RemoveMenu (ID_INPUT);
   input_removehotkeys ();
   addhotkeys ();
}

static void setup_mpg123name ()
{
   removehotkeys ();
   input_addhotkeys ();
   Lst.AddMenu (ID_INPUT);
   Lst.Input (mpg123name,48,(COLS - 50) >> 1,LINES >> 1,"Set mpg123 command",A_BOLD,WHITE,YELLOW);
   Lst.RemoveMenu (ID_INPUT);
   input_removehotkeys ();
   removehotkeys ();
}

static void setup_setaudiotype ()
{
   removehotkeys ();
   input_addhotkeys ();
   Lst.AddMenu (ID_INPUT);
   Lst.Input (audiotype,30,(COLS - 32) >> 1,LINES >> 1,"Audio type",A_BOLD,WHITE,YELLOW);
   Lst.RemoveMenu (ID_INPUT);
   input_removehotkeys ();
   addhotkeys ();
}

static void setup_setplaylisttype ()
{
   removehotkeys ();
   input_addhotkeys ();
   Lst.AddMenu (ID_INPUT);
   Lst.Input (playlisttype,30,(COLS - 32) >> 1,LINES >> 1,"Playlist type",A_BOLD,WHITE,YELLOW);
   Lst.RemoveMenu (ID_INPUT);
   input_removehotkeys ();
   addhotkeys ();
}

static void setup_setfadetime ()
{
   char tmp[30];
   removehotkeys ();
   input_addhotkeys ();
   sprintf (tmp,"%d",fadingtime);
   Lst.AddMenu (ID_INPUT);
   Lst.Input (tmp,30,(COLS - 32) >> 1,LINES >> 1,"Set fading time",A_BOLD,WHITE,YELLOW);
   Lst.RemoveMenu (ID_INPUT);
   fadingtime = atoi (tmp);
   input_removehotkeys ();
   addhotkeys ();
}

static void setup_confirmdialogs ()
{
   char tmp[255];
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (confirmdialogs) strcpy (tmp,"Disable"); else strcpy (tmp,"Enable");
		strcat (tmp," confirm dialogs");
		if (Lst.Question ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED))
		  confirmdialogs = !confirmdialogs;
	 }
   else
	 {
		confirmdialogs = !confirmdialogs;
		strcpy (tmp,"Confirm dialogs ");
		if (confirmdialogs) strcat (tmp,"enabled"); else strcat (tmp,"disabled");
		Lst.Info ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void setup_mpg123frontend ()
{
   char tmp[255];
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (mpg123frontend) strcpy (tmp,"Disable"); else strcpy (tmp,"Enable");
		strcat (tmp," mpg123 player frontend");
		if (Lst.Question ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED))
		  mpg123frontend = !mpg123frontend;
	 }
   else
	 {
		mpg123frontend = !mpg123frontend;
		strcpy (tmp,"mpg123 player frontend ");
		if (mpg123frontend) strcat (tmp,"enabled"); else strcat (tmp,"disabled");
		Lst.Info ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void setup_intersessionmemory ()
{
   char tmp[255];
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (intersessionmemory) strcpy (tmp,"Disable"); else strcpy (tmp,"Enable");
		strcat (tmp," intersession memory");
		if (Lst.Question ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED))
		  intersessionmemory = !intersessionmemory;
	 }
   else
	 {
		intersessionmemory = !intersessionmemory;
		strcpy (tmp,"Intersession memory ");
		if (intersessionmemory) strcat (tmp,"enabled"); else strcat (tmp,"disabled");
		Lst.Info ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void setup_setinfo0 () { displaymode = 0; }
static void setup_setinfo1 () { displaymode = 1; }
static void setup_setinfo2 () { displaymode = 2; }
static void setup_setinfo3 () { displaymode = 3; }
static void setup_setinfo4 () { displaymode = 4; }

static void setup_displaymode ()
{
   removehotkeys ();
   Lst.SetAbort (TRUE);
   Lst.Menu (opt_setupinfo,"Display mode",WHITE,MAGENTA);
   Lst.SetAbort (FALSE);
   addhotkeys ();
}

static void setup_setfmt0 () { displayformat = 0; }
static void setup_setfmt1 () { displayformat = 1; }
static void setup_setfmt2 () { displayformat = 2; }
static void setup_setfmt3 () { displayformat = 3; }

static void setup_displayformat ()
{
   removehotkeys ();
   Lst.SetAbort (TRUE);
   Lst.Menu (opt_setupformat,"Display format",WHITE,MAGENTA);
   Lst.SetAbort (FALSE);
   addhotkeys ();
}

static void setup_setshortenmethod0 () { shortenmethod = 0; }
static void setup_setshortenmethod1 () { shortenmethod = 1; }

static void setup_shortenmethod ()
{
   removehotkeys ();
   Lst.SetAbort (TRUE);
   Lst.Menu (opt_setupshorten,"Shorten method",WHITE,MAGENTA);
   Lst.SetAbort (FALSE);
   addhotkeys ();
}

static void setup_helpdialogs ()
{
   char tmp[255];
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (helpdialogs) strcpy (tmp,"Disable"); else strcpy (tmp,"Enable");
		strcat (tmp," help dialogs");
		if (Lst.Question ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED))
		  helpdialogs = !helpdialogs;
	 }
   else
	 {
		helpdialogs = !helpdialogs;
		strcpy (tmp,"Help dialogs ");
		if (helpdialogs) strcat (tmp,"enabled"); else strcat (tmp,"disabled");
		Lst.Info ((COLS - strlen (tmp) - 4) >> 1,LINES >> 1,tmp,A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void save_setup_settings ()
{
   Config Cfg;
   char *string[STRNUM] = { audiopath, playlistpath, playername, audiotype, playlisttype, mpg123name };
   bool boolean[BOOLNUM] = { confirmdialogs, mpg123frontend, intersessionmemory, helpdialogs };
   int integer[INTNUM],chan[CHANNUM] = { fadingtime, displaymode, displayformat, shortenmethod };
   register int i;
   Lst.GetSettings (chan[0],chan[1],chan[2],chan[3],chan[4],chan[5],chan[6]);
   if (Cfg.Parse (_configfile (CONFIG_USER)) == CFG_ERROR_NONE || Cfg.Parse (_configfile (CONFIG_SYSTEM)) == CFG_ERROR_NONE)
	 {
		int type,value;
		bool state;
		char txt[CFG_LINE_LENGTH];
		for (i = 0; i < CHANNUM; i++)
		  {
			 if (Cfg.GetEntry (chanstr[i],type,value,state,txt)) if (type == CFG_TYPE_INTEGER) chan[i] = value;
		  }
		Cfg.RemoveAllEntries ();
	 }
   Cfg.RemoveAllEntries ();
   for (i = 0; i < STRNUM; i++) Cfg.AddString (strnam[i],string[i]);
   for (i = 0; i < BOOLNUM; i++) Cfg.AddBoolean (boolnam[i],boolean[i]);
   for (i = 0; i < INTNUM; i++) Cfg.AddInteger (intnam[i],integer[i]);
   if (Cfg.Save (_configfile (CONFIG_USER)) != CFG_ERROR_NONE)
	 {
		Cfg.RemoveAllEntries ();
		for (i = 0; i < STRNUM; i++) Cfg.AddString (strnam[i],string[i]);
		for (i = 0; i < BOOLNUM; i++) Cfg.AddBoolean (boolnam[i],boolean[i]);
		for (i = 0; i < INTNUM; i++) Cfg.AddInteger (intnam[i],integer[i]);
		for (i = 0; i < CHANNUM; i++) Cfg.AddInteger (chanstr[i],chan[i]);
		Cfg.Create (_configfile (CONFIG_USER),"Juice %s Configuration file",PROGRAM_VERSION);
	 }
}

static void setup_save ()
{
   Config Cfg;
   removehotkeys ();
   if (confirmdialogs)
	 {
		if (Lst.Confirm ((COLS - 28) >> 1,LINES / 6,"Save configuration",A_BOLD,WHITE,RED))
		  save_setup_settings ();
	 }
   else
	 {
		save_setup_settings ();
		Lst.Info ((COLS - 23) >> 1,LINES >> 1,"Configuration saved",A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
}

static void setup_main ()
{
   removehotkeys ();
   setup_addhotkeys ();
   Lst.AddMenu (ID_SETUP);
   Lst.Menu (opt_setupmenu,"Setup",WHITE,WHITE);
   Lst.RemoveMenu (ID_SETUP);
   setup_removehotkeys ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

/*
 * Edit dialogs
 */

static void edit_browse ()
{
   removehotkeys ();
   browse_addhotkeys ();
   Lst.AddMenu (ID_BROWSE);
   Lst.Browse ();
   Lst.RemoveMenu (ID_BROWSE);
   browse_removehotkeys ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

static void edit_rearrange ()
{
   removehotkeys ();
   rearrange_addhotkeys ();
   Lst.AddMenu (ID_REARRANGE);
   Lst.Rearrange ();
   Lst.RemoveMenu (ID_REARRANGE);
   rearrange_removehotkeys ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

static void edit_shuffle ()
{
   removehotkeys ();
   Lst.Shuffle ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

/*
 * Track functions
 */

#define TRACKSTOP	0
#define TRACKEXEC	1

static const char *trackstr[] =
{
   "Stop playing track",
   "Couldn't execute player"
};

#define KILLTRACKS() \
  {																														\
	 if ((mpg123frontend && (mpg123_poll_status.playstat == 4)) || (((!mpg123frontend) && child_alive ())))				\
	   {																												\
		  if (confirmdialogs) result = Lst.Confirm ((COLS - 28) >> 1,LINES / 6,trackstr[TRACKSTOP],A_BOLD,WHITE,RED);	\
		  if (result)																									\
			{																											\
			   /*Lst.FadeDown ();*/																						\
			   if (mpg123frontend)																						\
				 {																										\
					mpg123_sendskip (FALSE);																			\
					mpg123_stop ();																						\
				 }																										\
			   else																										\
				 {																										\
					child_flush_stop ();				 /* Stop child_pty flush handler */								\
					child_destroy ();					 /* Close virtual console */									\
					while (child_alive ()) ;			 /* Wait for child to die */									\
				 }																										\
			}																											\
	   }																												\
  }

static void track_playcurrent ()
{
   bool result = TRUE;
   removehotkeys ();
   /* Kill current playing track if necessary */
   KILLTRACKS ();
   if (result)
	 {
		if (!mpg123frontend)
		  {
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
		  }
		/* Play new track */
		/*Lst.FadeUp ();*/
		if (mpg123frontend)
		  {
			 mpg123_sendskip (TRUE);
			 mpg123_sendtrack (Lst.tracks.EntryFilename (Lst.curtrack));
		  }
		else
		  {
			 if (!run_command_list (1,Lst.tracks.EntryFilename (Lst.curtrack)))
			   Lst.Info ((COLS - 27) >> 1,LINES >> 1,trackstr[TRACKEXEC],A_BOLD,WHITE,RED);
			 else child_flush_start ();					 /* Activate child_pty flush handler */
			 create_background ();
		  }
	 }
   addhotkeys ();
}

static void track_playupto ()
{
   bool result = TRUE;
   char **tracks;
   removehotkeys ();
   /* Kill current playing track if necessary */
   KILLTRACKS ();
   if (result)
	 {
		if (mpg123frontend)
		  {
			 mpg123_sendskip (TRUE);
			 for (int i = 0; i <= Lst.curtrack; i++) mpg123_sendtrack (Lst.tracks.EntryFilename (i));
		  }
		else
		  {
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
			 /* Create the list of tracks to play */
			 tracks = (char **) malloc ((Lst.curtrack + 2) * sizeof (char *));
			 for (int i = 0; i <= Lst.curtrack; i++) tracks[i] = Lst.tracks.EntryFilename (i);
			 tracks[Lst.curtrack + 1] = NULL;
			 /* Play the selected tracks */
			 if (!run_command_null ((const char **) tracks))
			   Lst.Info ((COLS - 27) >> 1,LINES >> 1,trackstr[TRACKEXEC],A_BOLD,WHITE,RED);
			 else
			   child_flush_start ();					 /* Activate child_pty flush handler */
			 free (tracks);							 /* Free the list of tracks */
			 create_background ();
		  }
		/*Lst.FadeUp ();*/
	 }
   addhotkeys ();
}

static void track_playfrom ()
{
   bool result = TRUE;
   char **tracks;
   removehotkeys ();
   /* Kill current playing track if necessary */
   KILLTRACKS ();
   if (result)
	 {
		if (mpg123frontend)
		  {
			 mpg123_sendskip (TRUE);
			 for (int i = Lst.curtrack; i < Lst.tracks.NumEntries (); i++) mpg123_sendtrack (Lst.tracks.EntryFilename (i));
		  }
		else
		  {
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
			 /* Create the list of tracks to play */
			 tracks = (char **) malloc ((Lst.tracks.NumEntries () - Lst.curtrack + 1) * sizeof (char *));
			 for (int i = Lst.curtrack; i < Lst.tracks.NumEntries (); i++) tracks[i - Lst.curtrack] = Lst.tracks.EntryFilename (i);
			 tracks[Lst.tracks.NumEntries () - Lst.curtrack] = NULL;
			 /* Play the selected tracks */
			 if (!run_command_null ((const char **) tracks))
			   Lst.Info ((COLS - 27) >> 1,LINES >> 1,trackstr[TRACKEXEC],A_BOLD,WHITE,RED);
			 else child_flush_start ();					 /* Activate child_pty flush handler */
			 free (tracks);							 /* Free the list of tracks */
			 create_background ();
		  }
		/*Lst.FadeUp ();*/
	 }
   addhotkeys ();
}

static void track_playall ()
{
   bool result = TRUE;
   char **tracks;
   removehotkeys ();
   /* Kill current playing track if necessary */
   KILLTRACKS ();
   if (result)
	 {
		if (mpg123frontend)
		  {
			 mpg123_sendskip (TRUE);
			 for (int i = 0; i < Lst.tracks.NumEntries (); i++) mpg123_sendtrack (Lst.tracks.EntryFilename (i));
		  }
		else
		  {
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
			 /* Create the list of tracks to play */
			 tracks = (char **) malloc ((Lst.tracks.NumEntries () + 1) * sizeof (char *));
			 for (int i = 0; i < Lst.tracks.NumEntries (); i++) tracks[i] = Lst.tracks.EntryFilename (i);
			 tracks[Lst.tracks.NumEntries ()] = NULL;
			 /* Play the selected tracks */
			 if (!run_command_null ((const char **) tracks))
			   Lst.Info ((COLS - 27) >> 1,LINES >> 1,trackstr[TRACKEXEC],A_BOLD,WHITE,RED);
			 else child_flush_start ();					 /* Activate child_pty flush handler */
			 free (tracks);							 /* Free the list of tracks */
			 create_background ();
		  }
		/*Lst.FadeUp ();*/
	 }
   addhotkeys ();
}

static void track_playselected ()
{
   bool result = TRUE;
   char **tracks;
   int cnt;
   removehotkeys ();
   /* Kill current playing track if necessary */
   KILLTRACKS ();
   if (result)
	 {
		if (mpg123frontend)
		  {
			 mpg123_sendskip (TRUE);
			 for (int i = 0; i < Lst.tracks.NumEntries (); i++) if (Lst.SelectedTrack (i)) mpg123_sendtrack (Lst.tracks.EntryFilename (i));
		  }
		else
		  {
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
			 /* Create the list of tracks to play */
			 tracks = (char **) malloc ((Lst.NumSelectedTracks () + 1) * sizeof (char *));
			 cnt = 0;
			 for (int i = 0; i < Lst.tracks.NumEntries (); i++) if (Lst.SelectedTrack (i)) tracks[cnt++] = Lst.tracks.EntryFilename (i);
			 tracks[Lst.NumSelectedTracks ()] = NULL;
			 /* Play the selected tracks */
			 if (!run_command_null ((const char **) tracks))
			   Lst.Info ((COLS - 27) >> 1,LINES >> 1,trackstr[TRACKEXEC],A_BOLD,WHITE,RED);
			 else child_flush_start ();					 /* Activate child_pty flush handler */
			 free (tracks);							 /* Free the list of tracks */
			 create_background ();
		  }
		/*Lst.FadeUp ();*/
	 }
   addhotkeys ();
}

static void track_pause (int sig)
{
   /* Pause child */
   if (!track_paused)
	 {
		if (mpg123frontend) mpg123_pause (); else child_pause ();
		track_paused = TRUE;
	 }
}
static void track_pause () { track_pause (SIGSTOP); }

static void track_resume (int sig)
{
   /* Resume child */
   if (track_paused)
	 {
		if (mpg123frontend) mpg123_pause (); else child_resume ();
		track_paused = FALSE;
	 }
}
static void track_resume () { track_resume (SIGCONT); }

static void track_stop ()
{
   bool result = TRUE;
   if (confirmdialogs)
	 {
		removehotkeys ();
		result = Lst.Confirm ((COLS - 28) >> 1,LINES / 6,trackstr[TRACKSTOP],A_BOLD,WHITE,RED);
		addhotkeys ();
	 }
   if (result)
	 {
		if (mpg123frontend)
		  {
			 /* unpause first */
			 if (track_paused)
			   {
				  mpg123_pause ();
				  track_paused = FALSE;
			   }
			 mpg123_sendskip (FALSE);
			 mpg123_stop ();
		  }
		else
		  {
			 /* Stop child */
			 child_destroy ();
			 /* Clear the virtual console */
			 console_destroy ();
			 console_create ();
		  }
	 }
}

static void track_skip ()
{
   /* Skip a track */
   if (mpg123frontend)
	 {
		mpg123_sendskip (TRUE);
		mpg123_stop ();
	 }
   else child_skip ();
}

/*
 * Track dialogs
 */

static void edit_playtrack ()
{
   removehotkeys ();
   /* Is this a new track? */
   if ((mpg123frontend && ((mpg123_poll_status.playstat == 0) || (mpg123_poll_status.playstat == 3))) || ((!mpg123frontend) && (!child_alive ())))
	 {
		newtrack_addhotkeys ();
		Lst.AddMenu (ID_NEWTRACK);
		Lst.SetAbort (TRUE);
		Lst.Menu (opt_newtrack,"New track",WHITE,MAGENTA);
		Lst.RemoveMenu (ID_NEWTRACK);
		newtrack_removehotkeys ();
	 }
   /* We're already playing one */
   else
	 {
		if (track_paused)
		  {
			 stopresume_addhotkeys ();
			 Lst.AddMenu (ID_STOPRESUME);
			 Lst.SetAbort (TRUE);
			 Lst.Menu (opt_stopresumetrack,"Stop/Resume player",WHITE,MAGENTA);
			 Lst.RemoveMenu (ID_STOPRESUME);
			 stopresume_removehotkeys ();
		  }
		else
		  {
			 curtrack_addhotkeys ();
			 Lst.AddMenu (ID_CURTRACK);
			 Lst.SetAbort (TRUE);
			 Lst.Menu (opt_curtrack,"Current track",WHITE,MAGENTA);
			 Lst.RemoveMenu (ID_CURTRACK);
			 curtrack_removehotkeys ();
		  }
	 }
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

static void edit_switchmode ()
{
   removehotkeys ();
   if (mpg123frontend)
	 {
		mpg123_addhotkeys ();
		Lst.AddMenu (ID_MPG123);
		Lst.SetAbort (TRUE);
		Lst.MPG123Frontend (fatal_error,XCTRL ('o'));
		Lst.RemoveMenu (ID_MPG123);
		mpg123_removehotkeys ();
	 }   
   else
	 {
		attrset (A_NORMAL);
		erase ();
		refresh ();
		doupdate ();
		reset_shell_mode ();
		noecho ();
		keypad (stdscr,FALSE);
		endwin ();
		raw ();
		console_restore (STDERR_FILENO);
		child_flush_stop ();
		child_invoke (XCTRL ('o'));
		child_flush_start ();
		reset_prog_mode ();
		curs_set (CURSOR_INVISIBLE);
		keypad (stdscr,TRUE);
		touchwin (stdscr);
		create_background ();
	 }
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

/*
 * Playlist dialogs
 */

static void save_playlist (const char *filename)
{
   FILE *handle = fopen (filename,"w");
   fprintf (handle,"%s\n",PLAYLISTHEADER);
   for (register int i = 0; i < Lst.tracks.NumEntries (); i++) fprintf (handle,"%s\n",Lst.tracks.EntryFilename (i));
   fclose (handle);
}

static void load_track (const char *filename)
{
   EntryRec curentry;
   char *tmp,tmp2[256],tmpname[256];
   song_t song;
   id3_t id3;
   bool result1,result2;
   int i = strlen (filename);
   while ((filename[i] != '/') && (i > 0)) i--;
   if (i)
	 {
		i++;
		strcpy (tmpname,filename + i);
	 }
   else strcpy (tmpname,filename);
   Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
   Lst.tracks.shortname (&tmp,tmpname,60);
   curentry.filename = (char *) filename;
   result1 = describe (&song,curentry.filename) && (strlen (song.title) > 0);
   result2 = getid3 (&id3,curentry.filename) && (strlen (id3.title) > 0);
   if ((result1 && ((displaymode == 1) || (displaymode == 3))) || (result1 && (!result2) && (displaymode == 4)))
	 {
		strcpy (tmp2,song.title);
		switch (displayformat)
		  {
		   case 0:
			 if (strlen (song.artist) > 0)
			   {
				  strcat (tmp2," - ");
				  strcat (tmp2,song.artist);
			   }
			 if (strlen (song.album) > 0)
			   {
				  strcat (tmp2," (");
				  strcat (tmp2,song.album);
				  strcat (tmp2,")");
			   }
			 break;
		   case 1:
			 if (strlen (song.artist) > 0)
			   {
				  strcat (tmp2," - ");
				  strcat (tmp2,song.artist);
			   }
			 break;
		   case 2:
			 if (strlen (song.album) > 0)
			   {
				  strcat (tmp2," (");
				  strcat (tmp2,song.album);
				  strcat (tmp2,")");
			   }
			 break;
		   case 3:
			 break;
		  }
		Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
		Lst.tracks.shortname (&curentry.name,tmp2,60);	/* Maybe use both title and artist? */
	 }
   else if ((result2 && ((displaymode == 2) || (displaymode == 4))) || (result2 && (!result1) && (displaymode == 3)))
	 {
		strcpy (tmp2,id3.title);
		switch (displayformat)
		  {
		   case 0:
			 if (strlen (id3.artist) > 0)
			   {
				  strcat (tmp2," - ");
				  strcat (tmp2,id3.artist);
			   }
			 if (strlen (id3.album) > 0)
			   {
				  strcat (tmp2," (");
				  strcat (tmp2,id3.album);
				  strcat (tmp2,")");
			   }
			 break;
		   case 1:
			 if (strlen (id3.artist) > 0)
			   {
				  strcat (tmp2," - ");
				  strcat (tmp2,id3.artist);
			   }
			 break;
		   case 2:
			 if (strlen (id3.album) > 0)
			   {
				  strcat (tmp2," (");
				  strcat (tmp2,id3.album);
				  strcat (tmp2,")");
			   }
			 break;
		   case 3:
			 break;
		  }
		Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
		Lst.tracks.shortname (&curentry.name,tmp2,60);	/* Maybe use both title and artist? */
	 }
   else curentry.name = tmp;
   curentry.type = FT_FILE;
   if (fperm (curentry.filename,"r","r")) Lst.tracks.AddEntry (&curentry);
   free (tmp);
}

static void load_playlist (const char *filename)
{
   char *ch,curline[256];
   FILE *handle;
   if ((handle = fopen (filename,"r")) == NULL) return;
   ch = fgets (curline,256,handle);
   curline[strlen (curline) - 1] = '\0';
   if (strcmp (curline,PLAYLISTHEADER) != 0) return;
   while (!feof (handle))
	 {
		ch = fgets (curline,256,handle);
		curline[strlen (curline) - 1] = '\0';
		if (ch != NULL) load_track (curline);
	 }
   fclose (handle);
}

static void playlists_save ()
{
   removehotkeys ();
   save_addhotkeys ();
   Lst.AddMenu (ID_SAVE);
   if (*currentplaylist == '\0') strcpy (currentplaylist,playlistpath);
   Lst.ResetHotkeyFlag ();
   Lst.SaveFile (currentplaylist,256,playlisttype,"Save Playlist");
   Lst.RemoveMenu (ID_SAVE);
   save_removehotkeys ();
   Lst.SetAbort (FALSE);
   if (Lst.LastHotkey () == HOTKEY_NONE)			 /* User aborted? */
	 {
		save_playlist (currentplaylist);
		Lst.Info ((COLS - 18) >> 1,LINES >> 1,"Playlist saved",A_BOLD,WHITE,RED);
	 }
   addhotkeys ();
   Lst.SetAbort (TRUE);
}

static void playlists_load ()
{
   FILE *handle;
   bool some_files_missing = FALSE,result1,result2;
   song_t song;
   id3_t id3;
   removehotkeys ();
   load_addhotkeys ();
   Lst.AddMenu (ID_LOAD);
   strcpy (currentplaylist,playlistpath);
   Lst.ResetHotkeyFlag ();
   Lst.SelectFile (currentplaylist,256,playlisttype,"Load Playlist");
   Lst.RemoveMenu (ID_LOAD);
   load_removehotkeys ();
   Lst.SetAbort (FALSE);
   if (Lst.LastHotkey () == HOTKEY_NONE)		 /* User aborted? */
	 {
		if ((handle = fopen (currentplaylist,"r")) == NULL)
		  Lst.Info ((COLS - 38) >> 1,LINES >> 1,"Couldn't open playlist for reading",A_BOLD,WHITE,RED);
		else
		  {
			 char *ch,curline[256];
			 ch = fgets (curline,256,handle);
			 curline[strlen (curline) - 1] = '\0';
			 if (strcmp (curline,PLAYLISTHEADER) != 0)
			   Lst.Info ((COLS - 20) >> 1,LINES >> 1,"Playlist corrupt",A_BOLD,WHITE,RED);
			 else
			   {
				  EntryRec curentry;
				  char *tmp,tmp2[256];
				  while (!feof (handle))
					{
					   ch = fgets (curline,256,handle);
					   curline[strlen (curline) - 1] = '\0';
					   if (ch != NULL)
						 {
							char tmpname[256];
							int i;
							i = strlen (curline);
							while ((curline[i] != '/') && (i > 0)) i--;
							if (i)
							  {
								 i++;
								 strcpy (tmpname,curline + i);
							  }
							else strcpy (tmpname,curline);
							Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
							Lst.tracks.shortname (&tmp,tmpname,60);
							curentry.filename = curline;
							result1 = describe (&song,curentry.filename) && (strlen (song.title) > 0);
							result2 = getid3 (&id3,curentry.filename) && (strlen (id3.title) > 0);
							if ((result1 && ((displaymode == 1) || (displaymode == 3))) || (result1 && (!result2) && (displaymode == 4)))
							  {
								 strcpy (tmp2,song.title);
								 switch (displayformat)
								   {
									case 0:
									  if (strlen (song.artist) > 0)
										{
										   strcat (tmp2," - ");
										   strcat (tmp2,song.artist);
										}
									  if (strlen (song.album) > 0)
										{
										   strcat (tmp2," (");
										   strcat (tmp2,song.album);
										   strcat (tmp2,")");
										}
									  break;
									case 1:
									  if (strlen (song.artist) > 0)
										{
										   strcat (tmp2," - ");
										   strcat (tmp2,song.artist);
										}
									  break;
									case 2:
									  if (strlen (song.album) > 0)
										{
										   strcat (tmp2," (");
										   strcat (tmp2,song.album);
										   strcat (tmp2,")");
										}
									  break;
									case 3:
									  break;
								   }
								 Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
								 Lst.tracks.shortname (&curentry.name,tmp2,60);	/* Maybe use both title and artist? */
							  }
							else if ((result2 && ((displaymode == 2) || (displaymode == 4))) || (result2 && (!result1) && (displaymode == 3)))
							  {
								 strcpy (tmp2,id3.title);
								 switch (displayformat)
								   {
									case 0:
									  if (strlen (id3.artist) > 0)
										{
										   strcat (tmp2," - ");
										   strcat (tmp2,id3.artist);
										}
									  if (strlen (id3.album) > 0)
										{
										   strcat (tmp2," (");
										   strcat (tmp2,id3.album);
										   strcat (tmp2,")");
										}
									  break;
									case 1:
									  if (strlen (id3.artist) > 0)
										{
										   strcat (tmp2," - ");
										   strcat (tmp2,id3.artist);
										}
									  break;
									case 2:
									  if (strlen (id3.album) > 0)
										{
										   strcat (tmp2," (");
										   strcat (tmp2,id3.album);
										   strcat (tmp2,")");
										}
									  break;
									case 3:
									  break;
								   }
								 Lst.tracks.shortenmethod = shortenmethod == 0 ? METHOD_SPLIT : METHOD_TRUNCATE;
								 Lst.tracks.shortname (&curentry.name,tmp2,60);	/* Maybe use both title and artist? */
							  }
							else curentry.name = tmp;
							curentry.type = FT_FILE;
							if (!freadable (curentry.filename))
							  some_files_missing = TRUE;
							else
							  Lst.tracks.AddEntry (&curentry);
							free (tmp);
						 }
					}
				  if (some_files_missing)
					Lst.Info ((COLS - 43) >> 1,LINES >> 1,
							  "WARNING: Not all tracks could be loaded",A_BOLD,WHITE,RED);
				  else
					Lst.Info ((COLS - 19) >> 1,LINES >> 1,"Playlist loaded",A_BOLD,WHITE,RED);
			   }
			 fclose (handle);
		  }
	 }
   addhotkeys ();
   Lst.SetAbort (TRUE);
}

static void edit_playlists ()
{
   removehotkeys ();
   playlists_addhotkeys ();
   Lst.AddMenu (ID_PLAYLISTS);
   Lst.Menu (opt_playlists,"Playlists",WHITE,MAGENTA);
   Lst.RemoveMenu (ID_PLAYLISTS);
   playlists_removehotkeys ();
   addhotkeys ();
   Lst.SetAbort (FALSE);
}

/*
 * Intialize variables
 */

#define GETSTRING(id,var) \
  {																	\
	 if (!Cfg.GetEntry (strnam[id],type,value,state,txt))			\
	   {															\
		  var = (char *) malloc (strlen (strini[id]) + 1);			\
		  strcpy (var,strini[id]);									\
	   }															\
	 else															\
	   {															\
		  if (type != CFG_TYPE_STRING)								\
			{														\
			   var = (char *) malloc (strlen (strini[id]) + 1);		\
			   strcpy (var,strini[id]);								\
			}														\
		  else														\
			{														\
			   Cfg.RemoveEntry (strnam[id]);						\
			   var = (char *) malloc (strlen (txt) + 1);			\
			   strcpy (var,txt);									\
			}														\
	   }															\
  }

#define GETBOOLEAN(id,var) \
  {																\
	 if (!Cfg.GetEntry (boolnam[id],type,value,state,txt))		\
	   var = boolini[id];										\
	 else														\
	   {														\
		  if (type != CFG_TYPE_BOOLEAN)							\
			var = boolini[id];									\
		  else													\
			{													\
			   Cfg.RemoveEntry (boolnam[id]);					\
			   var = state;										\
			}													\
	   }														\
  }

#define GETINTEGER(id,var) \
  {																\
	 if (!Cfg.GetEntry (intnam[id],type,value,state,txt))		\
	   var = intini[id];										\
	 else														\
	   {														\
		  if (type != CFG_TYPE_INTEGER)							\
			var = intini[id];									\
		  else													\
			{													\
			   Cfg.RemoveEntry (intnam[id]);					\
			   var = value;										\
			}													\
	   }														\
  }

static void bootupconfig ()
{
   Config Cfg;
   if (Cfg.Parse (_configfile (CONFIG_USER)) != CFG_ERROR_NONE && Cfg.Parse (_configfile (CONFIG_SYSTEM)) != CFG_ERROR_NONE)
	 {
		audiopath = (char *) malloc (100);
		playlistpath = (char *) malloc (100);
		playername = (char *) malloc (100);
		audiotype = (char *) malloc (100);
		playlisttype = (char *) malloc (100);
		mpg123name = (char *) malloc (100);
		strcpy (audiopath,strini[AUDIOPATH]);
		strcpy (playlistpath,strini[PLAYLISTPATH]);
		strcpy (playername,strini[PLAYERNAME]);
		strcpy (audiotype,strini[AUDIOTYPE]);
		strcpy (playlisttype,strini[PLAYLISTTYPE]);
		strcpy (mpg123name,strini[MPG123NAME]);
		confirmdialogs = boolini[CONFIRMDIALOGS];
		mpg123frontend = boolini[MPG123FRONTEND];
		intersessionmemory = boolini[INTERSESSIONMEMORY];
		helpdialogs = boolini[HELPDIALOGS];
		fadingtime = intini[FADINGTIME];
		displaymode = intini[DISPLAYMODE];
		displayformat = intini[DISPLAYFORMAT];
		shortenmethod = intini[SHORTENMETHOD];
	 }
   else
	 {
		int type,value;
		bool state;
		char txt[CFG_LINE_LENGTH];
		/*** STRING ***/
		GETSTRING (AUDIOPATH,audiopath);
		GETSTRING (PLAYLISTPATH,playlistpath);
		GETSTRING (PLAYERNAME,playername);
		GETSTRING (AUDIOTYPE,audiotype);
		GETSTRING (PLAYLISTTYPE,playlisttype);
		GETSTRING (MPG123NAME,mpg123name);
		/*** BOOLEAN ***/
		GETBOOLEAN (CONFIRMDIALOGS,confirmdialogs);
		GETBOOLEAN (MPG123FRONTEND,mpg123frontend);
		GETBOOLEAN (INTERSESSIONMEMORY,intersessionmemory);
		GETBOOLEAN (HELPDIALOGS,helpdialogs);
		/*** INTEGER ***/
		GETINTEGER (FADINGTIME,fadingtime);
		GETINTEGER (DISPLAYMODE,displaymode);
		GETINTEGER (DISPLAYFORMAT,displayformat);
		GETINTEGER (SHORTENMETHOD,shortenmethod);
		displaymode = (displaymode >= 0) && (displaymode <= 4) ? displaymode : intini[DISPLAYMODE];				/* Error checking */
		displayformat = (displayformat >= 0) && (displayformat <= 3) ? displayformat : intini[DISPLAYFORMAT];	/* Error checking */
		shortenmethod = (shortenmethod >= 0) && (shortenmethod <= 1) ? shortenmethod : intini[SHORTENMETHOD];	/* Error checking */
	 }
}

#undef GETSTRING
#undef GETBOOLEAN
#undef GETINTEGER

/*
 * Error checking routines
 */

#ifdef DEBUG
void fatal_error1 (const char *msg, ...)
{
   va_list ap;
   Lst.Update ();
   Lst.Close ();
   fputs ("FATAL ERROR: ",stderr);
   va_start (ap,msg);
   vfprintf (stderr,msg,ap);
   va_end (ap);
   fputs ("\n",stderr);
   abort ();
}
#endif

/* This switch back to text mode, print the error message and exit */
void fatal_error (const char *msg, ...)
{
   va_list ap;
   Lst.Update ();
   Lst.Close ();
//   echo ();                                      /* Enable character echoing */
//   attrset (A_NORMAL);                           /* Set window attributes to normal display */
//   clear ();                                     /* Clear the screen */
//   curs_set (CURSOR_NORMAL);                     /* Set cursor to normal */
//   refresh ();                                   /* Update screen */
//   endwin ();                                    /* Restore tty modes */
   fputs ("FATAL ERROR: ",stderr);
   va_start (ap,msg);
   vfprintf (stderr,msg,ap);
   va_end (ap);
   fputs ("\n",stderr);
   if (mpg123frontend)
	 {
		if (track_paused)
		  {
			 mpg123_pause ();
			 track_paused = FALSE;
		  }
		mpg123_quit ();
		mpg123_poll_quit ();
	 }
   else
	 {
		child_flush_stop ();
		child_destroy ();
		console_destroy ();
		destroy_split_args ();
	 }
   exit (1);
}

/* Do some error checking */
static void do_error_checking ()
{
   static const char *warnings[] =
	 {
		"WARNING: Couldn't access audiopath - setting to default",
		"WARNING: Couldn't access playlistpath - setting to default",
		"WARNING: Some/All help dialogs have been disabled; I couldn't open them",
		"WARNING: Display mode out of range - setting to default",
		"WARNING: Display format out of range - setting to default",
		"WARNING: Shorten method out of range - setting to default"
	 };
   static const char *errors[] =
	 {
		"Screen size is smaller than 80x25",
		"Terminal doesn't have color capabilities",
		"Cannot access root (/) directory"
	 };
   bool helpfiles_missing = FALSE;
   /* Check screen size */
   if ((COLS < 80) || (LINES < 25)) fatal_error (errors[0]);
   /* Check for color */
   if (!has_colors ()) fatal_error (errors[1]);
   /* Check default audio path */
   if (!dexist (audiopath))
	 {
		free (audiopath);
		audiopath = (char *) malloc (2 * sizeof (char));
		strcpy (audiopath,"/");
		Lst.Info ((COLS - strlen (warnings[0]) - 4) >> 1,LINES >> 1,warnings[0],A_BOLD,WHITE,RED);
	 }
   else if (!dexist ("/")) fatal_error (errors[2]);
   /* Check default playlist path */
   if (!dexist (playlistpath))
	 {
		free (playlistpath);
		playlistpath = (char *) malloc (2 * sizeof (char));
		strcpy (playlistpath,"/");
		Lst.Info ((COLS - strlen (warnings[1]) - 4) >> 1,LINES >> 1,warnings[1],A_BOLD,WHITE,RED);
	 }
   else if (!dexist ("/")) fatal_error (errors[2]);
   if ((displaymode < 0) || (displaymode > 4))
	 {
		displaymode = 3;
		Lst.Info ((COLS - strlen (warnings[3]) - 4) >> 1,LINES >> 1,warnings[3],A_BOLD,WHITE,RED);
	 }
   if ((displayformat < 0) || (displayformat > 3))
	 {
		displayformat = 0;
		Lst.Info ((COLS - strlen (warnings[4]) - 4) >> 1,LINES >> 1,warnings[4],A_BOLD,WHITE,RED);
	 }
   if ((shortenmethod < 0) || (shortenmethod > 4))
	 {
		shortenmethod = 0;
		Lst.Info ((COLS - strlen (warnings[5]) - 4) >> 1,LINES >> 1,warnings[5],A_BOLD,WHITE,RED);
	 }
   /* Check help files */
   if (helpdialogs)
	 {
		main_has_help = TRUE;
		setup_has_help = TRUE;
		mixer_has_help = TRUE;
		browse_has_help = TRUE;
		rearrange_has_help = TRUE;
		playlists_has_help = TRUE;
		save_has_help = TRUE;
		load_has_help = TRUE;
		newtrack_has_help = TRUE;
		curtrack_has_help = TRUE;
		stopresume_has_help = TRUE;
		help_has_help = TRUE;
		path_has_help = TRUE;
		input_has_help = TRUE;
		mpg123_has_help = TRUE;
		if (!freadable (mainhelp)) { main_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (setuphelp)) { setup_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (mixerhelp)) { mixer_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (browsehelp)) { browse_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (rearrangehelp)) { rearrange_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (playlistshelp)) { playlists_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (savehelp)) { save_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (loadhelp)) { load_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (newtrackhelp)) { newtrack_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (curtrackhelp)) { curtrack_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (stopresumehelp)) { stopresume_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (helphelp)) { help_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (pathhelp)) { path_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (inputhelp)) { input_has_help = FALSE; helpfiles_missing = TRUE; }
		if (!freadable (mpg123help)) { mpg123_has_help = FALSE; helpfiles_missing = TRUE; }
		if (helpfiles_missing) Lst.Info ((COLS - strlen (warnings[2]) - 4) >> 1,LINES >> 1,warnings[2],A_BOLD,WHITE,RED);
	 }
   else
	 {
		main_has_help = FALSE;
		setup_has_help = FALSE;
		mixer_has_help = FALSE;
		browse_has_help = FALSE;
		rearrange_has_help = FALSE;
		playlists_has_help = FALSE;
		save_has_help = FALSE;
		load_has_help = FALSE;
		newtrack_has_help = FALSE;
		curtrack_has_help = FALSE;
		stopresume_has_help = FALSE;
		help_has_help = FALSE;
		path_has_help = FALSE;
		input_has_help = FALSE;
		mpg123_has_help = FALSE;
	 }
}

static void showhelp ()
{
   fprintf (stderr,
			"\nJuice - Dialog based frontend for mpg123 and other players.\n"
			"Version %s. Written and copyrights by Abraham van der Merwe.\n"
			"THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY! USE AT YOUR OWN RISK!\n\n"
			"USAGE: juice [[file1 [file2 [...]]]]\n"
			"       juice -h | --help\n\n",PROGRAM_VERSION);
   exit (1);
}

/*
 * Main program
 */

int main (int argc,char *argv[])
{
   if ((argc == 2) && ((strcmp (argv[1],"-h") == 0) || (strcmp (argv[1],"--help") == 0))) showhelp ();
   for (int i = 1; i < argc; i++) if (!fperm (argv[1],"r","r")) showhelp ();
   Lst.Init ();
   /* Check configuration file */
   if (!freadable (_configfile (CONFIG_SYSTEM)))
	 fatal_error ("System-wide configuration file not readable by user: %s",_configfile (CONFIG_SYSTEM));
   /* Catch all quit-related signals */
   signal (SIGINT,quit_handler);
   signal (SIGQUIT,quit_handler);
   signal (SIGTERM,quit_handler);
   /* ...and all stop signals */
   signal (SIGSTOP,track_pause);
   signal (SIGCONT,track_resume);
   /* Initialize */
   track_paused = FALSE;
   *currentplaylist = '\0';
   bootupconfig ();
   if (argc > 1) for (int i = 1; i < argc; i++) load_track (argv[i]);
   else if (intersessionmemory) load_playlist (savedplaylist ());
   create_background ();
   if (!mpg123frontend)
	 console_create ();
   if (mpg123frontend)
	 {
		if (!mpg123_start (mpg123name)) fatal_error ("Couldn't execute mpg123 (%s)",mpg123name);
		if (!mpg123_poll (fatal_error)) fatal_error ("Failed to install poll-handler");
	 }
   else split_cmd_args (playername);
   /* Do error checking */
   do_error_checking ();
   /* Main menu */
   edit_addhotkeys ();
   Lst.AddMenu (ID_EDIT);
   Lst.SetWrapper (WRAPPER_BROWSE,&edit_browse);
   Lst.SetWrapper (WRAPPER_REARRANGE,&edit_rearrange);
   Lst.SetWrapper (WRAPPER_SHUFFLE,&edit_shuffle);
   Lst.SetWrapper (WRAPPER_PLAYLISTS,&edit_playlists);
   Lst.SetFunction (FUNCTION_SWITCHMODE,&edit_switchmode);
   Lst.SetFunction (FUNCTION_PLAYTRACK,&edit_playtrack);
   Lst.SetFunction (FUNCTION_PLAYSELECTED,&track_playselected);
   Lst.Edit ();
   /* Program won't ever reach this, but in case */
   kill (getpid (),SIGQUIT);
   usleep (DELAY);
   /* Now we're in trouble */
   Lst.Close ();
   return -1;
}

#endif
