# Copyright (C) 2007 Samuel Abels <http://debain.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import goocanvas
import gobject
import calendar
import pango
import util

class EventItem(goocanvas.Group):
    """
    A canvas item representing an event.
    """
    def __init__(self, cal, **kwargs):
        """
        Constructor.
        """
        goocanvas.Group.__init__(self)

        self.cal        = cal
        self.x          = kwargs.get('x')
        self.y          = kwargs.get('y')
        self.width      = kwargs.get('width')
        self.height     = kwargs.get('height')
        self.bg_color   = kwargs.get('bg_color')
        self.text_color = kwargs.get('text_color')
        self.font_size  = kwargs.get('font_size')
        self.event      = kwargs.get('event')
        self.type       = kwargs.get('type', 'leftright')

        # Create canvas items.
        self.box  = goocanvas.Rect(parent = self)
        self.box2 = goocanvas.Rect(parent = self)
        self.text = goocanvas.Text(parent = self)

        if self.x is not None:
            self.update()


    def update(self):
        if self.event.all_day or self.cal.zoom == self.cal.ZOOM_MONTH:
            self.update_all_day_event()
        else:
            self.update_event()


    def update_event(self):
        text_padding       = 4
        style              = self.cal.get_style()
        font_descr         = style.font_desc.copy()
        if self.font_size is not None:
            font_descr.set_absolute_size(self.font_size * pango.SCALE)
        self.font          = font_descr.to_string()
        caption            = self.event.record["description"]
        the_event_bg_color = self.event.bg_color

        # Choose text color.
        if self.event.text_color is None:
            the_event_text_color = self.text_color
        else:
            the_event_text_color = self.event.text_color

        # Draw the event background.
        radius  = 4
        if self.type == 'mid':
            self.box2.set_property('visibility', goocanvas.ITEM_INVISIBLE)
            radius = 0
        elif self.type == 'topbottom':
            self.box2.set_property('visibility', goocanvas.ITEM_INVISIBLE)
        elif self.type == 'top':
            self.box2.set_property('visibility', goocanvas.ITEM_VISIBLE)
            self.box2.set_property('y', self.y)
        elif self.type == 'bottom':
            self.box2.set_property('visibility', goocanvas.ITEM_VISIBLE)
            self.box2.set_property('y', self.y + self.height - radius)

        if the_event_bg_color is not None:
            self.box.set_property('x',            self.x)
            self.box.set_property('y',            self.y)
            self.box.set_property('width',        self.width)
            self.box.set_property('height',       self.height)
            self.box.set_property('radius_x',     radius)
            self.box.set_property('radius_y',     radius)
            self.box.set_property('stroke_color', the_event_bg_color)
            self.box.set_property('fill_color',   the_event_bg_color)

            # Box 2 hides the rounded corners of box1.
            self.box2.set_property('x',            self.x)
            self.box2.set_property('width',        self.width)
            self.box2.set_property('height',       radius)
            self.box2.set_property('stroke_color', the_event_bg_color)
            self.box2.set_property('fill_color',   the_event_bg_color)

        # Print the event name into the title box.
        self.text.set_property('x',          self.x + 6)
        self.text.set_property('y',          self.y + 2)
        self.text.set_property('font',       self.font)
        self.text.set_property('text',       caption)
        self.text.set_property('fill_color', the_event_text_color)

        # Clip the text.
        x2, y2 = self.x + self.width, self.y + self.height,
        path = 'M%s,%s L%s,%s L%s,%s L%s,%s Z' % (self.x, self.y,
                                                  self.x, y2,
                                                  x2,     y2,
                                                  x2,     self.y)
        self.text.set_property('clip_path',  path)


    def update_all_day_event(self):
        text_padding       = max(self.height / 4.5, 4)
        text_height        = max(self.height - 2 * text_padding, 10)
        style              = self.cal.get_style()
        font_descr         = style.font_desc.copy()
        font_descr.set_absolute_size(text_height * pango.SCALE)
        self.font          = font_descr.to_string()
        caption            = self.event.record["description"]
        the_event_bg_color = self.event.bg_color

        # Choose text color.
        if self.event.text_color is None:
            the_event_text_color = self.text_color
        else:
            the_event_text_color = self.event.text_color

        # Draw the event background.
        radius = self.height / 4
        if self.type == 'mid':
            self.box2.set_property('visibility', goocanvas.ITEM_INVISIBLE)
            radius = 0
        elif self.type == 'leftright':
            self.box2.set_property('visibility', goocanvas.ITEM_INVISIBLE)
        elif self.type == 'right':
            self.box2.set_property('visibility', goocanvas.ITEM_VISIBLE)
            self.box2.set_property('x', self.x)
        elif self.type == 'left':
            self.box2.set_property('visibility', goocanvas.ITEM_VISIBLE)
            self.box2.set_property('x', self.x + self.width - radius)

        if the_event_bg_color is not None:
            self.box.set_property('x',            self.x)
            self.box.set_property('y',            self.y)
            self.box.set_property('width',        self.width)
            self.box.set_property('height',       self.height - 4)
            self.box.set_property('radius_x',     radius)
            self.box.set_property('radius_y',     radius)
            self.box.set_property('stroke_color', the_event_bg_color)
            self.box.set_property('fill_color',   the_event_bg_color)

            # Box 2 hides the rounded corners of box1.
            self.box2.set_property('y',            self.y)
            self.box2.set_property('width',        radius)
            self.box2.set_property('height',       self.height - 4)
            self.box2.set_property('stroke_color', the_event_bg_color)
            self.box2.set_property('fill_color',   the_event_bg_color)

        # Print the event name into the title box.
        self.text.set_property('x',          self.x + 6)
        self.text.set_property('y',          self.y + 2)
        self.text.set_property('font',       self.font)
        self.text.set_property('text',       caption)
        self.text.set_property('fill_color', the_event_text_color)

        # Clip the text.
        x2, y2 = self.x + self.width, self.y + self.height,
        path = 'M%s,%s L%s,%s L%s,%s L%s,%s Z' % (self.x, self.y,
                                                  self.x, y2,
                                                  x2,     y2,
                                                  x2,     self.y)
        self.text.set_property('clip_path',  path)
