/* backup.c
 *
 * Copyright (C) 1999,2000,2001 by Jason Day
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <utime.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <gdbm.h>
#include <dirent.h>

#include <pi-dlp.h>
#include <pi-file.h>

#include <prefs.h>

#include "backup.h"
#include "libplugin.h"


static char *RCSID = "$Id: backup.c,v 1.10 2002/10/03 18:39:45 jday Exp $";


/* Global variables */
static gboolean needs_save;
static gboolean needs_load;
static GtkWidget *active_clist;
static GtkWidget *inactive_clist;
static GtkWidget *backup_new_button;
static GtkWidget *every_sync_button;
static GtkWidget *daily_button;
static GtkWidget *weekly_button;
static GtkWidget *monthly_button;
static GtkWidget *archive_spinner;


/* Local static functions */
static int get_backup_file_name (char *file, char *full_name, int max_size);
static int get_archive_file_name (char *arch, char *file, char *full_name, int max_size);
static int display_databases (void);
static int check_backup_dir (void);
static void update_prefs_gui (void);
static int archive_dir_select (const struct dirent *entry);
static int expire_archives (void);
static int expire_archive (char *dir);
static int skip_backup (void);
static void move_selected_item (GtkCList *src, GtkCList *dest);
static void move_all_items (GtkCList *src, GtkCList *dest);
static int dbm_move_items (char *src_name, char *dest_name, GList *node);
static void filename_make_legal(char *s);
static void cb_move_all_to_inactive (GtkWidget *widget, int data);
static void cb_move_to_inactive (GtkWidget *widget, int data);
static void cb_move_all_to_active (GtkWidget *widget, int data);
static void cb_move_to_active (GtkWidget *widget, int data);
static void cb_toggle_button (GtkWidget *widget, gpointer data);
static void cb_archives (GtkWidget *widget, gpointer data);

/* from jpilot/utils.h */
int get_home_file_name(char *file, char *full_name, int max_size);


/* This plugin was designed to work with version 0.99 or greater of jpilot */
void plugin_version (int *major_version, int *minor_version) {
    *major_version = 0;
    *minor_version = 99;
}

int plugin_get_name (char *name, int len) {
    strncpy (name, "Backup " VERSION, len);
    return 0;
}

int plugin_get_menu_name (char *name, int len) {
    strncpy (name, "Backup", len);
    return 0;
}

int plugin_get_help_name (char *name, int len) {
    strncpy(name, "About Backup", len);
    return 0;
}

int plugin_help (char **text, int *width, int *height) {
   *text = strdup(
	   "Backup plugin for J-Pilot\n"
           "version " VERSION "\n"
	   "by Jason Day (c) 1999-2002.\n"
	   "jasonday@worldnet.att.net\n"
	   "http://jasonday.home.att.net/\n"
	   );

   /* Specifying 0 for width and height lets GTK decide */
   *height = 0;
   *width = 0;

   return 0;
}

int plugin_startup (jp_startup_info *info) {
    jp_init();
    jp_logf (JP_LOG_DEBUG, "Backup: plugin_startup\n");

    /*Check to see if ~/.jpilot/Backup is there, or create it */
    jp_logf (JP_LOG_DEBUG, "calling check_backup_dir\n");
    if (check_backup_dir()) {
        return 1;
    }

    jp_logf (JP_LOG_DEBUG, "Backup: Loading prefs\n");
    jp_pref_init (backup_prefs, NUM_BPREFS);
    if (jp_pref_read_rc_file (PREFS_FILE, backup_prefs, NUM_BPREFS) < 0) {
        jp_logf (JP_LOG_WARN, "Backup: Unable to load preferences file " PREFS_FILE "\n");
    }
    else {
        jp_logf (JP_LOG_DEBUG, "Backup: loaded preferences from " PREFS_FILE "\n");
    }

    needs_save = needs_load = FALSE;

    return 0;
}

int plugin_sync (int sd) {
    struct pi_file *pi_fp;
    char full_name[256];
    char db_copy_name[MAX_DBNAME + 5];
    int start;
    struct DBInfo info;
    time_t ltime;
    struct tm *now;
    char arch[28];
    char all_arch[256];
    char current_arch[256];
    char last_arch[256];
    char temp_str[256];
    GDBM_FILE active_dbf;
    GDBM_FILE inactive_dbf;
    datum key;
    datum content;
    time_t mtime;
    int ret;
    FILE *manifest;
    long backup_new;


    /* see if it's time to make a backup */
    if (skip_backup()) {
        jp_logf (JP_LOG_GUI, "Skipping backup\n");
        return 0;
    }

    /* create the archive directory */
    time (&ltime);
    now = localtime (&ltime);
    sprintf (arch, "Archive_%4d-%02d-%02d@%02d:%02d:%02d",
             now->tm_year + 1900,
             now->tm_mon + 1,
             now->tm_mday,
             now->tm_hour,
             now->tm_min,
             now->tm_sec);
    get_backup_file_name (arch, current_arch, 255);
    if (mkdir (current_arch, 0755)) {
        /* Can't create directory */
        jp_logf (JP_LOG_FATAL, "Can't create directory %s\n", current_arch);
        return 1;
    }
    get_backup_file_name ("LatestArchive", last_arch, 255);
    get_backup_file_name ("AllArchive", all_arch, 255);

    /* open the active dbm file */
    get_backup_file_name (ACTIVE_DBM, full_name, 255);
    active_dbf = gdbm_open (full_name, 512, GDBM_WRCREAT, 0644, 0);
    if (!active_dbf) {
        /* Can't open or create dbm file */
        jp_logf (JP_LOG_FATAL, "Can't open dbm file %s\n", full_name);
        return 1;
    }
    /* open the inactive dbm file */
    get_backup_file_name (INACTIVE_DBM, full_name, 255);
    inactive_dbf = gdbm_open (full_name, 512, GDBM_WRCREAT, 0644, 0);
    if (!inactive_dbf) {
        /* Can't open or create dbm file */
        jp_logf (JP_LOG_FATAL, "Can't open dbm file %s\n", full_name);
        return 1;
    }

    /* open the manifest file */
    get_archive_file_name (current_arch, MANIFEST, full_name, 255);
    manifest = fopen (full_name, "w");
    if (!manifest) {
        jp_logf (JP_LOG_WARN,
                 "Cannot create manifest file %s.\n"
                 "Archive directory %s cannot be automatically expired.\n",
                 full_name, current_arch);
    }

    jp_get_pref (backup_prefs, BPREF_BACKUP_NEW, &backup_new, NULL);
    start = 0;
    while (dlp_ReadDBList (sd, 0, dlpOpenRead, start, &info) > 0) {
        start = info.index + 1;

        key.dptr = info.name;
        key.dsize = strlen (info.name) + 1;

        /* see if it's in the inactive list */
        if (gdbm_exists (inactive_dbf, key)) {
            continue;
        }

        /*
         * fetch the modification time from the active list
         */
        content = gdbm_fetch (active_dbf, key);
        if (content.dptr) {
            mtime = (time_t)atoi (content.dptr);
        }
        else {
            /*
             * not contained in either of the databases; store it in
             * the correct one based on user pref
             */
            mtime = 0;

            sprintf (temp_str, "%ld", info.modifyDate);
            content.dptr = temp_str;
            content.dsize = strlen (temp_str) + 1;
            if (backup_new) {
                ret = gdbm_store (active_dbf, key, content, GDBM_INSERT);
                jp_logf (JP_LOG_DEBUG, "Storing %s in active database file\n", key.dptr);
            }
            else {
                ret = gdbm_store (inactive_dbf, key, content, GDBM_INSERT);
                jp_logf (JP_LOG_DEBUG, "Storing %s in inactive database file\n", key.dptr);
                continue;
            }
        }

        strncpy(db_copy_name, info.name, MAX_DBNAME);
	filename_make_legal (db_copy_name);
        db_copy_name[MAX_DBNAME] = '\0';
        if (info.flags & dlpDBFlagResource) {
            strcat (db_copy_name, ".prc");
        }
        else if (strncmp(db_copy_name + strlen(db_copy_name) - 4, ".pqa", 4)) {
            strcat (db_copy_name, ".pdb");
        }

        get_archive_file_name (current_arch, db_copy_name, full_name, 255);

        /* If modification times are the same then we don't need to fetch it */
        if (info.modifyDate == mtime) {
            jp_logf (JP_LOG_GUI, "Backup: %s is up to date, fetch skipped.\n", db_copy_name);
            get_archive_file_name (last_arch, db_copy_name, temp_str, 255);
            if (link (temp_str, full_name)) {
                jp_logf (JP_LOG_WARN, "Backup: Unable to link file %s, will fetch.\n", temp_str);
            }
            else {
                /* update the file manifest */
                if (manifest) {
                    fprintf (manifest, "%s\n", db_copy_name);
                }

                continue;
            }
        }

        jp_logf (JP_LOG_GUI, "Backup: Fetching '%s'... ", info.name);

        /* update the active dbm file */
        sprintf (temp_str, "%ld", info.modifyDate);
        content.dptr = temp_str;
        content.dsize = strlen (temp_str) + 1;
        ret = gdbm_store (active_dbf, key, content, GDBM_REPLACE);

        info.flags &= 0xff;

        pi_fp = pi_file_create (full_name, &info);
        if (pi_fp==0) {
            jp_logf (JP_LOG_WARN, "Failed, unable to create file %s\n", full_name);
            continue;
        }
        if (pi_file_retrieve (pi_fp, sd, 0) < 0) {
            jp_logf (JP_LOG_WARN, "Failed, unable to back up database\n");
        }
        else {
            jp_logf (JP_LOG_GUI, "OK\n");

            /* update the file manifest */
            if (manifest) {
                fprintf (manifest, "%s\n", db_copy_name);
            }
        }
        pi_file_close (pi_fp);

	/*
        get_archive_file_name (all_arch, db_copy_name, temp_str, 255);
        if (link (full_name, temp_str)) {
            jp_logf (JP_LOG_WARN, "Backup: Unable to create link for %s in %s\n", full_name, all_arch);
        }
	*/
    }

    /* update the last backup time */
    sprintf (temp_str, "%ld", ltime);
    jp_logf (JP_LOG_DEBUG, "setting last backup time: %s\n", temp_str);
    jp_set_pref (backup_prefs, BPREF_LAST_BACKUP, 0, temp_str);
    jp_pref_write_rc_file (PREFS_FILE, backup_prefs, NUM_BPREFS);
    needs_save = FALSE;
    needs_load = TRUE;

    /* close the files */
    gdbm_close (active_dbf);
    gdbm_close (inactive_dbf);
    if (manifest) {
        fclose (manifest);
    }

    /* update the latest archive link */
    unlink (last_arch);
    symlink (arch, last_arch);

    expire_archives();
    jp_logf (JP_LOG_GUI, "Backup: sync complete\n");

    return 0;
}

int plugin_post_sync() {
    jp_logf (JP_LOG_DEBUG, "Backup: plugin_post_sync\n");
    display_databases();
    return 0;
}


/*
 * This function is called by J-Pilot when the user selects this plugin
 * from the plugin menu, or from the search window when a search result
 * record is chosen.  In the latter case, unique ID will be set.  This
 * application should go directly to that record in the case.
 */
int plugin_gui (GtkWidget *vbox, GtkWidget *hbox, unsigned int unique_id) {
    GtkWidget *vbox1, *vbox2, *vbox0;
    GtkWidget *temp_hbox;
    //GtkWidget *temp_vbox;
    GtkWidget *button;
    GtkWidget *label;
    //GtkWidget *vscrollbar;
    GtkWidget *left_scrolled_window;
    GtkWidget *right_scrolled_window;
    GtkObject *adj;
    //time_t ltime;
    //struct tm *now;

    jp_logf(JP_LOG_DEBUG, "Backup: plugin gui started, unique_id=%d\n", unique_id);

    /* left and right main boxes */
    vbox1 = gtk_vbox_new(FALSE, 0);
    vbox2 = gtk_vbox_new(FALSE, 0);
    vbox0 = gtk_vbox_new(FALSE, 0);

    /* ----------------------------------- */
    /* Top portion of screen (preferences) */
    /* ----------------------------------- */

    /* Make a temporary hbox */
    temp_hbox = gtk_hbox_new (FALSE, 0);

    /* 'Backup new databases' button */
    backup_new_button = gtk_check_button_new_with_label ("Backup new databases");
    gtk_signal_connect (GTK_OBJECT (backup_new_button), "toggled",
                        GTK_SIGNAL_FUNC (cb_toggle_button),
                        GINT_TO_POINTER (BACKUP_NEW_BUTTON));
    gtk_box_pack_start (GTK_BOX(temp_hbox), backup_new_button, FALSE, FALSE, 5);

    /* Archive spinner */
    label = gtk_label_new ("Archives: ");
    gtk_box_pack_start (GTK_BOX(temp_hbox), label, FALSE, FALSE, 5);

    adj = gtk_adjustment_new (1.0, 1.0, MAX_ARCHIVES, 1.0, 5.0, 0.0);
    archive_spinner = gtk_spin_button_new (GTK_ADJUSTMENT(adj), 0, 0);
    gtk_signal_connect (adj, "value-changed",
                        GTK_SIGNAL_FUNC (cb_archives),
                        NULL);
    gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (archive_spinner), TRUE);
    gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (archive_spinner), GTK_SHADOW_OUT);
    gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (archive_spinner), TRUE);
    gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON(archive_spinner), GTK_UPDATE_IF_VALID);
    gtk_box_pack_start (GTK_BOX(temp_hbox), archive_spinner, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX(vbox0), temp_hbox, FALSE, FALSE, 5);

    gtk_box_pack_start (GTK_BOX(vbox0), gtk_hseparator_new(), FALSE, FALSE, 0);

    /* Backup how often radio buttons */
    temp_hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("Run Backup:");
    gtk_box_pack_start (GTK_BOX(temp_hbox), label, FALSE, FALSE, 5);

    every_sync_button = gtk_radio_button_new_with_label (NULL, "Every Sync");
    gtk_signal_connect (GTK_OBJECT (every_sync_button), "toggled",
                        GTK_SIGNAL_FUNC (cb_toggle_button),
                        GINT_TO_POINTER (EVERY_SYNC_BUTTON));
    gtk_box_pack_start (GTK_BOX (temp_hbox), every_sync_button, TRUE, FALSE, 0);
    daily_button = gtk_radio_button_new_with_label(
        gtk_radio_button_group (GTK_RADIO_BUTTON (every_sync_button)),
        "Daily");
    gtk_signal_connect (GTK_OBJECT (daily_button), "toggled",
                        GTK_SIGNAL_FUNC (cb_toggle_button),
                        GINT_TO_POINTER (DAILY_BUTTON));
    gtk_box_pack_start (GTK_BOX (temp_hbox), daily_button, TRUE, FALSE, 0);
    weekly_button = gtk_radio_button_new_with_label(
        gtk_radio_button_group (GTK_RADIO_BUTTON (daily_button)),
        "Weekly");
    gtk_signal_connect (GTK_OBJECT (weekly_button), "toggled",
                        GTK_SIGNAL_FUNC (cb_toggle_button),
                        GINT_TO_POINTER (WEEKLY_BUTTON));
    gtk_box_pack_start (GTK_BOX (temp_hbox), weekly_button, TRUE, FALSE, 0);
    monthly_button = gtk_radio_button_new_with_label(
        gtk_radio_button_group (GTK_RADIO_BUTTON (weekly_button)),
        "Monthly");
    gtk_signal_connect (GTK_OBJECT (monthly_button), "toggled",
                        GTK_SIGNAL_FUNC (cb_toggle_button),
                        GINT_TO_POINTER (MONTHLY_BUTTON));
    gtk_box_pack_start (GTK_BOX(temp_hbox), monthly_button, TRUE, FALSE, 0);

    gtk_box_pack_start (GTK_BOX(vbox0), temp_hbox, FALSE, FALSE, 5);
    gtk_box_pack_start (GTK_BOX(vbox0), gtk_hseparator_new(), FALSE, FALSE, 0);

    temp_hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(temp_hbox), vbox1, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX(temp_hbox), vbox2, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX(vbox0), temp_hbox, TRUE, TRUE, 5);
    gtk_box_pack_start (GTK_BOX(hbox), vbox0, TRUE, TRUE, 5);

    /* ------------------------------------------------------------- */
    /* Middle portion of screen (active and inactive database lists) */
    /* ------------------------------------------------------------- */

    /* ------------------- */
    /* Left half of screen */
    /* ------------------- */

    /* Make a temporary hbox */
    temp_hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox1), temp_hbox, FALSE, FALSE, 0);

    label = gtk_label_new("Databases to Backup");
    gtk_box_pack_start(GTK_BOX(temp_hbox), label, TRUE, TRUE, 0);

    /* Scrolled Window */
    left_scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(left_scrolled_window), 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(left_scrolled_window),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start(GTK_BOX(vbox1), left_scrolled_window, TRUE, TRUE, 0);

    gtk_widget_set_usize(GTK_WIDGET(vbox1), 230, 0);

    /* Active Clist */
    active_clist = gtk_clist_new(1);
    gtk_clist_set_selection_mode(GTK_CLIST(active_clist), GTK_SELECTION_SINGLE);
    gtk_clist_set_column_width(GTK_CLIST(active_clist), 0, 150);
    gtk_clist_set_auto_sort (GTK_CLIST(active_clist), TRUE);
    gtk_container_add(GTK_CONTAINER(left_scrolled_window), GTK_WIDGET(active_clist));
    gtk_object_set_user_data (GTK_OBJECT(active_clist), (gpointer)ACTIVE_DBM);

    temp_hbox = gtk_hbox_new(FALSE, 0);

    button = gtk_button_new_with_label("Move All >>");
    gtk_box_pack_start(GTK_BOX(temp_hbox), button, TRUE, FALSE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(cb_move_all_to_inactive), NULL);

    button = gtk_button_new_with_label("Move >>");
    gtk_box_pack_start(GTK_BOX(temp_hbox), button, TRUE, FALSE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(cb_move_to_inactive), NULL);

    gtk_box_pack_start (GTK_BOX(vbox1), temp_hbox, FALSE, FALSE, 0);

    /* -------------------- */
    /* Right half of screen */
    /* -------------------- */

    temp_hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), temp_hbox, FALSE, FALSE, 0);

    label = gtk_label_new("Databases to Ignore");
    gtk_box_pack_start(GTK_BOX(temp_hbox), label, TRUE, TRUE, 0);

    /* Scrolled Window */
    right_scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(right_scrolled_window), 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(right_scrolled_window),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox2), right_scrolled_window, TRUE, TRUE, 0);

    gtk_widget_set_usize(GTK_WIDGET(vbox2), 230, 0);

    /* Inactive Clist */
    inactive_clist = gtk_clist_new(1);
    gtk_clist_set_selection_mode(GTK_CLIST(inactive_clist), GTK_SELECTION_SINGLE);
    gtk_clist_set_column_width(GTK_CLIST(inactive_clist), 0, 150);
    gtk_clist_set_auto_sort (GTK_CLIST(inactive_clist), TRUE);
    gtk_container_add(GTK_CONTAINER(right_scrolled_window), GTK_WIDGET(inactive_clist));
    gtk_object_set_user_data (GTK_OBJECT(inactive_clist), (gpointer)INACTIVE_DBM);

    temp_hbox = gtk_hbox_new(FALSE, 0);

    button = gtk_button_new_with_label("<< Move");
    gtk_box_pack_start(GTK_BOX(temp_hbox), button, TRUE, FALSE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(cb_move_to_active), NULL);

    button = gtk_button_new_with_label("<< Move All");
    gtk_box_pack_start(GTK_BOX(temp_hbox), button, TRUE, FALSE, 10);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(cb_move_all_to_active), NULL);

    gtk_box_pack_start (GTK_BOX(vbox2), temp_hbox, FALSE, FALSE, 0);

    gtk_widget_show_all(hbox);
    gtk_widget_show_all(vbox);

    display_databases();
    update_prefs_gui();

    return 0;
}

int plugin_gui_cleanup() {
    jp_logf (JP_LOG_DEBUG, "plugin_gui_cleanup()\n");

    jp_logf (JP_LOG_DEBUG, "plugin_gui_cleanup() - clearing clists\n");
    gtk_clist_clear (GTK_CLIST(active_clist));
    gtk_clist_clear (GTK_CLIST(inactive_clist));

    jp_logf (JP_LOG_DEBUG, "plugin_gui_cleanup() - freeing widgets\n");
    if (GTK_IS_WIDGET(active_clist)) {
        gtk_widget_destroy (active_clist);
    }
    if (GTK_IS_WIDGET(inactive_clist)) {
        gtk_widget_destroy (inactive_clist);
    }

    jp_logf (JP_LOG_DEBUG, "plugin_gui_cleanup() - NULLing widgets\n");
    active_clist = NULL;
    inactive_clist = NULL;

	return 0;
}


/*
 * Called when JPilot shuts down.
 */
int plugin_exit_cleanup() {
    jp_logf (JP_LOG_DEBUG, "plugin_exit_cleanup()\n");

    if (needs_save) {
        jp_pref_write_rc_file (PREFS_FILE, backup_prefs, NUM_BPREFS);
        needs_save = FALSE;
    }

    jp_free_prefs (backup_prefs, NUM_BPREFS);

	return 0;
}

static void update_prefs_gui() {
    long ivalue;
    //const char *svalue;

    jp_get_pref (backup_prefs, BPREF_BACKUP_NEW, &ivalue, NULL);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (backup_new_button), ivalue);

    jp_get_pref (backup_prefs, BPREF_NUM_ARCHIVES, &ivalue, NULL);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (archive_spinner), ivalue);

    jp_get_pref (backup_prefs, BPREF_BACKUP_WHEN, &ivalue, NULL);
    switch (ivalue) {
        case EVERY_SYNC:
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (backup_new_button), TRUE);
            break;

        case DAILY:
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (daily_button), TRUE);
            break;

        case WEEKLY:
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (weekly_button), TRUE);
            break;

        case MONTHLY:
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (monthly_button), TRUE);
            break;

        default:
            jp_logf (JP_LOG_WARN, "Invalid backup_when pref: %d\n", ivalue);
    }
}

/*
 * This is called by plugin_sync to see if we need to skip the sync based
 * on the user's preferences. We just check the last sync time, compare it
 * with the current time, then see if the required amount of time has
 * passed.
 *
 * Returns TRUE if the sync should be skipped,
 *    ie TRUE == do not sync, FALSE == perform sync
 */
static gboolean skip_backup() {
    time_t last_backup;
    time_t ltime;
    struct tm *timep;
    long ivalue;
    const char *svalue;
    gboolean rval = FALSE;   /* DON'T skip by default */


    /* Get the last sync time from the prefs */
    jp_get_pref (backup_prefs, BPREF_LAST_BACKUP, &ivalue, &svalue);
    last_backup = (time_t)atol (svalue);
    time (&ltime);
    jp_logf (JP_LOG_DEBUG, "Backup::skip_sync() - last_sync = %ld, now = %ld\n", last_backup, ltime);

    /*
     * Note that, despite what the man pages would seem to indicate, the
     * localtime function returns a pointer to a single globally allocated
     * structure which is overwritten with each call.
     */
    timep = localtime (&ltime);
    jp_logf (JP_LOG_DEBUG,
             "Backup::skip_backup() - now = %d/%d/%d %d:%d:%d\n",
             timep->tm_mon + 1, timep->tm_mday, timep->tm_year + 1900,
             timep->tm_hour, timep->tm_min, timep->tm_sec);

    timep = localtime (&last_backup);
    jp_logf (JP_LOG_DEBUG,
             "Backup::skip_backup() - last backup = %d/%d/%d %d:%d:%d\n",
             timep->tm_mon + 1, timep->tm_mday, timep->tm_year + 1900,
             timep->tm_hour, timep->tm_min, timep->tm_sec);

    /* Get the backup-when pref, and decide if we need to sync. */
    jp_get_pref (backup_prefs, BPREF_BACKUP_WHEN, &ivalue, NULL);
    switch (ivalue) {
        case EVERY_SYNC:
            rval = FALSE;
            break;

        case DAILY:
            timep->tm_mday++;
            if (mktime (timep) > ltime) {
                rval = TRUE;
            }
            jp_logf (JP_LOG_DEBUG,
                     "Backup::skip_backup() - adjusted last backup = %d/%d/%d %d:%d:%d\n",
                     timep->tm_mon + 1, timep->tm_mday, timep->tm_year + 1900,
                     timep->tm_hour, timep->tm_min, timep->tm_sec);
            jp_logf (JP_LOG_DEBUG, "ltime = %ld, mktime = %ld\n", ltime, mktime (timep));
            break;

        case WEEKLY:
            timep->tm_mday += 7;
            if (mktime (timep) > ltime) {
                rval = TRUE;
            }
            break;

        case MONTHLY:
            timep->tm_mon++;
            if (mktime (timep) > ltime) {
                rval = TRUE;
            }
            break;

        default:
            jp_logf (JP_LOG_WARN,
                         "Unrecognized pref value for backup_when: %d\n",
                         ivalue);
    }

    return rval;
}

/*
 * Used by expire_archives() to list archive directories.
 */
static int archive_dir_select (const struct dirent *entry) {
    int i, j, k;
    int x, y, z;

    return sscanf (entry->d_name,
                   "Archive_%4d-%2d-%2d@%2d:%2d:%2d",
                   &i, &j, &k, &x, &y, &z);
}

/*
 * Expires the given archive, ie, deletes the directory.
 */
static int expire_archive (char *dir) {
    FILE *fp;
    char full_name[256];
    char line[256];
    char *pc;

    jp_logf (JP_LOG_GUI, "Expiring %s\n", dir);

    get_archive_file_name (dir, MANIFEST, full_name, 255);
    fp = fopen (full_name, "r");
    if (!fp) {
        jp_logf (JP_LOG_WARN,
                 "Can't open manifest file %s.\n"
                 "Please delete archive directory %s by hand.\n",
                 full_name, dir);
        return -1;
    }

    while (!feof (fp)) {
        if (fgets (line, 256, fp)) {

            /* chop off the newline */
            if ((pc = (char *)index (line, '\n'))) {
                pc[0] = '\0';
            }

            get_archive_file_name (dir, line, full_name, 255);
            if (unlink (full_name) < 0) {
                perror ("unlink");
                jp_logf (JP_LOG_WARN,
                         "Can't delete archive file %s.\n"
                         "Please delete archive directory %s by hand.\n",
                         full_name, dir);
            }
        }
    }

    fclose (fp);
    get_archive_file_name (dir, MANIFEST, full_name, 255);
    unlink (full_name);

    if (rmdir (dir) < 0) {
        perror ("rmdir");
        jp_logf (JP_LOG_WARN,
                 "Can't remove archive directory %s.\n"
                 "Please delete by hand.\n",
                 dir);
    }

	return 0;
}

/*
 * Checks to see how many archive directories there currently are, and
 * if there are more than set in BPREF_NUM_ARCHIVES, expires the oldest
 * director(y|ies).
 */
static int expire_archives() {
    struct dirent **namelist;
    char temp_str[256];
    int n, i, j;
    long ivalue;
    char backup_dir[260];


    get_home_file_name(BACKUP_DIR_NAME, backup_dir, 256);

    jp_logf (JP_LOG_GUI, "Expiring old archives...\n");

    n = scandir (backup_dir, &namelist, archive_dir_select, alphasort);
    if (n < 0) {
        perror("scandir");
        jp_logf (JP_LOG_WARN,
                 "Unable to scan backup directory %s.\n"
                 "Hence, unable to expire archives\n",
                 backup_dir);
        return -1;
    }

    jp_get_pref (backup_prefs, BPREF_NUM_ARCHIVES, &ivalue, NULL);
    jp_logf (JP_LOG_DEBUG, "Backup: expire_archives - pref: %d, %d archives exist.\n", ivalue, n);
    for (i = 0, j = n; j > ivalue; i++, j--) {
        get_backup_file_name (namelist[i]->d_name, temp_str, 255);
        expire_archive (temp_str);
        free (namelist[i]);
    }

    /* free the remaining items in namelist */
    while (i < n) {
        free (namelist[i++]);
    }

    if (namelist) {
        free (namelist);
    }
    return 0;
}

/*
 * Loads the active and inactive databases from their respective dbm files
 * and inserts them into the CLists.
 */
static int display_databases() {
    char full_name[256];
    GDBM_FILE dbf;
    datum key, nextkey;

    jp_logf (JP_LOG_DEBUG, "Backup: display databases\n");

    /*
     * Let's make sure the CLists have been created before we go and
     * try to update them. Thanks to Martin Sjoelin for pointing this
     * out.
     */
    if (!GTK_IS_CLIST(active_clist) || !GTK_IS_CLIST(inactive_clist)) {
        jp_logf (JP_LOG_DEBUG, "Backup: GUI not created, skipping update\n");
        return 0;
    }

    gtk_clist_freeze (GTK_CLIST(active_clist));
    gtk_clist_clear (GTK_CLIST(active_clist));
    gtk_clist_freeze (GTK_CLIST(inactive_clist));
    gtk_clist_clear (GTK_CLIST(inactive_clist));

    /* open the active dbm file */
    get_backup_file_name (ACTIVE_DBM, full_name, 255);
    dbf = gdbm_open (full_name, 512, GDBM_READER, 0644, 0);
    if (dbf) {
        /* load the active databases */
        key = gdbm_firstkey (dbf);
        if (key.dptr) {
            jp_logf (JP_LOG_DEBUG, "Retrieved %s from active database file\n", key.dptr);
            gtk_clist_append (GTK_CLIST(active_clist), &key.dptr);

            while (key.dptr) {
                nextkey = gdbm_nextkey (dbf, key);
                if (nextkey.dptr) {
                    jp_logf (JP_LOG_DEBUG, "Retrieved %s from active database file\n", nextkey.dptr);
                    gtk_clist_append (GTK_CLIST(active_clist), &nextkey.dptr);
                }
                free (key.dptr);
                key = nextkey;
            }
        }
        else {
            jp_logf (JP_LOG_GUI, "No active databases found\n");
        }

        gdbm_close (dbf);
    }
    else {
        jp_logf (JP_LOG_WARN, "Can't open dbm file %s\n", full_name);
    }


    /* open the inactive dbm file */
    get_backup_file_name (INACTIVE_DBM, full_name, 255);
    dbf = gdbm_open (full_name, 512, GDBM_READER, 0644, 0);
    if (dbf) {
        /* load the inactive databases */
        key = gdbm_firstkey (dbf);
        if (key.dptr) {
            jp_logf (JP_LOG_DEBUG, "Retrieved %s from inactive database file\n", key.dptr);
            gtk_clist_append (GTK_CLIST(inactive_clist), &key.dptr);

            while (key.dptr) {
                nextkey = gdbm_nextkey (dbf, key);
                if (nextkey.dptr) {
                    jp_logf (JP_LOG_DEBUG, "Retrieved %s from inactive database file\n", nextkey.dptr);
                    gtk_clist_append (GTK_CLIST(inactive_clist), &nextkey.dptr);
                }
                free (key.dptr);
                key = nextkey;
            }
        }
        else {
            jp_logf (JP_LOG_GUI, "No inactive databases found\n");
        }

        gdbm_close (dbf);
    }
    else {
        jp_logf (JP_LOG_WARN, "Can't open dbm file %s\n", full_name);
    }

    gtk_clist_unselect_all (GTK_CLIST(active_clist));
    gtk_clist_unselect_all (GTK_CLIST(inactive_clist));

    gtk_clist_thaw (GTK_CLIST(active_clist));
    gtk_clist_thaw (GTK_CLIST(inactive_clist));

    return 0;
}


/*
 * Moves the selected item in the from CList to the to CList.
 */
static void move_selected_item (GtkCList *src, GtkCList *dest) {
    guint row;
    GList *node;
    gchar *text;
    gpointer src_name;
    gpointer dest_name;

    src_name = gtk_object_get_user_data (GTK_OBJECT(src));
    dest_name = gtk_object_get_user_data (GTK_OBJECT(dest));

    node = src->selection;
    if (node) {
        row = GPOINTER_TO_UINT(node->data);
        gtk_clist_get_text (src, row, 0, &text);
        node = NULL;
        node = g_list_append (node, text);
        dbm_move_items (src_name, dest_name, node);
        g_list_free (node);
        gtk_clist_append (dest, &text);
        gtk_clist_remove (src, row);
    }
}

/*
 * Moves all items in the src CList to the dest CList.
 */
static void move_all_items (GtkCList *src, GtkCList *dest) {
    gchar *text;
    gpointer src_name;
    gpointer dest_name;
    GList *list = NULL;
    int i;

    /*
     * Get the DBM filenames from the CList user data.
     */
    src_name = gtk_object_get_user_data (GTK_OBJECT(src));
    dest_name = gtk_object_get_user_data (GTK_OBJECT(dest));

    /*
     * Freeze the CLists, so we don't get an annoying flicker.
     */
    gtk_clist_freeze (src);
    gtk_clist_freeze (dest);

    /*
     * Iterate over each of the items in the src...
     */
    for (i = 0; gtk_clist_get_text (src, i, 0, &text); i++) {
        /* Append the item to the dest CList and store it in the GList */
        gtk_clist_append (dest, &text);
        list = g_list_append (list, text);
    }

    /* Move the items in the DBM files */
    dbm_move_items (src_name, dest_name, list);
    g_list_free (list);

    gtk_clist_clear (src);
    gtk_clist_thaw (src);
    gtk_clist_thaw (dest);
}


/*
 * Checks for the existence of the backup directory, and creates it if
 * necessary.
 */
static int check_backup_dir() {
    struct stat statb;
    char backup_dir[260];
    char test_file[260];
    FILE *out;

    get_home_file_name(BACKUP_DIR_NAME, backup_dir, 256);

    if (stat(backup_dir, &statb)) {
        /*directory isn't there, create it */
        if (mkdir(backup_dir, 0777)) {
            /*Can't create directory */
            jp_logf(JP_LOG_WARN, "Can't create directory %s\n", backup_dir);
            return 1;
        }
        if (stat(backup_dir, &statb)) {
            jp_logf(JP_LOG_WARN, "Can't create directory %s\n", backup_dir);
            return 1;
        }
    }

    /*Is it a directory? */
    if (!S_ISDIR(statb.st_mode)) {
        jp_logf(JP_LOG_WARN, "%s doesn't appear to be a directory.\n"
                    "I need it to be.\n", backup_dir);
        return 1;
    }

    /*Can we write in it? */
    get_backup_file_name ("test", test_file, 256);
    jp_logf (JP_LOG_DEBUG, "Trying to open %s for write\n", test_file);
    out = fopen(test_file, "w+");
    if (!out) {
        jp_logf(JP_LOG_WARN, "I can't write files in directory %s\n", backup_dir);
    }
    else {
        fclose(out);
        unlink(test_file);
    }

    return 0;
}

/*
 * Appends the filename specified by file to the Backup directory path
 * and stores the result in full_name, unless the resulting string would
 * be greater than max_size.
 */
static int get_backup_file_name (char *file, char *full_name, int max_size)
{
    char backup_dir[260];

    get_home_file_name(BACKUP_DIR_NAME, backup_dir, 256);

    if (strlen (backup_dir) > (max_size - strlen (file) - 2)) {
        jp_logf (JP_LOG_WARN, "filename %s is too big\n", file);
        return 1;
    }

    sprintf (full_name, "%s/%s", backup_dir, file);
    return 0;
}

/*
 * Appends the filename specified by file to the filename specified by
 * arch, which is assumed to be the full path of a directory in the
 * Backup directory. The result is stored in full_name, unless the resulting
 * string would be greater than max_size.
 */
static int get_archive_file_name (char *arch, char *file, char *full_name, int max_size)
{

    if (strlen (arch) > (max_size - strlen (file) - 2)) {
        jp_logf (JP_LOG_WARN, "filename %s is too big\n", file);
        return 1;
    }

    sprintf (full_name, "%s/%s", arch, file);
    return 0;
}

/*
 * Copied from sync.c in J-Pilot
 */
static void filename_make_legal(char *s)
{
   char *p;
   for (p=s; *p; p++) {
      if (*p=='/') {
         *p='?';
      }
   }
}

/*
 * Moves the given items from one DBM file to the other.
 */
static int dbm_move_items (char *src_name, char *dest_name, GList *node) {
    char src_dbf_name[256];
    char dest_dbf_name[256];
    GDBM_FILE src_dbf;
    GDBM_FILE dest_dbf;
    datum key;
    datum content;
    int ret;

    /* open the src dbm file */
    get_backup_file_name (src_name, src_dbf_name, 255);
    src_dbf = gdbm_open (src_dbf_name, 512, GDBM_WRCREAT, 0644, 0);
    if (!src_dbf) {
        /* Can't open or create dbm file */
        jp_logf (JP_LOG_FATAL, "Can't open dbm file %s\n", src_dbf_name);
        return 1;
    }

    /* open the dest dbm file */
    get_backup_file_name (dest_name, dest_dbf_name, 255);
    dest_dbf = gdbm_open (dest_dbf_name, 512, GDBM_WRCREAT, 0644, 0);
    if (!dest_dbf) {
        /* Can't open or create dbm file */
        jp_logf (JP_LOG_FATAL, "Can't open dbm file %s\n", dest_dbf_name);
        return 1;
    }

    while (node) {
        gchar *text = (gchar *)node->data;

        jp_logf (JP_LOG_DEBUG,
                     "dbm_move_items() - from: %s, to: %s, text: %s\n",
                     src_dbf_name,
                     dest_dbf_name,
                     text);

        key.dptr = text;
        key.dsize = strlen (text) + 1;

        /* fetch the content */
        content = gdbm_fetch (src_dbf, key);
        if (!content.dptr) {
            /* sanity check, shouldn't happen */
            jp_logf (JP_LOG_WARN, "Key %s has no content!\n", text);
            content.dptr = "0";
            content.dsize = 2;
        }

        /* store it in the dest dbm */
        ret = gdbm_store (dest_dbf, key, content, GDBM_INSERT);

        /* delete it from the src dbm */
        ret = gdbm_delete (src_dbf, key);

        node = node->next;
    }

    /* close the dbm files */
    gdbm_close (src_dbf);
    gdbm_close (dest_dbf);

    return 0;
}

/* ----------------------------------------------------------------------- */
/*                               Callbacks                                 */
/* ----------------------------------------------------------------------- */

static void cb_move_all_to_inactive (GtkWidget *widget, int data) {
    move_all_items (GTK_CLIST(active_clist), GTK_CLIST(inactive_clist));
}

static void cb_move_to_inactive (GtkWidget *widget, int data) {
    move_selected_item (GTK_CLIST(active_clist), GTK_CLIST(inactive_clist));
}

static void cb_move_all_to_active (GtkWidget *widget, int data) {
    move_all_items (GTK_CLIST(inactive_clist), GTK_CLIST(active_clist));
}

static void cb_move_to_active (GtkWidget *widget, int data) {
    move_selected_item (GTK_CLIST(inactive_clist), GTK_CLIST(active_clist));
}

static void cb_toggle_button (GtkWidget *widget, gpointer data) {
    if (needs_load) {
        if (jp_pref_read_rc_file (PREFS_FILE, backup_prefs, NUM_BPREFS) < 0) {
            jp_logf (JP_LOG_WARN, "Backup: Unable to load preferences file " PREFS_FILE "\n");
        }
        else {
            jp_logf (JP_LOG_DEBUG, "Backup: loaded preferences from " PREFS_FILE "\n");
        }

        needs_load = FALSE;
    }

    needs_save = FALSE;
    switch (GPOINTER_TO_INT (data)) {
        case BACKUP_NEW_BUTTON:
            jp_logf (JP_LOG_DEBUG, "backup new button\n");
            jp_set_pref (backup_prefs,
                         BPREF_BACKUP_NEW,
                         GTK_TOGGLE_BUTTON (widget)->active,
                         NULL);
            break;

        case EVERY_SYNC_BUTTON:
            jp_logf (JP_LOG_DEBUG, "every sync button\n");
            jp_set_pref (backup_prefs,
                         BPREF_BACKUP_WHEN,
                         EVERY_SYNC,
                         NULL);
            break;

        case DAILY_BUTTON:
            jp_logf (JP_LOG_DEBUG, "daily button\n");
            jp_set_pref (backup_prefs,
                         BPREF_BACKUP_WHEN,
                         DAILY,
                         NULL);
            break;

        case WEEKLY_BUTTON:
            jp_logf (JP_LOG_DEBUG, "weekly button\n");
            jp_set_pref (backup_prefs,
                         BPREF_BACKUP_WHEN,
                         WEEKLY,
                         NULL);
            break;

        case MONTHLY_BUTTON:
            jp_logf (JP_LOG_DEBUG, "monthly button\n");
            jp_set_pref (backup_prefs,
                         BPREF_BACKUP_WHEN,
                         MONTHLY,
                         NULL);
            break;


        default:
            jp_logf (JP_LOG_WARN, "Invalid button data: %d\n",
                         GPOINTER_TO_INT (data));
            needs_save = FALSE;
    }
}

static void cb_archives (GtkWidget *widget, gpointer data) {
    int n;

    if (needs_load) {
        if (jp_pref_read_rc_file (PREFS_FILE, backup_prefs, NUM_BPREFS) < 0) {
            jp_logf (JP_LOG_WARN, "Backup: Unable to load preferences file " PREFS_FILE "\n");
        }
        else {
            jp_logf (JP_LOG_DEBUG, "Backup: loaded preferences from " PREFS_FILE "\n");
        }

        needs_load = FALSE;
    }

    n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (archive_spinner));
    jp_set_pref (backup_prefs, BPREF_NUM_ARCHIVES, n, NULL);
    needs_save = TRUE;
}
