/*************************************************************************
 * $Id: mod_env.c,v 1.6 2000/04/04 23:37:34 dpotter Exp $
 *
 * mod_env.c - environment support
 *
 * Copyright (C) 2000 by David E. Potter <dp-irmp3@dpotter.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "tools.h"
#include "config.h"
#include "log.h"
#include "mod.h"
#include "mod_env.h"


/*************************************************************************
 * GLOBALS
 */


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_env = {
	mod_env_deinit,		// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	NULL,			// update
	mod_env_message,	// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * RECEIVE MESSAGE
 *
 * We have to handle these macros in an interesting way.  We can't just
 * parse up all the commands and then throw them on the msgqueue, because
 * some of the commands rely on other queued messages to complete their
 * work.  This would cause some commands to execute out of order.
 *
 * Instead, we add only the first command to the msgqueue, then we add
 * an env_queue msgqueue.  The env_queue command contains the rest of
 * the list of instructions.  We recursively perform this action until
 * we have no more instructions to perform.
 *
 * More work, but it guarantees that all commands will be executed in the
 * correct order.  Handy for things like beep disable;shuffle on;beep enable
 */

void mod_env_message (int msgtype, char *msg)
{
	char *c1,*c2=NULL,*nextcmd;
	char envname[256];	
	char envcmd[1024]; 	// a reasonable size for an expanded environment command.

	// handle input messages
	strcpy(envcmd,msg);
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		// process environment commands
		if (c1 && !strcasecmp(c1, "environment")) {
			c2 = c1 ? strtok(NULL, " \t") : NULL;
			if (c2 != NULL) {
				envcmd[0]=0; 		// clear the string
				strcpy(envname,"environment_");
				strcat(envname,c2);
				log_printf(LOG_NORMAL, "mod_env_message(): received %s.\n",envname);
				strcpy(envcmd,config_getstr(envname, ""));
				if(strlen(envcmd) > 0) {
					nextcmd=strchr(envcmd,';');
					if(nextcmd != NULL) {
						nextcmd[0]=(char) 0;
						nextcmd++;
					}
					mod_sendmsg(MSGTYPE_INPUT,envcmd);
					if (nextcmd != NULL) mod_sendmsgf(MSGTYPE_INPUT,"env_queue %s",nextcmd);
				}
			}	
		} else if (c1 && !strcasecmp(c1,"env_queue")) {
			if (strlen(envcmd) > strlen(msg)) c2 = &(envcmd[strlen(msg)+1]);
			log_printf(LOG_DEBUG, "mod_env_message(): continuing environment command: %s.\n",c2);
			nextcmd=strchr(c2,';');
			if(nextcmd != NULL) {
				nextcmd[0]=(char) 0;
				nextcmd++;
			}
			mod_sendmsg(MSGTYPE_INPUT,c2);
			if (nextcmd != NULL) mod_sendmsgf(MSGTYPE_INPUT,"env_queue %s",nextcmd);
		}
	} 
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_env_init (void)
{
	log_printf(LOG_DEBUG, "mod_env_init(): initializing\n");

	// register our module
	mod_register(&mod_env);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_env_deinit (void)
{
	log_printf(LOG_DEBUG, "mod_env_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
