/***************************************************************************
 *   Copyright (C) 2004 by Michael Schulze                                 *
 *   mike.s@genion.de                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "itunesdbparser.h"
#include <qdatastream.h>
#include <qfileinfo.h>

#include <kdebug.h>

namespace itunesdb {

class FileCloser {
    QFile & file;
public:
    FileCloser( QFile& toClose ) : file( toClose ) {}
    ~FileCloser() {
        file.close();
    }
};

ItunesDBParser::ItunesDBParser( ItunesDBListener& ituneslistener)
{
    listener= &ituneslistener;
}


ItunesDBParser::~ItunesDBParser()
{
}


void ItunesDBParser::seekRelative(QDataStream& stream, uint numbytes)
{
    if ( numbytes ) {
        char* buffer = new char[ numbytes ];
        stream.readRawBytes(buffer, numbytes);
        delete [] buffer;
    }
}

/*!
    \fn itunesdb::ItunesDBParser::parse(QFile& file)
 */
void ItunesDBParser::parse(QFile& file)
{
    Q_UINT32 magic, headerlen, filesize;

    PlaylistItem current_playlistitem;
    Track current_track;

    ListItem * listitem = NULL;

    listener->parseStarted();

    if( !file.exists() || !file.open( IO_ReadOnly)) {
        listener->handleError( file.name()+ " could not be opened!");
        return;    // no need to cleanup
    }

    FileCloser closer( file );

    QDataStream stream( &file);
    stream.setByteOrder( QDataStream::LittleEndian);

    stream >> magic;
    if( magic != 0x6462686D) {
        listener->handleError( file.name() + " is not an itunesDB file");
        return;
    }

    stream >> headerlen;
    stream >> filesize;

    // skip the rest of the header
    seekRelative(stream, headerlen- 12);

    // begin with actual parsing
    while( !stream.atEnd()) {
        Q_UINT32 blocktype;
        Q_UINT32 blocklen;

        stream >> blocktype;

        switch( blocktype) {
        case 0x6473686D: {    // mhsd - yeah, good to know
            Q_UINT32 type;
            stream >> blocklen;

            // iTunesDB has two large blocks - one with the tracks and one with playlists
            // both begin with this tag. Since the data entrys themself have different block
            // types we can safely ignore this information here ...
            if( listitem != NULL) {    // OK, so now we're at the end of the first mshd
                handleItem( *listitem);
                listitem = NULL;
            }
            stream >> type;     // skip mhsd size
            stream >> type;
            listener->handleDataSet( type );

            seekRelative(stream, blocklen- 16);
            }
            break;

        case 0x746C686D: {    // mhlt: number of tracks (and other metainfo?)
            Q_UINT32 numtracks;
            stream >> blocklen;
            if( listitem != NULL) {
                handleItem( *listitem);
                listitem = NULL;
            }
            stream >> numtracks;
            listener->setNumTracks( numtracks);
            seekRelative(stream, blocklen- 12);    // skip the rest
            }
            break;

        case 0x7469686D: {    // mhit - start of a track
            if( listitem != NULL) {
                handleItem( *listitem);
            }

            current_track = Track();
            current_track.readFromStream(stream);
            listitem = &current_track;
            }
            break;

        case 0x646F686D: {    // mhod
            Q_UINT32 type;
            stream >> blocklen;
            stream >> blocklen;

            uint ucs2len = ( blocklen - 40 ) / sizeof( Q_UINT16 );

            stream >> type;

            if( type > itunesdb::MHOD_DESCRIPTION_TEXT) {   // can't handle that at the moment
                seekRelative( stream, blocklen - 16 );  // skip the whole block
                break;    // ignore
            }

            Q_UINT16 * buffer = new Q_UINT16 [ucs2len+1];

            seekRelative(stream, 24);    // skip stuff
            for ( uint i = 0; i < ucs2len; ++i ) {
              Q_UINT16 h;
              stream >> h;
              buffer[i] = h;
            }

            buffer[ucs2len]= 0;

            if( listitem != NULL) {
                // TODO check if we really need this anymore
                // KFromToLittleEndian(buffer, buffer, ucs2len);
                listitem->setItemProperty( QString::fromUcs2(buffer), (ItemProperty)type);
            }

            delete [] buffer;
            }
            break;
        case 0x706C686D: {    // mhlp
            Q_UINT32 numplaylists;
            if( listitem != NULL) {
                handleItem( *listitem);
                listitem = NULL;
            }
            stream >> blocklen;
            stream >> numplaylists;
            listener->setNumPlaylists( numplaylists);
            seekRelative(stream, blocklen- 12);
            }
            break;

        case 0x7079686D: {    // mhyp: Playlist
            if( listitem != NULL) {
                handleItem( *listitem);
            }
            if( !current_playlist.getTitle().isEmpty()) {
                // there's a playlist waiting to be handed over to the listener
                listener->handlePlaylist( current_playlist);
            }
            current_playlist = Playlist();
            current_playlist.readFromStream( stream );
            listitem = &current_playlist;    // TODO working with a copy is wrong here
            }
            break;
        case 0x7069686D: {    // mhip: PlaylistItem
            Q_UINT32 itemid;

            stream >> blocklen;
            if( listitem != NULL) {
                handleItem( *listitem);
            }

            seekRelative(stream, 16);
            stream >> itemid;

            current_playlistitem = PlaylistItem( itemid);
            listitem = &current_playlistitem;

            seekRelative(stream, blocklen- 28);
            }
            break;
        default:
            listener->handleError( QString( "unknown tag found! stop parsing"));
            return;
        }
    }

    if( listitem != NULL) {
        handleItem( *listitem);
        listitem = NULL;
    }
    if( !current_playlist.getTitle().isEmpty()) {
        // there's a playlist waiting to be handed over to the listener
        listener->handlePlaylist( current_playlist);
    }
    listener->parseFinished();
}

void ItunesDBParser::parsePlaycount(QFile& file) {
    Q_UINT32 magic, numtracks, entrysize, headersize;
    Q_UINT32 playcount, lastplayed, bookmark, rating; //Items in the Entrys

    if( !file.exists() || !file.open( IO_ReadOnly)) {
        // listener->handleError( file.name()+ " could not be opened!"); // this should be optional
        return;    // no need to cleanup
    }

    FileCloser closer( file );

    QDataStream stream( &file);
    stream.setByteOrder( QDataStream::LittleEndian );

    stream >> magic;
    if( magic != ITUNESDB_MHDP) { //Needed this defines, otherwise my eyes hurt ;)
        // listener->handleError( file.name() + " is not an Playcount File");	// this should be optional
        return;
    }

    stream >> headersize;
    stream >> entrysize;
    stream >> numtracks;
    if (entrysize < IPOD_FIRMWARE_V13) {
        // listener->handleError( "You use a old firmware version, go get a new one :-)"); // this should be optional
        return;
    }
    if (QFileInfo(file).size() != (headersize + entrysize*numtracks)) {
        // listener->handleError( file.name() + " is corrupt"); //Actually this is not an error, but a bad sign for the hd
        //listener->handleError( "File System error, kiss your data goodbye as long as you could!"); 
        return;
    }

    seekRelative(stream, headersize-16);    // skip empty header
    //This i is (@see OTG Playlists) an INDEX in the mhdxtxft ;)
    for (unsigned int i = 0; i < numtracks; i++) {
        QString message;
        stream >> playcount;
        stream >> lastplayed;
        stream >> bookmark;
        if (entrysize > IPOD_FIRMWARE_V13) {
            stream >> rating;
            seekRelative(stream, entrysize - 16);    // skip unknown information
        } else {
            rating = 0;
            seekRelative(stream, entrysize - 12);    // skip unknown information
        }

        //Yes you got it!
        //As of iTunes4.9 these fields are NO longer 0 but have a value so this is pointless
        if (rating || playcount) {
            listener->handlePlaycount( i, lastplayed ? lastplayed - MAC_EPOCH_DELTA : 0, rating, playcount, bookmark );
        }
    }
}

void ItunesDBParser::parseOTG(QFile& file)
{
    Q_UINT32 magic, numtracks, filesize, idx;
    PlaylistItem current_playlistitem;
    ListItem * listitem = NULL;
    Track current_track;

    if( !file.exists() || !file.open( IO_ReadOnly)) {
        listener->handleError( file.name()+ " could not be opened!");
        return;    // no need to cleanup
    }

    FileCloser closer( file );

    QDataStream stream( &file);
    stream.setByteOrder( QDataStream::LittleEndian);

    stream >> magic;
    if( magic != 0x6F70686D) { //"mhpo" files start with this
        listener->handleError( file.name() + " is not an OTG file");
        return;
    }

    current_playlist = Playlist();   // Make a shiny new playlist, note: Does not get a name
    current_playlist.setTitle( "kpod ruleZ!" ); //will be overwritten by handleOTGPlaylist

    stream >> filesize;
    stream >> magic; //Unknown entry, but interesing to know
    stream >> numtracks;
    stream >> magic; //Unknown entry, but interesing to know

    //Hmmm the index returned here are not normal indizes but offsets in the MODT (?) 
    //
    for (unsigned int i = 0; i < numtracks; i++) {
	    stream >> idx;
            current_playlistitem = PlaylistItem( idx );
            listitem = &current_playlistitem;
	    handleItem( *listitem );
    }
    listitem = NULL;
    listener->handleOTGPlaylist( current_playlist );

    //listener->parseFinished();
}

}

/*!
    \fn itunesdb::ItunesDBParser::handleItem( ListItem& item)
 */
void itunesdb::ItunesDBParser::handleItem( ListItem& item)
{
    switch( item.getType()) {
    case ITEMTYPE_TRACK: {
        Track * pTrack = dynamic_cast <Track *> (&item);
        if(pTrack == NULL || pTrack->getID() == 0)
            break;
        pTrack->doneAddingData();
        listener->handleTrack( *pTrack);
        }
        break;
    case ITEMTYPE_PLAYLIST: {
        Playlist * pPlaylist = dynamic_cast <Playlist *> (&item);
        if( pPlaylist == NULL)
            break;
        pPlaylist->doneAddingData();
        // listener->addPlaylist( *pPlaylist);
        }
        break;
    case ITEMTYPE_PLAYLISTITEM: {
        PlaylistItem * pPlaylistitem = dynamic_cast <PlaylistItem *> (&item);
        if( pPlaylistitem == NULL)
            break;
        pPlaylistitem->doneAddingData();
        current_playlist.addPlaylistItem( *pPlaylistitem);
        }
        break;
    }
}

