/***************************************************************************
 *   Copyright (C) 2004 by Andrew de Quincey                               *
 *   adq@lidskialf.net                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ipoddevicedetails.h"
#include <qtextstream.h>



#define MAX_DETAILS_STRING_LEN 255


IPodDeviceDetails::IPodDeviceDetails(const QString& filename) {
  this->filename = filename;
  dirty = false;
}

IPodDeviceDetails::~IPodDeviceDetails() {
}

bool IPodDeviceDetails::load() {
  QFile detailsFile(filename);
  if (detailsFile.exists() && detailsFile.open(IO_ReadOnly)) {
    name = readDeviceInfoString(detailsFile, 0);
    owner = readDeviceInfoString(detailsFile, 0x200);
    comment = readDeviceInfoString(detailsFile, 0x400);
    detailsFile.close();
    dirty = false;
    return true;
  }
  
  // if we failed, mark it as dirty so we try and save a new one
  dirty = true;
  return false;
}

bool IPodDeviceDetails::save() {
  if (!dirty) return true;

  QFile detailsFile(filename);
  
  if (detailsFile.open(IO_WriteOnly)) {
    unsigned int length;
    QTextStream ostream(&detailsFile);
    ostream.setEncoding(QTextStream::RawUnicode);
    
    length = name.length();
    detailsFile.at(0x000);
    detailsFile.putch(length & 0xff);
    detailsFile.putch(length >> 8);
    ostream << name;

    length = owner.length();
    detailsFile.at(0x200);
    detailsFile.putch(length & 0xff);
    detailsFile.putch(length >> 8);
    ostream << owner;
    
    length = comment.length();
    detailsFile.at(0x400);
    detailsFile.putch(length & 0xff);
    detailsFile.putch(length >> 8);
    ostream << comment;
    
    // ensure file is the correct length
    detailsFile.at(0x5ff);
    detailsFile.putch(0);
    
    detailsFile.close();
    dirty = false;
    return true;
  }
  
  // failure!
  return false;
}

QString IPodDeviceDetails::getName() {
  return name;
}

QString IPodDeviceDetails::getOwner() {
  return owner;
}

QString IPodDeviceDetails::getComment() {
  return comment;
}

void IPodDeviceDetails::setName(const QString& name) {
  this->name = name;
  this->name.truncate(MAX_DETAILS_STRING_LEN);
  dirty = true;
}

void IPodDeviceDetails::setOwner(const QString& owner) {
  this->owner = owner;
  this->owner.truncate(MAX_DETAILS_STRING_LEN);
  dirty = true;
}

void IPodDeviceDetails::setComment(const QString& comment) {
  this->comment = comment;
  this->comment.truncate(MAX_DETAILS_STRING_LEN);
  dirty = true;
}


QString IPodDeviceDetails::readDeviceInfoString(QFile& file, int position) {
  // get the string size
  file.at(position);
  unsigned int stringSize = file.getch() | (file.getch() << 8);
  if (stringSize > MAX_DETAILS_STRING_LEN) return "";
  
  // read in the raw data
  QByteArray stringData(stringSize * 2);
  file.readBlock(stringData.data(), stringSize * 2);
  
  // decode the raw data into a string
  QTextIStream stringStream(stringData);
  stringStream.setEncoding(QTextStream::RawUnicode);
  QString string = stringStream.read(); 
  
  // strip any \0 chars
  int i = string.find((char) 0);
  if (i >= 0) string.truncate(i);
  
  // done
  return string;
}
