#
# iPodder player handling module
#

import platform
import os
import logging
import re

log = logging.getLogger('iPodder')

class CannotInvoke(AssertionError): 
    """Player objects raise this in __init__ if they can't be invoked 
    on this system."""
    pass

class Player(object): 
    """Generic dict-style interface to media players."""

    def __init__(self): 
        """Raise CannotInvoke if you can't be used."""
        object.__init__(self)
        raise NotImplementedError

    def append_and_create(self, filename, playlistname, playnow=True): 
        """Add the tune to the playlist. Create the list if necessary.
        Don't add the tune if it's there already."""
        raise NotImplementedError

    def get_rating(self, filename, playlistname): 
        """Get the rating (0-100) for a particular entry in a 
        particular playlist. Probably iTunes specific, but perhaps 
        not."""
        raise NotImplementedError

    def playlist_filenames(self, playlistname): 
        """Return a list of files referred to by a particular 
        playlist."""
        raise NotImplementedError

    def play_file(self,filename): 
        """Play a file."""
        raise NotImplementedError

    def sync(self): 
        """Synchronise changes to the media player."""
        raise NotImplementedError
        
def makeOS9abspath(path):
  """Returns an ":" delimited, OS 9 style absolute path."""
  import Carbon.File, os.path
  rv = []
  fss = Carbon.File.FSSpec(path)
  while 1:
    vrefnum, dirid, filename = fss.as_tuple()
    rv.insert(0, filename)
    if dirid == 1: break
    fss = Carbon.File.FSSpec((vrefnum, dirid, ""))
  if len(rv) == 1:
    rv.append('')
  return ':'.join(rv)

class iTunes(Player): 
    """Player interface to iTunes."""

    def __init__(self): 
        """Initialise the iTunes player interface."""
        # What OS are we on? Cache the result in the class object. 
        try: 
            plat = self.__class__.platform
        except AttributeError: 
            plat = platform.system()
            if plat in ['Windows', 'Microsoft', 'Microsoft Windows']: 
                plat = 'windows'
            elif plat in ['Darwin']: 
                plat = 'darwin'
            else: 
                raise CannotInvoke, "Unknown platform %s" % plat
        self.__class__.platform = plat

        self.safeplaylistchars = " '0123456789abcdefghijklmnopqrstuvwxyz"
        
        for methodname in ['append_and_create', 'sync', 'get_rating', 
                           'init', 'playlist_filenames', 'play_file']: 
            try: 
                platmethodname = '%s_%s' % (methodname, plat)
                platmethod = getattr(self, platmethodname)
                setattr(self, methodname, platmethod)
            except AttributeError: 
                pass # expose the NotImplementedError version from the base
                
        self.init()
        
    def init_darwin(self): 
        """Initialise the Player object for operation on Darwin."""
        pass
    
    def execute_applescript_darwin(self, script): 
        """Execute some AppleScript on Darwin."""
        scriptlines = [line.strip() for line in script.split('\n')]
        scriptlines = [line for line in scriptlines if line]
        
        cmd = '/usr/bin/osascript %s >/dev/null' % ' '.join(
               ['-e "%s"'  % re.sub(r'([$`"\\])',r'\\\1',line)
                for line in scriptlines])
        
        log.debug("Running command: %s", cmd)
        result = os.system(cmd)
        log.debug("Return code was %d", result)
            
    def append_and_create_darwin(self, filename, playlistname, playnow=True): 
        """Add the tune to the playlist. Create the list if necessary.
        Don't add the tune if it's there already."""

        playlistname = sanitize(playlistname, self.safeplaylistchars)
        log.debug("Ensuring playlist %s has file %s", playlistname, filename)
        song = makeOS9abspath(filename)

        # Can anyone refactor this script to eliminate the duplication? 
        script = """\
            tell application "System Events" 
              if exists process "iTunes" then 
                tell application "iTunes" 
                  if (not (exists user playlist "%(playlist)s")) then 
                    make new playlist with properties {name:"%(playlist)s"}
                  end if
                  add "%(song)s" to playlist "%(playlist)s" 
                end tell 
              else 
                tell application "iTunes" 
                  set visible of front window to false 
                    if (not (exists user playlist "%(playlist)s")) then 
                      make new playlist with properties {name:"%(playlist)s"} 
                    end if
                    add "%(song)s" to playlist "%(playlist)s" 
                end tell 
              end if 
            end tell
            """ % {
                'song': song, 
                'playlist': playlistname,
                }
        self.execute_applescript_darwin(script)

    def sync_darwin(self): 
        """Synchronise changes to the media player."""
        return # the script is lacking some parameters...
        
        # Can anyone fix this script? 
        # Let me know if you really need the parameters. - gtk
        script = """
            tell application "iTunes"
              repeat with i from 1 to the count of source
                if the kid of source i is iPod then
                  set thePod to the name of source i as text
                end if
              end repeat
              copy (get a reference to (every track in "%(playlist)s")) to theTracks
              duplicate theTracks to playlist "%(playlist)s" in source thePod
            end tell
            """ % {
                'playlist': "no idea what to put here"
                }
        self.execute_applescript_darwin(script)  

    # def get_rating_darwin(self, filename, playlistname)
    # def playlist_filenames_darwin(self, playlistname)
    # def play_file_darwin(self, filename)

    def invoke_windows(self): 
        return self.__win32com.client.Dispatch(self.iface)
        
    def init_windows(self): 
        """Initialise the Player object for operation on Darwin."""
        self.iface = iface = "iTunes.Application"
        try: 
            import win32com.client
            self.__win32com = win32com
            self.invoke_windows()
        except ImportError, ex: 
            raise CannotInvoke, "Can't import win32com.client"
        except win32com.client.pythoncom.error, ex:  
            raise CannotInvoke, "Can't dispatch %s" % iface

    def post_track_add_windows(self, filename, playlistname, addresult, playnow=True): 
        """Do some post-add work."""
        if hasattr(addresult, 'Tracks'): 
            log.debug("addresult has Tracks!")
            try: 
                for idx in range(1, addresult.Tracks.Count+1): 
                    track = addresult.Tracks.Item(idx)
                    track.Grouping = 'Podcast'
            except: 
                log.exception("Failure trying to set grouping of added tracks.")
        else: 
            log.debug("addresult has no Tracks attribute")
        if playnow: 
            self.play_file_windows(filename)
        return
    
    def append_and_create_windows(self, filename, playlistname, playnow=True): 
        """Add the tune to the playlist. Create the list if necessary.
        Don't add the tune if it's there already."""
        playlistname = sanitize(playlistname, self.safeplaylistchars)
        log.debug("Ensuring playlist %s has file %s", playlistname, filename)
        try: 
            plitems = self.mine_playlist_windows(playlistname)
        except KeyError: 
            log.info("Creating new playlist %s", playlistname)
            iTunes = self.invoke_windows()
            iTunesPlaylist = iTunes.CreatePlaylist(playlistname)
            res = iTunesPlaylist.AddFile(filename)
            return self.post_track_add_windows(filename, playlistname, res, playnow)

        for location, item in plitems: 
            if location == filename:
                break
        else: 
            log.debug("Updating existing playlist %s", playlistname)
            iTunesPlaylist = self.get_playlist_windows(playlistname)
            res = iTunesPlaylist.AddFile(filename)
            return self.post_track_add_windows(filename, playlistname, res, playnow)
        
    def sync_windows(self):
        """Synchronise changes to the media player."""
        iTunes = self.invoke_windows()
        iTunes.UpdateIPod()
        log.debug("Asked iTunes to update the iPod.")
        # I'm not catching exceptions until they're reported. - gtk

    def get_playlist_windows(self, playlistname): 
        playlistname = sanitize(playlistname, self.safeplaylistchars)
        iTunes = self.invoke_windows()
        iTunesPlaylists = iTunes.LibrarySource.Playlists
        iTunesPlaylist = iTunesPlaylists.ItemByName(playlistname)
        if not iTunesPlaylist:
            raise KeyError
        return iTunesPlaylist

    def mine_playlist_windows(self, playlistname): 
        """Return a list of (filename, ob) pairs or raise KeyError."""
        iTunesPlaylist = self.get_playlist_windows(playlistname)

        res = []
        for j in range(1,len(iTunesPlaylist.Tracks)+1):
            item = iTunesPlaylist.Tracks.Item(j)
            res.append((item.Location, item))
        return res

    def get_rating_windows(self, filename, playlistname): 
        """Get the rating (0-100) for a particular entry in a 
        particular playlist (Windows version)."""
        # get the playlist; might raise KeyError
        plitems = self.mine_playlist_windows(playlistname)
        plitems = [item for location, item in plitems
                   if location == filename]
        count = len(plitems)
        if count < 0: 
            raise KeyError
        if count > 1: 
            log.warn("Playlist %s has %d copies of %s!", 
                     playlistname, count, filename)
        return plitems[0].Rating

    def playlist_filenames_windows(self, playlistname): 
        """Return a list of files referred to by a particular 
        playlist."""
        plitems = self.mine_playlist_windows(playlistname)
        return [location for location, item in plitems]

    def play_file_windows(self, filename, rude=False): 
        """Play a file."""
        iTunes = self.invoke_windows()
        if iTunes.PlayerState > 0: 
            playing = True
        else: 
            playing = False
        if not playing or rude: 
            # This adds the file to the library, which is a little rude if 
            # we're about to delete it. :) 
            log.debug("Asking iTunes to play...")
            iTunes.PlayFile(filename)

class WindowsMedia(Player): 
    def __init__(self): 
        """Initialise the Windows Media player interface."""
        self.iface = iface = "WMPlayer.OCX.7"
        try: 
            import win32com.client
            win32com.client.Dispatch(iface)
            self.__win32com = win32com
        except ImportError, ex: 
            raise CannotInvoke, "Can't import win32com.client"
        except win32com.client.pythoncom.error, ex:  
            raise CannotInvoke, "Can't dispatch %s" % iface
        self.safeplaylistchars = " 0123456789abcdefghijklmnopqrstuvwxyz"

    def append_and_create(self, filename, playlistname, playnow=True): 
        """Add the tune to the playlist. Create the list if necessary.
        Don't add the tune if it's there already."""
        playlistname = sanitize(playlistname, self.safeplaylistchars)
        log.debug("Ensuring playlist %s has file %s", playlistname, filename)
        file = filename
        file.replace("\\", "\\\\")
        try:
            wmp = self.__win32com.client.Dispatch(self.iface)
            try:
                playlistArray = wmp.playlistCollection.getByName(playlistname)
                if playlistArray.count == 0:
                    play_l = wmp.playlistCollection.newPlaylist(playlistname)
                else:
                    play_l = playlistArray.item(0)
                file_to_add = wmp.newMedia(file)
                play_l.appendItem(file_to_add)
                wmp.close()
            except:
                log.exception("Couldn't tell iTunes for Windows to update its playlist.")
                wmp.close()
        except:
            log.exception("Couldn't dispatch iTunes for Windows.")

    def sync(self): 
        """Synchronise changes to the media player."""
        pass # Not necessary

def sanitize(string, safechars): 
    """Sanitize the string according to the characters in `safe`."""
    # First, get the function's cache dict. 
    try: 
        safecache = sanitize.safecache
    except AttributeError: 
        safecache = sanitize.safecache = {}
    # Next, fetch or set a dict version of the safe string. 
    safehash = safecache.get(safechars)
    if safehash is None: 
        safehash = safecache[safechars] = {}
        for char in safechars: 
            safehash[char.lower()] = 1
    # Finally, sanitize the string.
    reslist = []
    for char in string: 
        lower = char.lower()
        if safehash.get(lower, 0): 
            reslist.append(char)
    return ''.join(reslist)
    
player_classes = [iTunes, WindowsMedia]

def all_player_types(): 
    """Return a list of all defined player classes, workable or not."""
    return [pclass.__name__ for pclass in player_classes]

def player_types(): 
    """Return a list of invokable player classes for this system."""
    valid = []
    log.debug("Looking for invokable players...")
    for pclass in player_classes: 
        name = pclass.__name__
        try: 
            player = pclass()
            log.debug("Successfully invoked player %s.", name)
            valid.append(name)
        except CannotInvoke: 
            log.debug("Can't invoke %s.", name)
    return valid

def get(name): 
    """Get a player object by class name. Returns None if it can't 
    be invoked. Raises KeyError if it doesn't exist."""
    matches = [pclass for pclass in player_classes 
               if pclass.__name__.lower() == name.lower()]
    assert len(matches) <= 1
    if not matches: 
        log.debug("Couldn't locate requested player class %s", name)
        raise KeyError
    pclass = matches[0]
    try: 
        return pclass()
    except CannotInvoke, ex: 
        log.debug("Couldn't invoke requested player class %s", name)
        return None

if __name__ == '__main__': 
    # test code
    import conlogging
    logging.basicConfig()
    handler = logging.StreamHandler()
    handler.formatter = conlogging.ConsoleFormatter("%(message)s", wrap=False)
    log.addHandler(handler)
    log.propagate = 0
    log.setLevel(logging.DEBUG)
    log.info("Defined player classes: %s", ', '.join(all_player_types()))
    log.info("Invokable player classes: %s", ', '.join(player_types()))

    # edj: debugging session leftovers
    #it = iTunes()
    #it.append_and_create_darwin("/", "testje")

    player = get(player_types()[0])
    
    
