// --------------------------------------------------------------------
// Creating Postscript output
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeimage.h"
#include "ipetext.h"
#include "ipevisitor.h"
#include "ipepainter.h"
#include "ipegroup.h"
#include "iperef.h"
#include "ipeutils.h"
#include "ipedoc.h"

#include "ipepswriter.h"
#include "ipefontpool.h"

static const char hexChar[17] = "0123456789abcdef";

// --------------------------------------------------------------------

IpePsPainter::IpePsPainter(const IpeStyleSheet *style, IpeStream &stream)
  : IpePdfPainter(style, stream)
{
  // Postscript has only one current color: we use iStroke for that
  iImageNumber = 1;
}

IpePsPainter::~IpePsPainter()
{
  //
}

void IpePsPainter::StrokePath()
{
  State &s = iState.back();
  State &sa = iActiveState.back();
  const IpeRepository *rep = StyleSheet()->Repository();
  if (s.iDashStyle && !s.iDashStyle.IsVoid()
      && s.iDashStyle != sa.iDashStyle) {
    sa.iDashStyle = s.iDashStyle;
    if (s.iDashStyle.IsSolid())
      iStream << "[] 0 d ";
    else
      iStream << rep->String(s.iDashStyle) << " d ";
  }
  if (s.iLineWidth && s.iLineWidth != sa.iLineWidth) {
    sa.iLineWidth = s.iLineWidth;
    iStream << rep->String(s.iLineWidth) << " w ";
  }
  IpeAttribute cap = s.iLineCap;
  if (!cap)
    cap = StyleSheet()->LineCap();
  if (cap != sa.iLineCap) {
    sa.iLineCap = cap;
    iStream << int(cap.Index()) << " J\n";
  }
  IpeAttribute join = s.iLineJoin;
  if (!join)
    join = StyleSheet()->LineJoin();
  if (join != sa.iLineJoin) {
    sa.iLineJoin = join;
    iStream << int(join.Index()) << " j\n";
  }
  if (s.iStroke != sa.iStroke) {
    sa.iStroke = s.iStroke;
    DrawColor(iStream, s.iStroke, "g", "rg");
  }
  iStream << "S";
}

void IpePsPainter::FillPath(bool eoFill, bool preservePath)
{
  State &s = iState.back();
  State &sa = iActiveState.back();
  if (s.iFill != sa.iStroke) {
    sa.iStroke = s.iFill;
    DrawColor(iStream, s.iFill, "g", "rg");
  }
  if (preservePath)
    iStream << "q ";
  iStream << (eoFill ? "f*" : "f");
  if (preservePath)
    iStream << " Q ";
}

void IpePsPainter::DoDrawPath()
{
  bool noStroke = Stroke().IsNull() || DashStyle().IsVoid();
  bool noFill = Fill().IsNullOrVoid();
  IpeAttribute w = WindRule();
  if (!w)
    w = StyleSheet()->WindRule();
  bool eoFill = !w.Index();
  if (noStroke && noFill) {
    iStream << "np";  // flush path
  } else if (noStroke) {
    FillPath(eoFill, false);
  } else if (noFill) {
    StrokePath();
  } else {
    FillPath(eoFill, true);
    StrokePath();
  }
  iStream << "\n";
}

void IpePsPainter::DoDrawBitmap(IpeBitmap bitmap)
{
  switch (bitmap.ColorSpace()) {
  case IpeBitmap::EDeviceGray:
    iStream << "/DeviceGray setcolorspace ";
    break;
  case IpeBitmap::EDeviceRGB:
    iStream << "/DeviceRGB setcolorspace ";
    break;
  case IpeBitmap::EDeviceCMYK:
    iStream << "/DeviceCMYK setcolorspace ";
    break;
  }
  iStream << Matrix() << " cm\n";
  iStream << "<< /ImageType 1\n";
  iStream << "   /Width " << bitmap.Width() << "\n";
  iStream << "   /Height " << bitmap.Height() << "\n";
  iStream << "   /BitsPerComponent " << bitmap.BitsPerComponent() << "\n";
  iStream << "   /Decode [ ";
  for (int i = 0; i < bitmap.Components(); ++i)
    iStream << "0 1 ";
  iStream << "]\n";
  iStream << "   /ImageMatrix [ " << bitmap.Width() << " 0 0 "
	  << -bitmap.Height() << " 0 " << bitmap.Height() << " ]\n";
  iStream << "   /DataSource currentfile /ASCII85Decode filter";
  if (bitmap.Filter() == IpeBitmap::EFlateDecode)
    iStream << " /FlateDecode filter\n";
  else if (bitmap.Filter() == IpeBitmap::EDCTDecode)
    iStream << " /DCTDecode filter\n";
  else
    iStream << "\n";
  iStream << ">>\n";
  iStream << "%%BeginIpeImage: " << iImageNumber << " "
	  << bitmap.Size() << "\n";
  bitmap.SetObjNum(iImageNumber++);
  iStream << "image\n";
  const char *p = bitmap.Data();
  const char *p1 = p + bitmap.Size();
  IpeA85Stream a85(iStream);
  while (p < p1)
    a85.PutChar(*p++);
  a85.Close();
  iStream << "%%EndIpeImage\n";
}

// --------------------------------------------------------------------

/*! \class IpePsWriter
  \brief Create Postscript file.

  This class is responsible for the creation of a Postscript file from the
  Ipe data. You have to create an IpePsWriter first, providing a stream
  that has been opened for (binary) writing and is empty.

*/

//! Create Postscript writer operating on this (open and empty) file.
IpePsWriter::IpePsWriter(IpeTellStream &stream, const IpeDocument *doc,
			 bool noColor)
  : iStream(stream), iDoc(doc), iNoColor(noColor)
{
  // nothing
}

// --------------------------------------------------------------------

//! Create the document header and prolog (the resources).
/*! Embeds no fonts if \c pool is 0.
  Returns false if a TrueType font is present. */
bool IpePsWriter::CreateHeader(IpeString creator, int pno, int view)
{
  const IpeFontPool *pool = iDoc->FontPool();
  const IpeDocument::SProperties &props = iDoc->Properties();

  // Check needed and supplied fonts, and reject Truetype fonts
  IpeString needed = "";
  IpeString neededSep = "%%DocumentNeededResources: font ";
  IpeString supplied = "";
  IpeString suppliedSep = "%%DocumentSuppliedResources: font ";
  if (pool) {
    for (IpeFontPool::const_iterator font = pool->begin();
	 font != pool->end(); ++font) {
      if (font->iType == IpeFont::ETrueType)
	return false;
      if (font->iStandardFont) {
	needed += neededSep;
	needed += font->iName;
	needed += "\n";
	neededSep = "%%+ font ";
      } else {
	supplied += suppliedSep;
	supplied += font->iName;
	supplied += "\n";
	suppliedSep = "%%+ font ";
      }
    }
  }

  iStream << "%!PS-Adobe-3.0 EPSF-3.0\n";

  iStream << "%%Creator: Ipelib " << IPELIB_VERSION;
  if (!creator.empty())
    iStream << " (" << creator << ")";
  iStream << "\n";
  // CreationDate is informational, no fixed format
  iStream << "%%CreationDate: " << props.iModified << "\n";
  // Should use level 1 if no images present?
  iStream << "%%LanguageLevel: 2\n";

  const IpePage *page = iDoc->page(pno);
  std::vector<bool> layers;
  page->MakeLayerTable(layers, view, false);
  IpeBBoxPainter painter(iDoc->StyleSheet());
  for (IpePage::const_iterator it = page->begin(); it != page->end(); ++it) {
    if (layers[it->Layer()])
      it->Object()->Draw(painter);
  }
  IpeRect r = painter.BBox();
  iStream << "%%BoundingBox: " // (x0, y0) (x1, y1)
	  << int(r.Min().iX) << " " << int(r.Min().iY) << " "
	  << int(r.Max().iX + 1) << " " << int(r.Max().iY + 1) << "\n";
  iStream << "%%HiResBoundingBox: " << r.Min() << " " << r.Max() << "\n";

  iStream << needed;
  iStream << supplied;
  if (props.iAuthor.size())
    iStream << "%%Author: " << props.iAuthor << "\n";
  if (props.iTitle.size())
    iStream << "%%Title: " << props.iTitle << "\n";
  iStream << "%%EndComments\n";
  iStream << "%%BeginProlog\n";
  // procset <name> <version (a real)> <revision (an integer)>
  // <revision> is upwards compatible, <version> not
  iStream << "%%BeginResource: procset ipe 6.0 " << IPELIB_VERSION << "\n";
  iStream << "/ipe 40 dict def ipe begin\n";
  iStream << "/np { newpath } def\n";
  iStream << "/m { moveto } def\n";
  iStream << "/l { lineto } def\n";
  iStream << "/c { curveto } def\n";
  iStream << "/h { closepath } def\n";
  iStream << "/re { 4 2 roll moveto 1 index 0 rlineto 0 exch rlineto\n";
  iStream << "      neg 0 rlineto closepath } def\n";
  iStream << "/d { setdash } def\n";
  iStream << "/w { setlinewidth } def\n";
  iStream << "/J { setlinecap } def\n";
  iStream << "/j { setlinejoin } def\n";
  iStream << "/cm { [ 7 1 roll ] concat } def\n";
  iStream << "/q { gsave } def\n";
  iStream << "/Q { grestore } def\n";
  iStream << "/g { setgray } def\n";
  iStream << "/G { setgray } def\n";  // used inside text objects
  if (!iNoColor) {
    iStream << "/rg { setrgbcolor } def\n";
    iStream << "/RG { setrgbcolor } def\n"; // used inside text objects
  }
  iStream << "/S { stroke } def\n";
  iStream << "/f* { eofill } def\n";
  iStream << "/f { fill } def\n";
  iStream << "/ipeMakeFont {\n";
  iStream << "  exch findfont\n";
  iStream << "  dup length dict begin\n";
  iStream << "    { 1 index /FID ne { def } { pop pop } ifelse } forall\n";
  iStream << "    /Encoding exch def\n";
  iStream << "    currentdict\n";
  iStream << "  end\n";
  iStream << "  definefont pop\n";
  iStream << "} def\n";
  iStream << "/ipeFontSize 0 def\n";
  iStream << "/Tf { dup /ipeFontSize exch store selectfont } def\n";
  iStream << "/Td { translate } def\n";
  iStream << "/BT { gsave } def\n";
  iStream << "/ET { grestore } def\n";
  iStream << "/TJ { 0 0 moveto { dup type /stringtype eq\n";
  iStream << " { show } { ipeFontSize mul -0.001 mul 0 rmoveto } ifelse\n";
  iStream << "} forall } def\n";
  iStream << "end\n";
  iStream << "%%EndResource\n";

  iStream << "%%EndProlog\n";
  iStream << "%%BeginSetup\n";
  iStream << "ipe begin\n";

  if (pool) {
    for (IpeFontPool::const_iterator font = pool->begin();
	 font != pool->end(); ++font) {

      if (font->iStandardFont) {
	iStream << "%%IncludeResource: font " << font->iName << "\n";
      } else {
	iStream << "%%BeginResource: font " << font->iName << "\n";
	EmbedFont(*font);
	iStream << "%%EndResource\n";
      }

      // Create font dictionary
      iStream << "/F" << font->iLatexNumber
	      << " /" << font->iName;
      IpeString sep = "\n[ ";
      for (int i = 0; i < 0x100; ++i) {
	if (i % 8 == 0) {
	  iStream << sep;
	  sep = "\n  ";
	}
	iStream << "/" << font->iEncoding[i];
      }
      iStream << " ]\nipeMakeFont\n";
    }
  }

  iStream << "%%EndSetup\n";
  return true;
}

//! Destructor.
IpePsWriter::~IpePsWriter()
{
  // nothing
}

//! Write all fonts to the Postscript file.
void IpePsWriter::EmbedFont(const IpeFont &font)
{
  // write unencrypted front matter
  iStream.PutRaw(font.iStreamData.data(), font.iLength1);
  // write encrypted section
  const char *p = font.iStreamData.data() + font.iLength1;
  const char *p1 = font.iStreamData.data() + font.iLength1 + font.iLength2;
  int i = 0;
  while (p < p1) {
    iStream.PutChar(hexChar[(*p >> 4) & 0x0f]);
    iStream.PutChar(hexChar[*p & 0x0f]);
    ++p;
    if (++i % 32 == 0)
      iStream << "\n";
  }
  if (i % 32 > 0)
    iStream << "\n";
  // write tail matter
  iStream.PutRaw(p1, font.iLength3);
}

// --------------------------------------------------------------------

//! Create contents and page stream for this page view.
void IpePsWriter::CreatePageView(int pno, int vno)
{
  const IpePage *page = iDoc->page(pno);
  // Create page stream
  IpePsPainter painter(iDoc->StyleSheet(), iStream);
  std::vector<bool> layers;
  page->MakeLayerTable(layers, vno, false);
  for (IpePage::const_iterator it = page->begin(); it != page->end(); ++it) {
    if (layers[it->Layer()])
      it->Object()->Draw(painter);
  }
  iStream << "showpage\n";
}

//! Create the trailer of the Postscsript file.
void IpePsWriter::CreateTrailer()
{
  iStream << "%%Trailer\n";
  iStream << "end\n"; // to end ipe dictionary
  iStream << "%%EOF\n";
}

// --------------------------------------------------------------------

class IpePercentStream : public IpeStream {
public:
  IpePercentStream(IpeStream &stream)
    : iStream(stream), iNeedPercent(true) { /* nothing */ }
  virtual void PutChar(char ch);

private:
  IpeStream &iStream;
  bool iNeedPercent;
};

void IpePercentStream::PutChar(char ch)
{
  if (iNeedPercent)
    iStream.PutChar('%');
  iStream.PutChar(ch);
  iNeedPercent = (ch == '\n');
}

//! Save Ipe information in XML format.
void IpePsWriter::CreateXml(IpeString creator, int compressLevel)
{
  IpePercentStream s1(iStream);
  if (compressLevel > 0) {
    iStream << "%%BeginIpeXml: /FlateDecode\n";
    IpeA85Stream a85(s1);
    IpeDeflateStream s2(a85, compressLevel);
    iDoc->SaveAsXml(s2, creator, true);
    s2.Close();
  } else {
    iStream << "%%BeginIpeXml:\n";
    iDoc->SaveAsXml(s1, creator, true);
    s1.Close();
  }
  iStream << "%%EndIpeXml\n";
}

// --------------------------------------------------------------------

