// --------------------------------------------------------------------
// Overlay for creating IpeArc objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipecreatearc.h"
#include "ipecanvas.h"

#include <qevent.h>
#include <qpainter.h>
//Added by qt3to4:
#include <QMouseEvent>
#include <QPaintEvent>


// --------------------------------------------------------------------

/*! \class CreateCircle
  \brief Overlay for creating IpeArc objects.
*/

//! Constructor starts selection.
IpeCreateCircle::IpeCreateCircle(QMouseEvent *, IpeCanvas *canvas,
				 IpeOverlayServices *services, TMode mode)
  : IpeOverlay(canvas), iServices(services), iMode(mode)
{
  iV[0] = iCanvas->Pos();
  iV[1] = iV[0];
  iCur = 1;
}

IpePath IpeCreateCircle::Create() const
{
  switch (iMode) {
  case ECenterRadius:
    return IpePath(iServices->OvSvcAttributes(),
		   iV[0], (iV[1] - iV[0]).Len());
  case EDiameter:
    return IpePath(iServices->OvSvcAttributes(),
		   0.5 * (iV[0] + iV[1]), 0.5 * (iV[1] - iV[0]).Len());
  case E3Points:
  default:
    if (iCur == 1 || iV[1] == iV[2] ) {
      return IpePath(iServices->OvSvcAttributes(),
		     0.5 * (iV[0] + iV[1]), 0.5 * (iV[1] - iV[0]).Len());
    } else {
      IpeLine l1(iV[0], (iV[1] - iV[0]).Normalized());
      if (IpeAbs(l1.Side(iV[2])) < 1e-9) {
	// Argh. We really need an eps here!
	return IpePath(iServices->OvSvcAttributes(),
		       IpeSegment(iV[0], iV[2]));
      } else {
	IpeLine l2(iV[1], (iV[2] - iV[1]).Normalized());
	IpeLine bi1(0.5 * (iV[0] + iV[1]), l1.Normal().Normalized());
	IpeLine bi2(0.5 * (iV[1] + iV[2]), l2.Normal().Normalized());
	IpeVector center;
	bi1.Intersects(bi2, center);
	return IpePath(iServices->OvSvcAttributes(), center,
		       (iV[0] - center).Len());
      }
    }
  }
}

void IpeCreateCircle::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(CreateColor());
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  painter.Push();
  Create().Draw(painter);
  painter.Pop();
}

void IpeCreateCircle::MouseMove(QMouseEvent *)
{
  iV[iCur] = iCanvas->Pos();
  iCanvas->UpdateOverlay();
}

void IpeCreateCircle::MousePress(QMouseEvent *)
{
  // refuse point identical to previous
  if (iCanvas->Pos() == iV[iCur - 1])
    return;

  iV[iCur++] = iCanvas->Pos();
  if (iCur == 3 || iMode != E3Points && iCur == 2) {
    // finished
    IpePath *obj = new IpePath(Create());
    iServices->OvSvcAddObject(obj);
    iCanvas->FinishOverlay();
  } else {
    iV[iCur] = iCanvas->Pos();
    iCanvas->UpdateOverlay();
  }
}

// --------------------------------------------------------------------

/*! \class IpeCreateArc
  \brief Overlay for creating IpeArc objects.
*/

//! Constructor starts selection.
IpeCreateArc::IpeCreateArc(QMouseEvent *, IpeCanvas *canvas,
			   IpeOverlayServices *services, TMode mode)
  : IpeOverlay(canvas), iServices(services), iMode(mode)
{
  iV[0] = iCanvas->Pos();
  iV[1] = iV[0];
  iCur = 1;
}

IpePath IpeCreateArc::Create() const
{
  if (iCur == 1) {
    if (iMode == E3Points)
      return IpePath(iServices->OvSvcAttributes(),
		     0.5 * (iV[0] + iV[1]), 0.5 * (iV[1] - iV[0]).Len());
    else
      return IpePath(iServices->OvSvcAttributes(),
		     iV[0], (iV[1] - iV[0]).Len());
  }
  IpeAngle alpha = (iV[1]-iV[0]).Angle().Normalize(0.0);
  switch (iMode) {
  case ECenterRightLeft:
    return IpePath(iServices->OvSvcAttributes(),
		   iV[0], (iV[1] - iV[0]).Len(),
		   alpha,
		   (iV[2]-iV[0]).Angle().Normalize(alpha));

  case ECenterLeftRight:
    return IpePath(iServices->OvSvcAttributes(),
		  iV[0], (iV[1] - iV[0]).Len(),
		   alpha,
		   (iV[2]-iV[0]).Angle().Normalize(alpha - IpeTwoPi));
  case E3Points:
  default: // make compiler happy
    {
      IpeLine l1(iV[0], (iV[1] - iV[0]).Normalized());
      // Argh. We really need an eps here!
      if (IpeAbs(l1.Side(iV[2])) < 1e-9 || iV[2] == iV[1]) {
	return IpePath(iServices->OvSvcAttributes(),
		       IpeSegment(iV[0], iV[2]));
      }
      IpeLine l2(iV[1], (iV[2] - iV[1]).Normalized());
      IpeLine bi1(0.5 * (iV[0] + iV[1]), l1.Normal());
      IpeLine bi2(0.5 * (iV[1] + iV[2]), l2.Normal());
      IpeVector center;
      bi1.Intersects(bi2, center);
      // DBG << l1.Side(iV[2]) << "\n";
      // DBG << " center: " << center << "\n";
      alpha = (iV[0]-center).Angle();
      double beta = (iV[1]-center).Angle().Normalize(alpha);
      IpeAngle gamma = (iV[2]-center).Angle().Normalize(alpha);
      if (gamma < beta)
	gamma.Normalize(alpha - IpeTwoPi);
      return IpePath(iServices->OvSvcAttributes(), center,
		     (iV[0] - center).Len(),
		     alpha, gamma);
    }
  }
}

void IpeCreateArc::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(CreateColor());
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  painter.Push();
  if (iMode == E3Points && iCur == 1) {
    painter.NewPath();
    painter.MoveTo(iV[0]);
    painter.LineTo(iV[1]);
    painter.DrawPath();
  } else
    Create().Draw(painter);
  painter.Pop();
}

void IpeCreateArc::MouseMove(QMouseEvent *)
{
  iV[iCur] = iCanvas->Pos();
  iCanvas->UpdateOverlay();
}

void IpeCreateArc::MousePress(QMouseEvent *)
{
  // refuse two identical points
  if (iCanvas->Pos() == iV[iCur-1])
    return;

  iV[iCur++] = iCanvas->Pos();
  if (iCur == 3) {
    IpePath *obj = new IpePath(Create());
    iServices->OvSvcAddObject(obj);
    iCanvas->FinishOverlay();
  } else {
    iV[iCur] = iCanvas->Pos();
    iCanvas->UpdateOverlay();
  }
}

// --------------------------------------------------------------------
