// --------------------------------------------------------------------
// Constructor for AppUi
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipedoc.h"
#include "ipeq.h"

#include "appui.h"
#include "widgets.h"

extern QPixmap ipeIcon(const char* name);

// --------------------------------------------------------------------

const char * fileNewText = QT_TR_NOOP(
"<qt>This button creates a new document. "
"You can also select the <b>New</b> command from the File menu.</qt>");
const char * fileOpenText = QT_TR_NOOP(
"<qt>This button opens a document from a file. "
"You can also select the <b>Open</b> command from the File menu.</qt>");
const char * fileSaveText = QT_TR_NOOP(
"<qt>This button saves the current document. If it is a new document, "
"you will be prompted for a file name. "
"You can also select the <b>Save</b> command from the File menu.</qt>");
const char * cutcopyText = QT_TR_NOOP(
"<qt>Cut, copy , and paste "
"pass Ipe objects to and from the system clipboard. "
"When pasting objects, they are inserted into the active layer. "
"You can examine Ipe objects by copying them to the clipboard, and "
"pasting into your favorite text editor.</qt>");
const char * strokeColorText = QT_TR_NOOP(
"<qt>Select the <b>stroke color</b> here."
"<p>The stroke color is used to outline the shape of objects, to draw "
"lines, curves, arcs, marks, and for text.  To draw filled objects without "
"an outline, set the stroke color to <tt>void</tt>.</p></qt>");
const char * fillColorText = QT_TR_NOOP(
"<qt>Select the <b>fill color</b> here."
"<p>The fill color is used for the interior of objects such as polygons or "
"circles.  It is not used for text or marks.  To draw unfilled objects, "
"that is, polygons or circles with an outline only, set the fill "
"color to <tt>void</tt>.</p></qt>");
const char * resolutionText = QT_TR_NOOP(
"<qt>You control the resolution of the screen display here. "
"The number indicates the number of screen pixels being used to "
"display one inch in the document."
"<p>By setting this to the actual screen resolution (that is, "
"the number of pixels per inch of your monitor), you can view a "
"document in its original size "
"(the size it will have when printed).</p></qt>");
const char * pageNumberText = QT_TR_NOOP(
"<qt>The current page number.</qt>");
const char * viewNumberText = QT_TR_NOOP(
"<qt>Each page of an Ipe document can consist of several <em>views</em>. "
"Each view becomes one PDF/Postscript page when saving in that format. "
"A view is defined by selecting some layers of the page that will be "
"presented together.</qt>");
const char * lineWidthText = QT_TR_NOOP(
"<qt>This setting determines the width of lines you create. "
"If you change the setting while you have objects "
"selected, the line width of the selected objects will be changed "
"accordingly.</qt>");
const char * lineDashText = QT_TR_NOOP(
"<qt>This setting determines the dash-dot pattern of lines that you create. "
"If you change the setting while you have objects "
"selected, the dash-dot pattern of the selected objects will be changed "
"accordingly.</qt>");
const char * textSizeText = QT_TR_NOOP(
"<qt>This setting determines the font size of text objects that you create. "
"If you change the setting while you have text objects "
"selected, their font size will be changed "
"accordingly.</qt>");
const char * arrowText = QT_TR_NOOP(
"<qt>This setting determines the arrow heads on polyline, spline, or arc "
"objects that you create.  If you change the setting while you have objects "
"selected, the arrows on the selected objects will be changed "
"accordingly.</qt>");
const char * arrowSizeText = QT_TR_NOOP(
"<qt>This setting determines the size of arrow heads for objects that you "
"create.  If you change the setting while you have objects selected, "
"the arrow heads on the selected objects will be resized.</qt>");
const char * markShapeText = QT_TR_NOOP(
"<qt>This setting determines the shape of mark objects that you "
"create. If you change the setting while you have mark objects selected, "
"their shape will be changed.</qt>");
const char * markSizeText = QT_TR_NOOP(
"<qt>This setting determines the size of mark objects that you "
"create.  If you change the setting while you have mark objects selected, "
"the marks will be resized.</qt>");
const char * gridSizeText = QT_TR_NOOP(
"<qt>This setting determines the distance between grid points in "
"the snapping grid.</qt");
const char * angleSizeText = QT_TR_NOOP(
"<qt>This setting determines the angle between consecutive lines "
"in the angular snapping mode.</qt>");
const char * layerListText = QT_TR_NOOP(
"<qt>This list shows the layers of the current page. "
"<p>The selected layers are in the current <em>view</em>. "
"When you create new objects, they are inserted into the <em>active layer</em>, "
"indicated by the angular brackets. "
"<p>The letters H, D, L, S stand for <em>hidden</em>, <em>dimmed</em>, "
"<em>locked</em>, and <em>no snapping</em>.</p>"
"</qt>");
const char * penguinText = QT_TR_NOOP(
"<qt>This is a picture of penguins."
"<p>It is not related to Tux, the Linux-penguin. "
"In fact, Ipe has used a penguin icon since 1993, "
"long before I had ever heard of Linux.</p></qt>");
const char * snapText = QT_TR_NOOP (
"<qt>These buttons select the various snapping modes.  When the button "
"is pressed in, the mode is on, otherwise it is off. From left to right, "
"the buttons enable the following modes:"
"<ul><li>Snapping to object vertices,"
"<li>Snapping to path boundaries,"
"<li>Snapping to intersection points (currently only of straight segments),"
"<li>Snapping to the grid,"
"<li>Angular snapping,"
"<li>Automatic angular snapping."
"</ul>See the manual for more details about snapping.</qt>");
const char * canvasText = QT_TR_NOOP(
"<qt>This is the canvas, where you create your drawing."
"<p>Most operations can be done with the left mouse button. Its "
"function depends on the current mode.</p>"
"<p>The right mouse button pops up the object menu (you can change it "
"to select objects in the <em>Preferences</em> dialog - in that case "
"you get the object menu by holding the Control key with the right "
"mouse button).</p>"
"<p>The following mouse shortcuts are available:"
"<ul><li>Ctrl+Left mouse: Stretch"
"<li>Ctrl+Shift+Left mouse: Scale"
"<li>Middle mouse: Move"
"<li>Ctrl+Middle mouse: Rotate"
"<li>Shift+Middle mouse: Pan"
"<li>Shift+Ctrl+Middle mouse: Horizontal/vertical move"
"</ul></p></qt>");

// --------------------------------------------------------------------

/*! \class AppUi
  \brief AppUi encapsulates the Ipe user interface.

*/
/*! The constructor does not create an Ipe document.
  Clients need to call either Load or NewDoc after construction,
  before calling show. */

AppUi::AppUi(QWidget* parent, Qt::WFlags f)
  : QMainWindow(parent, f)
{
  QPixmap newIcon = ipeIcon("new");
  QPixmap openIcon = ipeIcon("open");
  QPixmap saveIcon = ipeIcon("save");
  QPixmap undoIcon = ipeIcon("undo");
  QPixmap redoIcon = ipeIcon("redo");
  QPixmap cutIcon = ipeIcon("cut");
  QPixmap copyIcon = ipeIcon("copy");
  QPixmap pasteIcon = ipeIcon("paste");

  IpePreferences *prefs = IpePreferences::Static();

  // --------------------------------------------------------------------
  // Create menus
  // --------------------------------------------------------------------

  iFileMenu = menuBar()->addMenu(tr("&File"));
  iEditMenu = menuBar()->addMenu(tr("&Edit"));
  iSnapMenu = menuBar()->addMenu(tr("&Snap"));
  iModeMenu = menuBar()->addMenu(tr("&Mode"));
  iZoomMenu = menuBar()->addMenu(tr("&Zoom"));
  iLayerMenu = menuBar()->addMenu(tr("&Layer"));
  iViewMenu = menuBar()->addMenu(tr("&View"));
  iPageMenu = menuBar()->addMenu(tr("&Page"));
  iIpeletMenu = menuBar()->addMenu(tr("&Ipelets"));
  menuBar()->addSeparator();
  iHelpMenu = menuBar()->addMenu(tr("&Help"));

  // --------------------------------------------------------------------
  // Create toolbars
  // --------------------------------------------------------------------

  iFileTools = addToolBar(tr("File"));
  iResolutionTools = addToolBar(tr("Resolution"));
  iPageTools = addToolBar(tr("Page"));
  iSnapTools = addToolBar(tr("Snap"));
  addToolBarBreak();
  iObjectTools = addToolBar(tr("Objects"));

  // --------------------------------------------------------------------
  // Create docking windows
  // --------------------------------------------------------------------

  iPropertiesTools = new QDockWidget(tr("Properties"), this);
  addDockWidget(Qt::LeftDockWidgetArea, iPropertiesTools);
  iPropertiesTools->setAllowedAreas(Qt::LeftDockWidgetArea|
				    Qt::RightDockWidgetArea);

  iLayerTools = new QDockWidget(tr("Layers"), this);
  addDockWidget(Qt::LeftDockWidgetArea, iLayerTools);
  iLayerTools->setAllowedAreas(Qt::LeftDockWidgetArea|
			       Qt::RightDockWidgetArea);

  iBookmarkTools = new QDockWidget(tr("Bookmarks"), this);
  addDockWidget(Qt::RightDockWidgetArea, iBookmarkTools);
  iBookmarkTools->setAllowedAreas(Qt::LeftDockWidgetArea|
				  Qt::RightDockWidgetArea);

  // --------------------------------------------------------------------
  // File toolbar
  // --------------------------------------------------------------------

  QAction *newAct = new QAction(newIcon, tr("&New document"), this);
  newAct->setShortcut(Key("File|New document"));
  // newAct->setStatusTip(tr("Create a new document"));
  newAct->setWhatsThis(tr(fileNewText));
  connect(newAct, SIGNAL(triggered()), this, SLOT(NewDoc()));
  iFileTools->addAction(newAct);

  QAction *openAct = new QAction(openIcon, tr("Open file"), this);
  openAct->setShortcut(Key("File|Open file"));
  openAct->setWhatsThis(tr(fileOpenText));
  connect(openAct, SIGNAL(triggered()), this, SLOT(Load()));
  iFileTools->addAction(openAct);

  QAction *saveAct = new QAction(saveIcon, tr("Save file"), this);
  saveAct->setShortcut(Key("File|Save file"));
  saveAct->setWhatsThis(tr(fileSaveText));
  connect(saveAct, SIGNAL(triggered()), this, SLOT(Save()));
  iFileTools->addAction(saveAct);

  iFileTools->addSeparator();

  iCutAction = new IpeAction(ECmdCut, cutIcon, tr("Cut"), this);
  iCutAction->setShortcut(Key("Edit|Cut"));
  iCutAction->setWhatsThis(tr(cutcopyText));
  connect(iCutAction, SIGNAL(triggered(int)), this, SLOT(Cmd(int)));
  iFileTools->addAction(iCutAction);

  iCopyAction = new IpeAction(ECmdCopy, copyIcon, tr("Copy"), this);
  iCopyAction->setShortcut(Key("Edit|Copy"));
  iCopyAction->setWhatsThis(tr(cutcopyText));
  connect(iCopyAction, SIGNAL(triggered(int)), this, SLOT(Cmd(int)));
  iFileTools->addAction(iCopyAction);

  QAction *pasteAct = new IpeAction(ECmdPaste, pasteIcon, tr("Paste"), this);
  pasteAct->setShortcut(Key("Edit|Paste"));
  pasteAct->setWhatsThis(tr(cutcopyText));
  connect(pasteAct, SIGNAL(triggered(int)), this, SLOT(Cmd(int)));
  iFileTools->addAction(pasteAct);

  iFileTools->addSeparator();

  QAction *whatsThisAct = QWhatsThis::createAction();
  iFileTools->addAction(whatsThisAct);

  iFileTools->hide();

  // --------------------------------------------------------------------
  // Resolution tool bar
  // --------------------------------------------------------------------

  QAction *normalSizeAct = new QAction(tr("&Normal size"), this);
  normalSizeAct->setShortcut(Key("Zoom|Normal size"));
  connect(normalSizeAct, SIGNAL(triggered()), this, SLOT(NormalSize()));

  iResolution = new ZoomSpinBox();
  iResolution->setWhatsThis(tr(resolutionText));
  iResolution->setToolTip(tr("Screen resolution"));
  iResolution->installEventFilter(this);
  connect(iResolution, SIGNAL(valueChanged(int)),
	  this, SLOT(ResolutionChanged(int)));
  iResolutionTools->addWidget(iResolution);

  // --------------------------------------------------------------------
  // Page tool bar
  // --------------------------------------------------------------------

  iPageNumber = new IpeSpinBox();
  iPageNumber->setToolTip(tr("Current page"));
  iPageNumber->setWhatsThis(tr(pageNumberText));
  iPageNumber->setPrefix(tr("Page "));
  connect(iPageNumber, SIGNAL(valueChanged(int)),
	  this, SLOT(PageChanged(int)));
  iPageNumber->installEventFilter(this);

  iViewNumber = new IpeSpinBox();
  iViewNumber->setToolTip(tr("Current view"));
  iViewNumber->setWhatsThis(tr(viewNumberText));
  iViewNumber->setPrefix(tr("View "));
  connect(iViewNumber, SIGNAL(valueChanged(int)),
	  this, SLOT(ViewChanged(int)));
  iViewNumber->installEventFilter(this);

  iPageTools->addWidget(iPageNumber);
  iPageTools->addWidget(iViewNumber);

  // --------------------------------------------------------------------
  // Snap tool bar
  // --------------------------------------------------------------------

  QPixmap snapVtxIcon = ipeIcon("snapvtx");
  QPixmap snapBdIcon = ipeIcon("snapbd");
  QPixmap snapIntIcon = ipeIcon("snapint");
  QPixmap snapGridIcon = ipeIcon("snapgrid");
  QPixmap snapAngleIcon = ipeIcon("snapangle");
  QPixmap snapAutoIcon = ipeIcon("snapauto");

  iSnapActionGroup = new QActionGroup(this);

  iSnapAction[0] = new QAction(iSnapActionGroup);
  iSnapAction[0]->setText(tr("Snap to vertices"));
  iSnapAction[0]->setShortcut(Key("Snap|Vertices"));
  iSnapAction[0]->setIcon(snapVtxIcon);

  iSnapAction[1] = new QAction(iSnapActionGroup);
  iSnapAction[1]->setText(tr("Snap to boundary"));
  iSnapAction[1]->setShortcut(Key("Snap|Boundary"));
  iSnapAction[1]->setIcon(snapBdIcon);

  iSnapAction[2] = new QAction(iSnapActionGroup);
  iSnapAction[2]->setText(tr("Snap to intersection"));
  iSnapAction[2]->setShortcut(Key("Snap|Intersections"));
  iSnapAction[2]->setIcon(snapIntIcon);

  iSnapAction[3] = new QAction(iSnapActionGroup);
  iSnapAction[3]->setText(tr("Snap to grid"));
  iSnapAction[3]->setShortcut(Key("Snap|Grid"));
  iSnapAction[3]->setIcon(snapGridIcon);

  iSnapAction[4] = new QAction(iSnapActionGroup);
  iSnapAction[4]->setText(tr("Angular snap"));
  iSnapAction[4]->setShortcut(Key("Snap|Angular"));
  iSnapAction[4]->setIcon(snapAngleIcon);

  iSnapAction[5] = new QAction(iSnapActionGroup);
  iSnapAction[5]->setText(tr("Automatic angular snap"));
  iSnapAction[5]->setShortcut(Key("Snap|Automatic angular"));
  iSnapAction[5]->setIcon(snapAutoIcon);

  iGridSizeStack = new QStackedWidget(iSnapTools);
  iGridSizeStack->setToolTip(tr("Grid size"));
  iGridSizeStack->setWhatsThis(tr(gridSizeText));
  iGridSize = new QComboBox();
  connect(iGridSize, SIGNAL(activated(int)),
	  SLOT(GridSizeChanged(int)));
  iAbsGridSize = new IpeSpinBox(0, 4, 100, 2);
  connect(iAbsGridSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsGridSizeChanged(int)));
  iGridSize->installEventFilter(this);
  iAbsGridSize->installEventFilter(this);
  iGridSizeStack->addWidget(iGridSize);
  iGridSizeStack->addWidget(iAbsGridSize);

  iAngleSizeStack = new QStackedWidget(iSnapTools);
  iAngleSizeStack->setToolTip(tr("Angular snapping angle"));
  iAngleSizeStack->setWhatsThis(tr(angleSizeText));
  iAngleSize = new QComboBox();
  connect(iAngleSize, SIGNAL(activated(int)),
	  SLOT(AngleSizeChanged(int)));
  iAbsAngleSize = new DecimalSpinBox(3000, 360000, 2000);
  connect(iAbsAngleSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsAngleSizeChanged(int)));
  iAngleSize->installEventFilter(this);
  iAbsAngleSize->installEventFilter(this);
  iAngleSizeStack->addWidget(iAngleSize);
  iAngleSizeStack->addWidget(iAbsAngleSize);

  iSnapTools->addAction(iSnapAction[0]);
  iSnapTools->addAction(iSnapAction[1]);
  iSnapTools->addAction(iSnapAction[2]);
  iSnapTools->addSeparator();
  iSnapTools->addAction(iSnapAction[3]);
  iSnapTools->addWidget(iGridSizeStack);
  iSnapTools->addSeparator();
  iSnapTools->addAction(iSnapAction[4]);
  iSnapTools->addAction(iSnapAction[5]);
  iSnapTools->addWidget(iAngleSizeStack);

  iSnapActionGroup->setExclusive(false);

  for (int i = 0; i < 6; ++i) {
    iSnapAction[i]->setCheckable(true);
    iSnapAction[i]->setWhatsThis(tr(snapText));
    connect(iSnapAction[i], SIGNAL(toggled(bool)),
	    this, SLOT(SnapChanged(bool)));
  }

  // --------------------------------------------------------------------
  // Object creators tool bar
  // --------------------------------------------------------------------

  iModeActionGroup = new QActionGroup(this);
  connect(iModeActionGroup, SIGNAL(triggered(QAction*)),
	  SLOT(SetCreator(QAction*)));

  for (int i = 0; i < IpeOverlayFactory::Num; ++i) {
    QPixmap icon = ipeIcon(IpeOverlayFactory::PixmapName(i));
    iModeAction[i] = new QAction(iModeActionGroup);
    iModeAction[i]->setText(IpeOverlayFactory::Tooltip(i));
    iModeAction[i]->setIcon(icon);
    iModeAction[i]->setWhatsThis(IpeOverlayFactory::WhatsThis(i));
    iModeAction[i]->setShortcut(Key(IpeOverlayFactory::Context(i)));
    iModeAction[i]->setCheckable(true);
    iObjectTools->addAction(iModeAction[i]);
    if (i == 4)
      iObjectTools->addSeparator();
  }
  iModeActionGroup->setExclusive(true);

  // --------------------------------------------------------------------
  // Properties window
  // --------------------------------------------------------------------

  QWidget *prop = new QWidget;
  QGridLayout *layout = new QGridLayout(prop);
  prop->setLayout(layout);

  iStrokeColorStack = new QStackedWidget();
  iStrokeColorStack->setToolTip(tr("Stroke color"));
  iStrokeColorStack->setWhatsThis(tr(strokeColorText));
  iStrokeColor = new QComboBox();
  connect(iStrokeColor, SIGNAL(activated(int)),
	  SLOT(SetStrokeColorName(int)));
  iAbsStrokeColor = new QPushButton();
  // iStrokeColorStack->setMaximumHeight(iStrokeColor->sizeHint().height());
  connect(iAbsStrokeColor, SIGNAL(clicked()),
	  this, SLOT(SetStrokeColor()));
  iStrokeColor->installEventFilter(this);
  iAbsStrokeColor->installEventFilter(this);
  iStrokeColorStack->addWidget(iStrokeColor);
  iStrokeColorStack->addWidget(iAbsStrokeColor);

  iFillColorStack = new QStackedWidget();
  iFillColorStack->setToolTip(tr("Fill color"));
  iFillColorStack->setWhatsThis(tr(fillColorText));
  iFillColor = new QComboBox();
  connect(iFillColor, SIGNAL(activated(int)),
	  SLOT(SetFillColorName(int)));
  iAbsFillColor = new QPushButton();
  // iFillColorStack->setMaximumHeight(iFillColor->sizeHint().height());
  connect(iAbsFillColor, SIGNAL(clicked()),
	  this, SLOT(SetFillColor()));
  iFillColor->installEventFilter(this);
  iAbsFillColor->installEventFilter(this);
  iFillColorStack->addWidget(iFillColor);
  iFillColorStack->addWidget(iAbsFillColor);

  layout->addWidget(iStrokeColorStack, 0, 0);
  layout->addWidget(iFillColorStack, 0, 1);
  layout->setRowMinimumHeight(0, iStrokeColor->sizeHint().height());

  // --------------------------------------------------------------------

  iLineWidthStack = new QStackedWidget();
  iLineWidthStack->setToolTip(tr("Line width"));
  iLineWidthStack->setWhatsThis(tr(lineWidthText));
  iLineWidth = new QComboBox();
  connect(iLineWidth, SIGNAL(activated(int)),
	  SLOT(LineWidthChanged(int)));
  iAbsLineWidth = new DecimalSpinBox(0, 10000, 200);
  connect(iAbsLineWidth, SIGNAL(valueChanged(int)),
	  SLOT(AbsLineWidthChanged(int)));
  iLineWidth->installEventFilter(this);
  iAbsLineWidth->installEventFilter(this);
  iLineWidthStack->addWidget(iLineWidth);
  iLineWidthStack->addWidget(iAbsLineWidth);

  iDashStyle = new QComboBox();
  iDashStyle->setToolTip(tr("Line dash pattern"));
  iDashStyle->setWhatsThis(tr(lineDashText));
  connect(iDashStyle, SIGNAL(activated(int)),
	  SLOT(DashStyleChanged(int)));
  iDashStyle->installEventFilter(this);

  iArrowSizeStack = new QStackedWidget();
  iArrowSizeStack->setToolTip(tr("Arrow size"));
  iArrowSizeStack->setWhatsThis(tr(arrowSizeText));
  iArrowSize = new QComboBox();
  connect(iArrowSize, SIGNAL(activated(int)),
	  SLOT(ArrowSizeChanged(int)));
  iAbsArrowSize = new DecimalSpinBox(200, 100000, 1000);
  connect(iAbsArrowSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsArrowSizeChanged(int)));
  iArrowSize->installEventFilter(this);
  iAbsArrowSize->installEventFilter(this);
  iArrowSizeStack->addWidget(iArrowSize);
  iArrowSizeStack->addWidget(iAbsArrowSize);

  QPixmap arrowIconRight = ipeIcon("arrowRight");
  QPixmap arrowIconLeft = ipeIcon("arrowLeft");
  QPixmap arrowIconMid = ipeIcon("arrowMid");

  iBackwardArrow = new ArrowButton();
  iBackwardArrow->setArrow(arrowIconLeft);
  iBackwardArrow->setToolTip(tr("Backward arrow"));
  iBackwardArrow->setWhatsThis(tr(arrowText));
  iBackwardArrow->setFocusPolicy(Qt::NoFocus);

  QLabel *midArrow = new QLabel();
  midArrow->setPixmap(arrowIconMid);
  midArrow->setAlignment(Qt::AlignCenter);

  iForwardArrow = new ArrowButton();
  iForwardArrow->setArrow(arrowIconRight);
  iForwardArrow->setToolTip(tr("Forward arrow"));
  iForwardArrow->setWhatsThis(tr(arrowText));
  iForwardArrow->setFocusPolicy(Qt::NoFocus);

  connect(iBackwardArrow, SIGNAL(toggled(bool)),
	  this, SLOT(ArrowChanged(bool)));
  connect(iForwardArrow, SIGNAL(toggled(bool)),
	  this, SLOT(ArrowChanged(bool)));

  iAttributes.iForwardArrow = false;
  iAttributes.iBackwardArrow = false;

  QHBoxLayout *arrowLayout = new QHBoxLayout();
  arrowLayout->addStretch(100);
  arrowLayout->addWidget(iBackwardArrow);
  arrowLayout->addWidget(midArrow);
  arrowLayout->addWidget(iForwardArrow);
  arrowLayout->addStretch(100);

  layout->addWidget(iLineWidthStack, 1, 0);
  layout->addWidget(iDashStyle, 1, 1);
  layout->addLayout(arrowLayout, 2, 0);
  layout->addWidget(iArrowSizeStack, 2, 1);

  // --------------------------------------------------------------------

  QPixmap markIcon = ipeIcon("marks");

  iMarkShape = new QComboBox();
  iMarkShape->setToolTip(tr("Mark shape"));
  iMarkShape->setWhatsThis(tr(markShapeText));
  connect(iMarkShape, SIGNAL(activated(int)), SLOT(MarkShapeChanged(int)));
  iMarkShape->addItem(markIcon, tr("Circle"));
  iMarkShape->addItem(markIcon, tr("Disc"));
  iMarkShape->addItem(markIcon, tr("Box"));
  iMarkShape->addItem(markIcon, tr("Square"));
  iMarkShape->addItem(markIcon, tr("Cross"));
  iMarkShape->setCurrentIndex(1);
  iAttributes.iMarkShape = iMarkShape->currentIndex() + 1;
  iMarkShape->installEventFilter(this);

  iMarkSizeStack = new QStackedWidget();
  iMarkSizeStack->setToolTip(tr("Mark size"));
  iMarkSizeStack->setWhatsThis(tr(markSizeText));
  iMarkSize = new QComboBox();
  connect(iMarkSize, SIGNAL(activated(int)),
	  SLOT(MarkSizeChanged(int)));
  iAbsMarkSize = new DecimalSpinBox(200, 100000, 1000);
  connect(iAbsMarkSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsMarkSizeChanged(int)));
  iMarkSize->installEventFilter(this);
  iAbsMarkSize->installEventFilter(this);
  iMarkSizeStack->addWidget(iMarkSize);
  iMarkSizeStack->addWidget(iAbsMarkSize);

  iTextSizeStack = new QStackedWidget();
  iTextSizeStack->setToolTip(tr("Font size"));
  iTextSizeStack->setWhatsThis(tr(textSizeText));
  iTextSize = new QComboBox();
  connect(iTextSize, SIGNAL(activated(int)),
	  SLOT(TextSizeChanged(int)));
  iAbsTextSize = new DecimalSpinBox(4000, 50000, 1000);
  connect(iAbsTextSize, SIGNAL(valueChanged(int)),
	  SLOT(AbsTextSizeChanged(int)));
  iTextSize->installEventFilter(this);
  iAbsTextSize->installEventFilter(this);
  iTextSizeStack->addWidget(iTextSize);
  iTextSizeStack->addWidget(iAbsTextSize);

  layout->addWidget(iMarkShape, 3, 0);
  layout->addWidget(iMarkSizeStack, 3, 1);
  layout->addWidget(iTextSizeStack, 4, 0);

  QToolButton *ab = new QToolButton();
  ab->setFocusPolicy(Qt::NoFocus);
  connect(ab, SIGNAL(clicked()),
	  this, SLOT(ToggleAbsoluteAttributes()));

  QHBoxLayout *abLayout = new QHBoxLayout();
  abLayout->addStretch(100);
  abLayout->addWidget(ab);

  layout->addLayout(abLayout, 4, 1);

  layout->setRowStretch(5, 1);

  iPropertiesTools->setWidget(prop);

  // ------------------------------------------------------------
  // Bookmark and layer windows
  // ------------------------------------------------------------

  iBookmarks = new QListWidget();
  iBookmarks->setToolTip(tr("Bookmarks of this document"));
  iBookmarks->setFocusPolicy(Qt::NoFocus);
  connect(iBookmarks, SIGNAL(itemActivated(QListWidgetItem *)),
	  this, SLOT(BookmarkSelected(QListWidgetItem *)));
  iBookmarkTools->setWidget(iBookmarks);

  iLayerList = new LayerBox();
  iLayerList->setFocusPolicy(Qt::NoFocus);
  iLayerList->setToolTip(tr("Layers of this page"));
  iLayerList->setWhatsThis(tr(layerListText));
  iLayerTools->setWidget(iLayerList);

  // QLabel *penguins = new QLabel();
  // penguins->setPixmap(penguinIcon(iLayerList->sizeHint().width()));
  // penguins->setWhatsThis(tr(penguinText));

  // ------------------------------------------------------------
  // File menu
  // ------------------------------------------------------------

  QAction *newWinAct = new QAction(tr("New &Window"), this);
  newWinAct->setShortcut(Key("File|New Window"));
  connect(newWinAct, SIGNAL(triggered()), SLOT(NewWindow()));

  QAction *saveAsAct = new QAction(tr("Save &as..."), this);
  saveAsAct->setShortcut(Key("File|Save as"));
  saveAsAct->setWhatsThis(tr(fileSaveText));
  connect(saveAsAct, SIGNAL(triggered()), SLOT(SaveAs()));

  QAction *saveAsBitmapAct = new QAction(tr("Save as &bitmap"), this);
  saveAsBitmapAct->setShortcut(Key("File|Save as bitmap"));
  connect(saveAsBitmapAct, SIGNAL(triggered()), SLOT(SaveAsBitmap()));

  QAction *runLatexAct = new QAction(tr("Run &Latex"), this);
  runLatexAct->setShortcut(Key("File|Run Latex"));
  connect(runLatexAct, SIGNAL(triggered()), SLOT(RunLatex()));

  QAction *closeAct = new QAction(tr("&Close"), this);
  closeAct->setShortcut(Key("File|Close"));
  connect(closeAct, SIGNAL(triggered()), SLOT(close()));

  QAction *exitAct = new QAction(tr("E&xit"), this);
  exitAct->setShortcut(Key("File|Exit"));
  connect(exitAct, SIGNAL(triggered()), qApp, SLOT(closeAllWindows()));

  iFileMenu->addAction(newWinAct);
  iFileMenu->addAction(newAct);
  iFileMenu->addAction(openAct);
  iFileMenu->addAction(saveAct);
  iFileMenu->addAction(saveAsAct);
  iFileMenu->addAction(saveAsBitmapAct);
  iFileMenu->addSeparator();
  iFileMenu->addAction(runLatexAct);
  iFileMenu->addSeparator();
  iFileMenu->addAction(closeAct);
  iFileMenu->addAction(exitAct);

  // ------------------------------------------------------------
  // Edit menu
  // ------------------------------------------------------------

  iAbsoluteAttributesAction = new QAction(tr("&Absolute attributes"), this);
  iAbsoluteAttributesAction->setShortcut(Key(
					     "Edit|Absolute attributes"));
  iAbsoluteAttributesAction->setCheckable(true);
  connect(iAbsoluteAttributesAction, SIGNAL(toggled(bool)),
	  SLOT(AbsoluteAttributes(bool)));

  iUndoAction = new IpeAction(ECmdUndo, undoIcon, tr("&Undo"), this);
  iUndoAction->setShortcut(Key("Edit|Undo"));
  connect(iUndoAction, SIGNAL(triggered(int)), SLOT(UndoCmd(int)));

  iRedoAction = new IpeAction(ECmdRedo, redoIcon, tr("&Redo"), this);
  iRedoAction->setShortcut(Key("Edit|Redo"));
  connect(iRedoAction, SIGNAL(triggered(int)), SLOT(UndoCmd(int)));

  iDeleteAction = new IpeAction(ECmdDelete, tr("&Delete"), this);
  iDeleteAction->setShortcut(Key("Edit|Delete"));
  connect(iDeleteAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iGroupAction = new IpeAction(ECmdGroup, tr("&Group"), this);
  iGroupAction->setShortcut(Key("Edit|Group"));
  connect(iGroupAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iUngroupAction = new IpeAction(ECmdUngroup, tr("Ungroup"), this);
  iUngroupAction->setShortcut(Key("Edit|Ungroup"));
  connect(iUngroupAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iFrontAction = new IpeAction(ECmdFront, tr("&Front"), this);
  iFrontAction->setShortcut(Key("Edit|Front"));
  connect(iFrontAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iBackAction = new IpeAction(ECmdBack, tr("&Back"), this);
  iBackAction->setShortcut(Key("Edit|Back"));
  connect(iBackAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *forwardAction = new IpeAction(ECmdForward, tr("Forward"), this);
  forwardAction->setShortcut(Key("Edit|Forward"));
  connect(forwardAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *backwardAction = new IpeAction(ECmdBackward, tr("Backward"), this);
  backwardAction->setShortcut(Key("Edit|Backward"));
  connect(backwardAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *beforeAction = new IpeAction(ECmdBefore, tr("Just before"), this);
  beforeAction->setShortcut(Key("Edit|Before"));
  connect(beforeAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *behindAction = new IpeAction(ECmdBehind, tr("Just behind"), this);
  behindAction->setShortcut(Key("Edit|Behind"));
  connect(behindAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iDuplicateAction = new IpeAction(ECmdDuplicate, tr("&Duplicate"), this);
  iDuplicateAction->setShortcut(Key("Edit|Duplicate"));
  connect(iDuplicateAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *selectAllAct = new IpeAction(ECmdSelectAll,
					tr("Select &all"), this);
  selectAllAct->setShortcut(Key("Edit|Select all"));
  connect(selectAllAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iComposeAction = new IpeAction(ECmdComposePaths, tr("Co&mpose paths"), this);
  iComposeAction->setShortcut(Key("Edit|Compose paths"));
  connect(iComposeAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iJoinAction = new IpeAction(ECmdJoinPaths, tr("&Join paths"), this);
  iJoinAction->setShortcut(Key("Edit|Join paths"));
  connect(iJoinAction, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *insertTextBoxAct = new QAction(tr("Insert text box"), this);
  insertTextBoxAct->setShortcut(Key("Edit|Insert text box"));
  connect(insertTextBoxAct, SIGNAL(triggered()), SLOT(InsertTextBox()));

  QAction *editObjectAct = new QAction(tr("Edit object"), this);
  editObjectAct->setShortcut(Key("Edit|Edit"));
  connect(editObjectAct, SIGNAL(triggered()), SLOT(EditObject()));

  QAction *docPropsAct = new QAction(tr("D&ocument properties"), this);
  docPropsAct->setShortcut(Key("Edit|Document properties"));
  connect(docPropsAct, SIGNAL(triggered()), SLOT(EditDocProps()));

  QAction *styleSheetsAct = new QAction(tr("St&yle sheets"), this);
  styleSheetsAct->setShortcut(Key("Edit|Style sheets"));
  connect(styleSheetsAct, SIGNAL(triggered()), SLOT(StyleSheets()));

  QAction *updateStylesAct = new QAction(tr("Update style sheets"), this);
  updateStylesAct->setShortcut(Key("Edit|Update style sheets"));
  connect(updateStylesAct, SIGNAL(triggered()), SLOT(UpdateStyleSheets()));

  iEditMenu->addAction(iAbsoluteAttributesAction);
  iEditMenu->addSeparator();
  iEditMenu->addAction(iUndoAction);
  iEditMenu->addAction(iRedoAction);
  iEditMenu->addSeparator();
  iEditMenu->addAction(iCutAction);
  iEditMenu->addAction(iCopyAction);
  iEditMenu->addAction(pasteAct);
  iEditMenu->addAction(iDeleteAction);
  iEditMenu->addSeparator();
  iEditMenu->addAction(iGroupAction);
  iEditMenu->addAction(iUngroupAction);
  iEditMenu->addAction(iFrontAction);
  iEditMenu->addAction(iBackAction);
  iEditMenu->addAction(forwardAction);
  iEditMenu->addAction(backwardAction);
  iEditMenu->addAction(beforeAction);
  iEditMenu->addAction(behindAction);
  iEditMenu->addAction(iDuplicateAction);
  iEditMenu->addAction(selectAllAct);
  iEditMenu->addSeparator();
  iEditMenu->addAction(iComposeAction);
  iEditMenu->addAction(iJoinAction);
  iEditMenu->addSeparator();
  iEditMenu->addAction(insertTextBoxAct);
  iEditMenu->addAction(editObjectAct);
  iEditMenu->addSeparator();
  iEditMenu->addAction(docPropsAct);
  iEditMenu->addAction(styleSheetsAct);
  iEditMenu->addAction(updateStylesAct);

  connect(iEditMenu, SIGNAL(aboutToShow()), SLOT(AboutToShowEditMenu()));

  // --------------------------------------------------------------------
  // Object (Mode) menu
  // --------------------------------------------------------------------

  for (int i = 0; i < IpeOverlayFactory::Num; ++i)
    iModeMenu->addAction(iModeAction[i]);

  // --------------------------------------------------------------------
  // Snap menu
  // --------------------------------------------------------------------

  iAbsoluteSnapAction = new QAction(tr("&Absolute grid/angle size"), this);
  iAbsoluteSnapAction->setShortcut(Key("Snap|Absolute"));
  iAbsoluteSnapAction->setCheckable(true);
  connect(iAbsoluteSnapAction, SIGNAL(toggled(bool)),
	  SLOT(AbsoluteSnapping(bool)));

  QAction *setOriginAct = new IpeAction(ECmdSetOrigin, tr("Set origin"), this);
  connect(setOriginAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));
  setOriginAct->setShortcut(Key("Snap|Set origin"));

  QAction *setOriginSnapAct =
    new IpeAction(ECmdSetOriginSnap, tr("Set origin && snap"), this);
  setOriginSnapAct->setShortcut(Key("Snap|Set origin & snap"));
  connect(setOriginSnapAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  QAction *hideAxesAct = new IpeAction(ECmdResetOrigin, tr("Hide axes"), this);
  hideAxesAct->setShortcut(Key("Snap|Hide axes"));
  connect(hideAxesAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  QAction *setDirectionAct =
    new IpeAction(ECmdSetDirection, tr("Set direction"), this);
  setDirectionAct->setShortcut(Key("Snap|Set direction"));
  connect(setDirectionAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  QAction *resetDirectionAct =
    new IpeAction(ECmdResetDirection, tr("Reset direction"), this);
  resetDirectionAct->setShortcut(Key("Snap|Reset direction"));
  connect(resetDirectionAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  QAction *setLineAct = new IpeAction(ECmdSetLine, tr("Set line"), this);
  setLineAct->setShortcut(Key("Snap|Set line"));
  connect(setLineAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  QAction *setLineSnapAct =
    new IpeAction(ECmdSetLineSnap, tr("Set line && snap"), this);
  setLineSnapAct->setShortcut(Key("Snap|Set line & snap"));
  connect(setLineSnapAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  iSnapMenu->addAction(iAbsoluteSnapAction);
  iSnapMenu->addSeparator();
  for (int i = 0; i < 5; ++i)
    iSnapMenu->addAction(iSnapAction[i]);
  iSnapMenu->addSeparator();
  // iSnapMenu->addAction(setOriginAct);
  iSnapMenu->addAction(setOriginSnapAct);
  iSnapMenu->addAction(hideAxesAct);
  iSnapMenu->addAction(setDirectionAct);
  iSnapMenu->addAction(resetDirectionAct);
  iSnapMenu->addAction(setLineAct);
  iSnapMenu->addAction(setLineSnapAct);

  // --------------------------------------------------------------------
  // Zoom menu
  // --------------------------------------------------------------------

  iGridVisibleAction = new QAction(tr("&Grid visible"), this);
  iGridVisibleAction->setShortcut(Key("Zoom|Grid"));
  iGridVisibleAction->setCheckable(true);
  connect(iGridVisibleAction, SIGNAL(toggled(bool)),
	  SLOT(GridVisible(bool)));

  iCoordinatesVisibleAction = new QAction(tr("&Coordinates visible"), this);
  iCoordinatesVisibleAction->setShortcut(Key("Zoom|Coordinates"));
  iCoordinatesVisibleAction->setCheckable(true);
  connect(iCoordinatesVisibleAction, SIGNAL(toggled(bool)),
	  SLOT(CoordinatesVisible(bool)));

  QAction *mouseIn[3];
  mouseIn[0] = new IpeAction(0, tr("points"), this);
  mouseIn[1] = new IpeAction(1, tr("millimeters"), this);
  mouseIn[2] = new IpeAction(2, tr("inches"), this);
  QMenu *sub = new QMenu();
  for (int i = 0; i < 3; ++i) {
    connect(mouseIn[i], SIGNAL(triggered(int)), SLOT(SetMouseDisplayIn(int)));
    sub->addAction(mouseIn[i]);
  }
  sub->setTitle(QLatin1String("Coordinates in"));

  QAction *zoomInAct = new QAction(tr("Zoom &in"), this);
  zoomInAct->setShortcut(Key("Zoom|Zoom in"));
  connect(zoomInAct, SIGNAL(triggered()), iResolution, SLOT(stepUp()));

  QAction *zoomOutAct = new QAction(tr("Zoom &out"), this);
  zoomOutAct->setShortcut(Key("Zoom|Zoom out"));
  connect(zoomOutAct, SIGNAL(triggered()), iResolution, SLOT(stepDown()));

  QAction *fitPageAct = new QAction(tr("Fit &page"), this);
  fitPageAct->setShortcut(Key("Zoom|Fit page"));
  connect(fitPageAct, SIGNAL(triggered()), SLOT(FitPage()));

  QAction *fitObjectsAct = new QAction(tr("&Fit objects"), this);
  fitObjectsAct->setShortcut(Key("Zoom|Fit objects"));
  connect(fitObjectsAct, SIGNAL(triggered()), SLOT(FitObjects()));

  QAction *fitSelectionAct = new QAction(tr("Fit &selection"), this);
  fitSelectionAct->setShortcut(Key("Zoom|Fit selection"));
  connect(fitSelectionAct, SIGNAL(triggered()), SLOT(FitSelection()));

  QAction *panHereAct = new IpeAction(ECmdHere, tr("Pan &here"), this);
  panHereAct->setShortcut(Key("Snap|Pan here"));
  connect(panHereAct, SIGNAL(triggered(int)), SLOT(SnapCmd(int)));

  iZoomMenu->addAction(iGridVisibleAction);
  iZoomMenu->addAction(iCoordinatesVisibleAction);
  iZoomMenu->addMenu(sub);
  iZoomMenu->addSeparator();
  iZoomMenu->addAction(zoomInAct);
  iZoomMenu->addAction(zoomOutAct);
  iZoomMenu->addAction(normalSizeAct);
  iZoomMenu->addAction(fitPageAct);
  iZoomMenu->addAction(fitObjectsAct);
  iZoomMenu->addAction(fitSelectionAct);
  iZoomMenu->addSeparator();
  iZoomMenu->addAction(panHereAct);

  // --------------------------------------------------------------------
  // Layer menu
  // --------------------------------------------------------------------

  QAction *newLayerAct = new IpeAction(ECmdNewLayer, tr("&New layer"), this);
  newLayerAct->setShortcut(Key("Layer|New"));
  connect(newLayerAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *renameLayerAct =
    new IpeAction(ECmdRenameLayer, tr("&Rename layer"), this);
  renameLayerAct->setShortcut(Key("Layer|Rename"));
  connect(renameLayerAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *selectLayerAct =
    new IpeAction(ECmdSelectLayer, tr("&Select all in layer"), this);
  selectLayerAct->setShortcut(Key("Layer|Select all"));
  connect(selectLayerAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *moveToLayerAct =
    new IpeAction(ECmdMoveToLayer, tr("&Move objects to layer"), this);
  moveToLayerAct->setShortcut(Key("Layer|Move objects"));
  connect(moveToLayerAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  iLayerMenu->addAction(newLayerAct);
  iLayerMenu->addAction(renameLayerAct);
  iLayerMenu->addSeparator();
  iLayerMenu->addAction(selectLayerAct);
  iLayerMenu->addAction(moveToLayerAct);

  // --------------------------------------------------------------------
  // View menu
  // --------------------------------------------------------------------

  QAction *nextViewAct = new QAction(tr("Next vie&w"), this);
  nextViewAct->setShortcut(Key("View|Next view"));
  connect(nextViewAct, SIGNAL(triggered()), SLOT(NextView()));

  QAction *previousViewAct = new QAction(tr("&Previous view"), this);
  previousViewAct->setShortcut(Key("View|Previous view"));
  connect(previousViewAct, SIGNAL(triggered()), SLOT(PreviousView()));

  QAction *firstViewAct =
    new IpeAction(ECmdFirstView, tr("&First view"), this);
  firstViewAct->setShortcut(Key("View|First view"));
  connect(firstViewAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *lastViewAct =
    new IpeAction(ECmdLastView, tr("&Last view"), this);
  lastViewAct->setShortcut(Key("View|Last view"));
  connect(lastViewAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *newLayerNewViewAct =
    new IpeAction(ECmdNewLayerNewView, tr("&New layer, new view"), this);
  newLayerNewViewAct->setShortcut(Key("View|New layer, new view"));
  connect(newLayerNewViewAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *newViewAct = new IpeAction(ECmdNewView, tr("New &view"), this);
  newViewAct->setShortcut(Key("View|New view"));
  connect(newViewAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *deleteViewAct =
    new IpeAction(ECmdDeleteView, tr("&Delete view"), this);
  deleteViewAct->setShortcut(Key("View|Delete view"));
  connect(deleteViewAct, SIGNAL(triggered(int)), SLOT(Cmd(int)));

  QAction *editEffectsAct = new QAction(tr("&Edit effects"), this);
  editEffectsAct->setShortcut(Key("View|Edit effects"));
  connect(editEffectsAct, SIGNAL(triggered()), SLOT(EditEffects()));

  iViewMenu->addAction(nextViewAct);
  iViewMenu->addAction(previousViewAct);
  iViewMenu->addAction(firstViewAct);
  iViewMenu->addAction(lastViewAct);
  iViewMenu->addSeparator();
  iViewMenu->addAction(newLayerNewViewAct);
  iViewMenu->addAction(newViewAct);
  iViewMenu->addAction(deleteViewAct);
  iViewMenu->addSeparator();
  iViewMenu->addAction(editEffectsAct);

  // --------------------------------------------------------------------
  // Page menu
  // --------------------------------------------------------------------

  QAction *nextPageAct = new QAction(tr("&Next page"), this);
  connect(nextPageAct, SIGNAL(triggered()), iPageNumber, SLOT(stepUp()));
  nextPageAct->setShortcut(Key("Page|Next page"));

  QAction *previousPageAct = new QAction(tr("&Previous page"), this);
  connect(previousPageAct, SIGNAL(triggered()), iPageNumber, SLOT(stepDown()));
  previousPageAct->setShortcut(Key("Page|Previous page"));

  QAction *firstPageAct = new QAction(tr("&First page"), this);
  connect(firstPageAct, SIGNAL(triggered()), SLOT(FirstPage()));
  firstPageAct->setShortcut(Key("Page|First page"));

  QAction *lastPageAct = new QAction(tr("&Last page"), this);
  connect(lastPageAct, SIGNAL(triggered()), SLOT(LastPage()));
  lastPageAct->setShortcut(Key("Page|Last page"));

  QAction *insertPageAct = new QAction(tr("&Insert page"), this);
  connect(insertPageAct, SIGNAL(triggered()), SLOT(CreatePage()));
  insertPageAct->setShortcut(Key("Page|Insert page"));

  QAction *cutPageAct = new QAction(tr("&Cut page"), this);
  connect(cutPageAct, SIGNAL(triggered()), SLOT(CutPage()));
  cutPageAct->setShortcut(Key("Page|Cut page"));

  QAction *copyPageAct = new QAction(tr("Cop&y page"), this);
  connect(copyPageAct, SIGNAL(triggered()), SLOT(CopyPage()));
  copyPageAct->setShortcut(Key("Page|Copy page"));

  QAction *pastePageAct = new QAction(tr("P&aste page"), this);
  connect(pastePageAct, SIGNAL(triggered()), SLOT(PastePage()));
  pastePageAct->setShortcut(Key("Page|Paste page"));

  QAction *deletePageAct = new QAction(tr("&Delete page"), this);
  connect(deletePageAct, SIGNAL(triggered()), SLOT(DeletePage()));
  deletePageAct->setShortcut(Key("Page|Delete page"));

  QAction *editBookmarkAct = new QAction(tr("Edit &bookmarks"), this);
  editBookmarkAct->setShortcut(Key("View|Edit bookmarks"));
  connect(editBookmarkAct, SIGNAL(triggered()), SLOT(EditBookmarks()));

  iPageMenu->addAction(nextPageAct);
  iPageMenu->addAction(previousPageAct);
  iPageMenu->addAction(firstPageAct);
  iPageMenu->addAction(lastPageAct);
  iPageMenu->addSeparator();
  iPageMenu->addAction(insertPageAct);
  iPageMenu->addAction(cutPageAct);
  iPageMenu->addAction(copyPageAct);
  iPageMenu->addAction(pastePageAct);
  iPageMenu->addAction(deletePageAct);
  iPageMenu->addSeparator();
  iPageMenu->addAction(editBookmarkAct);

  // --------------------------------------------------------------------
  // Ipelet menu
  // --------------------------------------------------------------------

  iPasteBitmapAction = 0;
  for (uint i = 0; i < iIpeletMaster.size(); ++i) {
    Ipelet *ipelet = iIpeletMaster[i];
    IpeString keys = "Ipelet|";
    keys += ipelet->Label();
    if (ipelet->NumFunctions() > 1) {
      QMenu *pop = new QMenu(QString::fromUtf8(ipelet->Label()));
      bool isBitmap = (!qstrcmp(ipelet->Label(), "Insert image"));
      keys += "|";
      for (int j = 0; j < ipelet->NumFunctions(); ++j) {
	QAction *a = new IpeAction(i * 0x1000 + j,
				   QString::fromUtf8(ipelet->SubLabel(j)),
				   this);
	a->setShortcut(Key(keys + ipelet->SubLabel(j)));
	connect(a, SIGNAL(triggered(int)), SLOT(RunIpelet(int)));
	pop->addAction(a);
	if (isBitmap && j == 2)
	  iPasteBitmapAction = a;
      }
      if (isBitmap) {
	// personal treatment
	iFileMenu->insertMenu(runLatexAct, pop);
	iFileMenu->insertSeparator(runLatexAct);
      } else
	iIpeletMenu->addMenu(pop);
    } else {
      QAction *a = new IpeAction(i * 0x1000 + 1,
				 QString::fromUtf8(ipelet->Label()), this);
      a->setShortcut(Key(keys));
      connect(a, SIGNAL(triggered(int)), SLOT(RunIpelet(int)));
      iIpeletMenu->addAction(a);
    }
  }
  QAction *a = new QAction(tr("About ipelets"), this);
  a->setShortcut(Key("Ipelet|About ipelets"));
  iIpeletMenu->addAction(a);
  connect(a, SIGNAL(triggered()), SLOT(AboutIpelets()));

  // --------------------------------------------------------------------
  // Help menu
  // --------------------------------------------------------------------

  QAction *manualAct = new QAction(tr("Ipe &manual"), this);
  manualAct->setShortcut(Key("Help|Ipe manual"));
  connect(manualAct, SIGNAL(triggered()), SLOT(Manual()));

  QAction *preferencesAct = new QAction(tr("Pre&ferences"), this);
  preferencesAct->setShortcut(Key("Help|Preferences"));
  connect(preferencesAct, SIGNAL(triggered()), SLOT(EditPrefs()));

  QAction *aboutAct = new QAction(tr("&About Ipe..."), this);
  aboutAct->setShortcut(Key("Help|About Ipe"));
  connect(aboutAct, SIGNAL(triggered()), SLOT(About()));

  iHelpMenu->addAction(manualAct);
  iHelpMenu->addAction(whatsThisAct);
  iHelpMenu->addSeparator();
  iHelpMenu->addAction(preferencesAct);
  iHelpMenu->addSeparator();
  iHelpMenu->addAction(aboutAct);

  // --------------------------------------------------------------------

  iCanvas = new IpeCanvas(this, statusBar(), prefs->iDoubleBuffer,
			  this, prefs->iBitmapSize);
  setCentralWidget(iCanvas);
  iCanvas->setWhatsThis(tr(canvasText));

  iResolution->setValue(72000);

  iAbsoluteAttributes = false;
  iAbsoluteSnapping = false;
  SwitchAttributeUi();
  SwitchSnapUi();

  iSnapData.iSnap = IpeSnapData::ESnapNone;
  iSnapData.iGridSize = prefs->iGridSize;
  iSnapData.iGridVisible = prefs->iGridVisible;
  iSnapData.iSnapDistance = prefs->iSnapDistance;
  iSnapData.iSelectDistance = prefs->iSelectDistance;
  iSnapData.iWithAxes = false;
  iSnapData.iOrigin = IpeVector::Zero;
  iSnapData.iDir = 0.0;

  iDoc = 0;
  SetCreator(iModeAction[0]);

  iMouse = new QLabel(statusBar());
  iMouseIn = 0;
  statusBar()->addPermanentWidget(iMouse, 0);

  connect(prefs, SIGNAL(Changed()), this, SLOT(PreferencesChanged()));

  installEventFilter(this);
}

// --------------------------------------------------------------------

//! Destructor.
AppUi::~AppUi()
{
  delete iDoc;
}

// --------------------------------------------------------------------
