// --------------------------------------------------------------------
// Main function
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "appui.h"
#include "ipeq.h"
#include "ipeversion.h"
#include "ipecreatetext.h"

#include <cstdio>
#include <cstdlib>

using namespace std;

#ifdef WIN32
#include <windows.h>
#endif

static bool showDebug = false;

#ifdef WIN32
static void ipeDebugImpl(const char *msg)
{
  OutputDebugStringA(msg);
}
#else
static void ipeDebugImpl(const char *msg)
{
  if (showDebug)
    fprintf(stderr, "%s\n", msg);
}
#endif

// used from appui.cpp
void StartManual(QString url)
{
#if defined(WIN32)
  ShellExecuteA(0, "open", url.toLocal8Bit(), 0, 0, 0);
#else
  QByteArray browser = IpePreferences::Static()->iBrowser.toLatin1();
  QByteArray s = browser + " " + url.toLatin1();
  system(s + " &");
#endif
}

static void usage()
{
  fprintf(stderr,
	  "Usage:\n ipe [ Qt options ] [-debug] [-geom wxh+x+y] "
	  "[-sheet stylesheet][ filename ]\n");
  exit(1);
}

// The main function just starts the QApplication
int main(int argc, char *argv[])
{
  // Check Ipelib version mismatch
  if (IpelibVersion() != IPELIB_VERSION) {
    fprintf(stderr,
	    "Ipe has been compiled with header files for Ipelib %d\n"
	    "but is (probably dynamically) linked against libipe %d.\n"
	    "Check with 'ldd ipe' which libipe is being loaded, and either "
	    "replace it by the correct version or set LD_LIBRARY_PATH.\n",
	    IPELIB_VERSION, IpelibVersion());
    exit(99);
  }

  // set ipeDebug handler
  ipeDebugHandler = ipeDebugImpl;

  if (argc >= 2 && (!qstrcmp(argv[1], "-help") ||
		    !qstrcmp(argv[1], "--help")))
    usage();

  QApplication a( argc, argv );

  // must create QApplication before first call to IpePreferences::Static.
  IpePreferences *prefs = IpePreferences::Static();

  IpeDialogCreateText::SetEditorFont(prefs->iFont);

  QString lang = IpePreferences::Static()->iLanguage;
  ipeDebug("Current locale is '%s'", QLocale().name().toLatin1().data());
  ipeDebug("Current directory is '%s'",
	   QDir::current().path().toLatin1().data());

  QTranslator qt(0);
  if (!lang.isNull() && qt.load(QLatin1String("ipe-") + lang, prefs->iLangDir))
    a.installTranslator(&qt);

  AppUi::ReadKeyAssignment();

  bool geom = false;
  int w, h, x, y;
  QByteArray load;

  int i = 1;
  if (a.argc() >= i+2 && !qstrcmp(a.argv()[i], "-geom")) {
    if (sscanf(a.argv()[i+1], "%dx%d+%d+%d", &w, &h, &x, &y) == 4)
      geom = true;
    ipeDebug("Geometry: %dx%d at (%d,%d)", w, h, x, y);
    i += 2;
  }
  if (a.argc() >= i+2 && !qstrcmp(a.argv()[i], "-sheet")) {
    prefs->iStyleSheet2 = QLatin1String(a.argv()[i+1]);
    i += 2;
  }
  if (a.argc() >= i+1 && !qstrcmp(a.argv()[i], "-debug")) {
    showDebug = true;
    ++i;
  }
  if (a.argc() == i + 1) {
    load = a.argv()[i];
  } else if (a.argc() > i + 1)
    usage();

  AppUi *appui = new AppUi;

  if (load.isEmpty() || !appui->Load(QString::fromLocal8Bit(load)))
    appui->NewDoc();

  if (prefs->iMaximize) {
    appui->showMaximized();
  } else {
    if (geom)
      appui->setGeometry(x, y, w, h);
    appui->show();
  }

  appui->DefaultZoom();

  QObject::connect(&a, SIGNAL(lastWindowClosed()),
		   &a, SLOT(quit()));
  return a.exec();
}

// --------------------------------------------------------------------
