// --------------------------------------------------------------------
// Loading IML stream (and objects) from PDF file
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeiml.h"
#include "ipepdfloader_p.h"

#include <qfile.h>

// --------------------------------------------------------------------

IpePdfParser::IpePdfParser(IpePdfLoader &loader, IpeRepository *rep)
  : IpeImlParser(loader, rep)
{
  iXRef = loader.iDoc->getXRef();
}

IpeString IpePdfParser::PdfStream(int objNum)
{
  PdfObject streamObj;
  iXRef->fetch(objNum, 0, &streamObj);
  if (streamObj->isStream()) {
    Stream *str = streamObj->getStream()->getBaseStream();
    str->reset();
    IpeString data;
    int ch;
    while ((ch = str->getChar()) != EOF)
      data += ch;
    return data;
  } else
    return IpeString();
}

// --------------------------------------------------------------------

/*! \class IpePdfLoader
  \brief Loader for PDF files written by Ipe.

  Create an IpePdfLoader given a file name.
  Check that the file has been opened okay using IsOk().
  Check that it is a file written by Ipe using IsIpeFile().

 */

//! Create a loader object.
IpePdfLoader::IpePdfLoader(QString fname)
{
  iDoc = 0;
  iIpeStream = -1;

  InitXpdfLib();

  // iDoc takes ownership of GString
  iDoc = new PDFDoc(new GString(fname.utf8()));
  if (!iDoc->isOk())
    return;

  // try opening object # 1
  iDoc->getXRef()->fetch(1, 0, &iStreamObj);
  if (iStreamObj->isStream() && iStreamObj->streamIs("Ipe")) {
    iIpeStream = 1;
    iStreamObj->streamReset();
    return;
  } else
    iStreamObj->free();

  // otherwise, use /Ipe key in /Catalog (first preview versions of Ipe 6.0)
  PdfObject catObj;
  iDoc->getXRef()->getCatalog(&catObj);
  if (catObj->isNull() || !catObj->isDict("Catalog")) {
    return;
  }
  PdfObject ipeRefObj;
  catObj->dictLookupNF("Ipe", &ipeRefObj);
  if (ipeRefObj->isNull() || !ipeRefObj->isRef() ||
      ipeRefObj->getRef().gen != 0)
    // not an Ipe file
    return;
  iIpeStream = ipeRefObj->getRef().num;
  iDoc->getXRef()->fetch(iIpeStream, 0, &iStreamObj);
  iStreamObj->streamReset();
}

//! Destructor.
IpePdfLoader::~IpePdfLoader()
{
  delete iDoc;
}

//! Has PDF file been opened okay?
bool IpePdfLoader::IsOk() const
{
  return (iDoc && iDoc->isOk() && iIpeStream >= 0);
}

//! Read one character from Ipe stream in PDF document.
int IpePdfLoader::GetChar()
{
  return iStreamObj->streamGetChar();
}

// --------------------------------------------------------------------
