// --------------------------------------------------------------------
// The Mark object.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipemark.h"
#include "ipevisitor.h"
#include "ipepainter.h"

/*! \class IpeMark
  \ingroup obj
  \brief The mark object.

  A mark is a little marker for a "point". Several shapes are
  available: disks, circles, squares, crosses, etc.

*/

//! Create from XML stream.
IpeMark::IpeMark(IpeRepository *rep, const IpeXmlAttributes &attr,
		 IpeString /* data */)
  : IpeObject(rep, attr)
{
  iSize = rep->MakeScalar(IpeAttribute::EMarkSize, attr["size"]);
  IpeLex shapeLex(attr["shape"]);
  int shape;
  shapeLex >> shape;
  iShape = shape;
  // parse position
  IpeLex lex(attr["pos"]);
  lex >> iPos.iX >> iPos.iY;
}

//! Create at position.
IpeMark::IpeMark(const IpeAllAttributes &attr, const IpeVector &pos)
  : IpeObject(attr)
{
  iSize = attr.iMarkSize;
  iShape = attr.iMarkShape;
  iPos = pos;
}

//! Clone object
IpeObject *IpeMark::Clone() const
{
  return new IpeMark(*this);
}

//! Copy constructor.
IpeMark::IpeMark(const IpeMark &rhs)
  : IpeObject(rhs)
{
  iPos = rhs.iPos;
  iShape = rhs.iShape;
  iSize = rhs.iSize;
}

//! Return pointer to this object.
IpeMark *IpeMark::AsMark()
{
  return this;
}

//! Call VisitMark of visitor.
void IpeMark::Accept(IpeVisitor &visitor) const
{
  visitor.VisitMark(this);
}

void IpeMark::SaveAsXml(IpePainter &painter, IpeStream &stream,
			IpeString layer) const
{
  stream << "<mark";
  SaveAttributesAsXml(painter, stream, layer);
  stream << " pos=\"" << iPos.iX << " " << iPos.iY << "\"";
  if (painter.MarkShape() == 0)
    stream << " shape=\"" << int(iShape) << "\"";
  if (painter.MarkSize().IsNull())
    stream << " size=\"" << painter.Repository()->String(iSize) << "\"/>\n";
}

//! Draw marker.
void IpeMark::Draw(IpePainter &painter) const
{
  painter.Push();
  painter.Transform(Matrix());
  painter.Translate(iPos);
  painter.Untransform(false);

  painter.SetMarkSize(iSize);
  painter.SetMarkShape(iShape);
  painter.SetStroke(Stroke());
  painter.SetDashStyle(IpeAttribute::Solid());

  const IpeRepository *rep = painter.Repository();
  int shape = painter.MarkShape();
  double absSize = rep->ToScalar(painter.MarkSize());

  painter.SetLineWidth(IpeAttribute(IpeAttribute::ELineWidth, 0.2 * absSize));

  if (shape == EDisc || shape == ESquare) {
    painter.SetFill(painter.Stroke());
  } else {
    painter.SetFill(IpeAttribute::Void());
  }

  IpeVector size(absSize, absSize);
  IpeVector ll = IpeVector::Zero - 0.5 * size;
  if (shape == EBox || shape == ESquare) {
    IpeRect re(ll);
    re.AddPoint(ll + size);
    painter.Rect(re);
  } else if (shape == ECross) {
    painter.BeginPath(ll);
    painter.LineTo(ll + size);
    painter.EndPath();
    painter.BeginPath(ll + IpeVector(absSize, 0));
    painter.LineTo(ll + IpeVector(0, absSize));
    painter.EndPath();
  } else {
    IpeMatrix m; // = 1
    m.iA[0] = m.iA[3] = 0.5 * absSize;
    painter.Transform(m);
    painter.DrawEllipse();
  }
  painter.DrawPath();
  painter.Pop();
}

double IpeMark::Distance(const IpeVector &v, const IpeMatrix &m,
			 double /* bound */) const
{
  return (v - m * (Matrix() * Position())).Len();
}

void IpeMark::AddToBBox(IpeRect &box, const IpeMatrix &m) const
{
  IpeVector p = m * (Matrix() * Position());
  box.AddPoint(p);
}

void IpeMark::SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		      IpeVector &pos, double &bound) const
{
  SnapVertex(mouse, m * Matrix() * Position(), pos, bound);
}

//! Set shape of mark.
void IpeMark::SetShape(int shape)
{
  iShape = shape;
}

//! Set size of mark.
void IpeMark::SetSize(IpeAttribute size)
{
  iSize = size;
}

void IpeMark::CheckStyle(const IpeStyleSheet *sheet,
			  IpeAttributeSeq &seq) const
{
  IpeObject::CheckStyle(sheet, seq);
  CheckSymbol(iSize, sheet, seq);
}

// --------------------------------------------------------------------
