/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/window.h
//! @brief A GdkGLWindow C++ wrapper interface.
//!
//! Provides Window, an OpenGL window which is located on-screen.

#ifndef INTI_GDK_GL_WINDOW_H
#define INTI_GDK_GL_WINDOW_H

#ifndef INTI_GDK_DRAWABLE_H
#include <inti/gdk/drawable.h>
#endif

#ifndef INTI_GDK_GL_DRAWABLE_H
#include <inti/gdk/gl/drawable.h>
#endif

#ifndef __GDK_GL_WINDOW_H__
#include <gdk/gdkglwindow.h>
#endif

namespace Inti {

namespace Gdk {

class Window;

namespace GL {

class Config;
class WindowClass;

//! @class Window window.h inti/gdk/gl/window.h
//! @brief A GdkGLWindow C++ wrapper class.
//!
//! Window represents an OpenGL window which is located on-screen.

class Window : public Gdk::Drawable, public Gdk::GL::Drawable
{
	friend class G::Object;
	friend class WindowClass;

	Window(const Window&);
	Window& operator=(const Window&);

protected:
//! @name Constructors
//! @{

	explicit Window(GdkGLWindow *window, bool reference = true);
	//!< Construct a new Window from an existing GdkGLWindow.
	//!< @param window A pointer to a GdkGLWindow.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>window</EM> can be a newly created GdkGLWindow or an existing
	//!< GdkGLWindow. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	virtual ~Window();
	//!< Destructor.

//! @}

	static Pointer<Window> create(const Config& glconfig, const Gdk::Window& window, const int *attrib_list = 0);
	//!< Creates an on-screen rendering area.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param window The Gdk::Window to be used as the rendering area.
	//!< @param attrib_list This must be set to null or empty (first attribute of none).
	//!< @return The new Gdk::GL::Window.
	//!<
	//!< <BR>The <EM>attrib_list</EM> argument is currently unused. This must be set to
	//!< null or empty (first attribute of none, see GLX 1.3 spec).

	static Window* get_gl_window(Gdk::Window& window);
	//!< Gets the Gdk:GL::Window held by <EM>window</EM>.
	//!< @param window A Gdk::Window.
	//!< @return The Gdk::GL::Window.

	static Gdk::GL::Drawable* get_gl_drawable(Gdk::Window& window);
	//!< Gets the Gdk::GL::Drawable held by the window.
	//!< @param window A Gdk::GL::Window.
	//!< @return The Gdk::GL::Drawable.
	//!<
	//!< <BR>This method returns the result of get_gl_window() cast to a Gdk::GL::Drawable.

	static bool is_gl_capable(Gdk::Window& window);
	//!< Determines whether the window is OpenGL-capable.
	//!< @param window A Gdk::Window.
	//!< @return <EM>true</EM> if the window is OpenGL-capable, <EM>false</EM> otherwise.

	static Pointer<Window> set_gl_capability(Gdk::Window& window, const Config& glconfig, const int *attrib_list = 0);
	//!< Set the OpenGL-capability for <EM>window</EM>.
	//!< @param window The Gdk::Window to be used as the rendering area.
	//!< @param glconfig A Gdk::GL::Config.
	//!< @param attrib_list This must be set to null or empty (a first attribute of none).
	//!< @return The Gdk::GL::Window used by the window if it is successful, null otherwise.
	//!<
	//!< <BR>This method creates a new Gdk::GL::Window held by <EM>window</EM>. The <EM>attrib_list</EM>
	//!< argument is currently unused. This must be set to null or empty (a first attribute of none).

	static void unset_gl_capability(Gdk::Window& window);
	//!< Unset the OpenGL-capability for <EM>window</EM>.
	//!< @param window The Gdk::Window that was used as the rendering area.
	//!<
	//!< <BR>This method destroys the Gdk::GL::Window held by the window.

//! @name Accessors
//! @{

	GdkGLWindow* gdk_gl_window() const { return (GdkGLWindow*)instance; }
	//!< Get a pointer to the GdkGLWindow structure.

	GdkGLWindowClass* gdk_gl_window_class() const;
	//!< Get a pointer to the GdkGLWindowClass structure.

	operator GdkGLWindow* () const;
	//!< Conversion operator; safely converts a Window to a GdkGLWindow pointer.

	Gdk::Window* get_window() const;
	//!< Gets the Gdk::Window associated with the <EM>window</EM>.
	//!< @return The Gdk::Window associated with the window.
	//!<
	//!< <BR>Notice that Gdk::GL::Window is not Gdk::Window, but another Gdk::Drawable
	//!< which has an associated Gdk::Window.

//! @}
//! @name Methods
//! @{

	virtual void dispose();
	//!< Disposes of the OpenGL resources associated with the window
	//!< and decrements window's reference count.

//! @}
};

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_WINDOW_H


