/***************************************************************************
                       nurbssurface.cpp  -  description
                          -------------------                                         
 begin                : Thu Nov 18 1999                                           
 copyright            : (C) 1999 by Jon Anderson                         
 email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "nurbssurface.h"
#include "nurbscurve.h"
#include <nurbs++/nurbs.h>
#include "ctrlpt.h"
#include "material.h"
#include "texturematerial.h"
#include "mesh.h"
#include "face.h"
#include "nurbssurfaceparser.h"

int NurbsSurface::display_mode = GLU_FILL;
int NurbsSurface::TYPE = Typed::getUID();
int NurbsSurfaceParser::TYPE = Parser::setParser( NurbsSurface::TYPE, "surface", new NurbsSurfaceParser() );

NurbsSurface::NurbsSurface( Entity *p ) : Object( p )
{

  addType( TYPE );
  material = new Material();

  theNurbs = gluNewNurbsRenderer();

  usegs = vsegs = 15;
  recompute = true;
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_SAMPLING_METHOD, GLU_DOMAIN_DISTANCE );
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_SAMPLING_TOLERANCE, 75.0 );
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_U_STEP, usegs );
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_V_STEP, vsegs );

  gluNurbsProperty( theNurbs, ( GLenum ) GLU_DISPLAY_MODE, GLU_FILL );

  dirty = true;


}

NurbsSurface::~NurbsSurface()
{
  gluDeleteNurbsRenderer( theNurbs );
}

void NurbsSurface::setVSegs( int i )
{
      vsegs = i;
      recompute = true;
gluNurbsProperty( theNurbs, ( GLenum ) GLU_V_STEP, vsegs );
};

void NurbsSurface::setUSegs( int i )
{
      usegs = i;
      recompute = true;
    gluNurbsProperty( theNurbs, ( GLenum ) GLU_U_STEP, usegs );
};


void NurbsSurface::setMaterial( Material *m )
{
  material = m;
}

int NurbsSurface::draw( int d_options )
{

  GLfloat v[ 4 ];

  gluNurbsProperty( theNurbs, ( GLenum ) GLU_DISPLAY_MODE, display_mode );

  glEnable( GL_NORMALIZE );
  glEnable( GL_AUTO_NORMAL );
  gluBeginSurface( theNurbs );

  gluNurbsSurface( theNurbs, surface.knotU().n(), surface.knotU().memory(),
                   surface.knotV().n(), surface.knotV().memory(),
                   4,
                   4 * surface.ctrlPnts().rows(),
                   ( surface.ctrlPnts() ) [ 0 ] ->data,
                   surface.degreeU() + 1,
                   surface.degreeV() + 1,
                   GL_MAP2_TEXTURE_COORD_2 );

  gluNurbsSurface( theNurbs, surface.knotU().n(), surface.knotU().memory(),
                   surface.knotV().n(), surface.knotV().memory(),
                   4,
                   4 * surface.ctrlPnts().rows(),
                   ( surface.ctrlPnts() ) [ 0 ] ->data,
                   surface.degreeU() + 1,
                   surface.degreeV() + 1,
                   GL_MAP2_VERTEX_4 );

  gluEndSurface( theNurbs );
  glDisable( GL_NORMALIZE );


  if ( SelectMode::is( Vertex::TYPE ) )
  {     //need to draw verts too?

    drawControlPolygon();
    drawVerts();
  }

  return 0;

}

void NurbsSurface::setDirtyBox( bool flag )
{

  Entity::setDirtyBox( flag );

}

void NurbsSurface::sweep( NurbsCurve &p, NurbsCurve &c, NurbsCurve &sc, int k )
{

  int w;
  w += 10;
  surface = PLib::NurbsSurfacef();

  //if(sc == 0)
  surface.sweep( p.getCurve(), c.getCurve(), k );
  //else
  // surface.sweep(p.getCurve(), c.getCurve(), sc.getCurve(), k);

  setUpControlPoints();
  // recomputeSurface();
}

void NurbsSurface::revolve( NurbsCurve &c, Vector4 &a, Vector4 &p, float degrees )
{
  surface.makeFromRevolution( c.getCurve(), PLib::Point3Df( a.x, a.y, a.z ), PLib::Point3Df( p.x, p.y, p.z ), degrees );
  setUpControlPoints();
}

void NurbsSurface::revolve( NurbsCurve &c, Vector4 &p, float degrees )
{
  surface.makeFromRevolution( c.getCurve(), PLib::Point3Df( 0, 0, 0 ), PLib::Point3Df( p.x, p.y, p.z ), degrees );
  setUpControlPoints();
  // recomputeSurface();
}

void NurbsSurface::makeSphere( Vector4 p )
{
  surface.makeSphere( PLib::Point3Df( p.x, p.y, p.z ), -2 );
  setUpControlPoints();
  //  recomputeSurface();
}

/**Creates a surface by skinning the array of curves in the V direction
  */
void NurbsSurface::skinU( PLib::NurbsCurveArrayf &a, int d )
{
  surface.skinU( a, d );
  setUpControlPoints();
}

/**Creates a surface by skinning the array of curves in the U direction
  */
void NurbsSurface::skinV( PLib::NurbsCurveArrayf &a, int d )
{
  surface.skinV( a, d );
  setUpControlPoints();
}

/**Creates a Cap surface from a curve.
  *
  *It does this by splitting the curve in half, and then
  *skinning it.
  */
void NurbsSurface::makeCap( NurbsCurve &f )
{
  PLib::NurbsCurvef a, b;
  PLib::NurbsCurvef full_curve;
  full_curve = f.getWorldSpaceCurve();

  full_curve.splitAt( .5, a, b );
  //reverse the second curve.
  Vector_HPoint3Df v = b.ctrlPnts();
  Vector_HPoint3Df rv( v.n() );
  int j = 0;

  for ( int i = v.n() - 1; i >= 0; i-- )
  {
    rv[ j ] = v[ i ];
    j++;
  }

  b = PLib::NurbsCurvef( rv, b.knot(), b.degree() );

  //create a NurbsCurve Array
  PLib::NurbsCurvef pns[ 2 ];
  pns[ 0 ] = a;
  pns[ 1 ] = b;

  PLib::NurbsCurveArrayf pnca;
  pnca.init( pns, 2 );

  skinU( pnca, 1 );

}



void NurbsSurface::makeTorus( Vector4 p )
{
  surface.makeTorus( PLib::Point3Df( p.x, p.y, p.z ), 2, 1 );
  setUpControlPoints();
}

void NurbsSurface::makePlane( Vector4 p, int plane, int u, int v )
{
  Matrix_Point3Df Pts( u, v );
  PLib::Matrix < float > WPts( u, v );

  int x, y;
  x = y = 0;

  udegree = u - 1;
  vdegree = v - 1;

  if ( udegree > 3 )
    udegree = 3;

  if ( vdegree > 3 )
    vdegree = 3;

  for ( int i = -u / 2;i < u / 2;++i )
  {
    for ( int j = -v / 2;j < v / 2;++j )
    {
      switch ( plane )
      {

        case 0:    //XY
          Pts( x, y ) = PLib::Point3Df( i, j, 0 );
          break;

        case 1:    //XZ
          Pts( x, y ) = PLib::Point3Df( i, 0, -j );
          break;

        case 2:    //YZ
          Pts( x, y ) = PLib::Point3Df( 0, j, i );
          break;

        default:
          Pts( x, y ) = PLib::Point3Df( i, j, 0 );
          break;
      }

      WPts( x, y ) = 1;
      y++;
    }

    x++;
    y = 0;
  }

  surface.globalInterp( Pts, udegree, vdegree ); // = Plib::NurbsSurfacef(3, 3, uknot, vknot, Pts, WPts);
  //surface= PLib::NurbsSurfacef(3, 3, uknot, vknot, Pts, WPts);

  setUpControlPoints();
  //recomputeSurface();
}

int NurbsSurface::getNumUPoints()
{
  Matrix_HPoint3Df m = surface.ctrlPnts();
  return m.cols();
}

int NurbsSurface::getNumVPoints()
{
  Matrix_HPoint3Df m = surface.ctrlPnts();
  return m.rows();
}


void NurbsSurface::setUpControlPoints()
{
  int x;
  int y;

  Matrix_HPoint3Df m = surface.ctrlPnts();
  PLib::HPoint3Df temp;
  CtrlPt *c;

  verts.erase( verts.begin(), verts.end() );

  x = m.rows();
  y = m.cols();

  for ( int i = 0; i < x; ++i )
  {
    for ( int j = 0; j < y; ++j )
    {
      temp = m( i, j );
      c = createCtrlPt( temp.x() / temp.w(), temp.y() / temp.w(), temp.z() / temp.w() );
      c->setMapping( i, j );
    }
  }

  vdegree = surface.degreeV();
  udegree = surface.degreeU();

}

void NurbsSurface::moveControlPoint( CtrlPt *c, float x, float y, float z )
{
  surface.modCPby( c->i, c->j, PLib::HPoint3Df( x, y, z, 0 ) );

  PLib::HPoint3Df temp;
  temp = surface.ctrlPnts( c->i, c->j );
  c->setPosition( temp.x() / temp.w(), temp.y() / temp.w(), temp.z() / temp.w() );

}

void NurbsSurface::drawControlPolygon()
{

  glPushAttrib( GL_LIGHTING_BIT | GL_COLOR_BUFFER_BIT );
  glDisable( GL_LIGHTING );
  glColor4f( 0, 0, 1, 1 );


  glLineWidth( 2 );
  int i, j ;

  for ( i = 0;i < surface.ctrlPnts().rows();++i )
  {
    glBegin( GL_LINE_STRIP );

    for ( j = 0;j < surface.ctrlPnts().cols();++j )
      glVertex4fv( surface.ctrlPnts() ( i, j ).data ) ;

    glEnd() ;
  }

  for ( j = 0;j < surface.ctrlPnts().cols();++j )
  {
    glBegin( GL_LINE_STRIP );

    for ( i = 0;i < surface.ctrlPnts().rows();++i )
      glVertex4fv( surface.ctrlPnts() ( i, j ).data );

    glEnd() ;
  }

  glLineWidth( 1 );
  glPopAttrib();

}

Mesh * NurbsSurface::getMesh( int nu, int nv )
{
  Mesh * m = new Mesh( pos );
  PLib::HPoint3Df temp;

  //create vertices...
  Face *faces[ nu - 1 ][ nv - 1 ];
  Vertex *verts[ nu ][ nv ];

  for ( int i = 0; i < nu; i++ )
  {
    float u = ( float ) i / ( nu - 1 );

    for ( int j = 0; j < nv; j++ )
    {
      float v = ( float ) j / ( nv - 1 );
      temp = surface.hpointAt( u, v );
      verts[ i ][ j ] = m -> createVertex( temp.x() / temp.w(), temp.y() / temp.w(), temp.z() / temp.w() );
    }
  }

  for ( int i = 0; i < nu - 1; i++ )
  {
    float u1 = ( float ) i / ( nu - 1 );
    float u2 = ( float ) ( i + 1 ) / ( nu - 1 );

    for ( int j = 0; j < nv - 1; j++ )
    {
      float v1 = ( float ) j / ( nv - 1 );
      float v2 = ( float ) ( j + 1 ) / ( nv - 1 );

      faces[ i ][ j ] = m -> createFace( verts[ i ][ j ] ->getParentIndex(), verts[ i + 1 ][ j ] ->getParentIndex(),
                                         verts[ i + 1 ][ j + 1 ] ->getParentIndex(), verts[ i ][ j + 1 ] ->getParentIndex() );
      faces[ i ][ j ] -> setUVCoord( 0, u1, v1 );
      faces[ i ][ j ] -> setUVCoord( 1, u2, v1 );
      faces[ i ][ j ] -> setUVCoord( 2, u2, v2 );
      faces[ i ][ j ] -> setUVCoord( 3, u1, v2 );
    }
  }

  m->normalize();
  return m;

}

void NurbsSurface::copyFrom( Entity *rhs )
{
  *this = * static_cast < NurbsSurface * > ( rhs );
}

NurbsSurface & NurbsSurface::operator=( NurbsSurface &rhs )
{
  Object::operator=( rhs );
  surface = rhs.surface;
  setUpControlPoints();

  return *this;

}

Entity * NurbsSurface::clone()
{
  NurbsSurface * s = new NurbsSurface();
  *s = *this;
  return s;
}

vector < float > NurbsSurface::getUKnots()
{
  vector < float > uknots;

  PLib::Vector < float > knts;
  knts = surface.knotU();

  for ( int i = 0; i < knts.n(); i++ )
  {
    uknots.push_back( knts[ i ] );
  }

  return uknots;

}

vector < float > NurbsSurface::getVKnots()
{
  vector < float > vknots;

  PLib::Vector < float > knts;
  knts = surface.knotV();

  for ( int i = 0; i < knts.n(); i++ )
  {
    vknots.push_back( knts[ i ] );
  }

  return vknots;

}


void NurbsSurface::createSurface( vector < Vector4 > &p, int upts, int vpts,
                                  vector < float > &uknots,
                                  vector < float > &vknots,
                                  int udegree,
                                  int vdegree )
{

  Matrix_HPoint3Df pts( upts, vpts );

  PLib::Vector < float > uknot( uknots.size() );
  PLib::Vector < float > vknot( vknots.size() );

  for ( int i = 0; i < upts; i++ )
  {
    for ( int j = 0; j < vpts; j++ )
    {
      int x;
      x = i * upts + j;
      cerr << x << "=[" << i << "][" << j << "]" << endl;
      pts[ i ][ j ] = PLib::HPoint3Df( p[ x ].x, p[ x ].y, p[ x ].z, 1 );

    }
  }

  cerr << "Done with ctrl" << endl;

  for ( int i = 0; i < uknot.n(); i++ )
    uknot[ i ] = uknots[ i ];

  for ( int i = 0; i < vknot.n(); i++ )
    vknot[ i ] = vknots[ i ];


  cerr << "Setup knots" << endl;

  surface = PLib::NurbsSurfacef( udegree, vdegree, uknot, vknot, pts );

  cerr << "setup surface" << endl;

  //get the control points and add them to this nurbs list.
  setUpControlPoints();

  cerr << "Setup my ctrl pts." << endl;


}

void NurbsSurface::createSurface( int upts, int vpts, vector<float> &uknots, vector<float> &vknots, int udegree, int vdegree )
{
  Matrix_HPoint3Df pts( upts, vpts );

  PLib::Vector < float > uknot( uknots.size() );
  PLib::Vector < float > vknot( vknots.size() );

  Vector4 pt;

  for ( int i = 0; i < upts; i++ )
  {
    for ( int j = 0; j < vpts; j++ )
    {
      int x;
      x = i * upts + j;
      cerr << x << "=[" << i << "][" << j << "]" << endl;
      verts[x] -> getPosition( &pt );

      pts[ i ][ j ] = PLib::HPoint3Df( pt.x, pt.y, pt.z, 1 );
    }
  }

  cerr << "Done with ctrl" << endl;

  for ( int i = 0; i < uknot.n(); i++ )
    uknot[ i ] = uknots[ i ];

  for ( int i = 0; i < vknot.n(); i++ )
    vknot[ i ] = vknots[ i ];


  cerr << "Setup knots" << endl;

  surface = PLib::NurbsSurfacef( udegree, vdegree, uknot, vknot, pts );

  cerr << "setup surface" << endl;

  //delete the old verts.
  for ( int i = 0; i < ( int ) numVerts(); i++ )
  {
    delete verts[i];
  }


  verts.clear();
  //get the control points and add them to this nurbs list.
  setUpControlPoints();

  cerr << "Setup my ctrl pts." << endl;

}

CtrlPt * NurbsSurface::createCtrlPt( float x, float y, float z )
{
  CtrlPt * c = new CtrlPt( x, y, z, this );
  addVert( c );
  c -> setIndex( verts.size() - 1 );
  return c;
}



void NurbsSurface::recomputeSurface()
{
  /* delete[] p;
   delete[] n;
   
     p = new PLib::Point3Df[usegs][vsegs];
     n = new PLib::Point3Df[usegs][vsegs];
   
     for(int u=0; u<usegs; u++){
      for(int v=0; v<vsegs; v++){
       p[u][v] = surface.pointAt((float)u/(usegs-1), (float)v/(vsegs-1));
     n[u][v] = surface.normal((float)u/(usegs-1), (float)v/(vsegs-1));
    }
     }
     recompute = false;
     */
}
