/***************************************************************************
                       camera.cpp  -  description
                          -------------------
 begin                : Thu Nov 2 2000
 copyright            : (C) 2000 by Jon Anderson
 email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <GL/gl.h>
#include <GL/glu.h>
#include "camera.h"
#include <iostream>
#include <cassert>

Camera::Camera( const Vector4 &p, const Vector4 &look_at, const Vector3 &up,
                double _near, double _far, double _fov, double _right )
    : fov( _fov ), right( _right ), near( _near ), far( _far )
{
  pos = p;
  Vector3 direction( look_at.x - pos.x, look_at.y - pos.y, look_at.z - pos.z );
  direction.normalize();
  setOrientation( direction, up );
}


Camera::Camera( const Vector4 &p, const Vector3 &dir, const Vector3 &up,
                double _near, double _far, double _fov, double _right )
    : fov( _fov ), right( _right ), near( _near ), far( _far )
{
  pos = p;
  setOrientation( dir, up );
}

void Camera::setOrientation( const Vector3 &dir, const Vector3 &up )
{
  static const double DELTA = 1.0e-5;

  // Direction and Up vectors need to be normalized.
  assert( fabs( dir.length() - 1 ) < DELTA );
  assert( fabs( up.length() - 1 ) < DELTA );

  Vector3 axis = Vector3( 0, 0, -1 ) * dir;
  axis.normalize();
  double angle = acos( dir.dot( Vector3( 0, 0, -1 ) ) );
  // If the angle is too small, then the axis has also suffered a severe
  // floating point error and the rotation wouldn't look right anyway
  if ( fabs( angle ) > DELTA )
    quat = Quat( axis, angle );

  cerr << "Fixme: up direction is currently ignored.\n";

  //angle = acos(up.dot(quat.GetYAxis()));
  // Same as above
  //if (fabs(angle) > DELTA)
  //quat = Quat(Vector3(0, 0, 1), angle) * quat;

  matrices_dirty = true;
}

void Camera::advance( double x, double y, double z )
{
  // The negative sign for the z axis is there because the camera is poiting
  // along the negative z axis.
  Vector3 delta = x * quat.GetXAxis() + y * quat.GetYAxis() + -z * quat.GetZAxis();
  pos += Vector4( delta.x, delta.y, delta.z, 1 );

  matrices_dirty = true;
}

void Camera::zoom( double amount )
{
  if ( fov )
    fov += amount;
  else
    right += amount;

  matrices_dirty = true;
}

void Camera::orbit( double up_angle, double right_angle,
                    double x, double y, double z )
{
  // Orbiting of the camera works by rotating the camera's position around
  // the center and then adjusting the camera's orientation by the same
  // angle so that it still points to the center

  // Center around which we are orbiting
  Vector3 center( x, y, z );

  // These are the two quaternions used for the rotation of camera's position.
  // Note that the up rotation is done around the camera orientation's x axis,
  // while the right rotation is done around the absolute up axis. This is
  // because the up rotation is done along the longitude lines while
  // the right rotation is done along the latitude lines.
  Quat rotate_up( quat.GetXAxis(), up_angle * M_PI / 180 );
  Quat rotate_right( Vector3( 0, 1, 0 ), right_angle * M_PI / 180 );
  pos = Matrix44( rotate_right ) * ( Matrix44( rotate_up ) * ( pos - center ) ) + center;

  // The same reasoning as above applies the the adjustements of the camera's
  // orientation. The up rotation is done around the current orientations's
  // x axis, and thus is premultiplied, while the right rotation is done
  // around the world's up axis and thus is postmultiplied.
  quat = Quat( Vector3( 1, 0, 0 ), up_angle * M_PI / 180 ) * quat;
  quat = quat * Quat( Vector3( 0, 1, 0 ), right_angle * M_PI / 180 );

  matrices_dirty = true;
}

void Camera::setViewport( int x, int y, int width, int height )
{
  glViewport( x, y, width, height );
  aspect = ( double ) width / height;

  viewport[ 0 ] = x;
  viewport[ 1 ] = y;
  viewport[ 2 ] = width;
  viewport[ 3 ] = height;

  matrices_dirty = true;
}

void Camera::setupProjection()
{
  if ( fov )
    gluPerspective( fov, aspect, near, far );
  else
    glOrtho( -right, right, -right / aspect, right / aspect, near, far );
}

void Camera::setupModelview()
{
  // The quaternion quat contains the orientation of the camera in space.
  // In OpenGL, the camera sits in origin looking down the negative z
  // axis. To set the view up, we first need to translate the world by
  // the negative camera position (after this translation the camera
  // sits in the origin) and then rotate the world by the inverse of the
  // cameras orientation in the space.
  Matrix44 matrix( quat.Inverse() );
  glMultMatrixf( ( GLfloat* ) matrix.m );
  glTranslatef( -pos.x, -pos.y, -pos.z );
}

Vector4 Camera::unProjectPoint( int x, int y, float z )
{
  if ( matrices_dirty )
  {
    glGetDoublev( GL_MODELVIEW_MATRIX, modelview_matrix );
    glGetDoublev( GL_PROJECTION_MATRIX, projection_matrix );
    matrices_dirty = false;
  }

  // first invert the y coordinate
  int realy = viewport[ 3 ] - y - 1;

  GLdouble wx, wy, wz;

  gluUnProject( x, realy, z, modelview_matrix,
                projection_matrix, viewport, &wx, &wy, &wz );

  return Vector4( wx, wy, wz, 1 );
}

Vector3 Camera::projectPoint( const Vector4 &pt )
{
  if ( matrices_dirty )
  {
    glGetDoublev( GL_MODELVIEW_MATRIX, modelview_matrix );
    glGetDoublev( GL_PROJECTION_MATRIX, projection_matrix );
    matrices_dirty = false;
  }

  GLdouble winx, winy, winz;
  gluProject( pt.x / pt.w, pt.y / pt.w, pt.z / pt.w, modelview_matrix,
              projection_matrix, viewport, &winx, &winy, &winz );

  return Vector3( winx, viewport[ 3 ] - winy - 1, winz );
}
