#!/usr/bin/env python

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Modified by Guido Tabbernuk <boamaod@gmail.com>

import screenlets
from screenlets.options import StringOption, BoolOption
import cairo
import pango
import gtk
import decimal
from os import popen
from sys import argv
import gobject
import time
import datetime
#use gettext for translation
import gettext

_ = screenlets.utils.get_translator(__file__)

def tdoc(obj):
	obj.__doc__ = _(obj.__doc__)
	return obj

@tdoc

class FuriusMoonScreenlet(screenlets.Screenlet):
	"""Simple Lunar Phase Calendar, visit http://www.marcus-furius.com for details"""

	__name__ = 'FuriusMoonScreenlet'
	__version__ = '1.1.2+'
	__author__ = 'Marcus Furius <marcus_furius@hotmail.com>'
	__desc__ = __doc__
	
	__timeout = None
	srcdir = argv[0][:argv[0].find('FuriusMoonScreenlet.py')].strip()
	phase = '0'
	moon_image = ''
	hover = 'True'
	src_height= 96
	src_width = 96

	def __init__(self, **keyword_args):

		screenlets.Screenlet.__init__(self, width=self.src_width, height=self.src_height, uses_theme=True, 
				**keyword_args)
		self.theme_name = "default"
		self.__timeout = gobject.timeout_add(3600000, self.update)

	def on_init (self):
		self.add_default_menuitems()
		self.get_moon()

	def get_moon(self):

		today = datetime.date.today()
		year = int(today.strftime("%y"))
		month = int(today.strftime("%m"))
		day = int(today.strftime("%d"))

		description = [_("new (totally dark)"),
		      _("waxing crescent (increasing to full)"),
		      _("first quarter (increasing to full)"),
		      _("waxing gibbous (increasing to full)"),
		      _("full (full light)"),
		      _("waning gibbous (decreasing from full)"),
		      _("last quarter (decreasing from full)"),
		      _("waning crescent (decreasing from full)")]
		months = [_(" January "), _(" February "), _(" March "), _(" April "), _(" May "), _(" June "), _(" July "), _(" August "), _(" September "), _(" October "), _(" November "), _(" December ")]

		ageOfMoon = year % 19
		if ageOfMoon > 9:
		    ageOfMoon = (ageOfMoon - 19)
		    
		ageOfMoon = (ageOfMoon * 11) % 30
		    
		if month == 1:
		    month = 3
		elif month == 2:
	            month = 4
		else:
		    month = month
		    
		ageOfMoon = ageOfMoon + day + month - 8.3
		days_into_phase = ageOfMoon % 30
		    
		index = int((days_into_phase + 2) * 16/59.0)
		if index > 7:
		    index = 7
		status = description[index]
		    
		# light should be 100% 15 days into phase
		light = int(2 * days_into_phase * 100/29)
		if light > 100:
		    light = abs(light - 200)
		    
		date = "%s%s%s" % (today.strftime("%d"), months[int(today.strftime("%m"))-1], today.strftime("%Y"))

		days_into_phase = int(days_into_phase)
		self.phase = '%d' % days_into_phase 
		self.moon_image = self.srcdir + "moons/%s.png" %days_into_phase


	def __setattr__(self, name, value):
		# call Screenlet.__setattr__ in baseclass (ESSENTIAL!!!!)
		screenlets.Screenlet.__setattr__(self, name, value)
		if name == "theme_name":
			self.update()
	def update(self):
		self.get_moon()
		self.redraw_canvas()
		return True

		
	def on_draw (self, ctx):

		if self.theme:
			# set scale rel. to scale-attribute
			ctx.scale(self.scale, self.scale)
			# TEST: render example-bg into context (either PNG or SVG)
			self.theme.render(ctx, self.phase)
		else:
			d = decimal.Decimal
			ctx.scale(self.scale, self.scale)
			ctx.save()
			try:  
				src = cairo.ImageSurface.create_from_png(self.moon_image)

				self.src_height =  src.get_height()
				self.src_width = src.get_width()
				ctx.set_source_surface(src, 0, 0)
				ctx.paint()

			except:
				pass

	def on_draw_shape (self, ctx):
		self.on_draw(ctx)
	
	def menuitem_callback(self, widget, id):
		screenlets.Screenlet.menuitem_callback(self, widget, id)



if __name__ == "__main__":
	import screenlets.session
	screenlets.session.create_session(FuriusMoonScreenlet)
