/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef ICALCULATORPARSER_H
#define ICALCULATORPARSER_H


#include "iarray.h"
#include "istring.h"

class iCalculatorBase;


class iCalculatorParser
{

	friend class iCalculatorBase;

private:
	
	bool Parse(const iString &str);
	inline const iString& GetPseudoCode() const { return mCode; }

	//
	//  Get the error message and position
	//
	inline const iString& GetErrorMessage() const { return mErrorMessage; }
	inline int GetErrorPosition() const { return mErrorPosition; }

protected:

	iCalculatorParser(bool blank);
	virtual ~iCalculatorParser();

	bool AddOperator(const iString& symbol, const iString& code, int priority, bool binary);

	//
	//  Punctuation symbols
	//
	virtual const iString& SymbolSeparator() const;
	virtual const iString& SymbolOpeningBraket() const;
	virtual const iString& SymbolClosingBraket() const;
	virtual const iString& SymbolOpeningParenthesis() const;
	virtual const iString& SymbolClosingParenthesis() const;

	//
	//  Pseudo-code tokens
	//
	virtual const iString& TokenBeg() const;
	virtual const iString& TokenEnd() const;
	virtual const iString& TokenFun() const;
	virtual const iString& TokenNum() const;
	virtual const iString& TokenVar() const;

	//
	//  Other functions
	//
	virtual bool IsVariableLetter(char c) const;
	virtual bool IsVariableFirstLetter(char c) const;
	virtual bool IsRecognizedLiteral(const iString &str, int index, int length) const; // can be overwritten by a child to recognize special numbers
	virtual void SetErrorExtra(const iString &message, int pos);

	//
	//  Error-recording operations
	//
	class Logger
	{

	public:

		Logger(iCalculatorParser *owner, const iString &code);
		~Logger();

		void LogChange(const iString &o, const iString &n);

	private:

		iString mSavedCode;
		const iString& mCode;
		int mSavedLogLocation;
		iCalculatorParser *mOwner;
	};

	void SetError(const iString &message, const iString &context, int pos);

private:

	struct OperatorInfo
	{
		iString S;
		iString C;
		int P;
		bool B;
		OperatorInfo() : P(0), B(false) {}
	};

	void DecorateToken(iString &code, const iString &op, const iString &token);

	//
	//  Text manipulation helpers
	//
	int FindVariableToLeft(const iString &str, int index) const;
	int FindVariableToRight(const iString &str, int index) const;
	int FindLiteralToLeft(const iString &str, int index) const;
	int FindLiteralToRight(const iString &str, int index) const;
	int FindCompleteTokenToLeft(const iString &str, iString &token, int index) const;
	int FindCompleteTokenToRight(const iString &str, iString &token, int index) const;

	int FindOperatorMatch(const iString &str, const iArray<OperatorInfo> &ops, int index, int &kop) const;
	bool IsInsideAToken(const iString &str, int index) const;
	bool IsFullyParsed(const iString &str) const;

	//
	//  Parsing helpers
	//
	bool ParseSegment(iString &code, const iString &fun);
	bool ParseComponent(iString &code);
	bool ParseExpressions(iString &code);
	bool ParseSomeOperators(iString &code, const iArray<OperatorInfo> &ops);

	//
	//  Self-check
	//
	bool VerifyIntegrity();
	bool VerifyIntegrityHelper(const iString &token);
#ifdef I_CHECK1
	bool ValidatePseudoCode(const iString &code) const;
#endif

	//
	//  Reverse enginnering log for error identification
	//
	struct Change
	{
		iString Old;
		iString New;
		void Clear(){ Old.Clear(); New.Clear(); }
	};
	iArray<Change> mLog;

	int mErrorPosition;
	iString mErrorMessage;
	iString mExpression, mCode;
	iArray<OperatorInfo> mOperators;
	bool mIsVerified;
	int mMaxPriority;
};


inline bool iCalculatorParser::IsVariableLetter(char c) const
{
	//
	//  Variables may contain letters, numbers, and '_'.
	//
	return ((c>='0'&&c<='9') || (c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}


inline bool iCalculatorParser::IsVariableFirstLetter(char c) const
{
	//
	//  Variables must start with a letter or '_'.
	//
	return ((c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}

#endif  // ICALCULATORPARSER_H
