/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggpagecrosssection.h"


#include "icamera.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "irendertool.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdatatypeprovider.h"
#include "iggframeattachposition.h"
#include "iggframecurrentinstance.h"
#include "iggframedatavariablelist.h"
#include "iggframedoublebutton.h"
#include "iggframematerialproperties.h"
#include "iggframepaletteselection.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgetselectionboxsubject.h"

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


using namespace iParameter;
using namespace iParameter;


namespace iggPageCrossSection_Private
{
	//
	//  Helper widgets
	//
	class DownsampleCheckBox : public iggWidget
	{
		
	public:
		
		DownsampleCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Downsample while moving",1);
			this->SetBaloonHelp("Toggles whether the cross section is downsampled while moving.","When the cross section is moving across the box, the data are being constantly resampled. If the size of the dataset is too large, resampling will take too much time, and the position slider will not feel interactive, but will be \"dragging\" behind. To avoid this, if this box is checked, IFrIT will downsample the data to maintain the interactive performance.");
		}

		bool IsDown() const
		{
			return mSubject->IsDown();
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}
		
		ibgWidgetButtonSubject *mSubject;
	};
	

	class PositionSlider : public iggWidgetKeyPositionSlider
	{
		
		friend class ScanBoxButton;

	public:
		
		PositionSlider(iggFrame *parent) : iggWidgetKeyPositionSlider(8,"Position",iCrossSectionViewSubject::KeyLocation(true),RenderMode::Immediate,parent)
		{
			mDownsampleBuddy = 0;
			mSelfChecking = true;
		}

		void SetDownsampleBuddy(DownsampleCheckBox *b)
		{
			mDownsampleBuddy = b;
		}

	protected:
		
		virtual void OnInt1Body(int i)
		{
			iggWidgetKeyPositionSlider::OnInt1Body(i);

			if(mDownsampleBuddy->IsDown())
			{
				int oldrf, mod = -1;
				if(this->GetExecuteFlags() & ModuleOption::All) mod = -2;

				this->GetShell()->GetControlModule()->QueryValue(iCrossSectionViewSubject::KeySampleRate(),oldrf);
				int newrf = round(oldrf*sqrt(10.0*this->GetShell()->GetControlModule()->GetUpdateTime(mod)));
				if(newrf > 1)
				{
					iString ws;
					this->GetShell()->GetControlModule()->PackCommand(ws,iCrossSectionViewSubject::KeySampleRate(),newrf);
					this->GetShell()->GetControlModule()->Execute(ws,false,this->GetExecuteFlags());
				}
			}
		}

		virtual void OnVoid1Body()
		{
			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,iCrossSectionViewSubject::KeySampleRate(),1);
			this->GetShell()->GetControlModule()->Execute(ws,false,this->GetExecuteFlags());
			iggWidgetKeyPositionSlider::OnVoid1Body();
		}

		void GetRange(int &min, int &max)
		{
			mSubject->GetRange(min,max);
		}

		void Drive(int v)
		{
			mSubject->FreezeNumberOfDigits(true);
			mSubject->SetValue(v);
			this->OnInt1Body(v);
			mSubject->FreezeNumberOfDigits(false);
		}

		DownsampleCheckBox *mDownsampleBuddy;
	};


	class ScanBoxButton : public iggWidgetSimpleButton
	{
		
	public:
		
		ScanBoxButton(iggFrame *parent) : iggWidgetSimpleButton("Scan",parent)
		{
			this->SetBaloonHelp("Moves the cross section accross the bunding box in a given number of seconds","When this button is clicked, the current cross section moves accross the bounding box with the speed specified by the number of seconds.");
			mSteps = 10;
		}

		void SetBuddy(PositionSlider *buddy)
		{
			mBuddy = buddy;
		}

		void SetNumberOfSteps(int v)
		{
			if(v > 1) mSteps = v;
		}

		inline int GetNumberOfSteps() const { return mSteps; }

	protected:

		virtual void UpdateWidgetBody()
		{
			bool dataPresent;

			if(!this->GetShell()->GetControlModule()->QueryValue(iCrossSectionViewSubject::KeyDataPresent(),dataPresent))
			{
				this->Enable(false);
				return this->UpdateFailed();
			}

			this->Enable(dataPresent && (mBuddy!=0));
		}

		virtual void Execute()
		{
			if(mBuddy == 0)
			{
				IERROR_LOW("PositionSlider buddy must be set.");
				return;
			}
			
			int min, max;
			mBuddy->GetRange(min,max);

			//
			//  Optimization: direct access
			//
			iCrossSectionViewSubject *obj = this->GetShell()->GetControlModule()->GetViewModule()->GetCrossSectionViewSubject();
			if(obj == 0) return;

			double op = obj->GetLocation();

			int i;
			bool r = false;
			for(i=0; i<mSteps && !r; i++)
			{
				mBuddy->Drive(round(min+float(max-min)*i/(mSteps-1)));
				r = obj->GetOverTheEdgeFlag();
			}
			
			obj->SetLocation(op);
			obj->SetSampleRate(1);
			this->GetShell()->GetControlModule()->Render();

			mBuddy->UpdateWidget();
		}

		PositionSlider *mBuddy;
		int mSteps;
	};


	class ScanDurationSpinBox : public iggWidget
	{
		
	public:
		
		ScanDurationSpinBox(ScanBoxButton *button, iggFrame *parent) : iggWidget(parent)
		{
			mButton = button;

			mSubject = iggSubjectFactory::CreateWidgetSpinBoxSubject(this,2,1000,"the box in",1);
			mSubject->SetValue(10);

			this->SetBaloonHelp("Number of steps to scan the box in","Set the number of steps to scan the box with a cross section in.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetValue(mButton->GetNumberOfSteps());
		}

		virtual void OnInt1Body(int v)
		{
			mButton->SetNumberOfSteps(v);
		}

		ScanBoxButton *mButton;
		ibgWidgetSpinBoxSubject *mSubject;
	};


	class OrthogonalizeViewButton : public iggWidgetSimpleButton
	{
		
	public:
		
		OrthogonalizeViewButton(iggFrame *parent) : iggWidgetSimpleButton("Orthogonalize view",parent)
		{
			this->SetBaloonHelp("Re-orient the view along the principal directions","This button re-orients the view along the nearest principal directions (X, Y, or Z).");
		}

	protected:

		virtual void Execute()
		{
			this->GetShell()->GetControlModule()->GetViewModule()->GetRenderTool()->GetCamera()->OrthogonalizeView();
			this->GetShell()->GetControlModule()->GetViewModule()->Render();
		}
	};
};


using namespace iggPageCrossSection_Private;


iggPageCrossSection::iggPageCrossSection(iggFrameBase *parent) : iggPageObject(parent,ViewSubject::Id::CrossSection)
{
	const iImage *icon = iImageFactory::FindIcon("xsec.png");

	//
	//  Main page
	// ************************************************
	//
	iggFrame *page0 = new iggFrame(mBook,3);
	mBook->AddPage("Main",icon,page0);
	//
	//  Show
	//
	iggWidget *show = new iggWidgetShowButton(iCrossSectionViewSubject::Type(),page0);
	iggFrameCurrentInstance *ci1 = new iggFrameCurrentInstance(false,"Instance","Instance",iViewModule::KeyCrossSectionCurrent(),iViewModule::KeyCrossSectionMax(),page0);
	ci1->AddDependent(this);
	page0->AddLine(show,static_cast<iggWidget*>(0),ci1);
	page0->AddSpace(2);
	//
	//  Paint with and Direction
	//
	iggFrame *pf = new iggFrame(page0,2);
	iggFrameDataVariableList *pl = new iggFrameDataVariableList(mProvider,"Paint with...",iCrossSectionViewSubject::KeyVar(),0,pf,0);
	pl->Complete();
	pl->AddDependent(show);
	iggWidgetKeyRadioBox *cd = new iggWidgetKeyRadioBox(1,"Direction",0,iCrossSectionViewSubject::KeyDir(),pf);
	cd->InsertItem("X");
	cd->InsertItem("Y");
	cd->InsertItem("Z");
	pf->AddLine(pl,cd);
	page0->AddLine(pf);
	page0->AddSpace(2);
	//
	//  Method
	//
	iggFrame *mf = new iggFrame(page0,2);
	iggWidgetKeyRadioBox *cm = new iggWidgetKeyRadioBox(1,"Method",0,iCrossSectionViewSubject::KeyMethod(),mf);
	cm->InsertItem("Polygons");
	cm->InsertItem("Textures");
	mf->AddLine(cm,new iggWidgetKeyCheckBox("Interpolate data",iCrossSectionViewSubject::KeyInterpolateData(),mf));
	page0->AddLine(mf);
	page0->AddSpace(10);
	//
	//  Position
	//
	PositionSlider *cp = new PositionSlider(page0);
	DownsampleCheckBox *dc = new DownsampleCheckBox(page0);
	cp->SetDownsampleBuddy(dc);
	page0->AddLine(cp,2);
	page0->AddLine(dc);

	page0->AddSpace(2);
	page0->AddLine(new iggFrameAttachPosition(cp,iCrossSectionViewSubject::KeyAttachToMarker(),"",page0,true),1);
	page0->AddLine(new OrthogonalizeViewButton(page0));

	page0->AddSpace(10);

	page0->SetColStretch(1,10);
	page0->SetColStretch(2,5);

	//
	//  Paint page
	// ************************************************
	//
	iggFrame *page1 = new iggFrame(mBook,1);
	mBook->AddPage("Paint",icon,page1);
	//
	//  Book
	//
	iggFrameBook *pb = new iggFrameBook(page1);
	page1->AddLine(pb);
	//
	//  Palette page
	//
	iggFrame *pbpage0 = new iggFrame(pb,2);
	pb->AddPage("Palette",icon,pbpage0);

	iggFramePaletteSelection *ps = new iggFramePaletteSelection(false,iCrossSectionViewSubject::KeyPalette(),pbpage0,0);
	pbpage0->AddLine(ps);
	pbpage0->AddSpace(20);

	pbpage0->SetColStretch(0,10);
	pbpage0->SetColStretch(1,3);

	//
	//  Material page
	//
	iggFrame *pbpage1 = new iggFrame(pb,2);
	pb->AddPage("Material",icon,pbpage1);

	iggFrameMaterialProperties *mp = new iggFrameMaterialProperties(true,true,iCrossSectionViewSubject::Type(),pbpage1);
	pbpage1->AddLine(mp);

	pbpage1->AddSpace(10);
	pbpage1->SetColStretch(0,10);
	pbpage1->SetColStretch(1,3);

	//
	//  Instances page
	// ************************************************
	//
	iggFrame *page2 = new iggFrame(mBook,3);
	mBook->AddPage("Instances",icon,page2);
	//
	//  Instance #
	//
	iggFrameCurrentInstance *ci2 = new iggFrameCurrentInstance(false,"Instance","Instance",iViewModule::KeyCrossSectionCurrent(),iViewModule::KeyCrossSectionMax(),page2);
	ci2->AddDependent(this);
	page2->AddLine(ci2);
	page2->AddSpace(2);
	//
	//  Color
	//
	page2->AddLine(new iggFrameCreateDeleteButton(this,iCrossSectionViewSubject::Type(),true,"instance",page2));
	page2->AddSpace(10);

	page2->SetColStretch(2,10);

	//
	//  Special page
	// ************************************************
	//
	iggFrame *page3 = new iggFrame(mBook,2);
	mBook->AddPage("Special",icon,page3);
	
	pf = new iggFrame(page3,3);
	ScanBoxButton *sb = new ScanBoxButton(pf);
	sb->SetBuddy(cp);
	pf->AddLine(sb,new ScanDurationSpinBox(sb,pf),new iggWidgetTextArea("steps",pf));
	page3->AddLine(pf);
	page3->AddSpace(2);

	pf = new iggFrame("Place cross-section at...",page3,2);
	pf->AddLine(new iggWidgetTextArea("Minimum of the data",pf),new iggWidgetKeyExecIntButton(0,"Go",iCrossSectionViewSubject::KeySpecialLocation(),pf));
	pf->AddLine(new iggWidgetTextArea("Maximum of the data",pf),new iggWidgetKeyExecIntButton(1,"Go",iCrossSectionViewSubject::KeySpecialLocation(),pf));
	pf->AddLine(new iggWidgetTextArea("Center of the box",pf),new iggWidgetKeyExecIntButton(3,"Go",iCrossSectionViewSubject::KeySpecialLocation(),pf));
	pf->AddLine(new iggWidgetTextArea("Lower 1/4 of the box",pf),new iggWidgetKeyExecIntButton(2,"Go",iCrossSectionViewSubject::KeySpecialLocation(),pf));
	pf->AddLine(new iggWidgetTextArea("Upper 1/4 of the box",pf),new iggWidgetKeyExecIntButton(4,"Go",iCrossSectionViewSubject::KeySpecialLocation(),pf));
	page3->AddLine(pf);

	page3->AddSpace(10);
	page3->SetColStretch(1,10);
}

#endif
