/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  Abstract script
//


#ifndef ISCRIPT_H
#define ISCRIPT_H


#include <vtkObjectBase.h>


#define ISCRIPT_BACKWARD_COMPATIBLE // remove in the future

#include "iarray.h"
#include "ipointermacro.h"
#include "istring.h"

#include <vtkSetGet.h>

class iErrorStatus;
class iScriptObserver;

namespace iScriptKit
{
	class Calculator;
	class Value;
	class Prototype;
	class Statement;
	class GenericFlowControlStatement;
	class LoopFlowControlStatement;
	class VariableDeclarationStatement;
};


class iScript : public vtkObjectBase
{
	
	friend class iScriptKit::Calculator;
	friend class iScriptKit::Value;
	friend class iScriptKit::Prototype;
	friend class iScriptKit::Statement;
	friend class iScriptKit::GenericFlowControlStatement;
	friend class iScriptKit::LoopFlowControlStatement;
	friend class iScriptKit::VariableDeclarationStatement;
	friend class iScriptObserver;

	IPOINTER_AS_USER(ErrorStatus);

public:
	
	//
	//  Aliasing
	//
	struct AliasWord
	{
		iString Word;
		iString Alias;
	};


	vtkTypeMacro(iScript,vtkObjectBase);

	void SetText(const iString &s);
	inline const iString& GetText() const { return mText; }

	//
	//  Using the script 
	//
	bool Compile();
	void Execute(bool keepVariables = false);

	//
	//  Line-by-line driver components
	//
	bool StartExecute();
	bool ExecuteOneLine();
	void StopExecute(bool keepVariables = false);

	int GetThisLine() const;
	int GetNextLine() const;
	int GetNumberOfLines() const;

	bool IsDummyWord(const iString &s) const;
	bool IsCommandWord(const iString &s) const;
	bool IsVariableName(const iString &s) const;
	bool IsConstantName(const iString &s) const;

	bool IsReservedWord(const iString &s) const;

	inline int GetNumberOfReservedWords() const { return mNumStatementPrototypes+mConstants.Size()+mDummyWords.Size(); }
	virtual const iString& GetReservedWord(int i) const;

	void QueryVariables(iArray<const iScriptKit::Value*> &list) const;

	inline int GetErrorPosition() const { return mErrorPosition; }
	
	//
	//  Internal interrupt
	//
	bool CheckAbort(int cur, int num, int level) const;
	bool CheckAbort() const;

	//
	//  Calculators
	//
	iScriptKit::Calculator* GetCalculator() const { return mCalculator; }

	inline bool IsEmbedded() const { return mParent != 0; }

protected:

	iScript(iScript *parent);
	virtual ~iScript();

	//
	//  Script execution
	//
	void Reset();
	void RemoveVariables();
	bool CompileOneLine(int line);

	//
	//  Other helpers
	//
	iString GetText(int) const;
	int FindCommandWord(const iString &context, const iString &word, int index = 0) const;
	virtual bool IsCommandWordLetter(char c) const;
	virtual bool IsCommandWordFirstLetter(char c) const;
	bool IsValidVariableName(const iString &name) const;

	//
	//  Script creation
	//
	void CreateAliasWord(const iString &alias, const iString &word);
	void CreateDummyWord(const iString &word);
	
	void AddConstant(iScriptKit::Value *value);

	//
	//  Prototypes
	//
	template<class T>
	void AddPrototype(const iString &command);
	template<class T, class A>
	void AddPrototype(const iString &command, A arg);
	template<class T, class A1, class A2>
	void AddPrototype(const iString &command, A1 arg1, A2 arg2);

	//
	//  Factory method: a calculator can be replaced by a child. It is non-const so that a child's
	//  calculator can add new variables and constants.
	//
	virtual iScriptKit::Calculator* CreateCalculator();

	//
	//  Overwrite this to create special non-algebraic variables
	//
	virtual bool SatisfyRequestForVariables(const iString &vars);

	//
	//  Variable manipulation
	//
	iScriptKit::Value* FindVariable(const iString &name) const;
	void AddVariable(iScriptKit::Value *value);
	void RemoveVariable(const iString &name);

	//
	//  Changeable members
	//
	bool mCaseSensitive;
	bool mAllowChildAccess;

	iString mCommentString, mHeadOfLineContinuationString, mTailOfLineContinuationString, mAppendSeparator;

	iScript *mParent;  // cannot be const

private:

	//
	//  Convenient error reporting for components
	//
	void ReportError(const iString &message, int pos, int line);

	//
	//  Special access for FlowControl statements
	//
	void JumpToEntry(int entry);
	void SetLoopParameters(int cur, int num);

	//
	//  Prototypes
	//
	void RegisterPrototype(iScriptKit::Prototype *prototype);

	//
	//  Observers
	//
	void AddObserver(iScriptObserver *obs);
	void RemoveObserver(iScriptObserver *obs);

	bool mBlock, mIsCompiled;
	int mErrorPosition;

	iArray<AliasWord> mAliasWords;
	iSearchableArray<iString> mDummyWords;

	iArray<iScriptKit::Value*> mVariables;
	iArray<const iScriptKit::Value*> mConstants;

	int mNumStatementPrototypes, mNumAssignmentPrototypes;
	iArray<iScriptKit::Statement*> mCode;
	iArray<iScriptKit::Prototype*> mPrototypes;
	iArray<iScriptKit::GenericFlowControlStatement*> mStack;

	iString mText;
	int mThisEntry, mNextEntry;
	int mLoopCur, mLoopNum;  //  for debuggers

	iScriptKit::Calculator *mCalculator; // for rare calculations

	iSearchableArray<iScriptObserver*> mObservers;
};


//
//  Inheritable observer class
//
class iScriptObserver
{
	
public:

	void SetScript(iScript *parent);

	void OnScriptStart();
	void OnScriptStop(const iString &error);
	void OnScriptBeginLine(int line, const iString &text);
	void OnScriptEndLine(int line, const iString &text);
	bool OnScriptCheckAbort(int cur, int num, int level);

protected:

	iScriptObserver(iScript *s);
	virtual ~iScriptObserver();

	virtual void OnScriptStartBody() = 0;
	virtual void OnScriptStopBody(const iString &error) = 0;
	virtual void OnScriptBeginLineBody(int line, const iString &text) = 0;
	virtual void OnScriptEndLineBody(int line, const iString &text) = 0;
	virtual bool OnScriptCheckAbortBody(int cur, int num, int level) = 0;

private:

	iScript *mScript;
};

#endif // ISCRIPT_H
