/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iviewmodule.h"


#include "iactor.h"
#include "iactorcollection.h"
#include "ianimator.h"
#include "icamera.h"
#include "icameraorthoactor.h"
#include "icaptioninteractorstyle.h"
#include "iclipplane.h"
#include "icolorbars.h"
#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "icubeaxesactor.h"
#include "idata.h"
#include "idatahandler.h"
#include "idatareader.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "iinteractoreventrecorder.h"
#include "ikeyboardinteractorstyle.h"
#include "ilegend.h"
#include "ilightkit.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "imath.h"
#include "imeasuringbox.h"
#include "imeasuringboxinteractorstyle.h"
#include "iobjectfactory.h"
#include "ioverlayhelper.h"
#include "ipalette.h"
#include "iparallelmanager.h"
#include "iparticlesviewsubject.h"
#include "ipicker.h"
#include "ipointglyph.h"
#include "iposition.h"
#include "irendertool.h"
#include "iruler.h"
#include "ishell.h"
#include "isurfaceviewsubject.h"
#include "ishellfactory.h"
#include "itensorfieldviewsubject.h"
#include "itextactor.h"
#include "ivectorfieldviewsubject.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"
#include "ivolumeviewsubject.h"
#include "iwriter.h"

#include <vtkActor2D.h>
#include <vtkArrowSource.h>
#include <vtkCamera.h>
#include <vtkCellArray.h>
#include <vtkCylinderSource.h>
#include <vtkCubeSource.h>
#include <vtkDiskSource.h>
#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkInteractorStyleFlight.h>
#include <vtkInteractorStyleJoystickCamera.h>
#include <vtkInteractorStyleTrackballCamera.h>
#include <vtkInteractorStyleUnicam.h>
#include <vtkLight.h>
#include <vtkMapper.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkPicker.h>
#include <vtkPointData.h>
#include <vtkPointWidget.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkPropAssembly.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowCollection.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkTextProperty.h>
#include <vtkTimerLog.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "iviewfamilytemplate.h"


using namespace iParameter;


//
//  Define keys
//
IOBJECT_DEFINE_TYPE(iViewModule,ViewModule,vm,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iViewModule,Antialiasing,a,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,AxesBoxLabels,abl,String,3);
IOBJECT_DEFINE_KEY(iViewModule,AxesBoxRanges,abr,Float,6);
IOBJECT_DEFINE_KEY(iViewModule,AnimationOutput,ao,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,BoundingBox,bb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,BoundingBoxType,bbt,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,BackgroundColor,bg,Color,1);
IOBJECT_DEFINE_KEY(iViewModule,BackgroundImage,bi,String,1);
IOBJECT_DEFINE_KEY(iViewModule,BoxSize,bs,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,CameraAlignmentLabel,ca,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,ColorBars,cb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,CloneOfWindow,co,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlaneDirection,cpd,Float,3);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlaneDistance,cpl,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,DumpImage,di,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,FontScale,fs,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,FontType,ft,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,GlassClipPlane,gcp,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,OpenGLCoordinates,glc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,ImageFormat,if,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,InteractorStyle,is,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,ImageMagnification,ix,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,LightAngles,la,Float,2);
IOBJECT_DEFINE_KEY(iViewModule,Label,lb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelDigits,ld,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,LightIntensity,li,Float,3);
IOBJECT_DEFINE_KEY(iViewModule,LabelName,ln,String,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelOffset,lo,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelScale,ls,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelUnit,lu,String,1);
IOBJECT_DEFINE_KEY(iViewModule,MeasuringBox,mb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerLegend,ml,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerLegendPosition,mlp,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,MoveMarkerCaption,mmc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,NoClone,nc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerCurrent,omc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerMax,omm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,ParticlesCurrent,opc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,ParticlesMax,opm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,SurfaceCurrent,osc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,SurfaceMax,osm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,CrossSectionCurrent,oxc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,CrossSectionMax,oxm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,PlaceMarker,pm,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,Position,p,Int,2);
IOBJECT_DEFINE_KEY(iViewModule,PostScriptPaperFormat,psf,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,PostScriptOrientation,pso,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,Ruler,rl,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,RulerScale,rs,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,RulerTitle,rt,String,1);
IOBJECT_DEFINE_KEY(iViewModule,Size,s,Int,2);
IOBJECT_DEFINE_KEY(iViewModule,UpdateRate,r,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,StereoAlignmentMarkers,sam,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,Stereo,ss,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,StereoType,st,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,WindowNumber,wn,Int,1);

IOBJECT_DEFINE_KEY(iViewModule,JustifyLabelLeft,-lj,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,FontSize,-fsize,Int,1);  // for backward compatibility, read-only


#define IVIEWMODULE_DEFINE_VIEWSUBJECT(_name_) \
i##_name_##ViewSubject* iViewModule::Get##_name_##ViewSubject(int n) const \
{ \
	return static_cast<i##_name_##ViewSubject*>((n==-1)?mViewObjectFamilies[_ObjectType##_name_]->GetCurrentMember()->GetViewSubject():mViewObjectFamilies[_ObjectType##_name_]->GetMember(n)->GetViewSubject()); \
} // we can do C-style cast here since it is guaranteed to be correct


namespace iViewModule_Private
{
	//
	//  Helper functions
	//
	vtkActor2D* CreateBackground(iRenderTool *rt);
	iActor* CreateBox1(iRenderTool *rt);
	iActorCollection* CreateBox2(iRenderTool *rt);
	iActorCollection* CreateBox3(iRenderTool *rt);
	iActorCollection* CreateDebugProp(iRenderTool *rt);
	vtkActor2D* CreateIcon(iRenderTool *rt);
};


using namespace iViewModule_Private;


//
//  Main class
//
iViewModule* iViewModule::New(iControlModule *cm)
{
	int i;

	IERROR_ASSERT(cm);
	iViewModule *tmp = new iViewModule(cm); IERROR_ASSERT(tmp);
	iObjectFactory::InstallExtensions(tmp);

	iViewObjectFamily *fam;
	for(i=0; i<tmp->mExtensions.Size(); i++)
	{
		while((fam = static_cast<iViewModuleExtension*>(tmp->mExtensions[i])->AddNewViewObjectFamily()) != 0)
		{
			tmp->mViewObjectFamilies.Add(fam);
		}
	}

	iEdition::ApplySettings(tmp,iViewModule::Type());

	return tmp;
}


iViewModule::iViewModule(iControlModule *cm) : iExtendableObject("View Module"), mControlModule(cm)
{
	int i;
	
	mFontScale = 0;
	mFontType = _TextTypeBitmapArial;
	mBoxType = -1; // needs to be reset
	mImageMagnification = 1;
	mSavedProjection = 0;

	mIsDebug = false;
	mAntialiasing = true;
	mInImageRender = false;

	mRate = 0;
	mBoxSize = -1.0;
	mCloneOfModule = 0;
	mWinNum = 0;

	mIsBoundingBoxOn = true;
	mIsLabelOn = true;
	mIsCameraAlignmentLabelOn = false;
	mIsColorBarsOn = true;
	mIsMeasuringBoxOn = false;
	mIsRulerOn = false;
	mIsClipPlaneOn = false;
	mIsGlassClipPlaneOn = false;
	mIsStereoAlignmentMarkersOn = true;

	mLabelJustificationLeft = false;
	mStereoType = VTK_STEREO_RED_BLUE;

	mLabelName = "Record";
	mLabelUnit = "";
	mLabelScale = 1.0;
	mLabelOffset = 0.0;
	mLabelDigits = 2;

	int bc = 255;
	mBackgroundColor = iColor(bc,bc,bc);

	//
	//  Create render view
	//
	mRenderTool = 0; // must have this zeroing - who knows what happens inside the call to iExtensionFactory::CreateRenderTool
	mRenderTool = iExtensionFactory::CreateRenderTool(this); IERROR_ASSERT(mRenderTool);

	//
	//  Attach to Interactor Observer
	//
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::LeftButtonPressEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::LeftButtonReleaseEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::MiddleButtonPressEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::MiddleButtonReleaseEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::RightButtonPressEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::RightButtonReleaseEvent,this->GetControlModule()->GetInteractorObserver());
	mRenderTool->GetInteractor()->AddObserver(vtkCommand::MouseMoveEvent,this->GetControlModule()->GetInteractorObserver());

	//
	//  File reader, gateway, volume converter
	//
	mDataReader = iDataReader::New(this); IERROR_ASSERT(mDataReader);
	mAnimator = iAnimator::New(this); IERROR_ASSERT(mAnimator);
	
	//
	//  Image writer
	//
	mWriter = iWriter::New(this); IERROR_ASSERT(mWriter); 

	//
	//  Bounding boxes
	//
	mBox1Actor = CreateBox1(mRenderTool);
	mBox2Actor = CreateBox2(mRenderTool);
	mBox3Actor = CreateBox3(mRenderTool);
	
	mBox3AxesLabels = iCubeAxesActor::New(mRenderTool); IERROR_ASSERT(mBox3AxesLabels);
	mBox3AxesLabels->PickableOff();
	mBox3AxesLabels->VisibilityOff();
	mBox3AxesLabels->SetCamera(mRenderTool->GetCamera()->GetDevice());
	this->SetAxesBox("X","Y","Z",-1.0,1.0,-1.0,1.0,-1.0,1.0);
	mRenderTool->AddObject(mBox3AxesLabels);

	//
	//  Set bounding box
	//	
	this->SetBoundingBoxType(_BoundingBoxTypeDefault);

	//
	//  Ruler actor
	//
	mRulerActor = iRuler::New(this); IERROR_ASSERT(mRulerActor);
	mRulerActor->VisibilityOff();
	mRulerActor->PickableOff();
	
	mRenderTool->AddObject(mRulerActor);

	mCameraOrthoActor = iCameraOrthoActor::New(mRenderTool);
	mCameraOrthoActor->PickableOff();

	mRenderTool->AddObject(mCameraOrthoActor);

	//
	//  Text label actors
	//
	mLabelActor = iTextActor::New(mRenderTool); IERROR_ASSERT(mLabelActor);
	mLabelActor->SetText("");
	mLabelActor->SetBold(true);
	mLabelActor->SetHJustification(_TextJustRight);
	mLabelActor->SetVJustification(_TextJustTop);
	mLabelActor->SetPosition(0.95,0.9);
	mLabelActor->VisibilityOff();
	mLabelActor->PickableOff();
	mRenderTool->AddObject(mLabelActor);
	mIsLabelOn = true;

	//
	//  Measuring box
	//
	mMeasuringBox = iMeasuringBox::New(this); IERROR_ASSERT(mMeasuringBox);
	mMeasuringBox->VisibilityOff();
	mRenderTool->AddObject(mMeasuringBox);

	mInteractorStyleMeasuringBox = iMeasuringBoxInteractorStyle::New(); IERROR_ASSERT(mInteractorStyleMeasuringBox);
	mInteractorStyleMeasuringBox->SetMeasuringBox(mMeasuringBox);

	//
	//  Color bars
	//
	mColorBars = iColorBars::New(this); IERROR_ASSERT(mColorBars);
	mRenderTool->AddObject(mColorBars->GetActor(false));
	mRenderTool->AddObject(mColorBars->GetActor(true));

	//
	//  Clipping plane
	//
	mClipPlane = iClipPlane::New(this); IERROR_ASSERT(mClipPlane);

	//
	//  Progress callback functions
	//
	mObsProgress = iProgressEventObserver::New(this); IERROR_ASSERT(mObsProgress);
	
	mObsAbortRender = iAbortRenderEventObserver::New(this); IERROR_ASSERT(mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::AbortCheckEvent,mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::StartEvent,mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::EndEvent,mObsAbortRender);

	mObsSlaveAbortRender = iSlaveAbortRenderEventObserver::New(mObsAbortRender); IERROR_ASSERT(mObsSlaveAbortRender);

	//
	//  Interactor styles
	//
	mCurrentInteractorStyle = 0;

	vtkInteractorStyleFlight *isf = vtkInteractorStyleFlight::New();

	mInteractorStyle[_InteractorStyleTrackball] = vtkInteractorStyleTrackballCamera::New();
	mInteractorStyle[_InteractorStyleJoystick] = vtkInteractorStyleJoystickCamera::New();
	mInteractorStyle[_InteractorStyleFlight] = isf;
	mInteractorStyle[_InteractorStyleKeyboard] = iKeyboardInteractorStyle::New();
//	mInteractorStyle[_InteractorStyleKeyboard] = vtkInteractorStyleUnicam::New();

	for(i=0; i<__NumInteractorStyles; i++) IERROR_ASSERT(mInteractorStyle[i]);

	isf->SetAngleStepSize(0.1);
	isf->SetMotionStepSize(0.001);

	//
	//  Work around for a VTK bug: undefined angles in vtkInteractorStyleFlight.
	//
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(isf);
		isf->ForwardFly();
	}

	mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);

	//
	//  Debug props
	//
	mDebugActor = CreateDebugProp(mRenderTool);
	
	//
	// Icon
	//
	mIconActor = CreateIcon(mRenderTool);

	//
	//  Picker
	//
	mPicker = iPicker::New(this); IERROR_ASSERT(mPicker);
	mRenderTool->GetInteractor()->SetPicker(mPicker->GetHandler());

	//
	//  Event recorder & marker helpers
	//  
	mIsMarkerLegendOn = false;
	mMarkerLegendPosition = 0;

	mEventRecorder = iInteractorEventRecorder::New(); IERROR_ASSERT(mEventRecorder);
	mEventRecorder->SetInteractor(mRenderTool->GetInteractor());
	mEventRecorder->SetEnabled(1);

	mObsRecord = iRecordEventObserver::New(this); IERROR_ASSERT(mObsRecord);

	for(i=0; i<__NumInteractorStyles; i++) mInteractorStyle[i]->AddObserver(vtkCommand::InteractionEvent,mObsRecord);

	mMarkerPlacer = vtkPointWidget::New(); IERROR_ASSERT(mMarkerPlacer);
	mMarkerPlacer->XShadowsOn();
	mMarkerPlacer->YShadowsOn();
	mMarkerPlacer->ZShadowsOn();
	mMarkerPlacer->OutlineOff();
	mMarkerPlacer->TranslationModeOff();
	mMarkerPlacer->SetInteractor(mRenderTool->GetInteractor());
	mMarkerPlacer->GetProperty()->SetColor(0.0,0.0,0.0);
	mMarkerPlacer->GetProperty()->SetLineWidth(4.0);
//	mMarkerPlacer->GetSelectedProperty()->SetColor(1.0,0.0,0.0);
	mMarkerPlacer->GetSelectedProperty()->SetLineWidth(2.0);
	mMarkerPlacer->PlaceWidget(-1.0,1.0,-1.0,1.0,-1.0,1.0);

	mObsMarker = iMarkerEventObserver::New(this); IERROR_ASSERT(mObsMarker);

	mMarkerPlacer->AddObserver(vtkCommand::InteractionEvent,mObsMarker);
	mMarkerPlacer->AddObserver(vtkCommand::EndInteractionEvent,mObsMarker);

	//
	//  vtkLegendBoxActor is defined under earlier VTK versions, but
	//  markers cannot be positioned without the vtkPointWidgets, so the legend
	//  is useless then too.
	//
	mMarkerLegend = iLegend::New(mRenderTool); IERROR_ASSERT(mMarkerLegend);
	this->SetMarkerLegendPosition(mMarkerLegendPosition);
	mRenderTool->AddObject(mMarkerLegend);
	mMarkerLegend->VisibilityOff();

	mInteractorStyleCaption = iCaptionInteractorStyle::New(this); IERROR_ASSERT(mInteractorStyleCaption);

	//
	//  iViewSubjects
	//
	mMarkerFamily = iMarkerFamily::New(this); IERROR_ASSERT(mMarkerFamily);

	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iCrossSectionViewSubject::Type(),false));
	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iParticlesViewSubject::Type(),true));
	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iSurfaceViewSubject::Type(),false));
	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iTensorFieldViewSubject::Type(),true));
	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iVectorFieldViewSubject::Type(),true));
	mViewObjectFamilies.Add(iViewObjectFamily::New(this,iVolumeViewSubject::Type(),true));
	
	for(i=0; i<mViewObjectFamilies.Size(); i++) IERROR_ASSERT(mViewObjectFamilies[i]);

	//
	//  Timer
	//
	mUpdateTimer = vtkTimerLog::New(); IERROR_ASSERT(mUpdateTimer);
	mUpdateTime = 0.0f;

	//
	//  Render the scene
	//
	mRenderTool->GetCamera()->Reset();

	mRulerActor->SetBaseScale(mRenderTool->GetCamera()->GetDevice()->GetParallelScale());
	mMeasuringBox->SetBaseScale(mRenderTool->GetCamera()->GetDevice()->GetParallelScale());
	
	mRenderTool->GetInteractor()->SetDesiredUpdateRate(mRate);
	mRenderTool->GetInteractor()->SetStillUpdateRate(0.001);

	vtkMapper::SetResolveCoincidentTopologyToPolygonOffset();

	//
	//  Set the font size
	//
	this->SetFontScale(mFontScale);
	this->SetFontType(mFontType);
	this->SetAntialiasing(mAntialiasing);
	this->SetBackgroundColor(mBackgroundColor);

	mWriter->SetImageFormat(_ImageFormatPNG);
	mWriter->SetAnimationOutput(_AnimationOutputImage);

	//
	//  Initialize the interactor
	//
	//mRenderTool->GetInteractor()->Initialize();

	//
	//  Debug area
	//

}


iViewModule::~iViewModule()
{
	// Clean up
	int i;

	if(this->IsClone())
	{
		//
		//  Unregister with the parent first
		//
		mCloneOfModule->mClones.Remove(this);
	}
	//
	//  Check that all clones are already deleted
	//
#ifdef I_CHECK1
	if(mClones.Size() > 0) IERROR_LOW("Deleting a window with undeleted clones.");
#endif

	//
	//  Delete picker first in case it holds onto an object
	//
	mRenderTool->GetInteractor()->SetPicker(0);
	mPicker->Delete();

	for(i=0; i<mViewObjectFamilies.Size(); i++) mViewObjectFamilies[i]->Delete();
	//
	//  Legend needs to be deleted before markers - otherwise, markers get stuck somewhere and GarbareCollector crashes.
	//
	mRenderTool->RemoveObject(mMarkerLegend);
	mMarkerLegend->Delete();
	mMarkerFamily->Delete();
	
	mRenderTool->RemoveObject(mBox3AxesLabels);
	mBox3AxesLabels->Delete();
	mRenderTool->RemoveObject(mRulerActor);
	mRulerActor->Delete();
	mRenderTool->RemoveObject(mCameraOrthoActor);
	mCameraOrthoActor->Delete();
	mRenderTool->RemoveObject(mLabelActor);
	mLabelActor->Delete();
	mRenderTool->RemoveObject(mMeasuringBox);
	mMeasuringBox->Delete();
	mRenderTool->RemoveObject(mColorBars->GetActor(true));
	mRenderTool->RemoveObject(mColorBars->GetActor(false));
	mColorBars->Delete();

	mEventRecorder->SetInteractor(0);
	mEventRecorder->Delete();
	mObsRecord->Delete();
	mMarkerPlacer->Off();
	mMarkerPlacer->SetInteractor(0);
	mMarkerPlacer->Delete();
	mInteractorStyleCaption->Delete();

	mUpdateTimer->Delete();
	mAnimator->Delete();

	for(i=0; i<__NumInteractorStyles; i++) mInteractorStyle[i]->Delete();

	mRenderTool->RemoveObject(mBox1Actor);
	mBox1Actor->Delete();
	mBox2Actor->Delete();
	mBox3Actor->Delete();
	mDebugActor->Delete();

	mInteractorStyleMeasuringBox->Delete();

	mRenderTool->RemoveObject(mIconActor);
	mIconActor->Delete();

	mClipPlane->Delete();

	if(!this->IsClone()) 
	{
		mDataReader->Delete();
	}
	mWriter->Delete();

	mObsProgress->Delete();
	mObsSlaveAbortRender->Delete();
	mObsAbortRender->Delete();

	//
	//  Extensions may use RenderTool, so we need to delete them before we delete mRenderTool
	//
	while(mExtensions.Size() > 0) mExtensions.RemoveLast()->Delete();

	mRenderTool->Delete();

	//
	//  Check that all DataConsumers have been deleted too
	//
	if(mDataConsumers.Size() > 0)
	{
		IERROR_FATAL("Not all DataConsumers have been deleted.");
	}
}


void iViewModule::UpdateWindowNumber()
{
	int i;
	iString windowName = "IFrIT - Visualization Window";

	for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		if(this->GetControlModule()->GetViewModule(i) == this)
		{
			mWinNum = i;
			break;
		}
	}
	if(this->GetControlModule()->GetNumberOfViewModules() > 1) windowName += " #" + iString::FromNumber(mWinNum+1);

	mRenderTool->UpdateWindowName(windowName);
	this->ClearCache();
}


bool iViewModule::BecomeClone(iViewModule *v)
{
	int i;
	bool ret = true;

	if(v == 0)
	{
		//
		//	Unmake the clone
		//
		v = mDataReader->GetViewModule(); 
		if(v != this)
		{
#ifdef I_CHECK1
			if(v != mCloneOfModule)
			{
				IERROR_LOW("Clones configured incorrectly.");
			}
#endif
			mCloneOfModule = 0;
			v->mClones.Remove(this);
			//
			//  Re-create data reader
			//
			mDataReader = iDataReader::New(this); IERROR_ASSERT(mDataReader);
			mDataReader->CopyState(v->mDataReader);

			for(i=0; i<mViewObjectFamilies.Size(); i++) mViewObjectFamilies[i]->Reset();

			for(i=0; i<=mViewObjectFamilies[_ObjectTypeParticles]->GetMaxMemberIndex(); i++)
			{
				this->GetParticlesViewSubject(i)->BecomeClone(0);
			}
			//
			//  Un-show label
			//
			this->ShowLabel(mIsLabelOn); 
		}
	}
	else
	{
		//
		//	Make the clone
		//
		if(mDataReader->GetViewModule() == this)
		{
			mCloneOfModule = v;
			v->mClones.Add(this);
			//
			//  Link the Data Reader
			//
			mDataReader->Delete();
			mDataReader = v->mDataReader;
			//
			//  Copy iViewSubject families
			//
			ret = ret & mMarkerFamily->Copy(v->mMarkerFamily);

			for(i=0; i<mViewObjectFamilies.Size(); i++) ret = ret & mViewObjectFamilies[i]->Copy(v->mViewObjectFamilies[i]);

			//
			//  Particles family is never shown as a whole, so show individual members
			//
			for(i=0; i<=mViewObjectFamilies[_ObjectTypeParticles]->GetMaxMemberIndex(); i++)
			{
				this->GetParticlesViewSubject(i)->BecomeClone(v->GetParticlesViewSubject(i));
				this->GetParticlesViewSubject(i)->Show(v->GetParticlesViewSubject(i)->IsVisible());
			}
			//
			//  Copy Animator state
			//
			mAnimator->CopyState(v->mAnimator);
			//
			//  Copy camera state
			//
			mRenderTool->GetCamera()->CopyState(v->mRenderTool->GetCamera());
			//
			//  Show label
			//
			this->ShowLabel(v->IsLabelVisible()); 
		}
	}

	this->ClearCache();
	return ret;
}


void iViewModule::SetDebugMode(bool s)
{
	int i;

	if(s != mIsDebug)
	{
		mIsDebug = s;
		if(s)
		{
			
			mVisibleProps.Clear();
			for(i=0; i<mProps.Size(); i++) if(mProps[i]->GetVisibility())
			{
				mVisibleProps.Add(mProps[i]);
				mProps[i]->VisibilityOff();
			}
			
			mDebugActor->VisibilityOn();
		}
		else
		{
			
			mAnimator->SetDebugFlag(0);
			mDebugActor->VisibilityOff();
			
			for(i=0; i<mVisibleProps.Size(); i++)
			{
				mVisibleProps[i]->VisibilityOn();
			}
		}
		this->Render();
	}
	this->ClearCache();
}


void iViewModule::AddObject(vtkProp* p)
{
	mRenderTool->AddObject(p);
}


void iViewModule::RemoveObject(vtkProp* p)
{
	mRenderTool->RemoveObject(p);
}


vtkRenderWindowCollection* iViewModule::GetRenderWindowCollection(bool inited) const
{
	vtkRenderWindowCollection *c = mRenderTool->GetRenderWindowCollection();
	if(inited) c->InitTraversal();
	return c;
}


iParallelManager* iViewModule::GetParallelManager() const
{
	return this->GetControlModule()->GetParallelManager();
}


void iViewModule::Render()  
{
	//
	//  Activate counting for parallel performance
	//
	this->GetParallelManager()->StartCounters();
	mUpdateTimer->StartTimer();

	mObsAbortRender->SetInteractive(false);
	mRenderTool->Render();
	mObsAbortRender->SetInteractive(true);

	mUpdateTimer->StopTimer();
	mUpdateTime = mUpdateTimer->GetElapsedTime();
	this->UpdatePerformance();
	this->GetParallelManager()->StopCounters();
}


void iViewModule::UpdatePerformance()
{
	mObsAbortRender->PostFinished(); // manually driven
}


void iViewModule::UpdateAfterFileLoad()
{
	this->UpdateLabel();
	this->UpdatePerformance();
	//
	//  Also make sure the current particles have the data (if there is any particle data)
	//
	if(!mViewObjectFamilies[_ObjectTypeParticles]->GetCurrentMember()->IsThereData())
	{
		int i;
		for(i=0; i<=mViewObjectFamilies[_ObjectTypeParticles]->GetMaxMemberIndex(); i++)
		{
			if(mViewObjectFamilies[_ObjectTypeParticles]->GetMember(i)->IsThereData())
			{
				mViewObjectFamilies[_ObjectTypeParticles]->SetCurrentMemberIndex(i);
				break;
			}
		}
	}
}


void iViewModule::SetAxesBox(const iString &labelX, const iString &labelY, const iString &labelZ, float xMin, float xMax, float yMin, float yMax, float zMin, float zMax)
{
	mBox3AxesLabels->SetXLabel(labelX.ToCharPointer());
	mBox3AxesLabels->SetYLabel(labelY.ToCharPointer());
	mBox3AxesLabels->SetZLabel(labelZ.ToCharPointer());
	mBox3AxesLabels->SetRanges(xMin,xMax,yMin,yMax,zMin,zMax);
	this->ClearCache();
}


void iViewModule::SetFontScale(int s)
{
	if(s>-10 && s<10)
	{
		mFontScale = s; 
		this->ClearCache();
	}
}


void iViewModule::SetFontType(int s)
{
	mLabelActor->SetFontType(s); 
	if(s == mLabelActor->GetFontType())
	{
		mFontType = s; 
//		mRulerActor->SetFontType(s);
		mMeasuringBox->GetTextActor()->SetFontType(s);
		mColorBars->SetFontType(s);
		this->ClearCache();
	}
}


void iViewModule::ShowBoundingBox(bool s)
{
	this->ClearCache();

	mBox1Actor->VisibilityOff(); 
	mBox2Actor->VisibilityOff();
	mBox3Actor->VisibilityOff();
	mBox3AxesLabels->VisibilityOff();
	mIsBoundingBoxOn = s;

	if(!mIsBoundingBoxOn) return;
		
	switch(mBoxType)
	{
	case _BoundingBoxTypeDefault:
		{
			mBox1Actor->GetMapper()->ScalarVisibilityOn();
			mBox1Actor->GetProperty()->SetLineWidth(4.0);
			mBox1Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeClassic:
		{
			mBox2Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeHairThin:
		{
			mBox1Actor->GetMapper()->ScalarVisibilityOff();
			mBox1Actor->GetProperty()->SetLineWidth(1.0);
			mBox1Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeAxes:
		{
			mBox3Actor->VisibilityOn();
			mBox3AxesLabels->VisibilityOn();
			break;
		}
	}
}


void iViewModule::ShowRuler(bool s)
{
	mRulerActor->SetVisibility(s?1:0);
	mIsRulerOn = s;
	this->ClearCache();
}


void iViewModule::ShowColorBars(bool s)
{
	mColorBars->GetActor(false)->SetVisibility(s?1:0);
	mColorBars->GetActor(true)->SetVisibility(s?1:0);
	mIsColorBarsOn = s;
	this->ClearCache();
}


void iViewModule::ShowMeasuringBox(bool s)
{
	if(s)
	{
		mMeasuringBox->VisibilityOn(); 
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyleMeasuringBox);
	} 
	else 
	{
		mMeasuringBox->VisibilityOff();
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
	}
	mIsMeasuringBoxOn = s;
	this->ClearCache();
}


void iViewModule::ShowGlassClipPlane(bool s)
{
	mIsGlassClipPlaneOn = s;
	if(mIsClipPlaneOn) mClipPlane->SetGlassPlaneVisible(mIsGlassClipPlaneOn);
	this->ClearCache();
}


void iViewModule::ShowClipPlane(bool s)
{
	int i;
	for(i=0; i<mViewObjectFamilies.Size(); i++) mViewObjectFamilies[i]->ShowClipPlane(s);

	mIsClipPlaneOn = s;
	if(mIsClipPlaneOn) mClipPlane->SetGlassPlaneVisible(mIsGlassClipPlaneOn); else mClipPlane->SetGlassPlaneVisible(false); 
	this->ClearCache();
}


void iViewModule::ShowCameraAlignmentLabel(bool s)
{
	mIsCameraAlignmentLabelOn = s;
	mCameraOrthoActor->SetVisibility(s?1:0);
	this->ClearCache();
}


void iViewModule::ShowStereoAlignmentMarkers(bool s)
{
	mIsStereoAlignmentMarkersOn = s;
	mRenderTool->ShowStereoAlignmentMarkers(s);
	this->ClearCache();
}


void iViewModule::UpdateLabel(float *v)
{
	this->ShowLabel(mIsLabelOn,v);
	//
	//  Update my clones as well
	//
	int i;
	for(i=0; i<mClones.Size(); i++) mClones[i]->UpdateLabel(v);
}


void iViewModule::ShowLabel(bool s, float *pv)
{
	mIsLabelOn = s;
	if(s && mDataReader->IsFileAnimatable())
	{
		iString s;
		int r = mDataReader->GetRecordNumber();
		float v;
		iString unit(mLabelUnit);
		if(pv == 0)
		{
			if(fabs(mLabelScale) > 0.0) v = mLabelScale*(r-mLabelOffset); else
			{
				unit.Clear();
				v = 1.0e4/((r>0)?r:1)-1.0;
			}
		}
		else
		{
			v = *pv;
		}
		//
		//  Auto unit expansion
		//
		if(!unit.IsEmpty() && unit[0]=='.')
		{
			//
			//  Use SI prefixes (from wikipedia.org)
			//
			if(v > 1.0e24)  { v *= 1.0e-24; unit.Replace(0,1,"Y"); } else
			if(v > 1.0e21)  { v *= 1.0e-21; unit.Replace(0,1,"Z"); } else
			if(v > 1.0e18)  { v *= 1.0e-18; unit.Replace(0,1,"E"); } else
			if(v > 1.0e15)  { v *= 1.0e-15; unit.Replace(0,1,"P"); } else
			if(v > 1.0e12)  { v *= 1.0e-12; unit.Replace(0,1,"T"); } else
			if(v > 1.0e9)   { v *= 1.0e-9;  unit.Replace(0,1,"G"); } else
			if(v > 1.0e6)   { v *= 1.0e-6;  unit.Replace(0,1,"M"); } else
			if(v > 1.0e3)   { v *= 1.0e-3;  unit.Replace(0,1,"k"); } else
			if(v>0.0 && v<1.0e-23) { v *= 1.0e24;  unit.Replace(0,1,"y"); } else
			if(v>0.0 && v<1.0e-20) { v *= 1.0e21;  unit.Replace(0,1,"z"); } else
			if(v>0.0 && v<1.0e-17) { v *= 1.0e18;  unit.Replace(0,1,"a"); } else
			if(v>0.0 && v<1.0e-14) { v *= 1.0e15;  unit.Replace(0,1,"f"); } else
			if(v>0.0 && v<1.0e-11) { v *= 1.0e12;  unit.Replace(0,1,"p"); } else
			if(v>0.0 && v<1.0e-8)  { v *= 1.0e9;   unit.Replace(0,1,"n"); } else
			if(v>0.0 && v<1.0e-5)  { v *= 1.0e6;   unit.Replace(0,1,"u"); } else
			if(v>0.0 && v<1.0e-2)  { v *= 1.0e3;   unit.Replace(0,1,"m"); } else
			unit = unit.Part(1);

		}

		iString format = "%-." + iString::FromNumber(mLabelDigits) + "f";
		iString num = iString::FromNumber(v,format.ToCharPointer());
		//
		//  Beatify
		//
		int i = num.Find('.');
		if(i == -1) i = num.Length();
		while(i > 3)
		{
			i -= 3;
			num = num.Part(0,i) + "," + num.Part(i);
		}
		//
		//  Form label
		//
		s = mLabelName + "=" + num + unit;
		mLabelActor->SetText(s);
		mLabelActor->VisibilityOn(); 	
	} 
	else mLabelActor->VisibilityOff();
	this->ClearCache();
}


void iViewModule::JustifyLabelLeft(bool s)
{
	if(s)
	{
		if(!mLabelJustificationLeft)
		{
			mLabelJustificationLeft = true;
			float s[2];
			const float *p;
			mLabelActor->GetSize(s);
			mLabelActor->SetHJustification(_TextJustLeft);
			p = mLabelActor->GetPosition();
			mLabelActor->SetPosition(p[0]-s[0],p[1]);
		}
	}
	else
	{
		if(mLabelJustificationLeft)
		{
			mLabelJustificationLeft = false;
			float s[2];
			const float *p;
			mLabelActor->GetSize(s);
			mLabelActor->SetHJustification(_TextJustRight);
			p = mLabelActor->GetPosition();
			mLabelActor->SetPosition(p[0]+s[0],p[1]);
		}
	}
}


void iViewModule::SetUpdateRate(int r)
{
	r = (r<0) ? 0 : r;
	r = (r>60) ? 60 : r;
	mRate = r;
	mRenderTool->GetInteractor()->SetDesiredUpdateRate(mRate);
	this->ClearCache();
}


void iViewModule::SetInteractorStyle(int n)
{
	if(n>=0 && n<__NumInteractorStyles)
	{
		//
		//  Specials
		//
		if(n == _InteractorStyleFlight)
		{
			mSavedProjection = mRenderTool->GetCamera()->GetDevice()->GetParallelProjection();
			mRenderTool->GetCamera()->GetDevice()->ParallelProjectionOff();
		} 
		else if(mCurrentInteractorStyle == _InteractorStyleFlight)
		{
			mRenderTool->GetCamera()->GetDevice()->SetParallelProjection(mSavedProjection);
			mRenderTool->GetCamera()->Reset();
		}
		mCurrentInteractorStyle = n;
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
		this->ClearCache();
	}
}


void iViewModule::SetBackgroundColor(const iColor &bg)
{
	mBackgroundColor = bg;
	iColor fg = mBackgroundColor.Reverse();

	mRenderTool->SetBackground(mBackgroundColor);
	mMeasuringBox->SetColor(fg);
	mBox1Actor->GetProperty()->SetColor(fg.ToVTK());
	mAnimator->SetCameraPathColor(fg);
	mMarkerPlacer->GetProperty()->SetColor(fg.ToVTK());
	this->ClearCache();
}


void iViewModule::SetBackgroundImage(const iString &name)
{
	iImage image;

	if(image.LoadFromFile(name))
	{
		mBackgroundImage = name;
	}
	else
	{
		image.Clear();
		mBackgroundImage.Clear();
	}

	mRenderTool->SetBackground(image);
	this->ClearCache();
}


void iViewModule::SetBoundingBoxType(int s)
{
	if(s!=mBoxType && s>=0 && s<__NumBoundingBoxTypes)
	{
		mBoxType = s;
		this->ShowBoundingBox(mIsBoundingBoxOn);
		this->ClearCache();
	}		
}


void iViewModule::SetAntialiasing(bool s)
{
	mAntialiasing = s;
	mRenderTool->SetAntialiasing(s);
	this->ClearCache();
}


void iViewModule::UpdateIcon()
{
	int i;

	if(this->GetControlModule()->GetImageComposer()->InComposing()) return;

	if(this->GetControlModule()->GetNumberOfViewModules() == 1) 
	{
		mIconActor->VisibilityOff(); 
	}
	else 
	{
		if(mWinNum == this->GetControlModule()->GetCurrentViewModuleIndex())
		{
			//
			//  Current window
			//
			mIconActor->VisibilityOn(); 
			mIconActor->GetProperty()->SetOpacity(1.0);
			for(i=0; i<mClones.Size(); i++)
			{ 
				mClones[i]->mIconActor->VisibilityOn(); 
				mClones[i]->mIconActor->GetProperty()->SetOpacity(0.5);
				mClones[i]->Render();
			}
		}
		else
		{ 
			mIconActor->VisibilityOff();
			for(i=0; i<mClones.Size(); i++)
			{
				mClones[i]->mIconActor->VisibilityOff();
				mClones[i]->Render();
			}
		}
		this->Render();
	}
}


void iViewModule::SetImageMagnification(int m)
{
	if(m>0 && m<101)
	{
		mImageMagnification = m;
		this->GetControlModule()->GetImageComposer()->UpdateSize();
		this->ClearCache();
	}
}


int iViewModule::GetFullImageWidth()
{
	return this->GetControlModule()->GetImageComposer()->GetImageWidth();
}


int iViewModule::GetFullImageHeight()
{
	return this->GetControlModule()->GetImageComposer()->GetImageHeight();
}


int iViewModule::GetThisImageWidth()
{
	const int *s = mRenderTool->GetRenderWindowSize();
	return mImageMagnification*s[0];
}


int iViewModule::GetThisImageHeight()
{
	const int *s = mRenderTool->GetRenderWindowSize();
	return mImageMagnification*s[1];
}


void iViewModule::CreateImages(iStereoImageArray &images)
{
	images.Clear();
	//
	//  ask composer to composer the image
	//
	this->GetErrorStatus()->Monitor(this->GetControlModule()->GetImageComposer()->GetErrorStatus());
	this->GetControlModule()->GetImageComposer()->Compose(this,mImages);

	if(this->GetErrorStatus()->NoError())
	{
		int i;
		images.Clear();
		for(i=0; i<mImages.Size(); i++) images.Add(mImages[i]);
	}
}


void iViewModule::RenderImages(iStereoImageArray &images)
{
	//
	//  Prepare for render; do a few more things for better representation
	//
	int iconVis = mIconActor->GetVisibility();
	mIconActor->VisibilityOff();
	float box2LineWidth = mBox1Actor->GetProperty()->GetLineWidth();
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth*mImageMagnification);
	
	//
	//  do the render
	//
	mInImageRender = true;		
	mRenderTool->RenderImages(mImageMagnification,images);
	mInImageRender = false;

	//
	//  Reset to the present state
	//
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth);
	mIconActor->SetVisibility(iconVis);
		
	//
	// render again to restore the original view
	//
	mRenderTool->Render();
}


void iViewModule::RenderStereoImage(iStereoImage &image)
{
	//
	//  Prepare for render; do a few more things for better representation
	//
	int iconVis = mIconActor->GetVisibility();
	mIconActor->VisibilityOff();
	float box2LineWidth = mBox1Actor->GetProperty()->GetLineWidth();
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth*mImageMagnification);
	
	//
	//  do the render
	//
	mInImageRender = true;		
	mRenderTool->RenderStereoImage(mImageMagnification,image);
	mInImageRender = false;

	//
	//  Reset to the present state
	//
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth);
	mIconActor->SetVisibility(iconVis);
		
	//
	// render again to restore the original view
	//
	mRenderTool->Render();
}


void iViewModule::DumpImages(int m)
{
	static const iStereoImageArray dummy;
	this->DumpImages(dummy,m);
}


void iViewModule::DumpImages(const iStereoImageArray &images, int m)
{
	int i;
	iStereoImageArray tmp;

	if(images.Size() != mRenderTool->GetNumberOfActiveViews()) 
	{
		this->CreateImages(tmp);
		if(images.Size() != 0) IERROR_LOW("Incorrect number of images in ImageArray in DumpImages."); 
	}
	else
	{
		for(i=0; i<images.Size(); i++) tmp.Add(images[i]);
	}

	//
	// Create and write files
	//
	if(this->GetErrorStatus()->NoError())
	{
		mObsProgress->SetMode(iProgressEventObserver::_Writing);  // auto-driven
		mWriter->WriteFrame(tmp,m);
	}
	else
	{
		iConsole::Display(iConsole::_HighError,"Failed to create the image: "+this->GetErrorStatus()->Message());
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iViewModule::ExtendableObjectPackStateBody(iString &s) const
{
	int i; float fv[6]; iString sv[3];

	this->PackValue(s,KeyWindowNumber(),mWinNum);
	this->PackValue(s,KeyCloneOfWindow(),this->GetCloneOfWindow());
	this->PackValue(s,KeyNoClone(),!this->IsClone());

//	this->PackValue(s,KeyPosition(),win->GetPosition(),2);
	this->PackValue(s,KeySize(),mRenderTool->GetRenderWindowSize(),2);

	this->PackValue(s,KeyInteractorStyle(),mCurrentInteractorStyle);
	this->PackValue(s,KeyLabel(),mIsLabelOn);
	this->PackValue(s,KeyCameraAlignmentLabel(),mIsCameraAlignmentLabelOn);
	this->PackValue(s,KeyColorBars(),mIsColorBarsOn);
	this->PackValue(s,KeyBoundingBox(),mIsBoundingBoxOn);
	this->PackValue(s,KeyRuler(),mIsRulerOn);
	this->PackValue(s,KeyGlassClipPlane(),mIsGlassClipPlaneOn);
	this->PackValue(s,KeyClipPlane(),mIsClipPlaneOn);
	this->PackValue(s,KeyMeasuringBox(),mIsMeasuringBoxOn);
	this->PackValue(s,KeyAntialiasing(),mAntialiasing);
	this->PackValue(s,KeyJustifyLabelLeft(),mLabelJustificationLeft);
	this->PackValue(s,KeyMarkerLegend(),mIsMarkerLegendOn);
	this->PackValue(s,KeyStereo(),this->GetStereo());
	this->PackValue(s,KeyStereoAlignmentMarkers(),mIsStereoAlignmentMarkersOn);

	this->PackValue(s,KeyLabelName(),mLabelName);
	this->PackValue(s,KeyLabelUnit(),mLabelUnit);
	this->PackValue(s,KeyLabelScale(),mLabelScale);
	this->PackValue(s,KeyLabelOffset(),mLabelOffset);
	this->PackValue(s,KeyLabelDigits(),mLabelDigits);

	this->PackValue(s,KeyFontScale(),mFontScale);
	this->PackValue(s,KeyFontType(),mFontType);
	this->PackValue(s,KeyBoundingBoxType(),mBoxType);
	this->PackValue(s,KeyImageFormat(),mWriter->GetImageFormat());
	this->PackValue(s,KeyAnimationOutput(),mWriter->GetAnimationOutput());
	this->PackValue(s,KeyMarkerLegendPosition(),mMarkerLegendPosition);
	this->PackValue(s,KeyImageMagnification(),mImageMagnification);
	this->PackValue(s,KeyStereoType(),this->GetStereoType());

	this->PackValue(s,KeyBackgroundColor(),mBackgroundColor);
	this->PackValue(s,KeyBackgroundImage(),mBackgroundImage);

	this->PackValue(s,KeyUpdateRate(),mRate);
	this->PackValue(s,KeyRulerScale(),this->GetRulerScale());
	this->PackValue(s,KeyRulerTitle(),this->GetRulerTitle());
	this->PackValue(s,KeyOpenGLCoordinates(),this->GetOpenGLCoordinates());
	this->PackValue(s,KeyBoxSize(),this->GetBoxSize());

	this->PackValue(s,KeyClipPlaneDistance(),this->GetClipPlaneDistance());
	this->PackValue(s,KeyClipPlaneDirection(),this->GetClipPlaneDirection(),3);

	this->PackValue(s,KeyPostScriptPaperFormat(),mWriter->GetPostScriptPaperFormat());
	this->PackValue(s,KeyPostScriptOrientation(),mWriter->GetPostScriptOrientation());

	sv[0] = mBox3AxesLabels->GetXLabel();
	sv[1] = mBox3AxesLabels->GetYLabel();
	sv[2] = mBox3AxesLabels->GetZLabel();
	for(i=0; i<6; i++) fv[i] = mBox3AxesLabels->GetRanges()[i];
	this->PackValue(s,KeyAxesBoxLabels(),sv,3);
	this->PackValue(s,KeyAxesBoxRanges(),fv,6);

	//
	//  Light parameters
	//
	mRenderTool->GetLights()->GetAngles(fv);
	this->PackValue(s,KeyLightAngles(),fv,2);
	mRenderTool->GetLights()->GetIntensities(fv);
	this->PackValue(s,KeyLightIntensity(),fv,3);

	//
	//  Extra keys for data quering
	//
	this->PackValue(s,KeyMarkerCurrent(),mMarkerFamily->GetCurrentMemberIndex());
	this->PackValue(s,KeyCrossSectionCurrent(),mViewObjectFamilies[_ObjectTypeCrossSection]->GetCurrentMemberIndex());
	this->PackValue(s,KeyParticlesCurrent(),this->GetParticlesObject()->GetActiveDataTypeIndex());
	this->PackValue(s,KeySurfaceCurrent(),mViewObjectFamilies[_ObjectTypeSurface]->GetCurrentMemberIndex());

	i = mMarkerFamily->GetMaxMemberIndex();
	if(i==0 && !mMarkerFamily->GetMember(0)->IsVisible()) i = -1;
	this->PackValue(s,KeyMarkerMax(),i);
	this->PackValue(s,KeyParticlesMax(),this->GetParticlesObject()->GetNumberOfTypes()-1);
	this->PackValue(s,KeySurfaceMax(),mViewObjectFamilies[_ObjectTypeSurface]->GetMaxMemberIndex());
	this->PackValue(s,KeyCrossSectionMax(),mViewObjectFamilies[_ObjectTypeCrossSection]->GetMaxMemberIndex());

	this->PackValue(s,KeyPlaceMarker(),mMarkerPlacer->GetEnabled()!=0);
	this->PackValue(s,KeyMoveMarkerCaption(),mRenderTool->GetInteractor()->GetInteractorStyle()==mInteractorStyleCaption);
}


void iViewModule::ExtendableObjectUnPackStateBody(const iString &s)
{
	int i, iv[2]; bool b; float f, fv[6]; iColor c; iString w, sv[3];

	//
	//  Clone (since clones are always created after the parent window, this should work always
	//
	if(this->UnPackValue(s,KeyCloneOfWindow(),i)) if(i >= 0) BecomeClone(this->GetControlModule()->GetViewModule(i));

//	if(this->UnPackValue(s,KeyPosition(),iv,2)) win->SetPosition(iv);
	iv[0] = mRenderTool->GetRenderWindowSize()[0];
	iv[1] = mRenderTool->GetRenderWindowSize()[1];
	if(this->UnPackValue(s,KeySize(),iv,2))
	{
		mRenderTool->SetRenderWindowSize(iv[0],iv[1]);
		this->ClearCache();
	}

	//
	//  These two must be set before many other keys
	//
	if(this->UnPackValue(s,KeyOpenGLCoordinates(),b)) this->SetOpenGLCoordinates(b);
	if(this->UnPackValue(s,KeyBoxSize(),f)) this->SetBoxSize(f);

	if(this->UnPackValue(s,KeyInteractorStyle(),i)) this->SetInteractorStyle(i);
	if(this->UnPackValue(s,KeyLabel(),b)) this->ShowLabel(b);
	if(this->UnPackValue(s,KeyCameraAlignmentLabel(),b)) this->ShowCameraAlignmentLabel(b);
	if(this->UnPackValue(s,KeyColorBars(),b)) this->ShowColorBars(b);
	if(this->UnPackValue(s,KeyBoundingBox(),b)) this->ShowBoundingBox(b);
	if(this->UnPackValue(s,KeyRuler(),b)) this->ShowRuler(b);
	if(this->UnPackValue(s,KeyGlassClipPlane(),b)) this->ShowGlassClipPlane(b);
	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);
	if(this->UnPackValue(s,KeyMeasuringBox(),b)) this->ShowMeasuringBox(b);
	if(this->UnPackValue(s,KeyAntialiasing(),b)) this->SetAntialiasing(b);
//	if(this->UnPackValue(s,KeyJustifyLabelLeft(),b)) this->justifyLabelLeft(b);
	if(this->UnPackValue(s,KeyMarkerLegend(),b)) this->ShowMarkerLegend(b);
	if(this->UnPackValue(s,KeyStereo(),b)) this->SetStereo(b);
	if(this->UnPackValue(s,KeyStereoAlignmentMarkers(),b)) this->ShowStereoAlignmentMarkers(b);

	iObject::ReportMissingKeys(false);
	if(this->UnPackValue(s,KeyFontSize(),i)) this->SetFontScale(round(log(i/16.0)/0.3)); // for backward compatibility only
	iObject::ReportMissingKeys(true);

	if(this->UnPackValue(s,KeyFontScale(),i)) this->SetFontScale(i);
	if(this->UnPackValue(s,KeyFontType(),i)) this->SetFontType(i);
	if(this->UnPackValue(s,KeyBoundingBoxType(),i)) this->SetBoundingBoxType(i);
	if(this->UnPackValue(s,KeyImageFormat(),i)) mWriter->SetImageFormat(i);
	if(this->UnPackValue(s,KeyAnimationOutput(),i)) mWriter->SetAnimationOutput(i);
	if(this->UnPackValue(s,KeyMarkerLegendPosition(),i)) this->SetMarkerLegendPosition(i);
	if(this->UnPackValue(s,KeyImageMagnification(),i)) this->SetImageMagnification(i);
	if(this->UnPackValue(s,KeyStereoType(),i)) this->SetStereoType(i);
	
	if(this->UnPackValue(s,KeyLabelName(),w)) this->SetLabelName(w);
	if(this->UnPackValue(s,KeyLabelUnit(),w)) this->SetLabelUnit(w);
	if(this->UnPackValue(s,KeyLabelScale(),f)) this->SetLabelScale(f);
	if(this->UnPackValue(s,KeyLabelOffset(),f)) this->SetLabelOffset(f);
	if(this->UnPackValue(s,KeyLabelDigits(),i)) this->SetLabelDigits(i);

	if(this->UnPackValue(s,KeyBackgroundColor(),c)) this->SetBackgroundColor(c);
	if(this->UnPackValue(s,KeyBackgroundImage(),w)) this->SetBackgroundImage(w);

	if(this->UnPackValue(s,KeyUpdateRate(),i)) this->SetUpdateRate(i);
	if(this->UnPackValue(s,KeyRulerScale(),f)) this->SetRulerScale(f);
	if(this->UnPackValue(s,KeyRulerTitle(),w)) this->SetRulerTitle(w);

	mClipPlane->GetDirection(fv);
	if(this->UnPackValue(s,KeyClipPlaneDirection(),fv,3)) this->SetClipPlaneDirection(fv);
	if(this->UnPackValue(s,KeyClipPlaneDistance(),f)) this->SetClipPlaneDistance(f);

	if(this->UnPackValue(s,KeyPostScriptPaperFormat(),i)) mWriter->SetPostScriptPaperFormat(i);
	if(this->UnPackValue(s,KeyPostScriptOrientation(),i)) mWriter->SetPostScriptOrientation(i);

	sv[0] = mBox3AxesLabels->GetXLabel();
	sv[1] = mBox3AxesLabels->GetYLabel();
	sv[2] = mBox3AxesLabels->GetZLabel();
	for(i=0; i<6; i++) fv[i] = mBox3AxesLabels->GetRanges()[i];
	if(this->UnPackValue(s,KeyAxesBoxLabels(),sv,3) || this->UnPackValue(s,KeyAxesBoxRanges(),fv,6))
	{
		this->SetAxesBox(sv[0],sv[1],sv[2],fv[0],fv[1],fv[2],fv[3],fv[4],fv[5]);
	}

	//
	//  Light parameters
	//
	mRenderTool->GetLights()->GetAngles(fv);
	if(this->UnPackValue(s,KeyLightAngles(),fv,2))
	{
		mRenderTool->GetLights()->SetAngles(fv[0],fv[1]);
		this->ClearCache();
	}
	mRenderTool->GetLights()->GetIntensities(fv);
	if(this->UnPackValue(s,KeyLightIntensity(),fv,3))
	{
		mRenderTool->GetLights()->SetIntensities(fv[0],fv[1],fv[2]);
		this->ClearCache();
	}

	//
	//  Current instances
	//
	if(this->UnPackValue(s,KeyMarkerCurrent(),i))
	{
		mMarkerFamily->SetCurrentMemberIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyCrossSectionCurrent(),i))
	{
		mViewObjectFamilies[_ObjectTypeCrossSection]->SetCurrentMemberIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyParticlesCurrent(),i))
	{
		this->GetParticlesObject()->SetActiveDataTypeIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeySurfaceCurrent(),i))
	{
		mViewObjectFamilies[_ObjectTypeSurface]->SetCurrentMemberIndex(i);
		this->ClearCache();
	}

	//
	//  "Action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyPlaceMarker(),b)) this->PlaceMarker(b);
	if(this->UnPackValue(s,KeyMoveMarkerCaption(),b)) this->MoveMarkerCaption(b);
	if(this->UnPackValue(s,KeyDumpImage(),b) && b) this->DumpImages();

	iObject::ReportMissingKeys(true);
}


void iViewModule::PlaceMarker(bool s)
{
	if(s)
	{
		mMarkerPlacer->SetPosition(mMarkerFamily->GetCurrentMember()->GetPosition()[0],mMarkerFamily->GetCurrentMember()->GetPosition()[1],mMarkerFamily->GetCurrentMember()->GetPosition()[2]);
		mMarkerPlacer->On();
	}
	else
	{
		mMarkerPlacer->Off();
	}
}


void iViewModule::MoveMarkerCaption(bool s)
{
	if(s)
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyleCaption);
	}
	else
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
	}
}


void iViewModule::ShowMarkerLegend(bool s)
{
	mIsMarkerLegendOn = s;
	if(s) 
	{
		this->BuildMarkerLegend();
		mMarkerLegend->VisibilityOn(); 
	}
	else mMarkerLegend->VisibilityOff();
	this->ClearCache();
}


void iViewModule::BuildMarkerLegend(int n)
{
	int i;
	if(n<0 || n>mMarkerFamily->GetMaxMemberIndex())
	{
		mMarkerLegend->SetNumberOfEntries(1+mMarkerFamily->GetMaxMemberIndex());
		for(i=0; i<=mMarkerFamily->GetMaxMemberIndex(); i++)
		{
			//
			//  Fix a bug in vtkLegendBoxActor
			//
			if(mMarkerFamily->GetMember(i)->GetCaptionText().Length() > 0) mMarkerLegend->SetEntry(i,mMarkerFamily->GetMember(i)->GetMarkerObject()->GetOutput(),mMarkerFamily->GetMember(i)->GetCaptionText().ToCharPointer(),mMarkerFamily->GetMember(i)->GetColor(0).ToVTK()); else mMarkerLegend->SetEntry(i,mMarkerFamily->GetMember(i)->GetMarkerObject()->GetOutput()," ",mMarkerFamily->GetMember(i)->GetColor(0).ToVTK());
		}
	}
	else
	{
		//
		//  Fix a bug in vtkLegendBoxActor
		//
		if(mMarkerFamily->GetMember(n)->GetCaptionText().Length() > 0) mMarkerLegend->SetEntry(n,mMarkerFamily->GetMember(n)->GetMarkerObject()->GetOutput(),mMarkerFamily->GetMember(n)->GetCaptionText().ToCharPointer(),mMarkerFamily->GetMember(n)->GetColor(0).ToVTK()); else mMarkerLegend->SetEntry(n,mMarkerFamily->GetMember(n)->GetMarkerObject()->GetOutput()," ",mMarkerFamily->GetMember(n)->GetColor(0).ToVTK());
	}
}


void iViewModule::SetMarkerLegendPosition(int p)
{
	if(p>=0 && p<=1)
	{
		mMarkerLegendPosition = p;
		switch (p)
		{
		case 0:
			{
				mMarkerLegend->SetPosition(0.73,0.02);
				break;
			}
		case 1:
			{
				mMarkerLegend->SetPosition(0.02,0.02);
				break;
			}
		}
		this->ClearCache();
	}
}


float iViewModule::GetMemorySize() const
{
	int i;
	float s = 0.0;

	for(i=0; i<mViewObjectFamilies.Size(); i++) s += mViewObjectFamilies[i]->GetMemorySize();
	s += mDataReader->GetMemorySize();

	return s;
}


//
//  Coordinate transformation functions
//
void iViewModule::SetOpenGLCoordinates(bool s)
{
	if(s == (mBoxSize<0.0)) return;
	if(s) 
	{
		mBoxSize = -fabs(mBoxSize); 
		mBox3AxesLabels->SetRanges(-1.0,1.0,-1.0,1.0,-1.0,1.0);
	}
	else 
	{
		mBoxSize = fabs(mBoxSize);
		mBox3AxesLabels->SetRanges(0.0,mBoxSize,0.0,mBoxSize,0.0,mBoxSize);
	}
	if(mBoxType == 3) this->Render();
	this->ClearCache();
}


void iViewModule::SetBoxSize(float s)
{ 
	if(s > 0.0)
	{
		if(mBoxSize < 0) mBoxSize = -s; else
		{
			mBoxSize = s;
			mBox3AxesLabels->SetRanges(0.0,s,0.0,s,0.0,s); 
			if(mBoxType == 3) this->Render();
		}
		this->ClearCache();
	}
}


void iViewModule::SetClipPlaneDistance(float l)
{
	mClipPlane->SetDistance(l);
	this->ClearCache();
}


float iViewModule::GetClipPlaneDistance() const
{
	return mClipPlane->GetDistance();
}


void iViewModule::SetClipPlaneDirection(const float *nd)
{
	mClipPlane->SetDirection(nd);
	this->ClearCache();
}


const float* iViewModule::GetClipPlaneDirection() const
{
	return mClipPlane->GetDirection();
}


//
//  DataConsumer registry functionality
//
void iViewModule::RegisterDataConsumer(iDataConsumer *c)
{
	mDataConsumers.AddUnique(c);
}


void iViewModule::UnRegisterDataConsumer(iDataConsumer *c)
{
	mDataConsumers.Remove(c);
}


void iViewModule::NotifyDataConsumers(const iDataSyncRequest &r)
{
	if(mCloneOfModule != 0)
	{
		//
		//  If we are a clone, pass the request to the primary window.
		//
		mCloneOfModule->NotifyDataConsumers(r);
	}
	else
	{
		//
		//  If we an individual window, notify our consumers and all our clones
		//
		this->NotifyDataConsumersBody(r);

		int i;
		for(i=0; i<mClones.Size(); i++)
		{
			mClones[i]->NotifyDataConsumersBody(r);
		}
	}
}


void iViewModule::NotifyDataConsumersBody(const iDataSyncRequest &r)
{
	int i;
	for(i=0; i<mDataConsumers.Size(); i++) if(mDataConsumers[i]->IsUsingData(r.Data(),false))
	{
		mDataConsumers[i]->SyncWithData(r);
	}
}


void iViewModule::SetRulerScale(float s)
{
	mRulerActor->SetScale(s);
	this->ClearCache();
}


float iViewModule::GetRulerScale() const
{
	return mRulerActor->GetScale();
}


void iViewModule::SetRulerTitle(const iString &s)
{
	mRulerActor->SetTitle(s);
	this->ClearCache();
}


const iString& iViewModule::GetRulerTitle() const
{
	return mRulerActor->GetTitle();
}


void iViewModule::SetStereo(bool s)
{
	mRenderTool->SetStereo(s);
	this->ClearCache();
}


bool iViewModule::GetStereo() const
{
	return 	mRenderTool->GetStereo();
}


void iViewModule::SetStereoType(int n)
{
	mRenderTool->SetStereoType(n);
	this->ClearCache();
}


int iViewModule::GetStereoType() const
{
	return mRenderTool->GetStereoType();
}


//
//  Access to RenderTool components
//
void iViewModule::SetRenderWindowSize(int w, int h)
{
	if(mRenderTool != 0) mRenderTool->SetRenderWindowSize(w,h);
	this->ClearCache();
}

 
const int* iViewModule::GetRenderWindowSize() const
{ 
	return mRenderTool->GetRenderWindowSize(); 
}


vtkRenderer* iViewModule::GetRenderer() const
{
	return mRenderTool->GetRenderer();
}


vtkRenderWindow* iViewModule::GetRenderWindow() const
{
	return mRenderTool->GetRenderWindow();
}


vtkRenderWindowInteractor* iViewModule::GetInteractor() const
{
	return mRenderTool->GetInteractor(); 
}


void iViewModule::SetLabelName(const iString &s)
{
	mLabelName = s;
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelUnit(const iString &s)
{ 
	mLabelUnit = s; 
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelScale(float v)
{
	mLabelScale = v;
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelOffset(float v)
{
	mLabelOffset = v;
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelDigits(int v)
{
	if(v>=0 && v<=7)
	{
		mLabelDigits = v;
		this->UpdateLabel();
		this->ClearCache();
	}
}


void iViewModule::ForceCrossSectionToUsePolygonalMethod(bool s)
{
	int k;

	for(k=0; k<=mViewObjectFamilies[_ObjectTypeCrossSection]->GetMaxMemberIndex(); k++)
	{
		this->GetCrossSectionViewSubject(k)->ForcePolygonalMethod(s);
	}
}


iMarker* iViewModule::GetMarker(int n) const
{
	return (n==-1) ? mMarkerFamily->GetCurrentMember() : mMarkerFamily->GetMember(n);
}


iViewObject* iViewModule::GetParticlesObject(int n) const
{ 
	return (n==-1) ? mViewObjectFamilies[_ObjectTypeParticles]->GetCurrentMember() : mViewObjectFamilies[_ObjectTypeParticles]->GetMember(n); 
}


IVIEWMODULE_DEFINE_VIEWSUBJECT(CrossSection);
IVIEWMODULE_DEFINE_VIEWSUBJECT(Particles);
IVIEWMODULE_DEFINE_VIEWSUBJECT(Surface);
IVIEWMODULE_DEFINE_VIEWSUBJECT(TensorField);
IVIEWMODULE_DEFINE_VIEWSUBJECT(VectorField);
IVIEWMODULE_DEFINE_VIEWSUBJECT(Volume);


//
//  Extension class
//
IOBJECTEXTENSION_DEFINE_ABSTRACT(iViewModule);

void iViewModuleExtension::Define()
{
	mMinViewObjectFamilyIndex = mRealOwner->GetNumberOfViewObjectFamilies();
	mMaxViewObjectFamilyIndex = mMinViewObjectFamilyIndex - 1;
}


iViewObjectFamily* iViewModuleExtension::AddNewViewObjectFamily()
{
	iViewObjectFamily *fam = this->CreateObjectFamily(mMaxViewObjectFamilyIndex-mMinViewObjectFamilyIndex+1);
	if(fam != 0) mMaxViewObjectFamilyIndex++;
	return fam;
}


//
//  Helper functions in a private namespace
//
namespace iViewModule_Private
{
	iActor* CreateBox1(iRenderTool *rt)
	{
		//
		//  Bounding box 1
		//
		static float x[8][3]={{-1,-1,-1}, {1,-1,-1}, {1,1,-1}, {-1,1,-1}, {-1,-1,1}, {1,-1,1}, {1,1,1}, {-1,1,1}};
		static vtkIdType lns[12][2]={{0,1}, {1,2}, {2,3}, {3,0}, {0,4}, {1,5}, {2,6}, {3,7}, {4,5}, {5,6}, {6,7}, {7,4}};

		vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT(pd);
		vtkPoints *points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(points);
		vtkCellArray *polys = vtkCellArray::New(); IERROR_ASSERT(polys);
		vtkFloatArray *scalars = vtkFloatArray::New(); IERROR_ASSERT(scalars);
		//
		// Load the point, cell, and data attributes.
		//
		int i;
		for (i=0; i<8; i++) points->InsertPoint(i,x[i]);
		for (i=0; i<8; i++) scalars->InsertTuple1(i,(i<4));
		for (i=0; i<12; i++) polys->InsertNextCell(2,lns[i]);
		//
		// We now assign the pieces to the vtkPolyData.
		//
		pd->SetLines(polys);
		polys->Delete();
		pd->SetPoints(points);
		points->Delete();
		pd->GetPointData()->SetScalars(scalars);
		scalars->Delete();
		//
		//  Create the data and assign to the mapper
		//
		pd->Update();
		//
		//  Configure the actor
		//
		iActor* box1Actor = iActor::New(); IERROR_ASSERT(box1Actor);
		box1Actor->PickableOff();
		box1Actor->VisibilityOff();
		//	mBox1Actor->SetScalarRange(0,1);
		box1Actor->GetProperty()->SetAmbient(1.0);
		box1Actor->GetProperty()->SetDiffuse(0.0);
		box1Actor->GetProperty()->SetSpecular(0.0);
		box1Actor->GetProperty()->SetLineWidth(4.0);
		box1Actor->GetProperty()->SetColor(0.0,0.0,0.0);
		box1Actor->SetInput(pd);
		pd->Delete();

		rt->AddObject(box1Actor);

		return box1Actor;
	}


	iActorCollection* CreateBox2(iRenderTool *rt)
	{
		//
		//  Bounding box 2
		//
		float diam = 0.02;
		iActor *actor[12];
		vtkCylinderSource *cs;

		int i;
		for(i=0; i<12; i++) 
		{
			actor[i] = iActor::New(); IERROR_ASSERT(actor[i]);
			actor[i]->PickableOff();
			cs = vtkCylinderSource::New(); IERROR_ASSERT(cs);
			cs->SetResolution(6);
			cs->SetHeight(2.0+2.0*diam);
			cs->SetRadius(diam);
			cs->Update();
			actor[i]->SetInput(cs->GetOutput());
			cs->Delete();
		}

		for(i=0; i<4; i++) 
		{
			actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
			actor[i]->RotateX(90.0);
		}
		for(i=4; i<8; i++) actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
		for(i=8; i<12; i++) 
		{
			actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
			actor[i]->RotateZ(90.0);
		}

		actor[ 0]->SetPosition(-1.0,-1.0,0.0);
		actor[ 1]->SetPosition(-1.0, 1.0,0.0);
		actor[ 2]->SetPosition( 1.0,-1.0,0.0);
		actor[ 3]->SetPosition( 1.0, 1.0,0.0);
		actor[ 4]->SetPosition(-1.0,0.0,-1.0);
		actor[ 5]->SetPosition(-1.0,0.0, 1.0);
		actor[ 6]->SetPosition( 1.0,0.0,-1.0);
		actor[ 7]->SetPosition( 1.0,0.0, 1.0);
		actor[ 8]->SetPosition(0.0,-1.0,-1.0);
		actor[ 9]->SetPosition(0.0,-1.0, 1.0);
		actor[10]->SetPosition(0.0, 1.0,-1.0);
		actor[11]->SetPosition(0.0, 1.0, 1.0);

		iActorCollection* box2Actor = iActorCollection::New(); IERROR_ASSERT(box2Actor);
		for(i=0; i<12; i++) 
		{
			actor[i]->GetProperty()->SetAmbient(0.2);
			actor[i]->GetProperty()->SetDiffuse(0.2);
			actor[i]->GetProperty()->SetSpecular(0.2);
			box2Actor->AddActor(actor[i]);
			rt->AddObject(actor[i]);
			actor[i]->Delete();
		}
		box2Actor->SetVisibility(false);

		return box2Actor;
	}


	iActorCollection* CreateBox3(iRenderTool *rt)
	{
		//
		//  Bounding box 3
		//
		float diam = 0.02;
		iActor *actor[12];
		vtkArrowSource *as;
		vtkCylinderSource *cs;

		int i;
		for(i=0; i<3; i++) 
		{
			actor[i] = iActor::New(); IERROR_ASSERT(actor[i]);
			actor[i]->PickableOff();
			as = vtkArrowSource::New(); IERROR_ASSERT(as);
			as->SetShaftResolution(6);
			as->SetTipResolution(6);
			as->SetTipLength(0.15);
			as->SetShaftRadius(0.5*diam);
			as->SetTipRadius(1.5*diam);
			as->Update();
			actor[i]->SetInput(as->GetOutput());
			as->Delete();
			actor[i]->SetAxisScale(2.5,2.5,2.5);
		}

		diam *= 0.75;
		for(i=3; i<12; i++) 
		{
			actor[i] = iActor::New(); IERROR_ASSERT(actor[i]);
			actor[i]->PickableOff();
			cs = vtkCylinderSource::New(); IERROR_ASSERT(cs);
			cs->SetResolution(6);
			cs->SetHeight(2.0+2.0*diam);
			cs->SetRadius(diam);
			actor[i]->SetInput(cs->GetOutput());
			cs->Delete();
		}

		actor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
		actor[0]->RotateY(-90.0);
		actor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
		actor[1]->RotateZ(90.0);
		actor[2]->GetProperty()->SetColor(1.0,0.0,0.0);

		for(i=3; i<6; i++) 
		{
			actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
			actor[i]->RotateX(90.0);
		}
		for(i=6; i<9; i++) actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
		for(i=9; i<12; i++) 
		{
			actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
			actor[i]->RotateZ(90.0);
		}

		actor[ 0]->SetPosition(-1.0,-1.0,-1.0);
		actor[ 1]->SetPosition(-1.0,-1.0,-1.0);
		actor[ 2]->SetPosition(-1.0,-1.0,-1.0);

		actor[ 3]->SetPosition(-1.0, 1.0,0.0);
		actor[ 4]->SetPosition( 1.0,-1.0,0.0);
		actor[ 5]->SetPosition( 1.0, 1.0,0.0);

		actor[ 6]->SetPosition(-1.0,0.0, 1.0);
		actor[ 7]->SetPosition( 1.0,0.0,-1.0);
		actor[ 8]->SetPosition( 1.0,0.0, 1.0);

		actor[ 9]->SetPosition(0.0, 1.0,-1.0);
		actor[10]->SetPosition(0.0,-1.0, 1.0);
		actor[11]->SetPosition(0.0, 1.0, 1.0);

		iActorCollection* box3Actor = iActorCollection::New(); IERROR_ASSERT(box3Actor);
		for(i=0; i<12; i++) 
		{
			actor[i]->GetProperty()->SetAmbient(0.2);
			actor[i]->GetProperty()->SetDiffuse(0.2);
			actor[i]->GetProperty()->SetSpecular(0.2);
			box3Actor->AddActor(actor[i]);
			rt->AddObject(actor[i]);
			actor[i]->Delete();
		}
		box3Actor->SetVisibility(false);

		return box3Actor;
	}


	iActorCollection* CreateDebugProp(iRenderTool *rt)
	{
		//
		//  Debug prop
		//
		float diam = 0.2;
		iActor *actor[3];
		vtkArrowSource *as;

		int i;
		for(i=0; i<3; i++) 
		{
			actor[i] = iActor::New(); IERROR_ASSERT(actor[i]);
			as = vtkArrowSource::New(); IERROR_ASSERT(as);
			as->SetShaftResolution(6);
			as->SetTipResolution(18);
			as->SetTipLength(0.5);
			as->SetShaftRadius(0.7*diam);
			as->SetTipRadius(1.3*diam);
			as->Update();
			actor[i]->SetInput(as->GetOutput());
			as->Delete();
			actor[i]->SetScale(0.5);
			actor[i]->GetProperty()->SetAmbient(0.25);
			actor[i]->GetProperty()->SetDiffuse(0.25);
			actor[i]->GetProperty()->SetSpecular(0.5);
			actor[i]->PickableOff();
		}

		actor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
		actor[0]->RotateY(-90.0);
		actor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
		actor[1]->RotateZ(90.0);
		actor[2]->GetProperty()->SetColor(1.0,0.0,0.0);

		actor[ 0]->SetPosition(0.0,0.0,0.0);
		actor[ 1]->SetPosition(0.0,0.0,0.0);
		actor[ 2]->SetPosition(0.0,0.0,0.0);

		iActorCollection* debugActor = iActorCollection::New(); IERROR_ASSERT(debugActor);
		for(i=0; i<3; i++) 
		{
			debugActor->AddActor(actor[i]);
			rt->AddObject(actor[i]);
			actor[i]->Delete();
		}
		debugActor->VisibilityOff();

		return debugActor;
	}

	vtkActor2D* CreateIcon(iRenderTool *rt)
	{
		//
		// Icon
		//
		vtkActor2D* actor = vtkActor2D::New(); IERROR_ASSERT(actor);
		vtkPolyDataMapper2D *mapper = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mapper);
		vtkCoordinate *c = vtkCoordinate::New(); IERROR_ASSERT(c);
		c->SetCoordinateSystemToNormalizedViewport();
		mapper->SetTransformCoordinate(c);
		c->Delete();

		//
		//  Data for the icon
		//
		vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT(pd);
		vtkPoints *points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(points);
		vtkCellArray *lines = vtkCellArray::New(); IERROR_ASSERT(lines);
		vtkCellArray *triangles = vtkCellArray::New(); IERROR_ASSERT(triangles);
		vtkUnsignedCharArray *scalars = vtkUnsignedCharArray::New(); IERROR_ASSERT(scalars);
		scalars->SetNumberOfComponents(3);
		scalars->SetNumberOfTuples(18);
		unsigned char *color = scalars->GetPointer(0);

		//
		// Load the point, cell, and data attributes.
		//
		int i;
		double x[3];
		vtkIdType verts[3];

		double d = 0.05;
		x[2] = 0.0;
		for(i=0; i<6; i++)
		{
			x[0] = x[1] = 0.0;
			points->InsertNextPoint(x);
			x[0] = d*cos(0.52359877559829887307710723054658*(2*i));
			x[1] = d*sin(0.52359877559829887307710723054658*(2*i));
			points->InsertNextPoint(x);
			x[0] = d*cos(0.52359877559829887307710723054658*(2*i+1));
			x[1] = d*sin(0.52359877559829887307710723054658*(2*i+1));
			points->InsertNextPoint(x);

			verts[0] = 3*i + 0;
			verts[1] = 3*i + 1;
			verts[2] = 3*i + 2;
			triangles->InsertNextCell(3,verts);
			
			verts[0] = 3*i + 0;
			verts[1] = 3*i + 1;
			lines->InsertNextCell(2,verts);
			verts[0] = 3*i + 1;
			verts[1] = 3*i + 2;
			lines->InsertNextCell(2,verts);
			verts[0] = 3*i + 2;
			verts[1] = 3*i + 0;
			lines->InsertNextCell(2,verts);

			color[9*i+0+0] = color[9*i+3+0] = color[9*i+6+0] = (i<2 || i>4) ? 255 : 0;
			color[9*i+0+1] = color[9*i+3+1] = color[9*i+6+1] = (i>0 && i<4) ? 255 : 0;
			color[9*i+0+2] = color[9*i+3+2] = color[9*i+6+2] = (i>2 && i<6) ? 255 : 0;
		}

		//
		// We now assign the pieces to the vtkPolyData.
		//
		pd->SetPolys(triangles);
		triangles->Delete();
		pd->SetLines(lines);
		lines->Delete();
		pd->SetPoints(points);
		points->Delete();
		pd->GetPointData()->SetScalars(scalars);
		scalars->Delete();

		//
		//  Create the data and assign to the mapper
		//
		pd->Update();
		mapper->SetInput(pd);
		pd->Delete();

		actor->SetMapper(mapper);
		mapper->Delete();
		actor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
		actor->SetPosition(0.05,0.95);
		actor->GetProperty()->SetLineWidth(2.0);
		actor->VisibilityOff();
		actor->PickableOff();
		actor->SetLayerNumber(10);

		rt->AddObject(actor);

		return actor;
	}
};

