/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iruler.h"


#include "icamera.h"
#include "ierror.h"
#include "ioverlayhelper.h"
#include "irendertool.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkTextMapper.h>
#include <vtkTextProperty.h>


iRuler* iRuler::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iRuler(vm);
}


iRuler::iRuler(iViewModule *vm) : mOverlayHelper(vm->GetRenderTool()), mViewModule(vm)
{
	mStarted = false;
	mScale = mBaseScale = 1.0;
	mLineWidth = this->AxisActor->GetProperty()->GetLineWidth();
	mTickLength = TickLength;

	this->LabelMappers[0]->SetTextProperty(this->LabelTextProperty);
	this->LabelMappers[1]->SetTextProperty(this->LabelTextProperty);
	//this->LabelTextProperty->SetJustificationToCentered();
	this->LabelTextProperty->SetVerticalJustificationToTop();
	this->TitleTextProperty->SetJustificationToCentered();
	this->TitleTextProperty->SetVerticalJustificationToBottom();

	this->SetPoint1(0.25,0.9);
	this->SetPoint2(0.75,0.9);
	this->SetRange(0.0,1.0);
	this->AdjustLabelsOff();
	this->SetNumberOfLabels(2);

	this->PickableOff();
}


iRuler::~iRuler()
{
}


void iRuler::ReleaseGraphicsResources(vtkWindow *win)
{
	this->vtkAxisActor2D::ReleaseGraphicsResources(win);
}


int iRuler::RenderOverlay(vtkViewport* viewport)
{
	vtkCamera *cam = mOverlayHelper->GetCamera(viewport);
	if(cam==0 || cam->GetParallelProjection()==0) return 0;

	return this->vtkAxisActor2D::RenderOverlay(viewport);
}


int iRuler::RenderOpaqueGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = mOverlayHelper->GetCamera(viewport);
	if(cam==0 || cam->GetParallelProjection()==0) return 0;
	
	int mag = mOverlayHelper->GetRenderingMagnification();

	if(mag == 1)
	{
		if(!mStarted)
		{
			mStarted = true;
			mBaseScale = cam->GetParallelScale();
		}

		float s;
		if(cam->GetParallelProjection() != 0)
		{
			s = cam->GetParallelScale()/mBaseScale;
		}
		else
		{
			s = cam->GetDistance()*(1.0e-35+tan(0.5*iMath::Deg2Rad(cam->GetViewAngle())))/mBaseScale;
		}
		
		double as[2];
		viewport->GetAspect(as);
		mDx = 0.5/mBaseScale*as[1]/as[0];

		if(!this->GetViewModule()->GetOpenGLCoordinates()) s *= this->GetViewModule()->GetBoxSize();
		this->SetRange(0.0,s);
		mScale = s;
		
		this->SetPoint1(0.5-mDx,0.9);
		this->SetPoint2(0.5+mDx,0.9);

		this->AxisActor->GetProperty()->SetLineWidth(mLineWidth);
		this->SetTickLength(mTickLength);
	}
	else
	{
		int winij[2];
		mOverlayHelper->ComputePositionShiftsUnderMagnification(winij);
		
		this->SetPoint1(mag*(0.5-mDx)-winij[0],mag*0.9-winij[1]);
		this->SetPoint2(mag*(0.5+mDx)-winij[0],mag*0.9-winij[1]);
		
		this->AxisActor->GetProperty()->SetLineWidth(mLineWidth*mag);
		this->SetTickLength(mag*mTickLength);
	}
	
	this->GetProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
	this->LabelTextProperty->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
	this->TitleTextProperty->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());

	this->GetOverlayHelper()->UpdateTextProperty(this->LabelTextProperty);
	this->GetOverlayHelper()->UpdateTextProperty(this->TitleTextProperty);

	this->vtkAxisActor2D::RenderOpaqueGeometry(viewport);

	//
	//  This must be after RenderOpaqueGeometry
	//
	this->LabelTextProperty->SetFontSize(mOverlayHelper->GetFontSize(viewport,mag));
	this->TitleTextProperty->SetFontSize(mOverlayHelper->GetFontSize(viewport,mag*1.2));
	this->TitleMapper->GetTextProperty()->ShallowCopy(this->TitleTextProperty);
	int *s = viewport->GetSize();
	this->TitleActor->SetPosition(0.5*s[0],0.91*s[1]);

	return 1;
}


void iRuler::SetBaseScale(float s)
{
	if(s > 0.0)
	{
		mStarted = true;
		mBaseScale = s;
	}
}


void iRuler::SetScale(float s)
{
	if(!mStarted)
	{
		mStarted = true;
		mBaseScale = this->GetViewModule()->GetRenderer()->GetActiveCamera()->GetParallelScale();
	}

	if(s > 0.0)
	{
		if(!this->GetViewModule()->GetOpenGLCoordinates()) s /= this->GetViewModule()->GetBoxSize();
		mOverlayHelper->GetRenderTool()->GetCamera()->GetDevice()->SetParallelScale(s*mBaseScale);
	}
}


void iRuler::SetTitle(const iString &title)
{
	mTitle = title;
	this->vtkAxisActor2D::SetTitle(title.ToCharPointer());
	if(mTitle.IsEmpty()) this->TitleVisibilityOff(); else this->TitleVisibilityOn();
}

