/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggshell.h"


#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "ifile.h"
#include "iversion.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include "iggdialog.h"
#include "iggextensionwindow.h"
#include "iggframe.h"
#include "iggmainwindow.h"
#include "iggrenderwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetmisc.h"

#include "ibgdialogsubject.h"
#include "ibgextensionwindowsubject.h"
#include "ibgframesubject.h"
#include "ibgmenuwindowsubject.h"
#include "ibgrenderwindowsubject.h"
#include "ibgshellsubject.h"
#include "ibgwindowhelper.h"

#if defined(I_DEBUG) && defined(I_EDITION)
#include <vtkTimerLog.h>
#endif

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;


namespace iggShell_Private
{
	class DialogFlashWindow : public iggDialog
	{

	public:

		DialogFlashWindow(iShell *shell) : iggDialog(shell,_DialogUnattached|_DialogBlocking|_DialogNoTitleBar,0,"",0,3,false)
		{
			mFrame->AddLine(new iggWidgetTextArea("%b%+   IFrIT - Ionization FRont Interactive Tool   ",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("Version "+iVersion::GetVersion(),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea(iEdition::GetEditionName(),mFrame),3);

			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper,0);

			mMessage = new iggWidgetTextArea("Starting...",mFrame);
			mFrame->AddLine(mMessage,3);

			this->ResizeContents(100,100);
		}

		virtual void Show(bool s)
		{
			mFlipper->Reset();
			iggDialog::Show(s);
		}

		void OnInitAtom()
		{
			mFlipper->Advance();
		}

		void PlaceOnDesktop(int dw, int dh)
		{
			int wg[4];
			mSubject->GetWindowGeometry(wg,true);
			wg[0] = (dw-wg[2]-20)/2;
			wg[1] = (dh-wg[3]-20)/2;
			mSubject->SetWindowGeometry(wg);
		}

		void DisplayMessage(const iString &text)
		{
			mMessage->SetText(text);
		}

	protected:

		virtual bool CanBeClosed()
		{
			return false; // cannot be closed
		}

		iggWidgetTextArea *mMessage;
		iggWidgetLogoFlipper *mFlipper;
	};
};


iggShell::iggShell(const iString &type, int argc, char **argv) : iShell(argc,argv)
{
	mType = type;
	mSubject = iggSubjectFactory::CreateShellSubject(this,argc,argv);
	if(mSubject == 0) exit(1);

	//
	//  Default settings
	//
	mShowFlashWindow = true;
	mAccept8BitDisplay = mStartDocked = false;

	mMainWindow = 0; // very important!!! This tells other components that the main
					 // window does not exist yet.
	mFlashWindow = 0;
}


void iggShell::Exit()
{
	mSubject->Exit();
}


#ifdef I_DEBUG
void iggShell::OutputDebugMessage(const iString &message) const
{
	mSubject->OutputDebugMessage(message);
}
#endif


void iggShell::GetDesktopDimensions(int &w, int &h) const
{
	mSubject->GetDesktopDimensions(w,h);
}


void iggShell::OutputStartupMessage(const iString &message)
{
	mSubject->OutputStartupMessage(message);
}


void iggShell::DefineSpecificCommandLineOptions()
{
	//
	//  Define generic GUI-specific command-line options
	//
	this->AddCommandLineOption("nf",false,"do not show the splash screen at start-up");
	if(mSubject->CanRecognize8BitDisplay()) this->AddCommandLineOption("8",false,"accept an 8-bit display");
	if(mSubject->CanDockWindows()) this->AddCommandLineOption("d",false,"start IFrIT with windows docked");
	if(mSubject->CanRenderOffScreen()) this->AddCommandLineOption("b",true,"execute a control script from a file <arg> in the screenless mode");
	if(mSubject->CanChangeFontSize()) this->AddCommandLineOption("fs",true,"increase/decrease window font size");
}


iggShell::~iggShell()
{
}


void iggShell::PrepareForConstruction()
{
	if(mScriptFileName.IsEmpty())
	{
		//
		//  Check for the 8-bit display
		//
		if(!mAccept8BitDisplay && mSubject->CanRecognize8BitDisplay() && mSubject->CanRecognize8BitDisplay(true))
		{
			this->OutputStartupMessage("You are not using a True Color (24-bit) monitor.\n"
				"Visualizations will be crappy without the True Color support.\n"
				"If you wish to continue anyway, specify -8 as an option.");
			exit(0);
		}
	}
	else
	{
		mShowFlashWindow = false;
	}
#if defined(I_DEBUG) && defined(I_EDITION)
	this->OutputDebugMessage("Starting...");
#ifdef IVTK_4
	mTotalTime = vtkTimerLog::GetCurrentTime();
#else
	mTotalTime = vtkTimerLog::GetUniversalTime();
#endif
#endif
	//
	//  Create and show message window
	//
	if(mShowFlashWindow)
	{
		mFlashWindow = new iggShell_Private::DialogFlashWindow(this); IERROR_ASSERT(mFlashWindow);
		//
		//  calculate the position of the flash window
		//
		int dw, dh;
		mSubject->GetDesktopDimensions(dw,dh);
		//
		//  Allow for the two-monitor desktop
		//
		if(dw > 2*dh)
		{
			dw /= 2;
		}
		mSubject->ProcessEvents(true);
		iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->PlaceOnDesktop(dw,dh);
		mFlashWindow->Show(true);
		mSubject->ProcessEvents(true);
		iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->DisplayMessage("Creating visualization window...");
	}
}


void iggShell::ConstructorBody()
{
	if(mShowFlashWindow)
	{
		iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->DisplayMessage("Creating widgets...");
		mSubject->ProcessEvents(false);
	}

#if defined(I_DEBUG) && defined(I_EDITION)
	iConsole::Instance()->StartTimer();
#endif
	mMainWindow = new iggMainWindow(this);
#if defined(I_DEBUG) && defined(I_EDITION)
	iConsole::Instance()->StopTimer("Creating main window");
	iConsole::Instance()->StartTimer();
#endif
	mMainWindow->StartInitialization();
#if defined(I_DEBUG) && defined(I_EDITION)
	iConsole::Instance()->StopTimer("Initializing main window");
#endif
}

  
void iggShell::DestructorBody()
{
	//
	//  Hide windows to avoid extra updates
	//
	mMainWindow->ShowAll(false);
	mMainWindow->Block(true); // block all events
	delete mMainWindow;
	mMainWindow = 0;  // tell other objects that the MainWindow does not exist any more
}


void iggShell::LoadStateFile(const iString &fname)
{
	if(mShowFlashWindow)
	{
		iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->DisplayMessage("Loading the state file...");
		mSubject->ProcessEvents(false);
	}
	if(!this->GetControlModule()->LoadStateFromFile(fname))
	{
		mMainWindow->PopupWindow("The state file is corrupted.\n Default values of parameters will be used.",_PopupWindowError);
	}
}


void iggShell::Start()
{
	const int topOffset = 64;
	const int frameXoffDefault = 13;
	const int frameYoffDefault = 25;
	int frameXoff = 1, frameYoff = 1;
	int leftOffset = 0;

	//
	//  If we are running in a batch mode, show no windows.
	//
	if(!mScriptFileName.IsEmpty()) return;

#if defined(I_DEBUG) && defined(I_EDITION)
	iConsole::Instance()->StartTimer();
#endif

	if(mStartDocked) mMainWindow->mDocked = true;
	mMainWindow->PreShowInitialization();
	if(mStartDocked) mMainWindow->mDocked = false;
	mSubject->ProcessEvents(false);
		
	if(mShowFlashWindow)
	{
		iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->DisplayMessage("Showing windows...");
		mSubject->ProcessEvents(false);
	}
	//
	//  Show visualization windows
	//
	if(mStateFileName.IsEmpty())
	{
		//
		// Calculate the left offset of the visualization window (it is the only one)
		//
		int dw, dh;
		mSubject->GetDesktopDimensions(dw,dh);
		//
		//  Allow for the two-monitor desktop
		//
		if(dw > 2*dh)
		{
			dw /= 2;
		}

		int mwg[4], vwg[4];
		iggRenderWindow *visWindow = iPointerCast<iggRenderWindow,vtkRenderWindow>(mControlModule->GetViewModule()->GetRenderWindow());

		//
		//  squeeze windows to the minimum possible size
		//
		mwg[0] = mwg[1] = mwg[2] = mwg[3] = 0;
		mMainWindow->GetSubject()->SetWindowGeometry(mwg);
		mMainWindow->GetExtensionWindow()->GetSubject()->SetWindowGeometry(mwg);
		mSubject->ProcessEvents(false);

		//
		//  Show all windows
		//
		if(mStartDocked)
		{
			//
			//  If we start in the docked mode, trust the window manager
			//
			mMainWindow->DockWindows(true,false);
			mMainWindow->GetSubject()->GetWindowGeometry(mwg);

			bool redo = false;
			if(mwg[2] < (4*mwg[3]/3))
			{
				mwg[2] = (4*mwg[3]/3);
				redo = true;
			}

			leftOffset = (dw-mwg[2]-20)/2;
			if(leftOffset < 0) leftOffset = 0;
			if(mwg[0]<leftOffset || mwg[1]<topOffset)
			{
				mwg[0] = leftOffset;
				mwg[1] = topOffset;
				redo = true;
			}
			if(redo) mMainWindow->GetSubject()->SetWindowGeometry(mwg);

			mMainWindow->ShowAll(true);
#if defined(I_DEBUG) && defined(I_EDITION)
			iConsole::Instance()->StopTimer("Showing windows");
			iConsole::Instance()->StartTimer();
#endif
		}
		else
		{
			//
			//  Place the windows on the screen
			//
			mMainWindow->GetSubject()->GetWindowGeometry(mwg,true);
			visWindow->GetSubject()->GetWindowGeometry(vwg,true);

			leftOffset = (dw-vwg[2]-mwg[2]-20)/2;
			if(leftOffset < 0) leftOffset = 0;
			//
			//  Move the visual module window
			//
			vwg[0] = leftOffset;
			vwg[1] = topOffset;
			visWindow->GetSubject()->SetWindowGeometry(vwg);

			//
			//  Show the visualization window and give a chance to the Window Manager to decorate the window
			//			     
			mControlModule->Render(_RenderModeAll);
			mSubject->ProcessEvents(false);
#if defined(I_DEBUG) && defined(I_EDITION)
			iConsole::Instance()->StopTimer("Showing render window");
			iConsole::Instance()->StartTimer();
#endif

			//
			//  Get the frame geometry
			//
			mSubject->ProcessEvents(true);
			visWindow->GetSubject()->GetFrameSize(frameXoff,frameYoff);
			if(frameXoff == 0) frameXoff = frameXoffDefault; // X11 failed to assign a proper frame
			if(frameYoff == 0) frameYoff = frameYoffDefault; // X11 failed to assign a proper frame

			mwg[0] = vwg[0] + vwg[2] + frameXoff;
			mwg[1] = topOffset;
			mMainWindow->GetSubject()->SetWindowGeometry(mwg,true);
	
			int ewg[4], ewg2[4];
			ewg[0] = leftOffset;
			ewg[1] = vwg[1] + vwg[3] + frameYoff;
			ewg[2] = vwg[2];
			ewg[3] = mwg[3] - vwg[3] - frameYoff;
			mMainWindow->GetExtensionWindow()->GetSubject()->SetWindowGeometry(ewg,true);
			mMainWindow->GetExtensionWindow()->GetSubject()->GetWindowGeometry(ewg2,true);
			if(ewg2[3] > ewg[3])
			{
				mMainWindow->GetSubject()->GetWindowGeometry(mwg,true);
				mwg[3] += (ewg2[3]-ewg[3]);
				mMainWindow->GetSubject()->SetWindowGeometry(mwg,true);
			}

			//
			//  Show the main window and give a chance to the Window Manager to decorate the window
			//			     
			mMainWindow->GetSubject()->Show(true);
			mSubject->ProcessEvents(false);
#if defined(I_DEBUG) && defined(I_EDITION)
			iConsole::Instance()->StopTimer("Showing main window");
			iConsole::Instance()->StartTimer();
#endif

			mMainWindow->GetExtensionWindow()->GetSubject()->Show(true);
			mSubject->ProcessEvents(false);
#if defined(I_DEBUG) && defined(I_EDITION)
			iConsole::Instance()->StopTimer("Showing extension window");
			iConsole::Instance()->StartTimer();
#endif
		}
	}
	else
	{
		mControlModule->Render(_RenderModeAll);
		mMainWindow->GetSubject()->Show(true);
		mMainWindow->GetExtensionWindow()->GetSubject()->Show(true);
	}
	
	mSubject->CompleteStartUp();

	if(mShowFlashWindow)
	{
		//
		//  Remove the flash window
		//
		mFlashWindow->Show(false);
		delete mFlashWindow;
		mFlashWindow = 0;
	}

	mSubject->ProcessEvents(false);
	mMainWindow->PostShowInitialization();

#if defined(I_DEBUG) && defined(I_EDITION)
	iConsole::Instance()->StopTimer("Mopping up");
#ifdef IVTK_4
	mTotalTime = vtkTimerLog::GetCurrentTime() - mTotalTime;
#else
	mTotalTime = vtkTimerLog::GetUniversalTime() - mTotalTime;
#endif
	this->OutputDebugMessage("Done: "+iString::FromNumber(mTotalTime));
#endif
}


void iggShell::RunBody()
{
	if(mScriptFileName.IsEmpty())
	{
		mSubject->EnterEventLoop();
	}
	else
	{
		if(!mSubject->CanRenderOffScreen())
		{
			mSubject->OutputStartupMessage("This shell cannot execute a script in a batch mode.");
			return;
		}
		//
		//  Execute the batch script
		//
		iString fname;

		if(mScriptFileName[0] == '+')
		{
			fname = mControlModule->GetFileName(_EnvironmentScript,mScriptFileName.Part(1));
		}
		else
		{
			fname = mScriptFileName;
		}
		iFile F(fname);

		if(!F.Open(iFile::_Read,iFile::_Text)) 
		{
			mSubject->OutputStartupMessage(iString("File ")+fname+" cannot be open for reading. Abort.");
			exit(1);
		}

		mSubject->OutputStartupMessage("Reading the file...");
		iString text, line;
		while(F.ReadLine(line)) text += "\n" + line;
		text += "\n";

		iControlScript *cs = mControlModule->GetControlScript();
		mSubject->OutputStartupMessage("Compiling the script...");

		cs->Run(text,true);
		if(cs->GetErrorStatus()->IsError())
		{
			mSubject->OutputStartupMessage(iString("Syntax error in script, line ")+iString::FromNumber(cs->GetCurrentLine())+" : "+cs->GetErrorMessage());
			return;
		}

		//
		//  Set image method to pixmap
		//
		iggRenderWindow::SetRenderIntoMemory(true);

		mSubject->OutputStartupMessage("Running the script...");
		cs->Run(text);
		if(cs->GetErrorStatus()->IsError())
		{
			mSubject->OutputStartupMessage(iString("Runtime error in script, line ")+iString::FromNumber(cs->GetCurrentLine())+" : "+cs->GetErrorMessage());
		}
	}
}


bool iggShell::AnalyseOneExtendedCommandLineOption(const struct Option &o)
{

	if(o.Name == "fs")
	{
		int s;
		bool ok;
		if((o.Value[0]=='+' || o.Value[0]=='-') && (s=o.Value.ToInt(ok))!=0 && ok)
		{
			ibgWindowHelper::ChangeFontSize(s);
			return true;
		}
		else
		{
			this->OutputStartupMessage("Invalid font size specification; should be in the form +<number> or -<number>. Abort.");
			exit(1);
		}
	} 

	if(o.Name == "nf")
	{
		mShowFlashWindow = false;
		return true;
	} 

	if(o.Name == "8")
	{
		mAccept8BitDisplay = true;
		return true;
	}

	if(o.Name == "d")
	{
		mStartDocked = true;
		return true;
	} 
	
	if(o.Name == "b")
	{
		if(iFile::IsLoadable(o.Value))
		{
			mScriptFileName = o.Value;
			return true;
		}
		else
		{
			mSubject->OutputStartupMessage("File "+o.Value+" is not accessible. Abort.");
			exit(1);
		}
	}
	return false;
}


//
//  State saving
//
bool iggShell::LoadShellStateFromFile(iFile &F)
{
	iString ws, line;
	//
	//  Restore the state of the main window
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);

		if(ws == iggMainWindow::Type().FullName()) 
		{
			ws = line.Section(" ",1);
			mMainWindow->UnPackCompleteState(ws);
		}
	}
	return true;
}


bool iggShell::SaveShellStateToFile(iFile &F) const
{
	//
	//  Save the state of the main window 
	//
	iString ws;
	mMainWindow->PackCompleteState(ws);
	ws = iggMainWindow::Type().FullName() + " " + ws;
	return F.WriteLine(ws);
}


void iggShell::ProcessEvents(bool sync)
{
	mSubject->ProcessEvents(sync);
}


void iggShell::OnInitAtomBody(bool timed)
{
	if(timed && mFlashWindow!=0) iPointerCast<iggShell_Private::DialogFlashWindow,iggDialog>(mFlashWindow)->OnInitAtom();
}

