/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticledataconverter.h"


#include "idatalimits.h"
#include "idatastretch.h"
#include "ierror.h"
#include "ifunctionmapping.h"
#include "imath.h"
#include "iparticlegroup.h"
#include "ipiecewisefunction.h"
#include "ipointglyph.h"
#include "itune.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkCellData.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPiecewiseFunction.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"

using namespace iParameter;


iParticleDataConverter::iParticleDataConverter(iViewSubject *vo) : iGenericPolyDataToPolyDataFilter<vtkPolyDataToPolyDataFilter>(vo,1,true,true)
{
	mSize = 0.005;
	mPoint = iPointGlyph::New(vo); IERROR_ASSERT(mPoint);
	this->SetType(_PointTypePoint);
}


iParticleDataConverter::~iParticleDataConverter()
{
	mPoint->Delete();
}


int iParticleDataConverter::GetType() const
{
	return mPoint->GetType();
}


void iParticleDataConverter::SetType(int t)
{
	mPoint->SetType(t);
	this->Modified();
}


void iParticleDataConverter::SetSize(float s)
{
	if(s>0.0 && s<1000.1)
	{
		mSize = 0.005*s;
		this->Modified();
	}
}


void iParticleDataConverter::ProduceOutput()
{
	int i, j;
	vtkIdType lp, lv;
	vtkIdType nCell1, *pCell1, pCell2[999];
	double x0[3], x1[3], x2[3];
	
	vtkPolyData *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	output->ShallowCopy(input);

	if(mPoint->GetType() == _PointTypePoint) return;
	
	vtkPoints *inpPnts = input->GetPoints();
	if(inpPnts == 0) return;

	vtkCellArray *inpVrts = input->GetVerts();
	if(inpVrts == 0) return;

	//
	//  Direct access to the input data
	//
	vtkIdType nInpVrts = inpVrts->GetNumberOfCells();
	vtkIdType nInpPnts = inpPnts->GetNumberOfPoints();

	if(nInpVrts<=0 || nInpPnts<=0) return;

	int sop;
	switch(inpPnts->GetDataType())
	{
	case VTK_FLOAT:
		{
			sop = sizeof(float);
			break;
		}
	case VTK_DOUBLE:
		{
			sop = sizeof(double);
			break;
		}
	default: return;
	}

	//
	//  Direct access to the glyph data
	//
	mPoint->Update();

	vtkPoints *srcPnts = mPoint->GetOutput()->GetPoints();
	if(srcPnts->GetDataType() != VTK_FLOAT)
	{
		vtkErrorMacro("Source points are not floats.");
		return;
	}
	float *srcPntsArr = (float *)srcPnts->GetVoidPointer(0);

	bool srcCellPoly = true;
	vtkCellArray *srcClls = mPoint->GetOutput()->GetPolys();
	if(srcClls==0 || srcClls->GetNumberOfCells()==0)
	{
		srcClls = mPoint->GetOutput()->GetVerts();
		srcCellPoly = false;
	}
	if(srcClls==0 || srcClls->GetNumberOfCells()==0)
	{
		vtkErrorMacro("Incorrect input to iParticleDataConverter");
		return;
	}
	
	int sSrcNmls = 0;
	vtkFloatArray *srcNmls = 0;
	if(mPoint->GetOutput()->GetPointData()->GetNormals() != 0) 
	{
		srcNmls = iRequiredCast<vtkFloatArray>(mPoint->GetOutput()->GetPointData()->GetNormals());
		sSrcNmls = 1;
	}
	else if(mPoint->GetOutput()->GetCellData()->GetNormals() != 0) 
	{
		srcNmls = iRequiredCast<vtkFloatArray>(mPoint->GetOutput()->GetCellData()->GetNormals());
		sSrcNmls = -1;
	}
	float *srcNmlsArr = 0;
	if(srcNmls != 0) srcNmlsArr = (float *)srcNmls->GetVoidPointer(0);

	vtkIdType nSrcPnts = srcPnts->GetNumberOfPoints();
	vtkIdType nSrcClls = srcClls->GetNumberOfCells(); 
	vtkIdType nSrcCellMax = srcClls->GetMaxCellSize();

	bool rotate = mPoint->GetOriented();
	bool orient = false;

	//
	//  Check if lines specified too
	//
	vtkIdType nOutPntsOff = 0;
	vtkCellArray *inpLins = 0;
	float *inpPntsDir = 0;
	if(input->GetLines()!=0 && input->GetLines()->GetNumberOfCells()>0)
	{
		inpLins = input->GetLines();
		nOutPntsOff = nInpPnts;
		//
		//  If lines are present, arrows and cones are oriented along the lines
		//
		if(mPoint->GetType()==_PointTypeCone || mPoint->GetType()==_PointTypeArrow)
		{
			orient = true;
			inpPntsDir = new float[3*nInpPnts];
			if(inpPntsDir == 0)
			{
				vtkErrorMacro("There is not enough memory to convert the particle set data");
				return;
			}
			memset(inpPntsDir,0,3*nInpPnts*sizeof(float));
			inpLins->InitTraversal();
			while(inpLins->GetNextCell(nCell1,pCell1) != 0)
			{
#ifdef I_CHECK1
				if(nCell1 != 2)
				{
					IERROR_REPORT("Line contains more than 2 points!");
					break;
				}
#endif
				inpPnts->GetPoint(pCell1[0],x1);
				inpPnts->GetPoint(pCell1[1],x2);
				for(j=0; j<3; j++)
				{
					x0[j] = x2[j] - x1[j];
				}
				vtkMath::Normalize(x0);
				for(j=0; j<3; j++)
				{
					inpPntsDir[3*pCell1[0]+j] += x0[j];
					inpPntsDir[3*pCell1[1]+j] += x0[j];
				}
			}
			for(lp=0; lp<nInpPnts; lp++)
			{
				vtkMath::Normalize(inpPntsDir+3*lp);
			}
		}
	}

	//
	//  Create output data
	//
	vtkIdType nOutPnts = nInpVrts*nSrcPnts + nOutPntsOff;
	vtkIdType nOutClls = nInpVrts*nSrcClls;
	vtkPoints *outPnts = vtkPoints::New(inpPnts->GetDataType()); IERROR_ASSERT(outPnts);
	vtkCellArray *outClls = vtkCellArray::New(); IERROR_ASSERT(outClls);

	//	
	// Allocates and Sets MaxId
	//
	outPnts->SetNumberOfPoints(nOutPnts);

	//
	//  If not enough memory - revert to points
	//
	if(outPnts->GetVoidPointer(0)==0 || outClls->Allocate(outClls->EstimateSize(nOutClls,nSrcCellMax))==0) 
	{
		if(inpPntsDir != 0) delete [] inpPntsDir;
		outPnts->Delete();
		outClls->Delete();
		vtkErrorMacro("There is not enough memory to convert the particle set data");
		return;
	}
	
	//
	//  Output normals
	//
	vtkFloatArray *outNmls = vtkFloatArray::New(); IERROR_ASSERT(outNmls);
	outNmls->SetNumberOfComponents(3);
	float *outNmlsArr = 0;
	if(sSrcNmls > 0) 
	{
		// Allocates and Sets MaxId
		outNmls->SetNumberOfTuples(nOutPnts);
		outNmlsArr = (float *)outNmls->GetVoidPointer(0);
	} 
	else if(sSrcNmls < 0) 
	{
		// Allocates and Sets MaxId
		outNmls->SetNumberOfTuples(nOutClls);
		outNmlsArr = (float *)outNmls->GetVoidPointer(0);
	} 
	if(sSrcNmls!=0 && outNmlsArr==0) 
	{
		if(inpPntsDir != 0) delete [] inpPntsDir;
		outPnts->Delete();
		outClls->Delete();
		outNmls->Delete();
		vtkErrorMacro("There is not enough memory to convert the particle set data");
		return;
	}

	if(!this->ScalarsInit(input->GetPointData(),nOutPnts))
	{
		outPnts->Delete();
		outClls->Delete();
		outNmls->Delete();
		return;
	}

	//
	//  Attributes for scaling
	//
	int catt;
	float cattLo = 0.0, cattHi = 0.0;
	int cattSt = 0;
	catt = iParticleGroup::RequiredCast(this->GetViewSubject())->GetAttributeToSize();
	if(catt>=0 && this->GetLimits()->GetNumVars())
	{
		iParticleGroup *pg = iParticleGroup::RequiredCast(this->GetViewSubject());
		cattSt = pg->GetStretchToSize();
		cattLo = iDataStretch::ApplyStretch(pg->GetLowerLimitToSize(),cattSt,false);
		cattHi = iDataStretch::ApplyStretch(pg->GetUpperLimitToSize(),cattSt,true);
	}
	
	//
	//  Set fixed seed to make sure particles do not jump in an animation
	//
	vtkMath::RandomSeed(123456789);

	//
	//  If lines specified, add original points first
	//
	if(nOutPntsOff > 0) 
	{
		memcpy(outPnts->GetVoidPointer(0),inpPnts->GetVoidPointer(0),nInpPnts*3*sop);
		if(wScalarDimIn > 0) iTune::CopyFloatPointers(wScalarPtrOut,wScalarPtrIn,nInpPnts*wScalarDimIn);
		if(sSrcNmls != 0) 
		{
			memset(outNmlsArr,0,nInpPnts*3*sizeof(float));
//			for(lp=0; lp<nInpPnts; lp++)
//			{
//				for(j=0; j<3; j++) outNmlsArr[3*lp+j] = snArr[j];  // this is faster than memcpy!
//			}
		}
	}
	//
	//  Prepare for direct scaling
	//
	iParticleGroup *myParticles = iParticleGroup::RequiredCast(this->GetViewSubject());
	iPiecewiseFunction *pFun = myParticles->GetSizeFunction()->GetFunction();

	bool doDirect = myParticles->GetAttributeSizeDirect();
	float scale = 2.0*myParticles->GetAttributeSizeExtraFactor();

	//
	//  Main loop
	//
	double mat[3][3];
	double en, e[4];  // Euler quaternion
	float *s0 = 0, latt;
	double s, ct, st, cp, sp;
	vtkIdType llPnts = nOutPntsOff, llClls = 0;

	outClls->InitTraversal();
	inpVrts->InitTraversal();
	for(lv=0; lv<nInpVrts; lv++)
	{
		if(lv%1000 == 0)
		{
			this->UpdateProgress((float)lv/nInpVrts);
			if(this->GetAbortExecute()) break;
		}

		inpVrts->GetNextCell(nCell1,pCell1);
#ifdef I_CHECK1
		if(nCell1 != 1)
		{
			IERROR_REPORT("Vert contains more than 1 point!");
			break;
		}
#endif
		lp = pCell1[0];

		inpPnts->GetPoint(lp,x0);
		s = mSize;
		if(wScalarDimIn > 0)
		{
			s0 = wScalarPtrIn + wScalarDimIn*lp;
			if(catt >= 0)
			{
				if(doDirect)
				{
					//
					//  use the attribute as the precise value of the size scaled by scale
					//
					s = scale*s0[catt];
				}
				else
				{
					latt = iDataStretch::ApplyStretch(s0[catt],cattSt,false);
					latt = (latt-cattLo)/(cattHi-cattLo);
					s *= pFun->GetValue(latt);
				}
			}
		}
		
		if(rotate)
		{
			if(orient)
			{
				ct = inpPntsDir[3*lp+2];
				cp = inpPntsDir[3*lp+0];
				st = sqrt(1.0-ct*ct);
				sp = sqrt(1.0-cp*cp);
				if(inpPntsDir[3*lp+1] < 0.0) sp = -sp;
				mat[0][0] = ct*cp;
				mat[1][0] = ct*sp;
				mat[2][0] = -st;
				mat[0][1] = -sp;
				mat[1][1] = cp;
				mat[2][1] = 0.0;
				mat[0][2] = st*cp;
				mat[1][2] = st*sp;
				mat[2][2] = ct;
			}
			else
			{
				en = 0.0;
				for(j=0; j<4; j++)
				{
					e[j] = vtkMath::Random();
					en += e[j]*e[j];
				}
				en = 1.0/sqrt(en);
				for(j=0; j<4; j++) e[j] *= en;
				//
				//  There is a bug in VTK: the quaternion must be normalized, despite the claim to the opposite in the help
				//
				vtkMath::QuaternionToMatrix3x3(e,mat);
			}
		}
		
		srcClls->InitTraversal();
		for(i=0; i<nSrcClls; i++)
		{
			srcClls->GetNextCell(nCell1,pCell1);
			for(j=0; j<nCell1; j++) pCell2[j] = pCell1[j] + llPnts;
			outClls->InsertNextCell(nCell1,pCell2);
			if(sSrcNmls < 0) // cell normals
			{
				iTune::CopyFloatPointers(outNmlsArr+3*llClls,srcNmlsArr+3*i,3);
			}
			llClls++;
		}

		for(i=0; i<nSrcPnts; i++)
		{
			for(j=0; j<3; j++) x1[j] = srcPntsArr[3*i+j]; // cannot use iTune, these are of different type
			if(rotate)
			{
				vtkMath::Multiply3x3(mat,x1,x2);
				iTune::CopyDoublePointers(x1,x2,3); // this is the fastest assignment
			}
			for(j=0; j<3; j++) x1[j] = s*x1[j] + x0[j];
			outPnts->SetPoint(llPnts,x1);
			if(wScalarDimIn > 0) 
			{
				iTune::CopyFloatPointers(wScalarPtrOut+wScalarDimOut*llPnts,s0,wScalarDimIn);
			}
			if(sSrcNmls > 0) // point normals 
			{
				iTune::CopyFloatPointers(outNmlsArr+3*llPnts,srcNmlsArr+3*i,3);
			}
			llPnts++;
		}
	}
	
	if(inpPntsDir != 0) delete [] inpPntsDir;

	output->SetPoints(outPnts);
	outPnts->Delete();
	
	//
	//  remove verts
	//
	output->SetVerts(0);

	if(srcCellPoly)
	{
		output->SetPolys(outClls);
	}
	else
	{
		output->SetVerts(outClls);
	}
	outClls->Delete();
	
	this->ScalarsDone(output->GetPointData());
	
	if(sSrcNmls > 0) 
	{
		output->GetPointData()->SetNormals(outNmls);
	} 
	else if(sSrcNmls < 0) 
	{
		output->GetCellData()->SetNormals(outNmls);
	}
	else
	{
		output->GetPointData()->SetNormals(0);
		output->GetCellData()->SetNormals(0);
	}
	outNmls->Delete();
}


float iParticleDataConverter::GetMemorySize()
{
	if(this->GetType() == _PointTypePoint) return 0.0; else return this->iGenericPolyDataToPolyDataFilter<vtkPolyDataToPolyDataFilter>::GetMemorySize();
}

