/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IIMAGECOMPOSER_H
#define IIMAGECOMPOSER_H


#include "iviewmodulecomponent.h"
#include "ierrorstatusholder.h"
#include "ideletable.h"


#include "iarray.h"
#include "icolor.h"
#include "iimage.h"
#include "istring.h"

class iImageComposer;
class iShell;

//
//  Helper classes
//

class iImageComposerWindow : public iViewModuleComponent, public iDeletable
{

public:
	
	enum iImageComposerWindowType
	{
		UNDEFINED = 0,
		BACKGROUND = 1,
		FOREGROUND = 2
	};
	
	void SetViewModule(iViewModule *vm);
	inline iImageComposerWindowType GetType() const { return mType; }
	inline const iImageComposer* GetImageComposer() const { return mComposer; }

protected:

	iImageComposerWindow(iViewModule *vm, iImageComposer *ic) : iViewModuleComponent(vm)
	{ 
		mType = UNDEFINED;
		mComposer = ic;
	}
	virtual ~iImageComposerWindow(){}

	iImageComposer *mComposer;
	iImageComposerWindowType mType;
};


class iImageComposerBackgroundWindow : public iImageComposerWindow
{

public:

	static iImageComposerBackgroundWindow* New(int tx, int ty, iImageComposer *ic);

	inline bool IsEmpty() const { return (this->GetViewModule()==0 && mWallpaperImage.IsEmpty()); }

	int GetWidth() const;
	int GetHeight() const;
	int GetTileX() const { return mTileX; }
	int GetTileY() const { return mTileY; }
	const iString& GetWallpaperFile() const { return mWallpaperFile; }
	const iImage& GetWallpaperImage() const { return mWallpaperImage; }

	bool LoadWallpaperImage(const iString &s);
	
protected:

	iImageComposerBackgroundWindow(int tx, int ty, iImageComposer *ic);
	virtual ~iImageComposerBackgroundWindow(){}

	int mTileX, mTileY;
	iImage mWallpaperImage;
	iString mWallpaperFile;
};


class iImageComposerForegroundWindow : public iImageComposerWindow
{

public:

	static iImageComposerForegroundWindow* New(iViewModule *vm, iImageComposer *ic);

	inline bool IsEmpty() const { return (this->GetViewModule() == 0); }
	inline float GetScale() const { return mScale; }
	inline int GetPositionX() const { return mPosX; }
	inline int GetPositionY() const { return mPosY; }
	inline int GetBorderWidth() const { return mBorderWidth; }
	inline iColor GetBorderColor() const { return mBorderColor; }

	int GetWidth() const;
	int GetHeight() const;
	int GetScaledWidth() const;
	int GetScaledHeight() const;
	inline int GetImageWidth() const { return 2*mBorderWidth + this->GetScaledWidth(); }
	inline int GetImageHeight() const { return 2*mBorderWidth + this->GetScaledHeight(); }
	
	void SetScale(float s);
	void SetPosition(int x, int y);
	void SetPositionX(int x);
	void SetPositionY(int y);
	void SetBorderWidth(int w);
	void SetBorderColor(const iColor& c);

	void CorrectPosition();

protected:

	iImageComposerForegroundWindow(iViewModule *vm, iImageComposer *ic);
	virtual ~iImageComposerForegroundWindow(){}

	float mScale;
	iColor mBorderColor;
	int mBorderWidth;
	int mPosX, mPosY;
	
	static int mPosQuantum;
};

//
//  Main class
//
#include "iobject.h"
#include "icontrolmodulecomponent.h"
#include "ishellcomponent.h"

class iStereoImageArray;


class iImageComposer : public iObject, public iControlModuleComponent, public iShellComponent, public iErrorStatusHolder, public iDeletable
{

	friend class iShellFactory;

public:

	IOBJECT_DECLARE_SELF(iImageComposer,iObject);

	static iImageComposer* New(iControlModule *cm, iShell *s);
	static const iObjectType& Type();

	IOBJECT_DECLARE_GETSET1(ScaleBackground,bool);
	IOBJECT_DECLARE_GETSET1(InnerBorder,bool);
	IOBJECT_DECLARE_GETSET1(BorderColor,const iColor&);
	IOBJECT_DECLARE_GETSET1(BorderWidth,int);

	IOBJECT_DECLARE_GET(ImageWidth,mFullWidth,int);
	IOBJECT_DECLARE_GET(ImageHeight,mFullHeight,int);
	int GetTileWidth() const { return mTileWidth; }
	int GetTileHeight() const { return mTileHeight; }

	void SetNumTiles(int nx, int ny);
	int GetNumTilesX() const { return mNumTilesX; }
	int GetNumTilesY() const { return mNumTilesY; }
	static const iObjectKey& KeyNumTiles();

	int GetNumForegroundWindows() const { return mForegroundWindow.Size(); }
	static const iObjectKey& KeyNumForegroundWindows();

	inline iImageComposerBackgroundWindow* GetBackgroundWindow(int i, int j) const
	{
		return this->GetBackgroundWindow(i+mNumTilesX*j);
	}

	iImageComposerBackgroundWindow* GetBackgroundWindow(int i) const;
	iImageComposerForegroundWindow* GetForegroundWindow(int i) const;

	void AddForegroundWindow(iViewModule *v);
	void RemoveForegroundWindow(int n);
	void MoveToBack(int n);

	//
	//  Useful keys for command-interface 
	//
	static const iObjectKey& KeyBackgroundWindowViewModule();
	static const iObjectKey& KeyBackgroundWindowWallpaperFile();

	static const iObjectKey& KeyForegroundWindowViewModule();
	static const iObjectKey& KeyForegroundWindowPositionX();
	static const iObjectKey& KeyForegroundWindowPositionY();
	static const iObjectKey& KeyForegroundWindowScale();
	static const iObjectKey& KeyForegroundWindowBorderWidth();
	static const iObjectKey& KeyForegroundWindowBorderColor();

	//
	//  Other functions
	//
	void Compose(iViewModule *vm, iStereoImageArray &images);
	bool IsActive();

	void Update();  // update all
	void UpdateWindowList();
	void UpdateSize();

protected:

	iImageComposer(iControlModule *cm, iShell *s);
	virtual ~iImageComposer();

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

	bool mBlockUpdate;

	bool mInnerBorder, mScaleBackground;
	iColor mBorderColor;
	int mBorderWidth;
	int mFullWidth, mFullHeight;
	int mTileWidth, mTileHeight;
	int mNumTilesX, mNumTilesY;

	iImageComposerBackgroundWindow** mBackgroundWindow;

	iArray<iImageComposerForegroundWindow*> mForegroundWindow;
};


inline void iImageComposerForegroundWindow::SetPositionX(int x)
{
	this->SetPosition(x,mPosY);
	mComposer->ClearCache();
}


inline void iImageComposerForegroundWindow::SetPositionY(int y)
{
	this->SetPosition(mPosX,y);
	mComposer->ClearCache();
}

#endif // IIMAGECOMPOSER_H

