/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iqtwidgettrackballsubject.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "imath.h"
#include "ishell.h"
#include "itransform.h"
#include "iviewmodule.h"

#include "iggwidget.h"
#include "iggwidgetrendermodebutton.h"

#include "iqtassistant.h"
#include "iqtwidgetbuttonsubject.h"
#include "iqtwidgethelper.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>

#ifdef IQT_3
#include <qbitmap.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#else
#include <QtGui/QBitmap>
#include <QtGui/QGridLayout>
#include <QtGui/QLabel>
#include <QtGui/QMouseEvent>
#include <QtGui/QPainter>
#endif

//
//  Templates
//
#include "iarraytemplate.h"


namespace iqtWidgetTrackBall_Private
{
	const float PI = 3.1415927535f;
	const float TWOPI = 2*PI;
};


using namespace iqtWidgetTrackBall_Private;

//
// Constructs a iTrackball.
//
iqtWidgetTrackBallSubject::iqtWidgetTrackBallSubject(iggWidget *owner, bool followCamera, iggWidgetRenderModeButton* rmb) : QWidget(iqtAssistant::Convert(owner->GetParent())
#ifdef IQT_3
																																	,0,Qt::WRepaintNoErase
#endif
																																	), ibgWidgetTrackBallSubject(owner,followCamera,rmb), mTimerIds(0)
{
	mWidgetHelper = new iqtWidgetHelper(this,owner); IERROR_ASSERT_NULL_POINTER(mWidgetHelper);
	//
	//  Create a layout
	//
	if(this->layout() != 0) delete this->layout();
	QGridLayout *l = iqtAssistant::NewLayout(this,1);

	mPixmap = new QPixmap(this->size());
	mPixmapMask = new QBitmap(this->size());
	mPainter = new QPainter;

	this->setMaximumSize(mEarth.NumLat,mEarth.NumLat);
	this->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
	this->setMinimumSize(mFixedSize,mFixedSize);

	if(rmb != 0)
	{
#ifdef IQT_3
		l->expand(2,2);
#endif
		iqtWidgetButtonSubject *rmbs = iPointerCast<iqtWidgetButtonSubject,ibgWidgetButtonSubject>(rmb->GetSubject());
		iqtAssistant::SetParent(rmbs,this);
		l->addWidget(rmbs,0,1);
#ifdef IQT_3
		l->setColStretch(0,10);
#else
		l->setColumnStretch(0,10);
#endif
		l->setRowStretch(1,10);
	}

	mInUpdate = false;
}


iqtWidgetTrackBallSubject::~iqtWidgetTrackBallSubject()
{
	while(mTimerIds.Size() > 0) this->killTimer(mTimerIds.RemoveLast());

	delete mPixmapMask;
	delete mPixmap;
	delete mPainter;
}


void iqtWidgetTrackBallSubject::GetSizes(int &w0, int &h0, int &rad, int &cor)
{
	w0 = this->width()/2 - 1;
	h0 = this->height()/2 - 1;
	if(mResizable) rad = (w0 < h0) ? w0 : h0; else rad = mFixedSize/2 - 1;
	cor = mFixedSize/8;
}


void iqtWidgetTrackBallSubject::Repaint()
{
	this->update();
}


void iqtWidgetTrackBallSubject::mousePressEvent(QMouseEvent *e)
{
	int w0, h0, rad, cor;
	this->GetSizes(w0,h0,rad,cor);

	if(e!=0 && e->button()==Qt::LeftButton)
	{
		mOldX = e->x();
		mOldY = e->y();

		if(mOldX>w0+rad-cor && mOldY>h0+rad-cor)
		{
			mResizable = !mResizable;
			this->layout()->invalidate();
			this->update();
		}
		iPointerCast<iqtWidgetHelper,ibgWidgetHelper>(mWidgetHelper)->OnVoid1();
	}

	if(e!=0 && e->button()==Qt::RightButton)
	{
		mOldX = e->x();
		mOldY = e->y();
		mTimerIds.Add(this->startTimer(100));
	}
}


void iqtWidgetTrackBallSubject::mouseReleaseEvent(QMouseEvent *e)
{
	if(e!=0 && (e->button()==Qt::LeftButton || e->button()==Qt::RightButton))
	{
		iPointerCast<iqtWidgetHelper,ibgWidgetHelper>(mWidgetHelper)->OnBool1(true);
		iPointerCast<iqtWidgetHelper,ibgWidgetHelper>(mWidgetHelper)->OnVoid2();
		if(e->button() == Qt::RightButton)
		{
			while(mTimerIds.Size() > 0) this->killTimer(mTimerIds.RemoveLast());
		}
	}
}

	
void iqtWidgetTrackBallSubject::mouseMoveEvent(QMouseEvent *e)
{
	if(e == 0) return;
		
	this->UpdateTransform(e->x(),e->y());
	iPointerCast<iqtWidgetHelper,ibgWidgetHelper>(mWidgetHelper)->OnBool1(false);
}


void iqtWidgetTrackBallSubject::timerEvent(QTimerEvent *e)
{
	if(e == 0) return;

	int x0 = mOldX;
	int y0 = mOldY;
	mOldX = this->width()/2 - 1;
	mOldY = this->height()/2 - 1;

	int x = mOldX + (x0-mOldX)/3;  // slow down the motion
	int y = mOldY + (y0-mOldY)/3;

	this->UpdateTransform(x,y);
	iPointerCast<iqtWidgetHelper,ibgWidgetHelper>(mWidgetHelper)->OnBool1(false);

	mOldX = x0;
	mOldY = y0;
}


void iqtWidgetTrackBallSubject::paintEvent(QPaintEvent *)
{
	if(mInUpdate) return;

	mInUpdate = true;

	vtkLinearTransform *ct = 0;
	int i, j;
	int w0, h0, rad, cor;
	this->GetSizes(w0,h0,rad,cor);

	if(!mFollowCamera)
	{
		ct = mOwner->GetShell()->GetControlModule()->GetViewModule()->GetRenderer()->GetActiveCamera()->GetViewTransformObject()->GetLinearInverse();
	}

	//
	//  Expansion triangle parameters
	//
#ifdef IQT_3
	QPointArray t(3);
#else
	QPolygon t(3);
#endif
	t.setPoint(0,w0+rad-cor,h0+rad);
	t.setPoint(1,w0+rad,h0+rad-cor);
	if(mResizable)
	{
		t.setPoint(2,w0+rad-cor,h0+rad-cor);
	}
	else
	{
		t.setPoint(2,w0+rad,h0+rad);
	}
	//
	//   Mini-renderer
	//
#ifdef IQT_3
	if(mDoubleBuffer)
	{
		if(mPixmap->size() != this->size())
		{
			mPixmap->resize(this->size());
			mPixmapMask->resize(this->size());
			mOldRad = 0;
		}

		if(mOldRad != rad)
		{
			//
			//  Create mask
			//
			mPixmap->fill(this->foregroundColor());
			mPixmapMask->fill(Qt::color0);
			mPainter->begin(mPixmapMask);
			mPainter->setPen(Qt::color1);
			mPainter->setBrush(Qt::color1);
			mPainter->drawEllipse(w0-rad,h0-rad,2*rad+1,2*rad+1);
			//
			//  Expansion triangle
			//
			mPainter->drawPolygon(t);
			mPainter->end();
			mPixmap->setMask(*mPixmapMask);

			mOldRad = rad;
		}
		mPainter->begin(mPixmap);
	}
	else
#endif
	{
		mPainter->begin(this);
	}

	QColor c;
	float r2, lon, lat, dimFactor, n0[3], n[3];
	for(i=-rad; i<=rad; i++)
	{
		n0[0] = float(i)/rad;
		for(j=-rad; j<=rad; j++)
		{
			n0[1] = float(j)/rad;
			r2 = n0[0]*n0[0] + n0[1]*n0[1];
			if(r2 <= 1.0)
			{
				n0[2] = sqrt(1.0-r2);
				
				if(this->isEnabled())
				{
					mBallTransform->TransformVector(n0,n);
					if(ct != 0)	ct->TransformVector(n,n);

					lat = asin(n[2]);
					lon = atan2(n[1],n[0]);
					
					//
					//  Unmodified color
					//
					c = iqtAssistant::Convert(this->ColorAtLonLat(lon,lat)); 
				}
				else
				{
					c = Qt::gray;
				}
				
				//
				//  Light calculation
				//
				dimFactor = vtkMath::Dot(mLightDirection,n0);
				if(dimFactor < 0.1) dimFactor = (float)0.1;
				c.setRgb(round(c.red()*dimFactor),round(c.green()*dimFactor),round(c.blue()*dimFactor));
				
				//
				//  Draw the scene
				//
				mPainter->setPen(c);
				mPainter->drawPoint(w0+i,h0-j);
			}
		}
	}
	
#ifdef IQT_3
	if(mDoubleBuffer)
	{
		bitBlt(this,w0-rad,h0-rad,mPixmap,w0-rad,h0-rad,2*rad+1,2*rad+1,Qt::CopyROP,false); 
	}
	else
#endif
	{
		//
		//  Expansion triangle
		//
#ifdef IQT_3
		mPainter->setBrush(this->foregroundColor());
#else
		mPainter->setBrush(this->palette().color(this->backgroundRole()));
#endif
		mPainter->drawPolygon(t);
	}

	mPainter->end();

	mInUpdate = false;
}

