/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
  

#include "iggrenderwindow.h"


#include "icontrolmodule.h"
#include "icrosssectionviewobject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iopengl.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include "iggmainwindow.h"
#include "iggshell.h"

#include "ibgmainwindowsubject.h"
#include "ibgrenderwindowsubject.h"

#include <vtkMapper.h>

#ifdef IOPENGL_X11
#include <X11/Xlib.h>
#endif

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


bool iggRenderWindow::mRenderIntoMemory = false;


iggRenderWindow::iggRenderWindow(iViewModule *vm) : iViewModuleComponent(vm), iShellComponent(vm?vm->GetControlModule()->GetShell():0)
{
	mSubject = iggSubjectFactory::CreateRenderWindowSubject(this); IERROR_ASSERT_NULL_POINTER(mSubject);

	mInitialized = mResizing = mMoving = false;

	mMaxStep = mCurStep = 0;

	GenericContext = 0;
	GenericDisplayId = 0;

	this->DoubleBufferOn();

	//
	//  Connect to Main Window
	//
	mMainWindow = 0;
	const iggShell *s = iPointerCast<const iggShell,iShell>(this->GetShell());
	if(s!=0 && s->GetMainWindow()!=0) this->AttachToMainWindow(s->GetMainWindow());

	glPolygonOffset(1.0,1.0);
}


iggRenderWindow::~iggRenderWindow()
{
	const iggShell *s = iPointerCast<const iggShell,iShell>(this->GetShell());
	if(s!=0 && s->GetMainWindow()!=0)
	{
		s->GetMainWindow()->UnRegister(mSubject);
	}

	mSubject->Delete();
}


void iggRenderWindow::AttachToMainWindow(iggMainWindow *mw)
{
	if(mMainWindow==0 && mw!=0)
	{
		mMainWindow = mw;
		mMainWindow->Register(mSubject);
		mMainWindow->GetMainSubject()->AttachRenderWindow(this);
	}
}


void iggRenderWindow::Initialize()
{
	if(!mInitialized) 
	{
		this->MakeCurrent();
		this->OpenGLInit();

		mSubject->Initialize(!mRenderIntoMemory);
		
		this->Mapped = 1;
		mInitialized = true;
	}
}


//
//  Rendering process
//
void iggRenderWindow::Start() 
{
	// 
	// Initialize the QGLWidget part of the widget if it has not
	// been initialized so far. 
	//
	if(!mInitialized) this->Initialize();
}


void iggRenderWindow::Render()  
{
	if(!mInitialized) this->Initialize();

	if(mRenderIntoMemory)
	{
		this->RenderIntoMemory();
		return;
	}

	if(this->GetViewModule()->IsRenderingImage())
	{	
		//
		//  This way the magnified image creation do not show in actual window
		//
		if(!mSubject->IsReadyForDumpingImage()) return;
		vtkOpenGLRenderWindow::Render();
		if(mCurStep == 0) 
		{
			this->GetViewModule()->GetProgressEventObserver()->Started(iProgressEventObserver::_Rendering);
			this->GetViewModule()->GetProgressEventObserver()->SetProgress(0.0);
			mMaxStep = this->GetViewModule()->GetImageMagnification();
			mMaxStep *= mMaxStep;
		}
		float p = (float)mCurStep/mMaxStep;
		this->GetViewModule()->GetProgressEventObserver()->SetProgress(p);
		mCurStep++;
		if(mCurStep == mMaxStep)
		{
			this->GetViewModule()->GetProgressEventObserver()->SetProgress(1.0);
			this->GetViewModule()->GetProgressEventObserver()->Finished();
			mCurStep = 0;
		}
	}
	else
	{
		mSubject->Render();
	}
}


void iggRenderWindow::Frame() 
{
	if(!this->AbortRender && this->DoubleBuffer && this->SwapBuffers) // VTK wants us to swap buffers
	{
		mSubject->SwapBuffers();
	}
}


void iggRenderWindow::MakeCurrent()
{
	mSubject->MakeCurrent();
}


//
//  Size and position
//
int* iggRenderWindow::GetPosition()
{
	//
	//  Update VTK positions
	//
	int wg[4];
	mSubject->GetWindowGeometry(wg);
	if(Position[0]!=wg[0] || Position[1]!=wg[1])
	{
		Position[0] = wg[0];
		Position[1] = wg[1];
		this->MTime.Modified(); // do not invoke observers
	}
	return this->vtkOpenGLRenderWindow::GetPosition();
}
	

int* iggRenderWindow::GetSize()
{
	//
	//  Update VTK size
	//
	int wg[4];
	mSubject->GetWindowGeometry(wg);
	if(Size[0]!=wg[2] || Size[1]!=wg[3])
	{
		Size[0] = wg[2];
		Size[1] = wg[3];
		this->MTime.Modified(); // do not invoke observers
	}
	return this->vtkOpenGLRenderWindow::GetSize();
}


void iggRenderWindow::SetPosition(int x, int y)
{
	if(mMoving) return;

	if(this->Position[0]!=x || this->Position[1]!=y)
	{
		int wg[4];
		mSubject->GetWindowGeometry(wg);
		wg[0] = x;
		wg[1] = y;
		mMoving = true;
		mSubject->SetWindowGeometry(wg);
		mMoving = false;
//		this->vtkOpenGLRenderWindow::SetPosition(x,y); in vtkWindow::SetPosition(...) this->Modified() is in the wrong place!!!
		this->Position[0] = x;
		this->Position[1] = y;
		this->Modified();
	}
}
	

void iggRenderWindow::SetSize(int w, int h)
{
	if(mResizing) return;

	if(this->Size[0]!=w || this->Size[1]!=h)
    {
		int wg[4];
		mSubject->GetWindowGeometry(wg);
		wg[2] = w;
		wg[3] = h;
		mResizing = true;
		mSubject->SetWindowGeometry(wg);
		this->SetSizeBody(w,h);
		mResizing = false;
	}
}


void iggRenderWindow::SetSizeBody(int w, int h)
{
	if(mResizing) return;

	if(this->Size[0]!=w || this->Size[1]!=h)
    {
	    this->Size[0] = w;
		this->Size[1] = h;
		this->Modified();
    }
}


void iggRenderWindow::SetFullScreen(int arg) 
{
	//
	// We do not need to do anything if the FullScreen mode
	// is already set to the specified value. 
	//
	if(this->FullScreen==arg || !this->mInitialized) return;
	
	mSubject->SetFullScreen(arg != 0);

	// set the mode 
	this->FullScreen = arg;
	this->Modified();
}


int* iggRenderWindow::GetScreenSize()
{
	iPointerCast<iggShell,iShell>(this->GetShell())->GetDesktopDimensions(mScreenSize[0],mScreenSize[1]);
	return mScreenSize;
}


//
//  Window system access
//
void* iggRenderWindow::GetGenericContext()
{
	if(GenericContext == 0)
	{
#ifdef IOPENGL_WIN
		GenericContext = (void *)GetDC((HWND)this->GetGenericWindowId());
#else
		GenericContext = XCreateGC((Display *)this->GetGenericDisplayId(),(unsigned long)this->GetGenericWindowId(),0,0);
#endif
    }

	return GenericContext;
}


void* iggRenderWindow::GetGenericDisplayId()
{
	if(GenericDisplayId == 0)
	{
#ifdef IOPENGL_WIN
		GenericDisplayId = (void *)wglGetCurrentContext();
#else
		GenericDisplayId = XOpenDisplay((char *)0);
#endif
	}

	return GenericDisplayId;
}


void* iggRenderWindow::GetGenericWindowId()
{
	return mSubject->GetGenericWindowId();
}


void iggRenderWindow::SetWindowName(const char *s)
{
	mSubject->SetWindowName(s);
}


void iggRenderWindow::RenderIntoMemory()
{
	//
	// This does not work under magnification yet. Or does it?
	//
	const int *size = this->GetSize();

	//
	//  QGLWidget::renderPixmap does nor render textures under Unix - force polygonal mode for xsections
	//
	int k;
	for(k=0; k<=this->GetViewModule()->GetCrossSectionFamily()->GetMaxMemberIndex(); k++) this->GetViewModule()->GetCrossSectionFamily()->GetMember(k)->ForcePolygonalMethod(true);
	//
	//  Need to reset all existing display lists
	//
	vtkMapper::GlobalImmediateModeRenderingOn();

	//
	//  render unto a pixmap
	//
	int db = this->DoubleBuffer;
	this->DoubleBuffer = 0; // Render into the front buffer

	iImage image;
	image.Scale(size[0],size[1]);

	mSubject->RenderIntoMemory(image);

	//
	//  Put them back so that Magnifier could read them
	//
	this->SetPixelData(0,0,size[0]-1,size[1]-1,image.DataPointer(),1);
	this->DoubleBuffer = db;

	vtkMapper::GlobalImmediateModeRenderingOff();
	for(k=0; k<=this->GetViewModule()->GetCrossSectionFamily()->GetMaxMemberIndex(); k++) this->GetViewModule()->GetCrossSectionFamily()->GetMember(k)->ForcePolygonalMethod(false);
}

