/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggpagedata.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatastretch.h"
#include "idatasubject.h"
#include "ierror.h"
#include "iexpressionparser.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdatatypeprovider.h"
#include "iggdialog.h"
#include "iggextensionwindow.h"
#include "iggframeboxsize.h"
#include "iggframedatatypeselector.h"
#include "iggframedatatypewidgetlist.h"
#include "iggframedatavariablelist.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetreloading.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgethelper.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"
#include "iggwidgetkeylineedittemplate.h"
#include "iggwidgetkeyslidertemplate.h"


namespace iggPageData_Private
{
	//
	//  Helper widgets
	//
	class KeyArea : public iggWidgetKeyHandler<iString>
	{
		
	public:
		
		KeyArea(const iObjectKey &key, iggFrame *parent, int index) : iggWidgetKeyHandler<iString>(_WidgetTypeOther,key,_RenderModeNoRender,parent,index,0)
		{
			mSubject = iggSubjectFactory::CreateWidgetDisplayAreaSubject(this,"");
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void QueryValue(iString &val) const
		{
			IERROR_REPORT_ERROR("This function should never be called.");
		}

		virtual void UpdateValue(iString val)
		{
			mSubject->SetText(val);
		}

		ibgWidgetDisplayAreaSubject *mSubject;
	};


	class LimitsExpandButton : public iggWidgetSimpleButton
	{
		
	public:
		
		LimitsExpandButton(bool out, const iggPageData *page, iggFrame *parent, int index) : iggWidgetSimpleButton("",parent,true), mPage(page)
		{
			mOut = out;
			mIndex = index;

			if(out) mSubject->SetIcon(*iImageFactory::FindIcon("expand.png")); else mSubject->SetIcon(*iImageFactory::FindIcon("contract.png"));

			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			iString ws;
			
			if(mPage->GetProvider()->IsThereDataType())
			{
				int n, s;
				float fmin, fmax;

				const iObjectKey &keyNumVars = mPage->GetProvider()->Translate(iDataSubject::KeyNumVars());
				const iObjectKey &keyStretch = mPage->GetProvider()->Translate(iDataSubject::KeyStretch());
				const iObjectKey &keyName = mPage->GetProvider()->Translate(iDataSubject::KeyName());
				const iObjectKey &keyMin = mPage->GetProvider()->Translate(iDataSubject::KeyMin());
				const iObjectKey &keyMax = mPage->GetProvider()->Translate(iDataSubject::KeyMax());

				if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && n>mIndex && this->GetShell()->GetControlModule()->QueryValue(keyStretch,mIndex,s,n,-1,-1) && this->GetShell()->GetControlModule()->QueryValue(keyMin,mIndex,fmin,n) && this->GetShell()->GetControlModule()->QueryValue(keyMax,mIndex,fmax,n))
				{
					iString ws;
					float d;

					fmin = iDataStretch::ApplyStretch(fmin,s,false);
					fmax = iDataStretch::ApplyStretch(fmax,s,true);

					if(mOut)
					{
						if(s==_StretchLog && fmax-fmin>3.0f) d = 1.0f; else d = 0.5f*(fmax-fmin);
					}
					else
					{

						if(s==_StretchLog && fmax-fmin>3.0f) d = -1.0f; else d = -0.25f*(fmax-fmin);

						const iObjectKey &keyLL = mPage->GetProvider()->Translate(iDataSubject::KeyLowerLimit());
						const iObjectKey &keyUL = mPage->GetProvider()->Translate(iDataSubject::KeyUpperLimit());

						float fmid, fll, ful;
						if(this->GetShell()->GetControlModule()->QueryValue(keyLL,mIndex,fll,n) && this->GetShell()->GetControlModule()->QueryValue(keyMax,mIndex,ful,n))
						{
							fmid = 0.5f*(fll+ful);
						}
						else
						{
							fmid = 0.5f*(fmin+fmax);
						}
					}

					fmin = (fmin>-iMath::_LargeFloat || d<0.0f) ? iDataStretch::ResetStretch(fmin-d,s) : iDataStretch::ResetStretch(fmin,s);
					fmax = (fmax< iMath::_LargeFloat || d<0.0f) ? iDataStretch::ResetStretch(fmax+d,s) : iDataStretch::ResetStretch(fmax,s);

					this->GetShell()->GetControlModule()->PackCommand(ws,keyMax,mIndex,fmax);
					this->GetShell()->GetControlModule()->PackCommand(ws,keyMin,mIndex,fmin);
					this->GetShell()->GetControlModule()->Execute(ws,!mOut,iggWidgetKeyHandlerBase::GetGlobalExecuteFlags());
					mParent->UpdateWidget();
				}
			}
		}

		bool mOut;
		int mIndex;
		const iggPageData *mPage;
	};


	class LimitsStretchComboBox : public iggWidgetKeyStretchComboBox
	{
		
	public:
		
		LimitsStretchComboBox(const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyStretchComboBox(iDataSubject::KeyStretch(),parent,index,0), mPage(page)
		{
			this->AttachDataTypeProvider(mPage->GetProvider());
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			//
			//  Optimization - direct access
			//
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider()->GetActiveDataType());
			if(mIndex >= lim->GetNumVars()) return true;

			return iggWidgetKeyComboBox::UpdateWidgetBody();
		}

		virtual void OnInt1Body(int i)
		{
			if(mPage->GetProvider()->IsThereDataType())
			{
				iggWidgetKeyComboBox::OnInt1Body(i);
				mParent->UpdateWidget();
			}
		}

		const iggPageData *mPage;
	};


	class LimitsNameEdit : public iggWidgetKeyTextLineEdit
	{
		
	public:
		
		LimitsNameEdit(const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyTextLineEdit(false,"Name",iDataSubject::KeyName(),_RenderModeUseGlobal,parent,index,0), mPage(page)
		{
			this->AttachDataTypeProvider(mPage->GetProvider());
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider()->GetActiveDataType());
			if(lim != 0) this->Show(!lim->AreNamesFixed()); else this->Show(false);
			//
			//  Optimization - direct access
			//
			mSubject->SetText(lim->GetName(mIndex));
			return true;
//			return iggWidgetKeyTextLineEdit::UpdateWidgetBody();
		}

		virtual void OnVoid1Body()
		{
			if(mPage->GetProvider()->IsThereDataType())
			{
				iggWidgetKeyTextLineEdit::OnVoid1Body();
				mParent->UpdateWidget();
			}
		}

		const iggPageData *mPage;
	};


	class LimitsRangeSlider : public iggWidgetKeyVariableLimitsSlider
	{
		
	public:
		
		LimitsRangeSlider(bool low, const iggPageData *page, iggFrame *parent, int index) : iggWidgetKeyVariableLimitsSlider(page->GetProvider(),6,low?"Min":"Max",low?iDataSubject::KeyLowerLimit():iDataSubject::KeyUpperLimit(),0,0,_RenderModeUseGlobal,parent,index), mPage(page)
		{
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			//
			//  Optimization - direct access
			//
			iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider()->GetActiveDataType());
			if(mIndex >= lim->GetNumVars()) return true;

			mMin = lim->GetMin(mIndex);
			mMax = lim->GetMax(mIndex);
			mStretch = lim->GetStretch(mIndex);
			mResolution = this->ComputeResolution();
			this->SetRange(0,mResolution);
			return iggWidgetKeyFloatSlider::UpdateWidgetBody();
//			return iggWidgetKeyVariableLimitsSlider::UpdateWidgetBody();
		}

		const iggPageData *mPage;
	};


	class LimitsFrameBox : public iggFrame
	{
		
	public:
		
		LimitsFrameBox(const iggPageData *page, iggFrame *parent, int index) : iggFrame("",parent,3), mPage(page)
		{
			mIndex = index;
			mNeedsBaloonHelp = false;

			LimitsRangeSlider *vls1 = new LimitsRangeSlider(true,mPage,this,index);
			LimitsRangeSlider *vls2 = new LimitsRangeSlider(false,mPage,this,index);
			vls1->AddBuddy(vls2);
			this->AddLine(vls1,2,new LimitsExpandButton(false,mPage,this,index),1);
			this->AddLine(vls2,2,new LimitsExpandButton(true,mPage,this,index),1);
			mStretchBox = new LimitsStretchComboBox(mPage,this,index);
			this->AddLine(mStretchBox,1,new LimitsNameEdit(mPage,this,index),2);
			this->SetColStretch(1,10);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
//			int n;
//			iString s;
			bool ret;
			
			if(mPage->GetProvider()->IsThereDataType())
			{
				const iObjectKey &keyNumVars = mPage->GetProvider()->Translate(iDataSubject::KeyNumVars());
				const iObjectKey &keyName = mPage->GetProvider()->Translate(iDataSubject::KeyName());

				//
				//  Optimization - direct access
				//
				iDataLimits *lim = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLimits(mPage->GetProvider()->GetActiveDataType());
				if(mIndex < lim->GetNumVars())
//				if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && mIndex<n && this->GetShell()->GetControlModule()->QueryValue(keyName,mIndex,s,n))
				{
//					this->SetTitle(s);
					this->SetTitle(lim->GetName(mIndex));
					mStretchBox->Show(!lim->GetFixedStretch(mIndex));
					this->Show(true);
					ret = iggFrame::UpdateWidgetBody();
				}
				else
				{
					this->Show(false);
					ret = true;
				}
			}
			else
			{
				this->Show(false);
				ret = true;
			}
			return ret;
		}

		int mIndex;
		const iggPageData *mPage;
		LimitsStretchComboBox *mStretchBox;
	};


	class LimitsFrameSet : public iggFrameScroll
	{
		
	public:
		
		LimitsFrameSet(const iggPageData *page, iggFrame *parent) : iggFrameScroll(parent,false), mPage(page), mBoxes(0)
		{
			mNeedsBaloonHelp = false;
		}

		virtual void UpdateChildren()
		{
			int i, n;

			//
			//  This is expensive to update, make sure it does not update needlessly
			//
			if(!mNeedsUpdate) return;

			const iObjectKey &keyNumVars = mPage->GetProvider()->Translate(iDataSubject::KeyNumVars());

			if(this->GetShell()->GetControlModule()->QueryValue(keyNumVars,n) && n>mBoxes.Size())
			{
				for(i=mBoxes.Size(); i<n; i++)
				{
					mBoxes.Add(new LimitsFrameBox(mPage,this->GetContents(),i));
					this->GetContents()->AddLine(mBoxes[i]);
					this->GetContents()->AddSpace(5);
				}
			}
			for(i=n-1; i>=0; i--) mBoxes[i]->Show(false);
			iggFrameScroll::UpdateChildren();
			for(i=0; i<n; i++) mBoxes[i]->Show(true);
		}

	protected:

		const iggPageData *mPage;
		iArray<LimitsFrameBox*> mBoxes;
	};


	class FixedLimitsFrame : public iggFrameDataTypeWidgetList
	{

	public:

		FixedLimitsFrame(iggFrame *parent) : iggFrameDataTypeWidgetList("Fix limits for data types:",parent)
		{
			this->UpdateList();
		}

	protected:

		virtual iggWidget* CreateEntry(const iDataType &type) const
		{
			return new iggWidgetReloadingCheckBox(type,type.GetTextName(),this->Translate(iDataSubject::KeyFixedLimits(),type),this->EntryParent());
		}
	};


	//
	//  Shift slider
	//
	class ShiftingSlider : public iggWidgetKeyDoubleSlider
	{

	public:

		ShiftingSlider(const iString &title, const iObjectKey &key, iggFrame *parent, int index) : iggWidgetKeyDoubleSlider(-1.0,1.0,200,0,5,title,key,_RenderModeUseGlobal,parent,index)
		{
		}

	protected:

		virtual bool ExecuteControl(bool final)
		{
			this->QueryValue(mValue);
			this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->SetDataShift(mIndex,mValue);
			
			if(final || this->GetRenderMode()==_RenderModeImmediate)
			{
				this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->ShiftData();
				this->GetShell()->GetControlModule()->Execute("",true,(iggWidgetKeyHandlerBase::GetGlobalExecuteFlags()==0)?(_ObjectModeCurrent|_ModuleModeCurrent|_RenderModeClones):iggWidgetKeyHandlerBase::GetGlobalExecuteFlags());
			}
			return true;
		}
	};


	//
	//  Erase widgets
	//
	class EraseCheckBox : public iggWidget
	{

	public:

		EraseCheckBox(const iDataType &type, const iString &title, iggFrame *parent) : iggWidget(parent), mType(type)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,title,1);

			iString tt = "Check this box if you would like to erase these data.";
			this->SetBaloonHelp(tt);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			mSubject->SetDown(false);
			mWidgetHelper->Enable(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(mType));
			return true;
		}

		virtual void OnVoid1Body()
		{
			if(mSubject->IsDown())
			{
				this->GetMainWindow()->AddEraseDataType(mType);
			}
			else
			{
				this->GetMainWindow()->RemoveEraseDataType(mType);
			}
		}

		ibgWidgetButtonSubject *mSubject;
		const iDataType &mType;
	};


	class ErasePushButton : public iggWidgetSimpleButton
	{

	public:

		ErasePushButton(const iString &title, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			iString tt = "Click this button to erase selected data.";
			this->SetBaloonHelp(tt);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			this->Enable(this->GetMainWindow()->GetEraseDataInfo().Count()>0 && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(this->GetMainWindow()->GetEraseDataInfo()));
			return true;
		}

		virtual void Execute()
		{
			if(this->GetMainWindow()->AskForConfirmation("Are you sure you want to erase data?","Erase"))
			{
				this->GetMainWindow()->EraseData();
				this->Enable(false);
			}
		}
	};


	class EraseFrame : public iggFrameDataTypeWidgetList
	{

	public:

		EraseFrame(ErasePushButton *button, iggFrame *parent) : iggFrameDataTypeWidgetList("Select the data to erase:",parent)
		{
			mButton = button;
			this->UpdateList();
		}

	protected:

		virtual iggWidget* CreateEntry(const iDataType &type) const
		{
			iggWidget *w = new EraseCheckBox(type,type.GetTextName(),this->EntryParent());
			w->AddBuddy(mButton);
			return w;
		}

		ErasePushButton *mButton;
	};


	//
	//  Function calculator
	//
	class CalculatorLineEdit : public iggWidget
	{

	public:

		CalculatorLineEdit(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,false,0,"%bFunction:",0);
			mSubject->SetEditable(false);

			mNeedsBaloonHelp = false;
		}

		void SelectText(int start, int len = -1)
		{
			mSubject->SelectText(start,len);
		}

		const iString GetText() const
		{
			return mSubject->GetText();
		}

		void SetText(const iString &text)
		{
			mSubject->SetText(text);
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}

		ibgWidgetEntrySubject *mSubject;
	};


	class CalculatorButton : public iggWidgetSimpleButton
	{

	public:

		CalculatorButton(const iString &title, const iString &text, CalculatorLineEdit *lineedit, iggFrame *parent) : iggWidgetSimpleButton(title,parent)
		{
			mText = text;
			mLineEdit = lineedit;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			iString line = mLineEdit->GetText();

			if(mText.IsEmpty())
			{
				if(line.Part(line.Length()-6) == "Vector")
				{
					//
					//  Remove 'Vector'
					//
					line = line.Part(0,line.Length()-6);
				}
				else if(line.Part(line.Length()-4,3) == "Var")
				{
					//
					//  Remove 'Var#'
					//
					line = line.Part(0,line.Length()-4);
				}
				else if(line[line.Length()-1] == '(')
				{
					//
					//  Remove 'fun('
					//
					int i = line.Length() - 2;
					while(i>=0 && line[i]>='a' && line[i]<='z') i--;
					line = line.Part(0,i+1);
				}
				else
				{
					//
					//  Remove one symbol
					//
					line = line.Part(0,line.Length()-1);
				}
			}
			else
			{
				line += mText;
			}
			mLineEdit->SetText(line);
		}

		iString mText;
		CalculatorLineEdit *mLineEdit;
	};


	class CalculatorDialog : public iggDialog
	{

	public:

		CalculatorDialog(iggWidgetKeyTextLineEdit *le, iggMainWindow *mw) : iggDialog(mw,_DialogModal,0,"Array Calculator","sr.gg.da",2)
		{
			mLineEdit = le;

			mParser = new iExpressionParser; IERROR_ASSERT_NULL_POINTER(mParser);

			mParser->SetScalarVariableValue("Var1",1.0);
			mParser->SetScalarVariableValue("Var2",1.0);
			mParser->SetScalarVariableValue("Var3",1.0);
			mParser->SetVectorVariableValue("Vector",0.0,0.0,0.0);

			mCalculatorLineEdit = new CalculatorLineEdit(mFrame);
			mFrame->AddLine(mCalculatorLineEdit,2);

			iggFrame *f1 = new iggFrame(mFrame,4);
			f1->AddLine(
                new CalculatorButton("Variable 1","Var1",mCalculatorLineEdit,f1),
				new CalculatorButton("Variable 2","Var2",mCalculatorLineEdit,f1),
				new CalculatorButton("Variable 3","Var3",mCalculatorLineEdit,f1),
				new CalculatorButton("Vector field","Vector",mCalculatorLineEdit,f1));
			mFrame->AddLine(f1,2);

			iggFrame *f2 = new iggFrame(mFrame,5);
			f2->AddLine(
                new CalculatorButton("+","+",mCalculatorLineEdit,f2),
				new CalculatorButton("-","-",mCalculatorLineEdit,f2),
				new CalculatorButton("x","*",mCalculatorLineEdit,f2),
				new CalculatorButton("/","/",mCalculatorLineEdit,f2),
				new CalculatorButton("^","^",mCalculatorLineEdit,f2));
			mFrame->AddLine(f2,2);

			iggFrame *f3 = new iggFrame("",mFrame,3);
			f3->AddLine(
                new CalculatorButton("abs","abs(",mCalculatorLineEdit,f3),
				new CalculatorButton("+/-","-",mCalculatorLineEdit,f3),
				new CalculatorButton("||  ||","mag(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("log","log(",mCalculatorLineEdit,f3),
				new CalculatorButton("exp","exp(",mCalculatorLineEdit,f3),
				new CalculatorButton("sqrt","sqrt(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("cos","cos(",mCalculatorLineEdit,f3),
				new CalculatorButton("sin","sin(",mCalculatorLineEdit,f3),
				new CalculatorButton("tan","tan(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("arccos","acos(",mCalculatorLineEdit,f3),
				new CalculatorButton("arcsin","asin(",mCalculatorLineEdit,f3),
				new CalculatorButton("arctan","atan(",mCalculatorLineEdit,f3));
			f3->AddLine(
				new CalculatorButton("cosh","cosh(",mCalculatorLineEdit,f3),
				new CalculatorButton("sinh","sinh(",mCalculatorLineEdit,f3),
				new CalculatorButton("tanh","tanh(",mCalculatorLineEdit,f3));

			iggFrame *f4 = new iggFrame("",mFrame,3);
			f4->AddLine(
				new CalculatorButton("7","7",mCalculatorLineEdit,f4),
				new CalculatorButton("8","8",mCalculatorLineEdit,f4),
				new CalculatorButton("9","9",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("4","4",mCalculatorLineEdit,f4),
				new CalculatorButton("5","5",mCalculatorLineEdit,f4),
				new CalculatorButton("6","6",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("1","1",mCalculatorLineEdit,f4),
				new CalculatorButton("2","2",mCalculatorLineEdit,f4),
				new CalculatorButton("3","3",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("0","0",mCalculatorLineEdit,f4),
				new CalculatorButton(".",".",mCalculatorLineEdit,f4),
				new CalculatorButton("E","e",mCalculatorLineEdit,f4));
			f4->AddLine(
				new CalculatorButton("(",")",mCalculatorLineEdit,f4),
				new CalculatorButton(")",")",mCalculatorLineEdit,f4),
				new CalculatorButton("<-","",mCalculatorLineEdit,f4));
			mFrame->AddLine(f3,f4);
		}

		virtual ~CalculatorDialog()
		{
			delete mParser;
		}

		virtual void Show(bool s)
		{
			iString line;

			mLineEdit->QueryValue(line);
			mCalculatorLineEdit->SetText(line);
			iggDialog::Show(s);

			if(mCalculatorLineEdit->GetText() != line)
			{
				this->GetMainWindow()->RequestReloadData(iDataType::UniformScalars());
			}

		}

		virtual const iString& GetToolTip() const
		{
			static const iString tt = "Calculator allows you to do mathematical operations on scalar data.";
			return tt;
		}

	protected:

		virtual bool CanBeClosed()
		{
			iString line = mCalculatorLineEdit->GetText();

			if(line.IsEmpty())
			{
				mLineEdit->ExecuteControl(true);
				this->GetShell()->GetControlModule()->Render(_RenderModeClones);
				mFrame->GetMainWindow()->UpdateAll();
				return true;
			}
			
			if(mParser->CheckSyntax(line))
			{
				double tmp[3];
				if(mParser->GetScalarResult(tmp[0]))
				{
					mLineEdit->SetText(line);
					mLineEdit->ExecuteControl(true);
					mFrame->GetMainWindow()->UpdateAll();
					return true;
				}
				else if(mParser->GetVectorResult(tmp))
				{
					mFrame->GetMainWindow()->PopupWindow("Result is a vector",_PopupWindowError);
					return false;
				}
				else
				{
					mCalculatorLineEdit->SelectText(mParser->GetErrorPosition());
					mFrame->GetMainWindow()->PopupWindow(mParser->GetErrorMessage(),_PopupWindowError);
					return false;
				}
			}
			else
			{
				mCalculatorLineEdit->SelectText(mParser->GetErrorPosition());
				mFrame->GetMainWindow()->PopupWindow(mParser->GetErrorMessage(),_PopupWindowError);
				return false;
			}
		}

		iString mLine;
		iExpressionParser *mParser;
		iggWidgetKeyTextLineEdit *mLineEdit;
		CalculatorLineEdit *mCalculatorLineEdit;
	};
};


using namespace iggPageData_Private;


iggPageData::iggPageData(iggFrameBase *parent) : iggPageMain(parent)
{
	const iImage *icon = iImageFactory::FindIcon("data.png");

	mInfoBuffer = new iDataInfo[__NumData]; IERROR_ASSERT_NULL_POINTER(mInfoBuffer);
	mProvider = new iggKeywordDataTypeProvider("ScalarField",this); IERROR_ASSERT_NULL_POINTER(mProvider);

	mInfoBuffer[_DataOnBoundaryCondition] = iDataType::UniformScalars() + iDataType::UniformVectors() + iDataType::UniformTensors() + iDataType::BasicParticles();
	mInfoBuffer[_DataOnVoxelLocation] = iDataType::UniformScalars() + iDataType::UniformVectors() + iDataType::UniformTensors();
	mInfoBuffer[_DataOnCellToPointMode] = iDataType::UniformScalars() + iDataType::UniformVectors() + iDataType::UniformTensors();
	mInfoBuffer[_DataOnParticlesDownsample] = iDataType::BasicParticles();
	mInfoBuffer[_DataOnParticlesOrderInFileIsAttribute] = iDataType::BasicParticles();

	this->GetMainWindow()->GetExtensionWindow()->AddReloadingDataTypes(this);

	//
	//  Main page
	// ************************************************
	//
	iggFrame *page0 = new iggFrame(mBook,1);
	mBook->AddPage("Settings",icon,page0);
	//
	//  Book
	//
	iggFrameBook *sb = new iggFrameBook(page0);
	page0->AddLine(sb);
	//
	//  Global page
	//
	iggFrame *sbpage0 = new iggFrame(sb,2);
	sb->AddPage("Global",icon,sbpage0);
	
	sbpage0->AddLine(new iggFrameBoxSize(sbpage0));
	sbpage0->AddSpace(2);

	iggWidgetKeyRadioBox *orb = new iggWidgetKeyRadioBox(1,"Optimize for...",0,iControlModule::KeyOptimizationMode(),sbpage0);
	orb->InsertItem("Speed");
	orb->InsertItem("Memory");
	orb->InsertItem("Quality");
	sbpage0->AddLine(orb);
	sbpage0->AddSpace(2);

	iggWidgetReloadingRadioBox *bc = new iggWidgetReloadingRadioBox(mInfoBuffer[_DataOnBoundaryCondition],"Boundary conditions",iDataReader::KeyBoundaryConditions(),sbpage0);
	bc->InsertItem("Periodic");
	bc->InsertItem("Wall");
	sbpage0->AddLine(bc);
	sbpage0->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",sbpage0));
	sbpage0->AddSpace(2);

	iggWidgetReloadingPushButton *rb = new iggWidgetReloadingPushButton("Reload data",sbpage0);
	rb->AddDependent(bc);
	sbpage0->AddLine(rb);
	sbpage0->AddSpace(10);
	sbpage0->SetColStretch(1,10);
	//
	//  Field data page
	//
	iggFrame *sbpage1 = new iggFrame(sb,2);
	sb->AddPage("Field data",icon,sbpage1);

//	sbpage1->AddLine(new iggWidgetKeyCheckBox("Readjust limits to correspond to the data file.",iDataReader::KeyAdjustableLimits(),sbpage1));

	iggWidgetReloadingRadioBox *vl = new iggWidgetReloadingRadioBox(mInfoBuffer[_DataOnVoxelLocation],"Voxel location",iDataReader::KeyVoxelLocation(),sbpage1);
	vl->InsertItem("Vertex of a cell");
	vl->InsertItem("Center of a cell");
	sbpage1->AddLine(vl);
	sbpage1->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",sbpage1));

	iggWidgetReloadingRadioBox *cp = new iggWidgetReloadingRadioBox(mInfoBuffer[_DataOnCellToPointMode],"Cell-to-point method",iDataReader::KeyCellToPointMode(),sbpage1);
	cp->InsertItem("Fourier shift (slow)");
	cp->InsertItem("8-point average");
	cp->InsertItem("8-point median");
	cp->InsertItem("8-point minimum");
	cp->InsertItem("8-point maximum");
	cp->InsertItem("Corner projection");
	sbpage1->AddLine(cp);
	sbpage1->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",sbpage1));

	sbpage1->AddSpace(10);
	sbpage1->SetColStretch(1,10);
	//
	//  Particle data page
	//
	iggFrame *sbpage2 = new iggFrame(sb,2);
	sb->AddPage("Particle data",icon,sbpage2);

	iggFrame *df = new iggFrame("Downsample particles",sbpage2,2);

	iggWidgetReloadingSlider *dl = new iggWidgetReloadingSlider(mInfoBuffer[_DataOnParticlesDownsample],1,30,2,"Factor",iDataReader::KeyParticlesDownsampleFactor(),df);
	df->AddLine(dl,2);
	df->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",df));

	iggWidgetReloadingRadioBox *db = new iggWidgetReloadingRadioBox(mInfoBuffer[_DataOnParticlesDownsample],"Downsample mask",iDataReader::KeyParticlesDownsampleMode(),df);
	db->InsertItem("Regular");
	db->InsertItem("Random");
	db->InsertItem("Head of file");
	db->InsertItem("Tail of file");
	df->AddLine(db);
	df->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",df));
	df->SetColStretch(1,10);

	sbpage2->AddLine(df);
	sbpage2->AddSpace(10);

	sbpage2->AddLine(new iggWidgetReloadingCheckBox(mInfoBuffer[_DataOnParticlesOrderInFileIsAttribute],"Include the particle number as an attribute",iDataReader::KeyParticlesOrderInFileIsAttribute(),sbpage2));
	sbpage2->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",sbpage2));

	sbpage2->AddSpace(10);
	sbpage2->SetColStretch(1,10);

	//
	//  Limits page
	// ************************************************
	//
	iggFrame *page1 = new iggFrame(mBook,1);
	mBook->AddPage("Limits",icon,page1);
	//
	//  Book
	//
	iggFrameBook *lb = new iggFrameBook(page1);
	page1->AddLine(lb);
	//
	//  Scalar limits page
	//
	iggFrame *lbpage0 = new iggFrame(lb,1);
	lb->AddPage("Scalar limits",icon,lbpage0);

	lbpage0->AddLine(new LimitsFrameSet(this,lbpage0));
	//
	//  Fixed/adjustable page
	//
	iggFrame *lbpage1 = new iggFrame(lb,2);
	lb->AddPage("Fixed/Adjustable",icon,lbpage1);

	lbpage1->AddLine(new FixedLimitsFrame(lbpage1));
	lbpage1->AddSpace(10);
	lbpage1->SetRowStretch(0,8);
	lbpage1->SetColStretch(1,10);

	//
	//  Placement page
	// ************************************************
	//
	iggFrame *page2 = new iggFrame(mBook,3);
	mBook->AddPage("Placement",icon,page2);

	if(false)
	{
		//
		//  Book
		//
		iggFrameBook *pb = new iggFrameBook(page2);
		page2->AddLine(pb,2);
		//
		//  Shift page
		//
		iggFrame *pbpage0 = new iggFrame(pb,2);
		pb->AddPage("Shift",icon,pbpage0);
		
		pbpage0->AddLine(new ShiftingSlider("%bX",iDataReader::KeyShiftData(),pbpage0,0));
		pbpage0->AddLine(new ShiftingSlider("%bY",iDataReader::KeyShiftData(),pbpage0,1));
		pbpage0->AddLine(new ShiftingSlider("%bZ",iDataReader::KeyShiftData(),pbpage0,2));
		
		pbpage0->AddSpace(10);
		pbpage0->SetColStretch(0,10);
		pbpage0->SetColStretch(1,3);
		//
		//  Fit page
		//
		iggFrame *pbpage1 = new iggFrame(pb,2);
		pb->AddPage("Fit",icon,pbpage1);
		
		iggWidgetKeyRadioBox *fb = new iggWidgetKeyRadioBox(1,"Dimension to fit into the bounding box",0,iDataReader::KeyScaledDimension(),pbpage1);
		fb->InsertItem("Longest");
		fb->InsertItem("Shortest");
		fb->InsertItem("X");
		fb->InsertItem("Y");
		fb->InsertItem("Z");
		pbpage1->AddLine(fb);
		
		pbpage1->AddSpace(10);
		pbpage1->SetColStretch(1,10);
	}
	else
	{
		//
		//  No book - all on one page
		//
		//
		//  Shift page
		//
		iggFrame *sh = new iggFrame("Shift data",page2);
		sh->AddLine(new ShiftingSlider("%bX",iDataReader::KeyShiftData(),sh,0));
		sh->AddLine(new ShiftingSlider("%bY",iDataReader::KeyShiftData(),sh,1));
		sh->AddLine(new ShiftingSlider("%bZ",iDataReader::KeyShiftData(),sh,2));
		page2->AddLine(sh,2);
		page2->AddSpace(2);
		//
		//  Fit page
		//
		iggWidgetKeyRadioBox *fb = new iggWidgetKeyRadioBox(1,"Dimension to fit into the bounding box",2,iDataReader::KeyScaledDimension(),page2);
		fb->InsertItem("Longest");
		fb->InsertItem("Shortest");
		fb->InsertItem("X");
		fb->InsertItem("Y");
		fb->InsertItem("Z");
		page2->AddLine(fb);
		
		page2->AddSpace(10);
		page2->SetColStretch(1,10);
		page2->SetColStretch(2,3);
	}		

	//
	//  Erase page
	// ************************************************
	//
	iggFrame *page3 = new iggFrame(mBook,2);
	mBook->AddPage("Erase",icon,page3);

	//
	//  No book - all on one page
	//
	ErasePushButton *eb = new ErasePushButton("Erase selected data",page3);
	page3->AddLine(new EraseFrame(eb,page3));
	page3->AddLine(eb);

	page3->AddSpace(10);
	page3->SetRowStretch(0,10);
	page3->SetColStretch(1,10);

	//
	//  Transform page
	// ************************************************
	//
	iggFrame *page4 = new iggFrame(mBook,2);
	mBook->AddPage("Operate on scalar data",icon,page4);

	iggFrameDataVariableList *ol = new iggFrameDataVariableList(mProvider,"Assign result to...",iDataReader::KeyScalarFieldCalculatorOutputVariable(),0,page4,0);
	ol->Complete();
	iggFrame *of = new iggFrame(page4,1);
	of->AddSpace(10);
	iggWidgetKeyTextLineEdit *oe = new iggWidgetKeyTextLineEdit(true,"= ",iDataReader::KeyScalarFieldCalculatorFunction(),_RenderModeNoRender,of);
	of->AddLine(oe);
	of->AddSpace(10);
	page4->AddLine(ol,of);
	page4->AddSpace(2);

	CalculatorDialog *cd = new CalculatorDialog(oe,this->GetMainWindow());
	page4->AddLine(new iggWidgetLaunchButton(cd,"Launch calculator",page4));
	page4->AddLine(new iggWidgetTextArea("%b(May require reloading of data)",page4));

	page4->AddSpace(10);
	page4->SetColStretch(1,10);
	page4->SetColStretch(2,3);
}


iggPageData::~iggPageData()
{
	delete [] mInfoBuffer;
	delete mProvider;
}


void iggPageData::UpdateChidren()
{
	mBook->EnablePage(3,mProvider->GetActiveDataTypeIndex()==0U);
	iggPageMain::UpdateChildren();
}


iggFrame* iggPageData::CreateFlipFrame(iggFrameBase *parent) const
{
	return new iggFrameDataTypeSelector(mProvider,false,"Current data",parent);
}


void iggPageData::UpdateOnDataLoad()
{
	mProvider->FindAvailableData();
}


void iggPageData::AddReloadingType(int key, const iDataType &type)
{
	if(key>=0 && key<__NumData)
	{
		mInfoBuffer[key] += type;
	}
}

