/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  Toolkit-independent MainWindow class
//

#ifndef IGGMAINWINDOW_H
#define IGGMAINWINDOW_H


#include "iobject.h"
#include "iggmenuwindow.h"


#include "iarray.h"
#include "istring.h"

#include "iggparameter.h"

class iColor;
class iDataInfo;
class iDataType;
class iImage;

class iggDialog;
class iggDialogAbout;
class iggDialogAnimatingProgress;
class iggDialogBase;
class iggDialogDataExplorer;
class iggDialogEventRecorder;
class iggDialogFileSetExplorer;
class iggDialogHelp;
class iggDialogImageComposer;
class iggDialogLoadFile;
class iggDialogPaletteEditor;
class iggDialogParallelController;
class iggDialogPickerWindow;
class iggDialogRenderingProgress;
class iggDialogScriptDebugger;
class iggExtensionWindow;
class iggFrame;
class iggFrameBook;
class iggFrameFlip;
class iggFrameTopParent;
class iggPageData;
class iggPageParticles;
class iggPageView;
class iggWidget;
class iggWidgetProgressBar;
class iggWidgetTextBrowser;

class ibgMainWindowSubject;
class ibgWindowSubject;


class iggMainWindow : public iObject, public iggMenuWindow
{

	friend class iggShell;

public:

	virtual ~iggMainWindow();
	static const iObjectType& Type();

	//
	//  Register dependent windows
	//
	void Register(ibgWindowSubject *ws);
	void UnRegister(ibgWindowSubject *ws);

	//
	//  Access components
	//
	inline ibgMainWindowSubject* GetMainSubject() const { return mMainSubject; }

	inline iggExtensionWindow* GetExtensionWindow() const { return mExtensionWindow; }

	iggWidgetProgressBar* GetProgressBar() const;
	iggFrameFlip* GetDataTypeFrame() const { return mDataTypeFrame; }

	inline iggDialog* GetDialogDataExplorer() const { return mDialogDataExplorer; }
	inline iggDialog* GetDialogPaletteEditor() const { return mDialogPaletteEditor; };
	inline iggDialog* GetDialogPerformanceMeter() const { return mDialogPerformanceMeter; };

	inline iggDialogAnimatingProgress* GetDialogAnimatingProgress() const { return mDialogAnimatingProgress; };
	inline iggDialogEventRecorder* GetDialogEventRecorder() const { return mDialogEventRecorder; };
	inline iggDialogHelp* GetDialogHelp() const { return mDialogHelp; };
	inline iggDialogImageComposer* GetDialogImageComposer() const { return mDialogImageComposer; };
	inline iggDialogParallelController* GetDialogParallelController() const { return mDialogParallelController; };
	inline iggDialogPickerWindow* GetDialogPickerWindow() const { return mDialogPickerWindow; };
	inline iggDialogRenderingProgress* GetDialogRenderingProgress() const { return mDialogRenderingProgress; };
	inline iggDialogScriptDebugger* GetDialogScriptDebugger() const { return mDialogScriptDebugger; };

	inline iggPageParticles* GetParticlesPage() const { return mParticlesPage; }

	//
	//  User notification
	//
	int PopupWindow(const iString &text, int type = iggParameter::_PopupWindowMessage, const char *b0 = "Ok", const char *b1 = 0, const char *b2 = 0);
	int PopupWindow(const iggFrame *parent, const iString &text, int type = iggParameter::_PopupWindowMessage, const char *b0 = "Ok", const char *b1 = 0, const char *b2 = 0);

	void WriteToLog(const iString &prefix, const iString &message, const iColor &c = iColor(0,0,0));
	void ClearLog();

	inline bool IsIdiosyncratic() const { return mIdiosyncratic; }
	bool AskForConfirmation(const iString &message, const char *action);

	void ProcessEvents(bool sync = false) const;
	void AddTheme(const iString &name);

	//
	//  Data handling
	//
	void LoadData(const iDataType &type, const iString &filename, int mod = -1);
	void AfterLoadData(const iDataType &type, const iString &filename);

	void ReloadData();
	void AfterReloadData(const iDataType &type);
	inline const iDataInfo& GetReloadDataInfo() const { return mReloadDataInfo; }
	void AddReloadDataType(const iDataType &type);
	void RequestReloadData(const iDataInfo &info);

	void EraseData();
	void AfterEraseData(const iDataType &type);
	inline const iDataInfo& GetEraseDataInfo() const { return mEraseDataInfo; }
	void AddEraseDataType(const iDataType &type);
	void RemoveEraseDataType(const iDataType &type);

	//
	//  Multiple window settings
	//
	inline void SetWindowUnderFocusCurrent(bool s){ mWindowUnderFocusCurrent = s; }
	inline bool GetWindowUnderFocusCurrent() const { return mWindowUnderFocusCurrent; }

	//
	//  Update components
	//
	void UpdateOnPick();
	void UpdateMarkerWidgets();
	void UpdateParticleSetWidgets(const iImage *icon = 0);

	void ShowAll(bool s);
	void Block(bool s);
	void Exit();

	//
	//  Main book manipulation
	//
	void OpenBookPage(int n);
	void SetTabMode(int m);

	//
	//  Render window events
	//
	void OnRenderWindowMove(int wn);
	void OnRenderWindowResize(int wn);
	void OnRenderWindowFocusIn(int wn);
	void OnRenderWindowFocusOut(int wn);
	void OnRenderWindowEnter(int wn);
	void OnRenderWindowLeave(int wn);

	//
	//  Attached windows manipulation
	//
	void DisplayWindowsAsIcons();
	void DisplayWindowsAsWindows();
	void MoveWindows(bool all);

	inline bool IsDocked() const { return mDocked; }
		
protected:

	iggMainWindow(iShell *shell);

	void StartInitialization();
	void PreShowInitialization();
	void PostShowInitialization();

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

	virtual void UpdateContents();

	//
	//  Menu work
	//
	virtual void OnMenuBody(int id, bool on);
	void BuildMenus();

	//
	//  Misc functions
	//
	void ShowToolTips(bool s);
	void ShowInteractorHelp(bool s, iViewModule *vm = 0);
	void DockWindows(bool s, bool show);
	bool IsExitAllowed();
	void SetTheme(int n);

	void LogDataInfo(const iDataType &type, bool details);

	//
	//  Subject
	//
	ibgMainWindowSubject *mMainSubject;

	//
	//  Extension
	//
	iggExtensionWindow *mExtensionWindow;

	//
	//  Dialogs
	//
	iggDialog *mDialogAbout;
	iggDialog *mDialogAxesLabels;
	iggDialog *mDialogCommandLine;
	iggDialog *mDialogDataExplorer;
	iggDialog *mDialogDocking;
	iggDialog *mDialogFileSetExplorer;
	iggDialog *mDialogPaletteEditor;
	iggDialog *mDialogPerformanceMeter;

	iggDialogAnimatingProgress *mDialogAnimatingProgress;
	iggDialogEventRecorder *mDialogEventRecorder;
	iggDialogHelp *mDialogHelp;
	iggDialogImageComposer *mDialogImageComposer;
	iggDialogLoadFile *mDialogLoadFile;
	iggDialogParallelController *mDialogParallelController;
	iggDialogPickerWindow *mDialogPickerWindow;
	iggDialogRenderingProgress *mDialogRenderingProgress;
	iggDialogScriptDebugger *mDialogScriptDebugger;

#if defined(I_DEBUG) && defined(I_EDITION)
	iggDialog *mDialogDebugHelper;
#endif
	//
	//  Widget components
	//
	iggFrameTopParent *mBusyIndicatorFrame, *mVisitedFileListFrame;
	iggFrameFlip *mBaseFrame, *mDataTypeFrame;
	iggWidget *mBusyIndicator, *mVisitedFileList;
	iggFrameBook *mBook;
	iggPageView *mViewPage;
	iggPageParticles *mParticlesPage;
	iggPageData *mDataPage;
	iggWidgetTextBrowser *mLog;

	//
	//  Helper variables
	//
	iArray<iString> mThemeList;
	int mCurrentTheme;
	iPointerArray<ibgWindowSubject> mWindowList;
	iggWidgetProgressBar *mProgressBar;
	bool mDocked, mInteractorHelp, mToolTips, mOptionsAreGlobal, mWindowUnderFocusCurrent, mDetailLoadedDataInfo;
	bool mIdiosyncratic, mAttachWindows, mDoNotOfferToReload, mInitialized, mInMove, mTwoSubjectsAreTheSame;
	iDataInfo &mReloadDataInfo, &mEraseDataInfo;
	iString mStateFileName;
	int mInitialGeometry[4], mPrevPos[2], mProgressBarMode;

private:

	iggMainWindow(const iggMainWindow&); // Not implemented.
	void operator=(const iggMainWindow&);  // Not implemented.

	static const iObjectKey& KeyGeometry();
	static const iObjectKey& KeyExtensionGeometry();
	static const iObjectKey& KeyDocked();
	static const iObjectKey& KeyInteractorHelp();
	static const iObjectKey& KeyToolTips();
	static const iObjectKey& KeyOptionsAreGlobal();
	static const iObjectKey& KeyWindowUnderFocusCurrent();
	static const iObjectKey& KeyIsIdiosyncratic();
	static const iObjectKey& KeyAttachWindows();
	static const iObjectKey& KeyTabMode();
	static const iObjectKey& KeyTheme();
};

//
//  Useful Macros
//
#define FOR_EVERY_RENDER_WINDOW(window) \
vtkRenderWindowCollection *c_; iggRenderWindow *window; int k_, n_ = this->GetShell()->GetControlModule()->GetNumberOfViewModules(); \
for(k_=0;  k_<n_ && (c_=this->GetShell()->GetControlModule()->GetViewModule(k_)->GetRenderWindowCollection(true))!=0; k_++) while((window=iPointerCast<iggRenderWindow,vtkRenderWindow>(c_->GetNextItem())) != 0)

#endif  // IGGMAINWINDOW_H

