/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivectorfieldviewobject.h"


#include "iboundedplanesource.h"
#include "iboundeddisksource.h"
#include "iboundedspheresource.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "imarkerviewobject.h"
#include "ipicker.h"
#include "ireplicatedactor.h"
#include "istreamlinefilter.h"
#include "ivectorfieldglyphparallelpipeline.h"
#include "ivectorfieldglyphserialpipeline.h"
#include "ivectorfieldstreamlineparallelpipeline.h"
#include "ivectorfieldstreamlineserialpipeline.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkStructuredPoints.h>


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iVectorFieldViewObject,Vector,v);

IOBJECT_DEFINE_KEY(iVectorFieldViewObject,AttachSourceToMarker,am,OffsetInt,1);  //  for sphere and plane
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,LineDir,ld,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,LineLength,ll,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,LineQuality,lq,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,SourcePositionToPickedPoint,s2p,Bool,1); // action key
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,SourceDirection,sd,Float,3);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,SourceOpacity,so,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,NumberOfStreamLines,nl,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,ShowSourceObject,sso,Bool,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,SourceType,st,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,TubeRangeFactor,tr,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,TubeSize,ts,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewObject,TubeVariationFactor,tv,Float,1);

IOBJECT_DEFINE_DISTANCE_KEY(iVectorFieldViewObject,SourceSize,ss);
IOBJECT_DEFINE_POSITION_KEY(iVectorFieldViewObject,SourcePosition,sx);

//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iVectorFieldViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iVectorFieldViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(iVectorFieldViewObject,2);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(iVectorFieldViewObject,2);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(iVectorFieldViewObject,2);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_FIELD(iVectorFieldViewObject);

//
//  Templates
//
#include "iarraytemplate.h"

//
// iVectorFieldViewObject class
//
iVectorFieldViewObject* iVectorFieldViewObject::New(iViewModule *vm)
{
	iVectorFieldViewObject *tmp =  iPointerCast<iVectorFieldViewObject,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	IERROR_ASSERT_NULL_POINTER(tmp);
	//
	//  Create pipelines (must be created after the object is fully created)
	//	
	tmp->mGlyphPipeline = tmp->CreateGlyphParallelPipeline(); IERROR_ASSERT_NULL_POINTER(tmp->mGlyphPipeline);
	tmp->mStreamLinePipeline = tmp->CreateStreamLineParallelPipeline(); IERROR_ASSERT_NULL_POINTER(tmp->mStreamLinePipeline);
	tmp->mActors[0]->SetInput(tmp->mGlyphPipeline->GetOutput());

	vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(pd);
	pd->SetPoints(tmp->mStreamLineSourcePoints);
	pd->SetVerts(tmp->mStreamLineSourceVerts);
	pd->GetPointData()->SetNormals(tmp->mStreamLineSourceNorms);
	tmp->mStreamLinePipeline->SetInput(pd);
	pd->Delete();

	tmp->Reset();

	return tmp;
}


iVectorFieldViewObject::iVectorFieldViewObject(iViewModule *vm, const iDataInfo &info) : iFieldViewObject(vm,info,2,1U,_ViewObjectAlwaysShaded), mSourceSize(vm,true), mSourcePos(vm)
{
	mName = "iVectorFieldViewObject";
	mObjectType = _ObjectTypeVectorField;

	this->AddSecondaryDataType(iDataType::UniformScalars());

	if(mPrimaryDataInfo->Index(iDataType::UniformVectors()) == -1)
	{
		IERROR_REPORT_FATAL_ERROR("iVectorFieldViewObject is configured incorrectly.");
	}

	mMethod = _VectorMethodGlyph;
	mLineQuality = 0;
	mLineLength = 1.0;
	mLineDir = _StreamLineDirBothWays;
	mTubeSize = 1;
	mTubeRangeFactor = 10;
	mTubeVariationFactor = 0.01; 
	mColorByOffset = 0;

	mSourceShow = false;
	mSourceType = _VectorSourcePlane;
	mNumLines = 4;
	mSourceDir[0] = mSourceDir[1] = 0.0; mSourceDir[2] = 1.0;
	mSourceSize = 0.5;

	mColorBy = _VectorColorByNone;
	//
	//  Do VTK stuff
	//	
	mStreamLineSourcePoints = vtkPoints::New(VTK_DOUBLE); IERROR_ASSERT_NULL_POINTER(mStreamLineSourcePoints);
	mStreamLineSourceVerts = vtkCellArray::New(); IERROR_ASSERT_NULL_POINTER(mStreamLineSourceVerts);
	mStreamLineSourceNorms = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(mStreamLineSourceNorms);
	mStreamLineSourceNorms->SetNumberOfComponents(3);

	mSourceDisk = iBoundedDiskSource::New(this); IERROR_ASSERT_NULL_POINTER(mSourceDisk);
	mSourcePlane = iBoundedPlaneSource::New(this); IERROR_ASSERT_NULL_POINTER(mSourcePlane);
	mSourceSphere = iBoundedSphereSource::New(this); IERROR_ASSERT_NULL_POINTER(mSourceSphere);

	mSourceDisk->SetResolution(10);
	mSourcePlane->SetResolution(10);
	mSourceSphere->SetResolution(10);

	mSourceDisk->SetWithArrow(false);

	this->SetOpacity(1,0.5f);
 	mActors[1]->SetInput(mSourcePlane->GetOutput());

	this->UpdateMaterialProperties();
}


iVectorFieldViewObject::~iVectorFieldViewObject()
{
	for(int i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++) this->GetViewModule()->GetMarkerFamily()->GetMember(i)->DetachViewObject(this);
	mStreamLineSourcePoints->Delete();
	mStreamLineSourceVerts->Delete();
	mStreamLineSourceNorms->Delete();
	mSourceDisk->Delete();
	mSourcePlane->Delete();
	mSourceSphere->Delete();

	mGlyphPipeline->Delete();
	mStreamLinePipeline->Delete();
}


void iVectorFieldViewObject::FinishInitialization()
{
}


iVectorFieldGlyphParallelPipeline* iVectorFieldViewObject::CreateGlyphParallelPipeline()
{
	iVectorFieldGlyphParallelPipeline *tmp = new iVectorFieldGlyphParallelPipeline(this); IERROR_ASSERT_NULL_POINTER(tmp);
	tmp->FinishInitialization();
	return tmp;
}


iVectorFieldGlyphSerialPipeline* iVectorFieldViewObject::CreateGlyphSerialPipeline()
{
	return new iVectorFieldGlyphSerialPipeline(this);
}


iVectorFieldStreamLineParallelPipeline* iVectorFieldViewObject::CreateStreamLineParallelPipeline()
{
	iVectorFieldStreamLineParallelPipeline *tmp = new iVectorFieldStreamLineParallelPipeline(this); IERROR_ASSERT_NULL_POINTER(tmp);
	tmp->FinishInitialization();
	return tmp;
}


iVectorFieldStreamLineSerialPipeline* iVectorFieldViewObject::CreateStreamLineSerialPipeline()
{
	return new iVectorFieldStreamLineSerialPipeline(this);
}


void iVectorFieldViewObject::ResetPipelineInput(vtkImageData *input)
{
	mGlyphPipeline->SetInput(input);
	mStreamLinePipeline->SetGridInput(input);
}


void iVectorFieldViewObject::ResetPipeline()
{
	mStreamLineSourcePoints->Initialize();
	mStreamLineSourceVerts->Initialize();
	this->UpdatePipelines();
}


void iVectorFieldViewObject::UpdatePipelines()
{
	mGlyphPipeline->UpdateSize();
	mStreamLinePipeline->UpdatePipeline();
}


void iVectorFieldViewObject::UpdateColorByExtra()
{
	if(mColorByOffset > 0)
	{
		mActors[0]->SetScalarVisibility(mColorBy != _VectorColorByNone);
		float curSpacing;
		if(mGlyphPipeline->GetInput() != 0)
		{
			curSpacing = mGlyphPipeline->GetInput()->GetSpacing()[0];
		}
		else
		{
			curSpacing = 1.0;
		}
		switch (mColorBy) 
		{
		case _VectorColorByMagnitude:
			{ 
				mActors[0]->ColorByArrayComponent(0,1); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0);
				break;
			}
		case _VectorColorByVorticity:
			{
				mActors[0]->ColorByArrayComponent(0,2); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0,curSpacing);
				break;
			}
		case _VectorColorByDivergence:
			{
				mActors[0]->ColorByArrayComponent(0,3); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0,curSpacing);
				break;
			}
		}
	}
//	this->UpdatePipelines();
}


void iVectorFieldViewObject::UpdateAfterMarkerMove()
{
	if(mSourceType == _VectorSourceMarker)
	{
		int i;
		vtkIdType l[1];
		bool reset = false;

		if(mStreamLineSourcePoints->GetNumberOfPoints() != this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()+1)
		{
			mStreamLineSourcePoints->Initialize();
			mStreamLineSourceVerts->Initialize();
			mStreamLineSourceNorms->Initialize();
			mStreamLineSourcePoints->SetNumberOfPoints(this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()+1);
			reset = true;
		}

		for(i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++)
		{
			this->GetViewModule()->GetMarkerFamily()->GetMember(i)->AttachViewObject(this);
			mStreamLineSourcePoints->SetPoint(i,this->GetViewModule()->GetMarkerFamily()->GetMember(i)->GetPosition());
			if(reset)
			{
				//
				//  If we are not resetting, then the vert array did not change
				//
				l[0] = i;
				mStreamLineSourceVerts->InsertNextCell(1,l);
			}
		}

		mStreamLinePipeline->GetInput()->Modified();
	}
	else if(mAttachedMarker != 0)
	{
		mSourcePos = mAttachedMarker->GetPosition();
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::UpdateStreamLineSource()
{ 
	switch(mSourceType)
	{
	case _VectorSourceMarker:
		{
			this->UpdateAfterMarkerMove();
			break;
		}
	case _VectorSourceDisk:
	case _VectorSourcePlane:
	case _VectorSourceSphere:
		{
			int i, k;
			vtkIdType n, l;
			double x[3], rx, ry, rf;
			bool reset = false;
			
			n = mNumLines;
			if(n < 1) return;

			mSourceDisk->SetCenter(mSourcePos);
			mSourceDisk->SetNormal(mSourceDir);
			mSourceDisk->SetRadius(mSourceSize);
			mSourcePlane->SetCenter(mSourcePos);
			mSourcePlane->SetNormal(mSourceDir);
			mSourceSphere->SetCenter(mSourcePos);
			mSourceSphere->SetRadius(mSourceSize);

			if(mStreamLineSourcePoints->GetNumberOfPoints() != n)
			{
				mStreamLineSourcePoints->Initialize();
				mStreamLineSourceVerts->Initialize();
				mStreamLineSourceNorms->Initialize();
				mStreamLineSourcePoints->SetNumberOfPoints(n);
				mStreamLineSourceNorms->SetNumberOfComponents(3);
				mStreamLineSourceNorms->SetNumberOfTuples(n);
				reset = true;
			}

			l = 0;
			vtkMath::RandomSeed(123456789);
			//
			//  Find two basis vectors for the plane
			//
			float t, t1[3], t2[3], z[3];
			do
			{
				for(i=0; i<3; i++) t1[i] = 2*vtkMath::Random() - 1;
				vtkMath::Normalize(t1);
				vtkMath::Cross(mSourceDir,t1,t2);
			}
			while(t2[0]*t2[0]+t2[1]*t2[1]+t2[2]*t2[2] < 1.0e-10);
			
			t = vtkMath::Dot(mSourceDir,t1);
			for(i=0; i<3; i++) t1[i] -= t*mSourceDir[i];
			vtkMath::Normalize(t1);
			vtkMath::Normalize(t2);

			k = 0;
			rf = 1.0;
			double sz = mSourceSize;
			double pos[3];
			for(i=0; i<3; i++) pos[i] = mSourcePos[i];

			while(l < n)
			{
				//
				//  Pick up a random point on the source surface
				//
				switch(mSourceType)
				{
				case _VectorSourceDisk:
					{
						//
						//  Random point on the disk
						//
						rx = 2*vtkMath::Random() - 1;
						ry = 2*vtkMath::Random() - 1;
						if(rx*rx+ry*ry > 1.0)
						{
							x[0] = x[1] = x[2] = 2.0;
						}
						else
						{
							for(i=0; i<3; i++) 
							{
								x[i] = pos[i] + rf*sz*(rx*t1[i]+ry*t2[i]);
								z[i] = mSourceDir[i];
							}
						}
						break;
					}
				case _VectorSourcePlane:
					{
						//
						//  Random point on the plane
						//
						rx = 7.0*vtkMath::Random() - 3.5;
						ry = 7.0*vtkMath::Random() - 3.5;
						for(i=0; i<3; i++) 
						{
							x[i] = pos[i] + rf*(rx*t1[i]+ry*t2[i]);
							z[i] = mSourceDir[i];
						}
						break;
					}
				case _VectorSourceSphere:
					{
						//
						//  Random point on the sphere
						//
						z[2] = 2*vtkMath::Random() - 1;
						rx = sqrt(1.0-z[2]*z[2]);
						ry = 6.2831853*vtkMath::Random();
						z[0] = rx*cos(ry);
						z[1] = rx*sin(ry);
						for(i=0; i<3; i++)
						{
							x[i] = pos[i] + sz*z[i];
							if(x[i] < -1.0) x[i] += 2.0; 
							if(x[i] >  1.0) x[i] -= 2.0;
						}
						break;
					}
				}

				k++;

				if(k<100 && (x[0]<-1.0 || x[0]>1.0 || x[1]<-1.0 || x[1]>1.0 || x[2]<-1.0 || x[2]>1.0)) 
				{
					if(k > 20) rf *= 0.5;
					continue;
				}

				if(k >= 100)
				{
#ifdef I_CHECK1
					IERROR_REPORT_BUG;
#endif
					for(i=0; i<3; i++) x[i] = mSourcePos[i];
				}

				k = 0;
				rf = 1.0;
				mStreamLineSourcePoints->SetPoint(l,x); 
				mStreamLineSourceNorms->SetTuple(l,z); 
				if(reset) mStreamLineSourceVerts->InsertNextCell(1,&l); 
				l++;
			}
			break;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
		}
	}

	mStreamLinePipeline->GetInput()->Modified();
}


void iVectorFieldViewObject::SetShowSourceObject(bool s)
{
	if(s) mActors[1]->VisibilityOn(); else mActors[1]->VisibilityOff();
	mSourceShow = s;
	this->ClearCache();
}


void iVectorFieldViewObject::SetSourceType(int v)
{
	if(v>=0 && v<__NumVectorSources && v!=mSourceType && (v<_VectorSourceMarker || (this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()>0 || this->GetViewModule()->GetMarkerFamily()->GetMember(0)->IsVisible())))
	{
		mSourceType = v;
		switch(mSourceType)
		{
		case _VectorSourceDisk:
			{
				mActors[1]->SetInput(mSourceDisk->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		case _VectorSourcePlane:
			{
				mActors[1]->SetInput(mSourcePlane->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		case _VectorSourceSphere:
			{
				mActors[1]->SetInput(mSourceSphere->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		default:
			{
				this->SetShowSourceObject(false);
			}
		}
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetNumberOfStreamLines(int v)
{
	if(v>0 && v!=mNumLines)
	{
		mNumLines = v;
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetSourceOpacity(float v)
{
	if(v>0.0f && !(v>1.0f))
	{
		mOpacities[1] = v;
		mActors[1]->GetProperty()->SetOpacity(v);
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetMethod(int v)
{ 
	if(v>=0 && v<__NumVectorMethods && v!=mMethod) 
	{
		mMethod = v;
		switch(mMethod)
		{
		case _VectorMethodGlyph:
			{
				mColorByOffset = 0;
				mActors[0]->SetInput(mGlyphPipeline->GetOutput());
				break;
			}
		case _VectorMethodStreamLine:
		case _VectorMethodStreamTube:
		case _VectorMethodStreamBand:
			{
				mColorByOffset = 4;
				mActors[0]->SetInput(mStreamLinePipeline->GetOutput());
				break;
			}
		}
		this->UpdatePipelines();
		this->UpdateColorByExtra();
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetLineDir(int v)
{ 
	if(v>=0 && v<__NumStreamLineDirs && v!=mLineDir) 
	{
		mLineDir = v;
		mStreamLinePipeline->UpdateDirection();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::UpdateGlyphSize()
{ 
	mGlyphPipeline->UpdateSize();
}


void iVectorFieldViewObject::SetLineLength(float v)
{ 
	if(v>0.0 && v<100.0 && v!=mLineLength)
	{
		mLineLength = v;
		mStreamLinePipeline->UpdateLength();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::UpdateGlyphSampleRate()
{ 
	mGlyphPipeline->UpdateSampleRate();
}


void iVectorFieldViewObject::SetLineQuality(int v)
{ 
	if(v>=0 && v<10 && v!=mLineQuality) 
	{
		mLineQuality = v;
		mStreamLinePipeline->UpdateQuality();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetTubeSize(int v)
{ 
	if(v>=1 && v!=mTubeSize && v<=10) 
	{
		mTubeSize = v;
		mStreamLinePipeline->UpdateTubeSize();
		mActors[0]->GetProperty()->SetLineWidth(v);
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetTubeRangeFactor(float q)
{ 
	if(q>1.0 && q<100.1 && fabs(q/mTubeRangeFactor-1)>0.1 ) 
	{
		mTubeRangeFactor = q;
		mStreamLinePipeline->UpdateTubeRangeFactor();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetTubeVariationFactor(float q)
{ 
	if(q>1.0e-7 && q<1.0 && fabs(q/mTubeVariationFactor-1)>0.1 ) 
	{
		mTubeVariationFactor = q;
		mStreamLinePipeline->UpdateTubeVariationFactor();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::ShowBody(bool show)
{
	if(show)
	{
		this->UpdatePipelines();
		this->UpdateStreamLineSource();
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mActors[0]->VisibilityOff();
	}
}


float iVectorFieldViewObject::GetMemorySize() const
{
	float s = 0.0;
	s += mStreamLinePipeline->GetMemorySize();
	s += mGlyphPipeline->GetMemorySize();
	s += mStreamLineSourcePoints->GetActualMemorySize();
	s += mStreamLineSourceVerts->GetActualMemorySize();
	return s;
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVectorFieldViewObject::FieldViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyTubeSize(),mTubeSize);
	this->PackValue(s,KeyLineQuality(),mLineQuality);
	this->PackValue(s,KeyLineDir(),mLineDir);
	this->PackValue(s,KeyLineLength(),mLineLength);
	this->PackValue(s,KeyTubeRangeFactor(),mTubeRangeFactor);
	this->PackValue(s,KeyTubeVariationFactor(),mTubeVariationFactor);

	this->PackValue(s,KeyShowSourceObject(),mSourceShow);
	this->PackValue(s,KeySourceOpacity(),mOpacities[1]);
	this->PackValue(s,KeySourceType(),mSourceType);
	this->PackValue(s,KeyNumberOfStreamLines(),mNumLines);
	this->PackValue(s,KeySourceDirection(),mSourceDir,3);
	this->PackValue(s,KeyAttachSourceToMarker(),mAttachedMarkerIndex);  //  for sphere and plane

	this->PackValueDistance(s,KeySourceSize(),mSourceSize);
	this->PackValuePosition(s,KeySourcePosition(),mSourcePos);
}


void iVectorFieldViewObject::FieldViewObjectUnPackStateBody(const iString &s)
{
	int i; bool b; float f, fv[3];

	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	if(this->UnPackValue(s,KeyTubeSize(),i)) this->SetTubeSize(i);
	if(this->UnPackValue(s,KeyLineQuality(),i)) this->SetLineQuality(i);
	if(this->UnPackValue(s,KeyLineDir(),i)) this->SetLineDir(i);
	if(this->UnPackValue(s,KeyLineLength(),f)) this->SetLineLength(f);
	if(this->UnPackValue(s,KeyTubeRangeFactor(),f)) this->SetTubeRangeFactor(f);
	if(this->UnPackValue(s,KeyTubeVariationFactor(),f)) this->SetTubeVariationFactor(f);

	if(this->UnPackValue(s,KeyShowSourceObject(),b)) this->SetShowSourceObject(b);
	if(this->UnPackValue(s,KeySourceOpacity(),f)) this->SetSourceOpacity(f);
	if(this->UnPackValue(s,KeySourceType(),i)) this->SetSourceType(i);
	if(this->UnPackValue(s,KeyNumberOfStreamLines(),i)) this->SetNumberOfStreamLines(i);
	for(i=0; i<3; i++) fv[i] = mSourceDir[i];
	if(this->UnPackValue(s,KeySourceDirection(),fv,3)) this->SetSourceDirection(fv);

	if(this->UnPackValue(s,KeyAttachSourceToMarker(),i)) this->SetAttachSourceToMarker(i);  //  for sphere and plane

	if(this->UnPackValueDistance(s,KeySourceSize(),mSourceSize)) this->SetSourceSize(mSourceSize);
	if(this->UnPackValuePosition(s,KeySourcePosition(),mSourcePos)) this->SetSourcePosition(mSourcePos);
}


void iVectorFieldViewObject::UpdateAfterMarkerAdd(iMarkerViewObject *)
{
	this->UpdateAfterMarkerMove();
}


void iVectorFieldViewObject::UpdateAfterMarkerDelete(iMarkerViewObject *m)
{
	this->UpdateAfterMarkerMove();
	iViewObject::UpdateAfterMarkerDelete(m);
}


int iVectorFieldViewObject::GetDataOffset() const
{
	return 0;
}


void iVectorFieldViewObject::SetSourceSize(const iDistance &s)
{
	if(s < 1.0)
	{
		mSourceSize = s;
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}
	

void iVectorFieldViewObject::DetachFromMarker()
{
	if(mAttachedMarker != 0)
	{
		mAttachedMarker->DetachViewObject(this);
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
}


void iVectorFieldViewObject::SetSourcePosition(const iPosition &pos)
{
	this->DetachFromMarker();

	mSourcePos = pos;
	mSourcePos.CutToBounds();
	this->UpdateStreamLineSource();
	this->ClearCache();
}


void iVectorFieldViewObject::SetSourceDirection(const float n[3])
{
	int i;
	for(i=0; i<3; i++) mSourceDir[i] = n[i];
	this->UpdateStreamLineSource();
	this->ClearCache();
}


void iVectorFieldViewObject::SetAttachSourceToMarker(int m)
{
	int i;

	this->DetachFromMarker();

	if(m>=0 && m<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex())
	{
		mAttachedMarkerIndex = m;
		mAttachedMarker = this->GetViewModule()->GetMarkerFamily()->GetMember(mAttachedMarkerIndex);
		mAttachedMarker->AttachViewObject(this);

		for(i=0; i<3; i++) mSourcePos[i] = mAttachedMarker->GetPosition()[i];
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewObject::SetSourcePositionToPickedPoint()
{
	int i;
	double *x = this->GetViewModule()->GetPicker()->GetPickPosition();

	this->DetachFromMarker();

	for(i=0; i<3; i++) mSourcePos[i] = x[i];
	this->UpdateStreamLineSource();
	this->ClearCache();
}


void iVectorFieldViewObject::FieldViewObjectSyncWithData(int, void *)
{
	this->UpdatePipelines();
}

