/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "isolidviewobject.h"


#include "ibuffer.h"
#include "icontrolmodule.h"
#include "iclipplane.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ilookuptable.h"
#include "ipalette.h"
#include "ireplicatedactor.h"
#include "itune.h"
#include "iviewmodule.h"
#include "iviewobjectobserver.h"
#include "iviewobjectreplicator.h"

#include <vtkLookupTable.h>
#include <vtkProperty.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


#include "iparameter.h"
using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iSolidViewObject,SolidViewObject,-svo);

//
//  Inherited keys
//
IOBJECT_DEFINE_KEY(iSolidViewObject,Color,c,Color,0);
IOBJECT_DEFINE_KEY(iSolidViewObject,Opacity,o,Float,0);
IOBJECT_DEFINE_KEY(iSolidViewObject,Palette,p,OffsetInt,0);


//
// Main class
//
iSolidViewObject::iSolidViewObject(iViewModule *vm, const iDataInfo &info, int numActors, unsigned int flags) : iViewObject(vm,info,flags) , mActors(0), mOpacities(0.0), mPalettes(0), mColors(iColor()), mNoColor((flags&_ViewObjectNoColor)!=0U), mNoOpacity((flags&_ViewObjectNoOpacity)!=0U), mNoPalette((flags&_ViewObjectNoPalette)!=0U)
{
	int i;

	mName = "iSolidViewObject";

	if(numActors < 1)
	{
		IERROR_REPORT_FATAL_ERROR("SolidViewObject must have at least 1 Actor.");
	}

	iAbortRenderEventObserver *obsAbortRender = this->GetViewModule()->GetAbortRenderEventObserver();

	iActor *a;
	for(i=0; i<numActors; i++)
	{
		mOpacities.Add(1.0f);
		mPalettes.Add(0);
		mColors.Add(mDefaultColor);
		if(mNoReplicating)
		{
			a = iActor::New(); IERROR_ASSERT_NULL_POINTER(a);
		}
		else
		{
			a = iReplicatedActor::New(mReplicator); IERROR_ASSERT_NULL_POINTER(a);
		}
		a->SetScalarVisibility(false);
		a->SetPalette(this->GetViewModule()->GetControlModule()->GetPalette(0));
		a->ColorByArrayComponent(0,0); 
		a->SetPosition(0.0,0.0,0.0);
		a->VisibilityOff();
		a->GetProperty()->SetOpacity(1.0);
		a->GetProperty()->SetInterpolationToGouraud();
		a->GetProperty()->SetColor(mDefaultColor.ToVTK());
		a->AddMapperObserver(vtkCommand::ProgressEvent,obsAbortRender);
		this->GetViewModule()->AddObject(a);
		mActors.Add(a);
	}

	this->UpdateMaterialProperties();

	//
	//  Add observer to keep information about this object
	//
	mActors[0]->AddObserver(vtkCommand::UserEvent,mObjectObserver);
}


iSolidViewObject::~iSolidViewObject()
{
	int i;
	for(i=0; i<mActors.Size(); i++)
	{
		this->GetViewModule()->RemoveObject(mActors[i]);
		mActors[i]->Delete();
	}
}


void iSolidViewObject::UpdateLookupTables()
{
	int i;

	for(i=0; i<mActors.Size(); i++)
	{
		mActors[i]->GetLookupTable()->Modified();
	}
}


void iSolidViewObject::Reset()
{
	int i;
	this->ResetInput();
	this->SyncWithData(0,0);
	this->ResetPipeline();
	//
	//  Un-initialized, if needed
	//
	if(!mInit)
	{
		this->ShowColorBars(false);
		for(i=0; i<mActors.Size(); i++) mActors[i]->VisibilityOff();
		mInit = true;
	}
}


void iSolidViewObject::ShowClipPlane(bool s)
{
	int i;

	if(s && !mClipPlaneOn)
	{
		for(i=0; i<mActors.Size(); i++) mActors[i]->AddClippingPlane(this->GetViewModule()->GetClipPlane());
		mClipPlaneOn = true;
	} 
	if(!s && mClipPlaneOn) 
	{
		for(i=0; i<mActors.Size(); i++) mActors[i]->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
		mClipPlaneOn = false;
	}
	this->ClearCache();
}


void iSolidViewObject::ShowColorBars(bool show)
{
	if(mNoPalette || !this->IsVisible()) return;
	this->ShowColorBarsBody(show);
	this->ClearCache();
}


void iSolidViewObject::SetOpacity(int l, float o)
{ 
	if(mNoOpacity || o<0.0f || o>1.0f) return;

	if(l>=0 && l<mActors.Size())
	{
		mActors[l]->GetProperty()->SetOpacity(o);
		mOpacities[l] = o;
		this->ClearCache();
	}
}


void iSolidViewObject::SetColor(int l, const iColor &c)
{
	if(mNoColor) return;

	if(l>=0 && l<mActors.Size())
	{
		double v[3];
		iTune::CopyDoublePointers(v,c.ToVTK(),3);
		mActors[l]->GetProperty()->SetColor(v);
		mColors[l] = c;
		this->ClearCache();
	}
}


void iSolidViewObject::SetPalette(int l, int p)
{
	if(!mNoPalette && l>=0 && l<mActors.Size() && p>=0 && p<this->GetViewModule()->GetControlModule()->GetNumberOfPalettes())
	{
		this->ShowColorBars(false);
		mPalettes[l] = p;
		this->ShowColorBars(true);
		mActors[l]->SetPalette(this->GetViewModule()->GetControlModule()->GetPalette(p));
		this->ClearCache();
	}
}


void iSolidViewObject::UpdateMaterialProperties()
{
	int i;
	vtkProperty *prop;

	if(mShading)
	{
		for(i=0; i<mActors.Size(); i++)
		{
			prop = mActors[i]->GetProperty();
			prop->SetAmbient(mAmbient);
			prop->SetDiffuse(mDiffuse);
			prop->SetSpecular(mSpecular);
			prop->SetSpecularPower(mSpecularPower);
		}
	}
	else
	{
		for(i=0; i<mActors.Size(); i++)
		{
			prop = mActors[i]->GetProperty();
			prop->SetAmbient(1.0);
			prop->SetDiffuse(0.0);
			prop->SetSpecular(0.0);
			prop->SetSpecularPower(0.0);
		}
	}
}


bool iSolidViewObject::IsVisible() const
{
	return (mActors[0]->GetVisibility() != 0);
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iSolidViewObject::ViewObjectPackStateBody(iString &s) const
{
	if(!mNoColor) this->PackValue(s,KeyColor(),mColors.Data(),mColors.Size());
	if(!mNoOpacity) this->PackValue(s,KeyOpacity(),mOpacities.Data(),mOpacities.Size());
	if(!mNoPalette) this->PackValue(s,KeyPalette(),mPalettes.Data(),mPalettes.Size());

	this->SolidViewObjectPackStateBody(s);
}


void iSolidViewObject::ViewObjectUnPackStateBody(const iString &s)
{
	static iBuffer<int> ibuf;
	static iBuffer<float> fbuf;
	static iBuffer<iColor> cbuf;
	int i;

	if(!mNoColor)
	{
		cbuf.Fill(mColors.Data(),mColors.Size());
		if(this->UnPackValue(s,KeyColor(),cbuf,mColors.Size())) for(i=0; i<mColors.Size(); i++) this->SetColor(i,cbuf[i]);
	}

	if(!mNoOpacity)
	{
		fbuf.Fill(mOpacities.Data(),mOpacities.Size());
		if(this->UnPackValue(s,KeyOpacity(),fbuf,mOpacities.Size())) for(i=0; i<mOpacities.Size(); i++) this->SetOpacity(i,fbuf[i]);
	}

	if(!mNoPalette)
	{
		ibuf.Fill(mPalettes.Data(),mPalettes.Size());
		if(this->UnPackValue(s,KeyPalette(),ibuf,mPalettes.Size())) for(i=0; i<mPalettes.Size(); i++) this->SetPalette(i,ibuf[i]);
	}

	this->SolidViewObjectUnPackStateBody(s);
}

