/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticlesviewobject.h"


#include "ibuffer.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "ifunctionmapping.h"
#include "ihistogrammaker.h"
#include "iparticleconnector.h"
#include "iparticlegroup.h"
#include "iparticlesplitter.h"
#include "irangecollection.h"
#include "irangemapping.h"
#include "itune.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkCellArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"
#include "ifamilytemplate.h"
#include "iviewobjectfamilytemplate.h"


const int iParticlesViewObject::CurrentId = iMath::_LargeInt;
bool iParticlesViewObject::mUseFullState = true;


using namespace iParameter;


class iParticleGroupFamily : public iViewObjectFamily<iParticleGroup>
{
	
	friend class iParticlesViewObject;

protected:
		
	iParticleGroupFamily(iParticlesViewObject *p) : iViewObjectFamily<iParticleGroup>(p->GetViewModule())
	{
		mParent = p;
		mMemlist[0]->mParent = mParent;
		mMemlist[0]->mParentIndex = 0;
	}

	virtual int CreateMember()
	{
		int ret = iViewObjectFamily<iParticleGroup>::CreateMember();
		if(ret >= 0)
		{
			mMemlist[ret]->mParent = mParent;
			mMemlist[ret]->mParentIndex = ret;
		}
		return ret;
	}

	virtual bool DeleteMember(int i)
	{
		bool ret = iViewObjectFamily<iParticleGroup>::DeleteMember(i);
		if(ret)
		{
			int j;
			for(j=i; j<mMemlist.Size(); j++) mMemlist[j]->mParentIndex = j;
		}
		return ret;
	}

private:

	iParticlesViewObject *mParent;
};


//
//  Main class
//

IVIEWOBJECT_DEFINE_TYPE(iParticlesViewObject,Particles,p);

IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeToSplit,a,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeToColor,ac,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeSizeDirect,ad,Bool,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeSizeExtraFactor,ae,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeToSize,as,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeToConnect,atc,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,AttributeToSeparate,ats,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,Color,c,Color,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,CurrentGroup,cg,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,StretchToColor,stc,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,LowerLimitToColor,ltc,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,UpperLimitToColor,utc,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,StretchToSize,sts,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,LowerLimitToSize,lts,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,UpperLimitToSize,uts,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,LineWidth,lw,Int,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,MaxGroup,mg,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,Opacity,o,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,Palette,p,OffsetInt,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangeLowerLimit,rl,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangesMax,rmax,Float,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangesMin,rmin,Float,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangesStretch,rs,Int,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangesTiled,rt,Bool,1);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRangeUpperLimit,ru,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,FixedSize,s,Float,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,SizeFunction,sf,Any,0);
IOBJECT_DEFINE_KEY(iParticlesViewObject,Type,t,Int,0);

IOBJECT_DEFINE_KEY(iParticlesViewObject,SplitRanges,-r,Any,1);

//
//  Inherited keys (define them explicitly since they are now multi-component
//
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,ActiveDataType,dt,Int,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Ambient,ma,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Diffuse,md,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Specular,ms,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,SpecularPower,mp,Float,0);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Visible,vis,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,DataPresent,dp,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Ready,r,Bool,1);
IOBJECT_DEFINE_INHERITED_KEY(iViewObject,iParticlesViewObject,Shading,mh,Bool,0);

IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iParticlesViewObject);


iParticlesViewObject* iParticlesViewObject::New(iViewModule *vm)
{
	iParticlesViewObject *tmp = iPointerCast<iParticlesViewObject,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	//
	//  Attach the first group
	//
	tmp->mGroups->GetMember(0)->SetInput(tmp->mSplitter->GetOutput());

	tmp->Reset();

	return tmp;
}


iParticlesViewObject::iParticlesViewObject(iViewModule *vm, const iDataInfo &info) : iViewObject(vm,info,0U)
{
	mName = "iParticlesViewObject";
	mObjectType = _ObjectTypeParticles;

	vtkPolyData *input = dynamic_cast<vtkPolyData*>(this->GetData()); // accepts any PolyData
	if(input == 0)
	{
		IERROR_REPORT_FATAL_ERROR("iParticlesViewObject is configured incorrectly.");
	}

	mGroups = new iParticleGroupFamily(this); IERROR_ASSERT_NULL_POINTER(mGroups);
	mSplitter = iParticleSplitter::New(this); IERROR_ASSERT_NULL_POINTER(mSplitter);
}


iParticlesViewObject::~iParticlesViewObject()
{
	delete mGroups;
	mSplitter->Delete();
}


void iParticlesViewObject::FinishInitialization()
{
}


void iParticlesViewObject::BecomeClone(iParticlesViewObject *v)
{
	mSplitter->TakeOverData(v != 0);
	this->ClearCache();
}


void iParticlesViewObject::UpdateReplicas()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) 
	{
		mGroups->GetMember(i)->ReplicateAs(this);
	}
}


void iParticlesViewObject::ShowBody(bool s)
{
	mGroups->Show(s);
}


void iParticlesViewObject::Reset()
{
	mGroups->Reset();
	mSplitter->SetInput(iPointerCast<vtkPolyData,vtkDataSet>(this->GetData()));
	this->SyncWithData(0,0);
	this->ClearCache();
}


void iParticlesViewObject::ShowClipPlane(bool s)
{
	int i;
	mClipPlaneOn = s;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->ShowClipPlane(s);
	this->ClearCache();
}


void iParticlesViewObject::ShowColorBars(bool s)
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->ShowColorBars(s);
	this->ClearCache();
}


void iParticlesViewObject::PackStateBody(iString &s) const
{
	static iBuffer<float> fv1, fv2;
	int i, n = mGroups->GetMaxMemberIndex() + 1;
	float f1, f2;

	this->PackValue(s,KeyMaxGroup(),mGroups->GetMaxMemberIndex()); // arrays start with 1
	this->PackValue(s,KeyCurrentGroup(),mGroups->GetCurrentMemberIndex()); // arrays start with 1

	this->PackValue(s,KeyActiveDataType(),mActiveDataTypeIndex);
	this->PackValue(s,KeyClipPlane(),mClipPlaneOn);
	this->PackValue(s,KeyNumReplicas(),mNumReplicas,6);
	this->PackValue(s,KeyVisible(),this->IsVisible());
	this->PackValue(s,KeyDataPresent(),this->IsDataPresent());
	this->PackValue(s,KeyReady(),this->IsReady());

	this->PackValue(s,KeyAttributeToSplit(),mSplitter->GetAttributeToSplit());
	this->PackValue(s,KeySplitRangesStretch(),mSplitter->GetSplitRangesStretch());
	this->PackValue(s,KeySplitRangesTiled(),mSplitter->GetSplitRangesTiled());

	mSplitter->GetSplitRangesMinMax(f1,f2);
	this->PackValue(s,KeySplitRangesMin(),f1);
	this->PackValue(s,KeySplitRangesMax(),f2);

	for(i=0; i<n; i++)
	{
		mSplitter->GetSplitRangeLimits(i,fv1[i],fv2[i]);
	}
	this->PackValue(s,KeySplitRangeLowerLimit(),fv1,n);
	this->PackValue(s,KeySplitRangeUpperLimit(),fv2,n);

	if(mUseFullState) this->PackGroupStateBody(s,0,mGroups->GetMaxMemberIndex());
}


void iParticlesViewObject::UnPackStateBody(const iString &s0)
{
	static iString rep1 = iObjectKey::LeftBracket() + iObjectKey::RightBracket();
	static iString rep2 = iObjectKey::LeftBracket() + "0" + iObjectKey::RightBracket();
	static iString rep3 = iObjectKey::LeftBracket() + iString::FromNumber(1+CurrentId) + iObjectKey::RightBracket();
	static iBuffer<float> fv1, fv2;
	int i, n, v, d[6];
	bool b;

	//
	//  Unpack splitter attribute first, so that groups are created correctly
	//
	if(this->UnPackValue(s0,KeyAttributeToSplit(),i)) this->SetAttributeToSplit(i);
	//
	//  Make the size of the group family consistent
	//
	if(this->UnPackValue(s0,KeyMaxGroup(),v))
	{
		//
		//  Delete extra groups
		//
		while(v < mGroups->GetMaxMemberIndex()) this->DeleteGroup(mGroups->GetMaxMemberIndex());
		//
		//  Create missing groups
		//
		while(v > mGroups->GetMaxMemberIndex()) this->CreateGroup();
		this->ClearCache();
	}

	n = mGroups->GetMaxMemberIndex() + 1;

	if(this->UnPackValue(s0,KeyCurrentGroup(),v))
	{
		mGroups->SetCurrentMemberIndex(v); // arrays start with 1
		this->ClearCache();
	}

	//
	//  Special syntax: replace [] or [0] with [<curindex>].
	//
	iString sub, s = s0;
	sub = iObjectKey::LeftBracket() + iString::FromNumber(1+mGroups->GetCurrentMemberIndex()) + iObjectKey::RightBracket(); // arrays start with 1

	s.Replace(rep1,sub);
	s.Replace(rep2,sub);
	s.Replace(rep3,sub);

	if(this->UnPackValue(s,KeyActiveDataType(),i)) this->SetActiveDataTypeIndex(i);
	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);

	for(i=0; i<6; i++) d[i] = mNumReplicas[i];
	if(this->UnPackValue(s,KeyNumReplicas(),d,6)) this->SetNumReplicas(d);

	if(this->UnPackValue(s,KeySplitRangesStretch(),i))
	{
		mSplitter->SetSplitRangesStretch(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeySplitRangesTiled(),b))
	{
		mSplitter->SetSplitRangesTiled(b);
		this->ClearCache();
	}

	float f1, f2;
	if(this->UnPackValue(s,KeySplitRangesMin(),f1) && this->UnPackValue(s,KeySplitRangesMax(),f2))
	{
		mSplitter->SetSplitRangesMinMax(f1,f2);
		this->ClearCache();
	}

	for(i=0; i<n; i++)
	{
		mSplitter->GetSplitRangeLimits(i,fv1[i],fv2[i]);
	}
	if(this->UnPackValue(s,KeySplitRangeLowerLimit(),fv1,n) && this->UnPackValue(s,KeySplitRangeUpperLimit(),fv2,n)) for(i=0; i<n; i++)
	{
		mSplitter->SetSplitRangeLimits(i,fv1[i],fv2[i]);
		this->ClearCache();
	}
	
	if(mUseFullState) this->UnPackGroupStateBody(s,0,mGroups->GetMaxMemberIndex());
}


void iParticlesViewObject::PackCompleteState(iString &s) const
{
	bool u = mUseFullState;

	mUseFullState = true;
	mCacheInvalid = true;
	this->PackState(s);
	mUseFullState = u;
}


void iParticlesViewObject::UnPackCompleteState(const iString &s)
{
	bool u = mUseFullState;

	mUseFullState = true;
	this->UnPackState(s);
	mUseFullState = u;
}


void iParticlesViewObject::PackGroupStateBody(iString &s, int first, int last) const
{
	static iBuffer<int> iv;
	static iBuffer<bool> bv;
	static iBuffer<float> fv;
	static iBuffer<double> dv;
	static iBuffer<iColor> cv;
	static iBuffer<const iPiecewiseFunction*> pv;
	int i, n;
	
	if(last > mGroups->GetMaxMemberIndex()) last = mGroups->GetMaxMemberIndex();
	n = last - first + 1;
	if(n < 1) return;

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetType();
	this->PackValue(s,KeyType(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetPalette(0);
	this->PackValue(s,KeyPalette(),iv+first,n);

	for(i=first; i<=last; i++) cv[i] = mGroups->GetMember(i)->GetColor(0);
	this->PackValue(s,KeyColor(),cv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToColor();
	this->PackValue(s,KeyAttributeToColor(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSize();
	this->PackValue(s,KeyAttributeToSize(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetOpacity(0);
	this->PackValue(s,KeyOpacity(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetFixedSize();
	this->PackValue(s,KeyFixedSize(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToColor();
	this->PackValue(s,KeyStretchToColor(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToColor();
	this->PackValue(s,KeyLowerLimitToColor(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToColor();
	this->PackValue(s,KeyUpperLimitToColor(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToSize();
	this->PackValue(s,KeyStretchToSize(),iv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToSize();
	this->PackValue(s,KeyLowerLimitToSize(),fv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToSize();
	this->PackValue(s,KeyUpperLimitToSize(),fv+first,n);

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAttributeSizeDirect();
	this->PackValue(s,KeyAttributeSizeDirect(),bv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAttributeSizeExtraFactor();
	this->PackValue(s,KeyAttributeSizeExtraFactor(),fv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetLineWidth();
	this->PackValue(s,KeyLineWidth(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToConnect();
	this->PackValue(s,KeyAttributeToConnect(),iv+first,n);

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSeparate();
	this->PackValue(s,KeyAttributeToSeparate(),iv+first,n);

	for(i=first; i<=last; i++) pv[i] = mGroups->GetMember(i)->GetSizeFunction()->GetFunction();
	this->PackValuePiecewiseFunction(s,KeySizeFunction(),pv+first,n);

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetShading();
	this->PackValue(s,KeyShading(),bv+first,n);

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAmbient();
	this->PackValue(s,KeyAmbient(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetDiffuse();
	this->PackValue(s,KeyDiffuse(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecular();
	this->PackValue(s,KeySpecular(),fv+first,n);
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecularPower();
	this->PackValue(s,KeySpecularPower(),fv+first,n);
}


void iParticlesViewObject::UnPackGroupStateBody(const iString &s, int first, int last)
{
	static iBuffer<int> iv;
	static iBuffer<bool> bv;
	static iBuffer<float> fv;
	static iBuffer<double> dv;
	static iBuffer<iColor> cv;
	static iBuffer<iPiecewiseFunction*> pv;
	int i, n;

	if(last > mGroups->GetMaxMemberIndex()) last = mGroups->GetMaxMemberIndex();
	n = last - first + 1;
	if(n < 1) return;

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetType();
	if(this->UnPackValue(s,KeyType(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetType(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetPalette(0);
	if(this->UnPackValue(s,KeyPalette(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetPalette(0,iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) cv[i] = mGroups->GetMember(i)->GetColor(0);
	if(this->UnPackValue(s,KeyColor(),cv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetColor(0,cv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToColor();
	if(this->UnPackValue(s,KeyAttributeToColor(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToColor(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSize();
	if(this->UnPackValue(s,KeyAttributeToSize(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToSize(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetOpacity(0);
	if(this->UnPackValue(s,KeyOpacity(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetOpacity(0,fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetFixedSize();
	if(this->UnPackValue(s,KeyFixedSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetFixedSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToColor();
	if(this->UnPackValue(s,KeyStretchToColor(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetStretchToColor(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToColor();
	if(this->UnPackValue(s,KeyLowerLimitToColor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLowerLimitToColor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToColor();
	if(this->UnPackValue(s,KeyUpperLimitToColor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetUpperLimitToColor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetStretchToSize();
	if(this->UnPackValue(s,KeyStretchToSize(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetStretchToSize(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetLowerLimitToSize();
	if(this->UnPackValue(s,KeyLowerLimitToSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLowerLimitToSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetUpperLimitToSize();
	if(this->UnPackValue(s,KeyUpperLimitToSize(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetUpperLimitToSize(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetAttributeSizeDirect();
	if(this->UnPackValue(s,KeyAttributeSizeDirect(),bv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeSizeDirect(bv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAttributeSizeExtraFactor();
	if(this->UnPackValue(s,KeyAttributeSizeExtraFactor(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeSizeExtraFactor(fv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetLineWidth();
	if(this->UnPackValue(s,KeyLineWidth(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetLineWidth(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToConnect();
	if(this->UnPackValue(s,KeyAttributeToConnect(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToConnect(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) iv[i] = mGroups->GetMember(i)->GetAttributeToSeparate();
	if(this->UnPackValue(s,KeyAttributeToSeparate(),iv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAttributeToSeparate(iv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) pv[i] = mGroups->GetMember(i)->GetSizeFunction()->GetFunction();
	if(this->UnPackValuePiecewiseFunction(s,KeySizeFunction(),pv+first,n)) this->ClearCache(); // this unpacks into the appropriate function automatically.

	for(i=first; i<=last; i++) bv[i] = mGroups->GetMember(i)->GetShading();
	if(this->UnPackValue(s,KeyShading(),bv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetShading(bv[i]);
		this->ClearCache();
	}

	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetAmbient();
	if(this->UnPackValue(s,KeyAmbient(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetAmbient(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetDiffuse();
	if(this->UnPackValue(s,KeyDiffuse(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetDiffuse(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecular();
	if(this->UnPackValue(s,KeySpecular(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetSpecular(fv[i]);
		this->ClearCache();
	}
	
	for(i=first; i<=last; i++) fv[i] = mGroups->GetMember(i)->GetSpecularPower();
	if(this->UnPackValue(s,KeySpecularPower(),fv+first,n))
	{
		for(i=first; i<=last; i++) mGroups->GetMember(i)->SetSpecularPower(fv[i]);
		this->ClearCache();
	}
}


float iParticlesViewObject::GetMemorySize() const
{
	return mGroups->GetMemorySize() + mSplitter->GetMemorySize();
}


bool iParticlesViewObject::IsVisible() const
{
	return mGroups->IsVisible();
}


int iParticlesViewObject::GetMaxGroupIndex() const
{
	return mGroups->GetMaxMemberIndex();
}


int iParticlesViewObject::GetCurrentGroupIndex() const
{
	return mGroups->GetCurrentMemberIndex();
}


iParticleGroup* iParticlesViewObject::GetCurrentGroup() const
{
	return mGroups->GetCurrentMember();
}


iParticleGroup* iParticlesViewObject::GetGroup(int i) const
{
	return mGroups->GetMember(i);
}


iRangeMapping* iParticlesViewObject::GetSplitRanges()
{
	return mSplitter->GetSplitRanges();
}


void iParticlesViewObject::SetCurrentGroupIndex(int i)
{
	mGroups->SetCurrentMemberIndex(i);
}


int iParticlesViewObject::CreateGroup()
{ 
	if(!mSplitter->CreateGroup()) return -1;

	vtkPolyData *newOutput = mSplitter->GetOutput(mSplitter->GetNumberOfOutputs()-1);
	//
	//  Create a new piece
	//
	int oldGroup = mGroups->GetCurrentMemberIndex();
	int newGroup = mGroups->CreateMember();
	if(newGroup == -1)
	{
		mSplitter->DeleteGroup(mSplitter->GetNumberOfOutputs());
		return -1;
	}

	//
	//  Set the correct piece number
	//
	mGroups->GetMember(newGroup)->CopyState(mGroups->GetMember(oldGroup));
	mGroups->GetMember(newGroup)->SetInput(newOutput);

	mGroups->GetMember(newGroup)->Show(mGroups->GetMember(oldGroup)->IsVisible());
	mGroups->SetCurrentMemberIndex(newGroup);
	
	this->ClearCache();

	return newGroup;
}


bool iParticlesViewObject::DeleteGroup(int n)
{
	if(mSplitter->DeleteGroup(n))
	{
		this->ClearCache();
		return mGroups->DeleteMember(n);
	}
	else return false;
}


void iParticlesViewObject::UpdateColorLookupTable()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->UpdateColorLookupTable();
}


void iParticlesViewObject::UpdateMaterialProperties()
{
	int i;
	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) mGroups->GetMember(i)->UpdateMaterialProperties();
}


void iParticlesViewObject::SetAttributeToSplit(int a)
{
	int oldAtt = mSplitter->GetAttributeToSplit();
	mSplitter->SetAttributeToSplit(a);
	if(oldAtt != mSplitter->GetAttributeToSplit())
	{
		while(mGroups->GetMaxMemberIndex() > 0) this->DeleteGroup(mGroups->GetMaxMemberIndex());
		mGroups->GetMember(0)->UpdateLookupTables();
	}
	this->ClearCache();
}


int iParticlesViewObject::GetAttributeToSplit() const
{
	return mSplitter->GetAttributeToSplit();
}


void iParticlesViewObject::ViewObjectPackStateBody(iString &) const
{
#ifdef I_CHECK1
	IERROR_REPORT_ERROR("This function should never be called.");
#endif
}


void iParticlesViewObject::ViewObjectUnPackStateBody(const iString &)
{
#ifdef I_CHECK1
	IERROR_REPORT_ERROR("This function should never be called.");
#endif
}


bool iParticlesViewObject::CanBeShown() const
{
	return true;
}


void iParticlesViewObject::ViewObjectSyncWithData(int info, void *data)
{
	int i;

	for(i=0; i<=mGroups->GetMaxMemberIndex(); i++) 
	{
		mGroups->GetMember(i)->ViewObjectSyncWithData(info,data);
	}
}

