/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipalette.h"


#include "ierror.h"
#include "iimage.h"
#include "imath.h"
#include "ipiecewisefunction.h"

#include <vtkColorTransferFunction.h> 
#include <vtkLookupTable.h>

//
//  Templates
//
#include "iarraytemplate.h"


//
// Implementation of iPalette class
//
iPalette::iPalette()
{
	mName = "empty";

	mCTF = vtkColorTransferFunction::New(); IERROR_ASSERT_NULL_POINTER(mCTF);
	mLT = vtkLookupTable::New(); IERROR_ASSERT_NULL_POINTER(mLT);
	
	mFunR = iPiecewiseFunction::New(); IERROR_ASSERT_NULL_POINTER(mFunR);
	mFunG = iPiecewiseFunction::New(); IERROR_ASSERT_NULL_POINTER(mFunG);
	mFunB = iPiecewiseFunction::New(); IERROR_ASSERT_NULL_POINTER(mFunB);

	mCTF->SetColorSpaceToRGB();
	mLT->SetNumberOfTableValues(256);

	//
	//  Image representation
	//
	mImage = new iImage(3); IERROR_ASSERT_NULL_POINTER(mImage);
	mImageNeedsUpdate = true;
	mImageWidth = 256;
	mImageHeight = 32;
}


iPalette::~iPalette()
{
	delete mImage;

	mCTF->Delete(); 
	mLT->Delete(); 

	mFunR->Delete();
	mFunG->Delete();
	mFunB->Delete();
}


void iPalette::SetComponents(const iPiecewiseFunction *r, const iPiecewiseFunction *g, const iPiecewiseFunction *b)
{
	if(r != 0) mFunR->Copy(r);
	if(g != 0) mFunG->Copy(g);
	if(b != 0) mFunB->Copy(b);

	this->Update();
}

	
iColor iPalette::GetColor(int n) const
{
	n = ( n < 0 ) ? 0 : n;
	n = ( n > 255 ) ? 255 : n;
	float x = (float)n/255.0;
	float r = mFunR->GetValue(x);
	float g = mFunG->GetValue(x);
	float b = mFunB->GetValue(x);
	return iColor(round(255.0*r),round(255.0*g),round(255.0*b));
}


void iPalette::Update()
{
	int i, ir, ig, ib;
	float x;

	//
	//  Update vtkColorTransferFunction
	//
	mCTF->RemoveAllPoints();
	ir = ig = ib = 0;
	while(ir<mFunR->N() && ig<mFunG->N() && ib<mFunB->N())
	{
		x = mFunR->X(ir);
		if(x > mFunG->X(ig)) x = mFunG->X(ig);
		if(x > mFunB->X(ib)) x = mFunB->X(ib);
		
		mCTF->AddRGBPoint(255.0*x,mFunR->GetValue(x),mFunG->GetValue(x),mFunB->GetValue(x));
		
		if(fabs(x-mFunR->X(ir)) < 1.0e-4) ir++;
		if(fabs(x-mFunG->X(ig)) < 1.0e-4) ig++;
		if(fabs(x-mFunB->X(ib)) < 1.0e-4) ib++;
	}

	//
	//  Update vtkLookupTable
	//
	for(i=0; i<256; i++)
	{
		x = (float)i/255.0;
		mLT->SetTableValue(i,mFunR->GetValue(x),mFunG->GetValue(x),mFunB->GetValue(x));
	}

	mImageNeedsUpdate = true;
}


void iPalette::Copy(iPalette *p)
{
	mFunR->Copy(p->mFunR);
	mFunG->Copy(p->mFunG);
	mFunB->Copy(p->mFunB);
	mName = p->mName;

	this->Update();
}


void iPalette::Reverse()
{
	int i, j; 
	iPiecewiseFunction *f, *fn;
	
	fn = iPiecewiseFunction::New(); IERROR_ASSERT_NULL_POINTER(fn);
	if(fn == 0) return;

	for(j=0; j<3; j++)
	{
		switch(j)
		{
		case 0: { f = mFunR; break; }
		case 1: { f = mFunG; break; }
		default: f = mFunB; 
		}

		fn->Reset();
		fn->MovePoint(0,0.0,f->Y(f->N()-1));
		fn->MovePoint(1,1.0,f->Y(0));
		for(i=1; i<f->N()-1; i++)
		{
			fn->AddPoint(1.0-f->X(i),f->Y(i));
		}

		f->Copy(fn);
	}
	fn->Delete();

	this->Update();
}


const iImage* iPalette::GetImage()
{
	if(mImageNeedsUpdate)
	{
		mImageNeedsUpdate = false;
		//
		//  Update the image
		//
		mImage->Scale(mImageWidth,mImageHeight);
		unsigned char *dPtr = mImage->DataPointer();

		int i;
		float x;
		unsigned char r, g, b;

		//
		//  Fill first row
		//
		for(i=0; i<mImageWidth; i++)
		{
			x = (float)i/(mImageWidth-1);
			r = char(round(255.0*mFunR->GetValue(x)));
			g = char(round(255.0*mFunG->GetValue(x)));
			b = char(round(255.0*mFunB->GetValue(x)));
			dPtr[3*i+0] = r;
			dPtr[3*i+1] = g;
			dPtr[3*i+2] = b;
		}

		//
		//  Copy to the rest of rows
		//
		for(i=1; i<mImageHeight; i++) memcpy(dPtr+3*i*mImageWidth,dPtr,3*mImageWidth);
	}
	return mImage;
}
