/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ioutputwindow.h"

#include "ierror.h"

#include <vtkCriticalSection.h>

//
//  Templates
//
#include "iarraytemplate.h"


iOutputWindow* iOutputWindow::New(iShell *s)
{
	iOutputWindow *tmp = new iOutputWindow(s);
	//
	//  Take over VTK output window
	//
	vtkOutputWindow::SetInstance(tmp);
	tmp->Delete();
	return tmp;
}


iOutputWindow::iOutputWindow(iShell *s) : iShellComponent(s), mTextQueue(iString()), mErrorQueue(iString())
{
	mErrorMessageFirstSymbol = 0;
	mErrorMessageFirstString = 0;

	mQueue = 0;

	mMutex = vtkCriticalSection::New(); IERROR_ASSERT_NULL_POINTER(mMutex);
}


iOutputWindow::~iOutputWindow()
{
	mMutex->Delete();
}


void iOutputWindow::DisplayErrorText(const char* message)
{
	const char* tmp = FormattedErrorMessage(message);
	if(tmp == 0) return;

	if(mQueue > 0)
	{
		mMutex->Lock();
		mErrorQueue.Add(iString(tmp));
		mMutex->Unlock();
	}
	else iConsole::Instance()->DisplayErrorMessage(tmp,0,0);
}


void iOutputWindow::DisplayText(const char* message)
{
	if(mQueue > 0)
	{
		mMutex->Lock();
		mTextQueue.Add(iString(message));
		mMutex->Unlock();
	}
	else iConsole::Instance()->DisplayTextMessage(message);
}


void iOutputWindow::DisplayWarningText(const char* message)
{
	this->DisplayText(FormattedWarningMessage(message));
}


void iOutputWindow::DisplayGenericWarningText(const char* message)
{
	this->DisplayText(FormattedWarningMessage(message));
}


void iOutputWindow::DisplayDebugText(const char* message)
{
#ifdef I_DEBUG
	this->DisplayText(message);
#endif
}


const char* iOutputWindow::FormattedErrorMessage(const char* message)
{
	//
	//  Block some of the VTK error messages - not all are useful
	//
	const int nBlockError = 3;
	const char *classBlockError[] = { 
		"vtkDecimate",
		"vtkPolyDataNormals",
		"vtkWindowedSincPolyDataFilter"
	};
	const char *messgBlockError[] = {
		"No data to decimate!",
		"No data to generate normals for!",
		"No data to smooth!"
	};

	int i;
	for(i=0; i<nBlockError; i++)
	{
		if(strstr(message,classBlockError[i])!=0 && strstr(message,messgBlockError[i])!=0) return 0;
	}
	//
	//  Cut extra parts of VTK error messages
	//
	const char *outMessage;
	if(mErrorMessageFirstString == 0)
	{
		outMessage = message;
	} 
	else 
	{
		outMessage = strstr(message,mErrorMessageFirstString);
	}
	outMessage += mErrorMessageFirstSymbol;
	
	return outMessage;
}

	
const char* iOutputWindow::FormattedWarningMessage(const char* message)
{
	//
	//  Cut extra parts of VTK error messages
	//

	const char *outMessage = strstr(message,"IFRIT:START-HERE");
	if(outMessage == 0)
	{
		outMessage = message;
	}
	else
	{
		outMessage += strlen("IFRIT:START-HERE");
	}
	
	return outMessage;
}


void iOutputWindow::StartMessageQueue()
{ 
	if(mQueue > 0)
	{
#ifdef I_CHECK1
		IERROR_REPORT_ERROR("Competing parallel regions.");
#endif
	}
	mQueue++;
}

void iOutputWindow::FlashMessageQueue()
{
	int i;

	mQueue--;

	if(mQueue > 0)
	{
#ifdef I_CHECK1
		IERROR_REPORT_ERROR("Competing parallel regions.");
#endif
	}
	else
	{
		for(i=0; i<mTextQueue.Size(); i++)
		{
			iConsole::Instance()->DisplayTextMessage(mTextQueue[i].ToCharPointer());
		}
		
		for(i=0; i<mErrorQueue.Size(); i++)
		{
			iConsole::Instance()->DisplayErrorMessage(mErrorQueue[i].ToCharPointer(),0,0);
		}
		
		mTextQueue.Clear();
		mErrorQueue.Clear();
	}
}
