/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iorthotextureplanefilter.h"


#include "ierror.h"
#include "imath.h"

//
//  Templates
//
#include "igenericfiltertemplate.h"


iOrthoTexturePlaneFilter::iOrthoTexturePlaneFilter() 
{
	mTexture = 0;
	mOffX = mOffY = 0;
}


iOrthoTexturePlaneFilter::~iOrthoTexturePlaneFilter() 
{
}


void iOrthoTexturePlaneFilter::SetTexturePiece(vtkImageData *texture, int offx, int offy)
{
	mTexture = texture;
	mOffX = offx;
	mOffY = offy;

	this->Modified();
}


void iOrthoTexturePlaneFilter::ProduceOutput(vtkImageData *input, vtkPolyData *output)
{
	int dims[3], dimsTexture[3];
	double org[3], spa[3];

	output->Initialize();

	if(mTexture == 0) return;
	
    input->GetOrigin(org);
    input->GetSpacing(spa);
    input->GetDimensions(dims);

	mTexture->GetDimensions(dimsTexture);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dims[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;


	//	
	//  Fill in the mTexture
	//
    float *ptrIn = (float *)input->GetScalarPointer();
	float *ptrOut = (float *)mTexture->GetScalarPointer();

	if(ptrOut == 0)
	{
		vtkErrorMacro("Texture data has not been allocated properly.");
		return;
	}

	int ijk[3], ijkmin[3], ijknum[3];

	ijk[Axis] = 0;

	ijkmin[Uidx] = 0;
	ijkmin[Vidx] = 0;
	ijknum[Uidx] = dims[Uidx] - 1;
	ijknum[Vidx] = dims[Vidx] - 1;

	if(mOffX < 0) ijkmin[Uidx] -= mOffX;
	if(mOffY < 0) ijkmin[Vidx] -= mOffY;
	if(mOffX+ijknum[Uidx] > dimsTexture[0]) ijknum[Uidx] = dimsTexture[0] - mOffX;
	if(mOffY+ijknum[Vidx] > dimsTexture[1]) ijknum[Vidx] = dimsTexture[1] - mOffY;

	vtkIdType off1 = 0, off2 = 0, off3;
	switch(Axis)
	{
	case 0:
		{
			//
			//  Uidx = 1, Vidx = 2;
			//
			off1 = dims[0];
			off2 = dims[0]*dims[1];
			break;
		}
	case 1:
		{
			//
			//  Uidx = 2, Vidx = 0;
			//
			off1 = dims[0]*dims[1];
			off2 = 1;
			break;
		}
	case 2:
		{
			//
			//  Uidx = 0, Vidx = 1;
			//
			off1 = 1;
			off2 = dims[0];
			break;
		}
	}

	off3 = off1 + off2;

	vtkIdType lIn, lOut;
	for(ijk[Vidx]=ijkmin[Vidx]; ijk[Vidx]<ijknum[Vidx]; ijk[Vidx]++)
	{
		this->UpdateProgress((float)(ijk[Vidx]-ijkmin[Vidx])/(ijknum[Vidx]-ijkmin[Vidx]+1));
		if(this->GetAbortExecute()) break;

		lOut = dimsTexture[0]*(ijk[Vidx]+mOffY);
		for(ijk[Uidx]=ijkmin[Uidx]; ijk[Uidx]<ijknum[Uidx]; ijk[Uidx]++)
		{
			lIn = ijk[0] + dims[0]*(ijk[1]+(vtkIdType)dims[1]*ijk[2]);
			ptrOut[lOut+mOffX+ijk[Uidx]] = 0.25f*(ptrIn[lIn]+ptrIn[lIn+off1]+ptrIn[lIn+off2]+ptrIn[lIn+off3]);
		}
	}

	mTexture->Modified();
}

