/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "igriddataconverter.h"


#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "imath.h"
#include "iviewmodule.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


iGridDataConverter* iGridDataConverter::New(iViewObject *vo)
{
	return new iGridDataConverter(vo);
}


iGridDataConverter::iGridDataConverter(iViewObject *vo) : iViewObjectComponent(vo)
{
	int i;

	mCurVar = 0;
	mScalarStatus = _NoScalars;

	vtkImageData *d = vtkImageData::New(); IERROR_ASSERT_NULL_POINTER(d);
	this->AddInput(d);
	d->Delete();

	for(i=0; i<3; i++)
	{
		mInputs[i] = vtkImageData::New(); IERROR_ASSERT_NULL_POINTER(mInputs[i]);
		this->AddInput(mInputs[i]);
		mInputs[i]->Delete();
	}

	this->SetDataRank(0U);
}


iGridDataConverter::~iGridDataConverter()
{
}


void iGridDataConverter::SyncWithData(int, void *)
{
	this->Modified();
}


vtkImageData* iGridDataConverter::GetScalarInput()
{
	return mInputs[0];
}


vtkImageData* iGridDataConverter::GetVectorInput()
{
	return mInputs[1];
}


vtkImageData* iGridDataConverter::GetTensorInput()
{
	return mInputs[2];
}


void iGridDataConverter::SetScalarInput(vtkImageData *input)
{
	mInputs[0] = input;
	this->vtkProcessObject::SetNthInput(1,input);
	if(mDataRank == 0U)	this->vtkProcessObject::SetNthInput(0,input);
}


void iGridDataConverter::SetVectorInput(vtkImageData *input)
{
	mInputs[1] = input;
	this->vtkProcessObject::SetNthInput(2,input);
	if(mDataRank == 1U)	this->vtkProcessObject::SetNthInput(0,input);
}


void iGridDataConverter::SetTensorInput(vtkImageData *input)
{
	mInputs[2] = input;
	this->vtkProcessObject::SetNthInput(3,input);
	if(mDataRank == 2U)	this->vtkProcessObject::SetNthInput(0,input);
}


void iGridDataConverter::SetInput(vtkImageData *)
{
	vtkErrorMacro("Incorrect use of iGridDataConverter");
}


void iGridDataConverter::SetNthInput(int, vtkDataObject *)
{
	vtkErrorMacro("Incorrect use of iGridDataConverter");
}


void iGridDataConverter::SetDataRank(unsigned int r)
{
	if(r<3U && r!=mDataRank)
	{
		mDataRank = r;
		this->vtkProcessObject::SetNthInput(0,mInputs[r]);
		this->Modified();
	}
}


void iGridDataConverter::SetCurrentVar(int n)
{ 
	if(n >= 0)
	{
		mCurVar = n;
		this->Modified();
	}
}


iGridDataConverter::ScalarStatus iGridDataConverter::GetScalarStatus()
{
	this->Update();
	return mScalarStatus;
}


void iGridDataConverter::PrepareOutput(vtkImageData *, vtkStructuredPoints *output)
{
	vtkImageData *input = mInputs[mDataRank];
	int dims0[3];
	double org0[3], spa0[3];

	if(input==0 || input->GetPointData()==0) return;

	input->GetDimensions(dims0);
	input->GetSpacing(spa0);
	input->GetOrigin(org0);

	output->SetWholeExtent(0,dims0[0]-1,0,dims0[1]-1,0,dims0[2]-1);  // God only knows why these two are needed,
	output->SetUpdateExtent(0,dims0[0]-1,0,dims0[1]-1,0,dims0[2]-1); // but it took me a lot of debugging to find out they must be here
}


void iGridDataConverter::ProduceOutput(vtkImageData*, vtkStructuredPoints *output)
{
	vtkImageData *input = mInputs[mDataRank];
	vtkImageData *scalars = mInputs[0];
	int dims[3], dims0[3];
	double org[3], spa[3], org0[3], spa0[3];
	
	mScalarStatus = _NoScalars;

	output->Initialize();

	if(input==0 || input->GetPointData()==0) return;

	input->GetDimensions(dims0);
	input->GetSpacing(spa0);
	input->GetOrigin(org0);

	output->SetDimensions(dims0);
	output->SetSpacing(spa0);
	output->SetOrigin(org0);

	output->SetWholeExtent(0,dims0[0]-1,0,dims0[1]-1,0,dims0[2]-1);  // God only knows why these two are needed,
	output->SetUpdateExtent(0,dims0[0]-1,0,dims0[1]-1,0,dims0[2]-1); // but it took me a lot of debugging to find out they must be here

	switch(mDataRank)
	{
	case 0U:
		{
			break;
		}
	case 1U:
		{
			output->GetPointData()->SetVectors(input->GetPointData()->GetVectors());
			break;
		}
	case 2U:
		{
			output->GetPointData()->SetTensors(input->GetPointData()->GetTensors());
			break;
		}
	default: 
		{
			vtkErrorMacro("Incorrect PassMode");
			return;
		}
	}

	if(scalars==0 || scalars->GetPointData()==0) return;

	int n = scalars->GetPointData()->GetNumberOfArrays();
	if(n == 0) return;

	if(mCurVar<0 || mCurVar>=n)
	{
		return;
    }

	scalars->GetDimensions(dims);
	scalars->GetSpacing(spa);
	scalars->GetOrigin(org);

	if(dims[0]!=dims0[0] || dims[1]!=dims0[1] || dims[2]!=dims0[2] || vtkMath::Distance2BetweenPoints(spa,spa0)>iMath::_DoubleTolerance  || vtkMath::Distance2BetweenPoints(org,org0)>iMath::_DoubleTolerance)
	{
		//
		// Scalars are not compatible with vectors/tensors - exit
		//
		return;
	}

	if(scalars->GetPointData()->GetScalars() != 0)
	{
		output->GetPointData()->SetScalars(scalars->GetPointData()->GetScalars());
		mScalarStatus = _MultipleComponents;
	}
	else if(scalars->GetPointData()->GetArray(mCurVar) != 0)
	{
		output->GetPointData()->SetScalars(scalars->GetPointData()->GetArray(mCurVar));
		mScalarStatus = _SingleComponent;
	}
	else 
	{
		vtkErrorMacro("No appropriate data present");
	}
}
