/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "igenericparticlesdatasubject.h"


#include "idata.h"
#include "idatalimits.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "iparallel.h"
#include "iparallelworker.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>


//
//  Helper class for parallel executions
//
class iGenericParticlesDataHelper : protected iParallelWorker
{

public:

	iGenericParticlesDataHelper(iGenericParticlesDataSubject *subject);

	void ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	iGenericParticlesDataSubject *mSubject;

	bool *mPeriodic;
	float *mFarr1;
	double *mDarr1;

	int mItmp1;
	bool mBtmp1;
	long mLtmp1;
	float mFtmp1;
};


//
//  Main class
//
iGenericParticlesDataSubject::iGenericParticlesDataSubject(iDataReader *r, const iDataType &type) : iDataSubject(r,type)
{
	mHaveNormals = false;

	mDownsampleMode = 0;
	mDownsampleFactor = 1;

	mHelper = new iGenericParticlesDataHelper(this); IERROR_ASSERT_NULL_POINTER(mHelper);
}


iGenericParticlesDataSubject::~iGenericParticlesDataSubject()
{
	delete mHelper;
}


void iGenericParticlesDataSubject::SetDownsampleMode(int v)
{
	if(v>=0 && v<=3 && v!=mDownsampleMode)
	{
		mDownsampleMode = v;
	}
}


void iGenericParticlesDataSubject::SetDownsampleFactor(int v)
{
	if(v > 0)
	{
		mDownsampleFactor = v;
	}
}


void iGenericParticlesDataSubject::FinalizePolyData(vtkPolyData *data)
{
	if(data==0 || data->GetPoints()==0) return;

	int i;
	vtkIdType l, loff, ntot = data->GetNumberOfPoints();

	//
	//  Boundary condition
	//
	for(i=0; i<3; i++) this->SetDirectionPeriodic(i,this->IsBoxPeriodic());

	if(this->IsBoxPeriodic())
	{
		if(data->GetPoints()->GetDataType() == VTK_FLOAT)
		{
			float *xptrF = (float *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrF[loff] -= 2.0*floor(0.5*(1.0+xptrF[loff]));
				}
			}
		}
		else if(data->GetPoints()->GetDataType() == VTK_DOUBLE)
		{
			double *xptrD = (double *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrD[loff] -= 2.0*floor(0.5*(1.0+xptrD[loff]));
				}
			}
		}
		else
		{
			this->GetErrorStatus()->Set("Internal bug: ivalid points data type.");
		}
	}

	if(mHaveNormals)
	{
		vtkFloatArray *newNormals;
		newNormals = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(newNormals);
		newNormals->SetNumberOfComponents(3);
		// Allocates and Sets MaxId
		newNormals->SetNumberOfTuples(ntot);
		float *p = (float *)newNormals->GetVoidPointer(0);
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				p[3*l+0] = p[3*l+1] = 0.0f;
				p[3*l+2] = 1.0f;
			}
			data->GetPointData()->SetNormals(newNormals);
		}
		newNormals->Delete();
	}

	//
	//  Check for overflow
	//
	if(data->GetPointData()->GetScalars() != 0)
	{
		float *p = (float *)data->GetPointData()->GetScalars()->GetVoidPointer(0);
		int natt = data->GetPointData()->GetScalars()->GetNumberOfComponents();
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<natt; i++)
				{
					if(p[natt*l+i] < -iMath::_LargeFloat)
					{
						p[natt*l+i] = -iMath::_LargeFloat;
						mOverflow = true;
					}
					if(p[natt*l+i] >  iMath::_LargeFloat)
					{
						p[natt*l+i] =  iMath::_LargeFloat;
						mOverflow = true;
					}
				}
			}
		}
	}
}


void iGenericParticlesDataSubject::ShiftPolyData(vtkPolyData *data, int d, double dx)
{
	if(data==0 || data->GetPoints()==0 || d<0 || d>2) return;

	long n = data->GetPoints()->GetNumberOfPoints();
	float *pf = (float *)data->GetPoints()->GetVoidPointer(0);
	double *pd = (double *)data->GetPoints()->GetVoidPointer(0);
		
	if(pf!=0 && pd!=0)
	{
		mHelper->ShiftData(data->GetPoints()->GetDataType()==VTK_FLOAT,d,n,mPeriodic,dx,pf,pd);
		data->Modified();
	}
}


vtkIdType iGenericParticlesDataSubject::PrepareMask(vtkIdType ntot, int df, vtkIdType offm, vtkIdType offt)
{
	if(df < 1) df = 1;

	wMask.NumTotal = ntot;
	wMask.NumMasked = (ntot+df-1)/df;
	wMask.NumUnmasked = wMask.NumTotal - wMask.NumMasked;
	wMask.OffsetTotal = offt;
	wMask.OffsetMasked = offm;
	wMask.DownsampleFactor = df;
	wMask.RandomThreshold = 1.0/df;

	return wMask.NumMasked;
}


//
//  Helper class
//
iGenericParticlesDataHelper::iGenericParticlesDataHelper(iGenericParticlesDataSubject *subject) : iParallelWorker(subject->GetViewModule()->GetParallelManager())
{
	mSubject = subject;
}


void iGenericParticlesDataHelper::ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd)
{
	mBtmp1 = paf;
	mItmp1 = dim;
	mLtmp1 = n;
	mFtmp1 = dr;
	mFarr1 = pf;
	mDarr1 = pd;
	mPeriodic = per;

	this->ParallelExecute(1);
}


int iGenericParticlesDataHelper::ExecuteStep(int /*step*/, iParallel::ProcessorInfo &p)
{
	long l, kstp, kbeg, kend;

	iParallel::SplitRange(p,mLtmp1,kbeg,kend,kstp);

	int d = mItmp1;
	float dr = mFtmp1;

	if(mBtmp1)
	{
		float *x = mFarr1 + 3*kbeg;
		for(l=kbeg; l<kend; l++)
		{
			if(l%1000 == 0)
			{
				if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress((d+(float)(l-kbeg)/(kend-kbeg))/3.0);
				if(mSubject->GetObserver()->IsAborted()) return 2;
			}
			x[d] += 2.0*dr;
			if(mPeriodic[d])
			{
				if(x[d] >  1.0) x[d] -= 2.0;
				if(x[d] < -1.0) x[d] += 2.0;
			}
			x += 3;
		}
	}
	else
	{
		double *x = mDarr1 + 3*kbeg;
		for(l=kbeg; l<kend; l++)
		{
			if(l%1000 == 0)
			{
				if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress((float)(l-kbeg)/(kend-kbeg));
				if(mSubject->GetObserver()->IsAborted()) return 2;
			}
			x[d] += 2.0*dr;
			if(mPeriodic[d])
			{
				if(x[d] >  1.0) x[d] -= 2.0;
				if(x[d] < -1.0) x[d] += 2.0;
			}
			x += 3;
		}
	}

	return 0;
}

