/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icrosssectionviewobject.h"


#include "icolorbars.h"
#include "icontrolmodule.h"
#include "icrosssectionparallelpipeline.h"
#include "icrosssectionserialpipeline.h"
#include "idata.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ihistogram.h"
#include "ilookuptable.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"

#include <vtkLookupTable.h>
#include <vtkProperty.h>
#include <vtkStructuredPoints.h>
#include <vtkTexture.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iCrossSectionViewObject,CrossSection,x);

IOBJECT_DEFINE_KEY(iCrossSectionViewObject,Dir,d,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,InterpolateData,id,Bool,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,SpecialLocation,loc,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,OverTheEdgeFlag,oe,Bool,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,ResampleFactor,rs,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewObject,Var,v,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iCrossSectionViewObject,Pos,x);

//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iCrossSectionViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_SHADING(iCrossSectionViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(iCrossSectionViewObject,1);


//
// iCrossSectionViewObject class
//
iCrossSectionViewObject* iCrossSectionViewObject::New(iViewModule *vm)
{
	iCrossSectionViewObject *tmp = iPointerCast<iCrossSectionViewObject,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	IERROR_ASSERT_NULL_POINTER(tmp);
	//
	//  Create mPipeline (must be created after the ibject is fully created)
	//	
	tmp->mPipeline = tmp->CreateParallelPipeline(); IERROR_ASSERT_NULL_POINTER(tmp->mPipeline);
	tmp->mActors[0]->SetInput(tmp->mPipeline->GetOutput());
	tmp->mTexture->SetInput(tmp->mPipeline->GetTextureData());

	tmp->Reset();

	return tmp;
}


iCrossSectionViewObject::iCrossSectionViewObject(iViewModule *vm, const iDataInfo &info) : iSolidViewObject(vm,info,1,_ViewObjectNoColor|_ViewObjectNoOpacity|_ViewObjectNoReplicating), mPos(vm,false), mPosMax(vm,false), mVar(0)
{
	int i;

	mName = "iCrossSectionViewObject";
	mObjectType = _ObjectTypeCrossSection;

	if(mPrimaryDataInfo->Index(iDataType::UniformScalars()) == -1)
	{
		IERROR_REPORT_FATAL_ERROR("iCrossSectionViewObject is configured incorrectly.");
	}
	for(i=0; i<mPrimaryDataInfo->Count(); i++) mVar.Add(0);

	mDir = 2;
	mPos = -1.0;
	mPosMax = 1.0;
	
	mMethod = 1;
	mInterpolateData = true;

	mOverTheEdgeFlag = false;

	mResampleFactor = 1;
	mForcePolygonalMethod = false;

	//
	//  Do VTK stuff
	//	
	mTexture = vtkTexture::New(); IERROR_ASSERT_NULL_POINTER(mTexture);

	mPipeline = 0;

	mActors[0]->SetScalarVisibility(true);
	mActors[0]->ColorByArrayComponent(0,0);
	mActors[0]->GetProperty()->SetColor(1.0,1.0,1.0);
	mActors[0]->GetProperty()->SetOpacity(1.0);
	this->SetShading(false);

	mTexture->SetLookupTable(mActors[0]->GetLookupTable());
	mTexture->SetQualityTo32Bit();
}


iCrossSectionViewObject::~iCrossSectionViewObject()
{
	this->ShowColorBars(false);

	mTexture->Delete();
	mPipeline->Delete();
}


void iCrossSectionViewObject::FinishInitialization()
{
	this->SetMethod(mMethod);
	this->SetInterpolateData(mInterpolateData);
	this->SetDir(mDir);
}


iCrossSectionParallelPipeline* iCrossSectionViewObject::CreateParallelPipeline()
{
	iCrossSectionParallelPipeline *tmp = new iCrossSectionParallelPipeline(this); IERROR_ASSERT_NULL_POINTER(tmp);
	tmp->FinishInitialization();
	return tmp;
}


iCrossSectionSerialPipeline* iCrossSectionViewObject::CreateSerialPipeline()
{
	return new iCrossSectionSerialPipeline(this);
}


void iCrossSectionViewObject::ResetInput()
{
	mPipeline->SetInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetData()));
}


void iCrossSectionViewObject::ResetPipeline()
{
}


void iCrossSectionViewObject::SetMethod(int m)
{
	switch(m) 
	{
	case 0:
		{
			mActors[0]->SetTexture(0); 
			break;
		}
	case 1: 
		{
			mActors[0]->SetTexture(mTexture); 
			break;
		}
	default: return;
	}
	mMethod = m;
	mPipeline->UpdateMethod();
	this->ClearCache();
}


void iCrossSectionViewObject::ForcePolygonalMethod(bool s)
{
	mForcePolygonalMethod = s;
	mPipeline->UpdateMethod();
}


void iCrossSectionViewObject::ShowColorBarsBody(bool show)
{
	this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityCrossSection,this->GetVar(),this->GetActiveDataType(),mPalettes[0],show);
}


void iCrossSectionViewObject::SetDir(int d)
{ 
	if(d>=0 && d<3)
	{
		mDir = d;
		mPipeline->UpdateDir();
		this->ClearCache();
	}
}


void iCrossSectionViewObject::SetVar(int v)
{ 
	if(v>=0 && v<this->GetLimits()->GetNumVars() && v!=this->GetVar())
	{
		this->ShowColorBars(false);
		mVar[mActiveDataTypeIndex] = v;
		this->ShowColorBars(true);
		mPipeline->UpdateVar();
		this->SyncWithData(this->GetVar()+1,0);
		this->ClearCache();
	}
}


bool iCrossSectionViewObject::SetPos(double p, bool checkPosMax)
{
	iDistance tmp(this->GetViewModule(),false);
	tmp = p;
	return this->SetPos(tmp,checkPosMax);
}


bool iCrossSectionViewObject::SetPos(const iDistance &p, bool checkPosMax)
{ 
	bool ret = false;
	
	double spa[3], org[3];
	org[0] = org[1] = org[2] = 0.0;
	if(mDir == 0) mPosMax = 1.0; else 
	{
		int dims[3];
		mPipeline->GetInput()->GetDimensions(dims);
		if(dims[0]>0 && dims[1]>0 && dims[2]>0)
		{
			mPipeline->GetInput()->GetSpacing(spa);
			mPipeline->GetInput()->GetOrigin(org);
			mPosMax = org[mDir-1] + spa[mDir-1]*(dims[mDir-1]-1);
		}
		else mPosMax = 1.0;
	}
	
	mPos = p;
	if(mPos>mPosMax && checkPosMax)
	{
		mPos = mPosMax;
		ret = true;
	}
	if(mPos < -1.0) mPos = -1.0;

	mPipeline->UpdatePos();
	this->ClearCache();
	
	return ret;
}


void iCrossSectionViewObject::SetInterpolateData(bool s)
{
	mInterpolateData = s;
	if(mInterpolateData) 
	{
		mTexture->InterpolateOn(); 
		mActors[0]->GetProperty()->SetInterpolationToGouraud();
	}
	else 
	{
		mTexture->InterpolateOff(); 
		mActors[0]->GetProperty()->SetInterpolationToFlat();
	}
	mPipeline->UpdateInterpolateData();
	this->ClearCache();
}


void iCrossSectionViewObject::SetResampleFactor(int p)
{
	mResampleFactor = p;
	mPipeline->UpdateResampleFactor();
	this->ClearCache();
}


void iCrossSectionViewObject::ShowBody(bool show)
{
	if(show)
	{
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mActors[0]->VisibilityOff();
	}
}


float iCrossSectionViewObject::GetMemorySize() const
{
	return mPipeline->GetMemorySize();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iCrossSectionViewObject::SolidViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyInterpolateData(),mInterpolateData);

	this->PackValue(s,KeyVar(),this->GetVar());
	this->PackValue(s,KeyDir(),mDir);
	this->PackValue(s,KeyMethod(),mMethod);

	this->PackValue(s,KeyResampleFactor(),this->GetResampleFactor());
	this->PackValue(s,KeyOverTheEdgeFlag(),mOverTheEdgeFlag);

	this->PackValueDistance(s,KeyPos(),mPos);
}


void iCrossSectionViewObject::SolidViewObjectUnPackStateBody(const iString &s)
{
	int i; bool b;

	if(this->UnPackValue(s,KeyInterpolateData(),b)) this->SetInterpolateData(b);

	if(this->UnPackValue(s,KeyVar(),i)) this->SetVar(i);
	if(this->UnPackValue(s,KeyDir(),i)) this->SetDir(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	
	if(this->UnPackValue(s,KeyResampleFactor(),i)) this->SetResampleFactor(i);

	if(this->UnPackValueDistance(s,KeyPos(),mPos)) this->SetPos(mPos,false);

	//
	//  Special "action" keys
	//
#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //action keys are not part of the states
#endif

	if(this->UnPackValue(s,KeySpecialLocation(),i)) this->PlaceAtSpecialLocation(i); 

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
#endif
}


void iCrossSectionViewObject::PlaceAtSpecialLocation(int n)
{
	switch(n)
	{
	case 0:
	case 1:
		{
			if(this->IsDataPresent())
			{
				iDataExplorer *de = iDataExplorer::New(this->GetViewModule());
				if(de != 0)
				{
					de->SetActiveDataType(this->GetActiveDataType());
					de->SetInputComponent(this->GetVar());
					const iDataExplorer::Info &di = de->GetInfo(true);
					if(n == 0)
					{
						if(di.CellMin >= 0) this->SetPos(di.PosMin->OpenGLPosition(mDir));
					}
					else
					{
						if(di.CellMax >= 0) this->SetPos(di.PosMax->OpenGLPosition(mDir));
					}
					de->Delete();
				}
			}
			break;
		}
	}
}


bool iCrossSectionViewObject::CanBeShown() const
{
	return (this->GetVar() < this->GetLimits()->GetNumVars());
}


void iCrossSectionViewObject::ViewObjectSyncWithData(int info, void *)
{
	if(info==0 || info==1+this->GetVar())
	{
		// this replaces SetPal
		if(this->GetVar() < this->GetLimits()->GetNumVars())
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetVar());
		}
	}
}

