/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontrolmodule.h"


#include "ianimator.h"
#include "icamera.h"
#include "icolorbars.h"
#include "icontrolscript.h"
#include "icrosssectionviewobject.h"
#include "idata.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "imarkerviewobject.h"
#include "ipalette.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iparticlegroup.h"
#include "iparticlesviewobject.h"
#include "ipicker.h"
#include "ipiecewisefunction.h"
#include "irendertool.h"
#include "ishell.h"
#include "isurfaceviewobject.h"
#include "itensorfieldviewobject.h"
#include "iviewobjectfamily.h"
#include "iviewmodule.h"
#include "ivectorfieldviewobject.h"
#include "ivolumeviewobject.h"

#include <vtkCamera.h>
#include <vtkCommand.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>

//
//  templates
//
#include "iarraytemplate.h"
#include "ifamilytemplate.h"
#include "iviewobjectfamilytemplate.h"

//
//  macros
//
#include "icontrolmodulemacro.h"


using namespace iParameter;


int iControlModule::mIdCounter = 0;


IOBJECT_DEFINE_TYPE(iControlModule,ControlModule,cm,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iControlModule,AutoRender,ar,Bool,1);
IOBJECT_DEFINE_KEY(iControlModule,OptimizationMode,om,Int,1);
IOBJECT_DEFINE_KEY(iControlModule,SynchronizeCameras,sc,Bool,1);
IOBJECT_DEFINE_KEY(iControlModule,SynchronizeInteractors,si,Bool,1);


#define BACKWARD_COMPATIBLE
#define COMPATIBILITY_MODE	11


//
//  Helper functions hidden in a private namespace
//
namespace iControlModule_Private
{
	iPalette* Palette_Rainbow();
	iPalette* Palette_Temperature();
	iPalette* Palette_Greyscale();
	iPalette* Palette_Bluewhite();
	iPalette* Palette_Prizm();
	iPalette* Palette_Greenwhite();
	iPalette* Palette_Bluered();
	iPalette* Palette_Stern();
	iPalette* Palette_Haze();
	iPalette* Palette_Starlight();
	iPalette* Palette_3color();
	iPalette* Palette_4color();
};


using namespace iControlModule_Private;


//
//  Helper class. Probably should not be here.
//
class iInteractorEventObserver : public vtkCommand, protected iControlModuleComponent
{

public:
	
	static iInteractorEventObserver* New(iControlModule *cm);

	void SetDriver(iViewModule *vm);
	virtual void Execute(vtkObject *caller, unsigned long event, void *callData);

protected:

	iInteractorEventObserver(iControlModule *cm) : iControlModuleComponent(cm)
	{
		mInUpdate = false;
		mDriver = 0;
	}

	bool mInUpdate;
	iViewModule *mDriver;
};


iInteractorEventObserver* iInteractorEventObserver::New(iControlModule *cm)
{
	return new iInteractorEventObserver(cm);
}


void iInteractorEventObserver::SetDriver(iViewModule *vm)
{
	if(vm == mDriver) return;

	if(mDriver != 0)
	{
		mDriver->GetInteractor()->RemoveObserver(this);
	}

	if(vm != 0)
	{
		vm->GetInteractor()->AddObserver(vtkCommand::LeftButtonPressEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::LeftButtonReleaseEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::MiddleButtonPressEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::MiddleButtonReleaseEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::RightButtonPressEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::RightButtonReleaseEvent,this);
		vm->GetInteractor()->AddObserver(vtkCommand::MouseMoveEvent,this);
		mDriver = vm;
	}
}


void iInteractorEventObserver::Execute(vtkObject *caller, unsigned long event, void *callData)
{
	if(mInUpdate) return;
	
	vtkRenderWindowInteractor *iren = dynamic_cast<vtkRenderWindowInteractor *>(caller);
	if(this->GetControlModule()!=0 && iren!=0)
	{
		int i;
		mInUpdate = true;
		for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++) if(this->GetControlModule()->GetViewModule(i)->GetInteractor() != iren)
		{
			this->GetControlModule()->GetViewModule(i)->GetInteractor()->SetControlKey(iren->GetControlKey());
			this->GetControlModule()->GetViewModule(i)->GetInteractor()->SetShiftKey(iren->GetShiftKey());
			this->GetControlModule()->GetViewModule(i)->GetInteractor()->SetEventPosition(iren->GetEventPosition());
			this->GetControlModule()->GetViewModule(i)->GetInteractor()->SetKeySym(iren->GetKeySym());
			this->GetControlModule()->GetViewModule(i)->GetInteractor()->InvokeEvent(event,callData);
		}
		mInUpdate = false;
	}
}


//
//  Main class
//
iControlModule* iControlModule::New(iShell *s)
{
	iControlModule *cm = iExtensionFactory::CreateControlModule(s);
	if(cm != 0) cm->FinishInitialization();
	return cm;
}


iControlModule::iControlModule(iShell *s) : iShellComponent(s), mPalList(0)
{
	mId = mIdCounter++;

	mInExecute = false;

	mBlockBroadcasts = true;
	mAutoRender = mSyncInteractors = mExecuteOk = false;

	this->UpdateDataDir();

	//
	//  Palettes must be created before the ViewModule family is created, since they are used then
	//
	this->CreateDefaultPalettes();

	//
	//  Parallel performance - this must be first, since many components are parallel workers as well
	//
	mParallelManager = iParallelManager::New(this); IERROR_ASSERT_NULL_POINTER(mParallelManager);
	mParallelObserver = iParallelUpdateEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mParallelObserver);
	mParallelManager->AddObserver(iParallel::InformationEvent,mParallelObserver);

	mCurrentSubject = 0;

	//
	//  Start with NULL view module family
	//
	mViewModules = 0;
}


void iControlModule::FinishInitialization()
{
	//
	//  Create observer for synchronizing interactors
	//
	mInteractorObserver = iInteractorEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mInteractorObserver);
	//
	//  Create script - must be created before VM family, since VM Animators will be children of CM script
	//
	mControlScript = iControlScript::New(this);
	//
	//  ViewModule family must be created after ControlModule is fully created
	//
	mViewModules = iFamily<iViewModule,iControlModule>::New(this); IERROR_ASSERT_NULL_POINTER(mViewModules);
	if(mSyncInteractors) mInteractorObserver->SetDriver(mViewModules->GetMember(0));
	//
	//  Composer must be created after view modules
	//
	mComposer = iImageComposer::New(this,this->GetShell()); IERROR_ASSERT_NULL_POINTER(mComposer);
	mViewModules->GetMember(0)->SetImageComposer(mComposer);
	//
	//  Allow broadcasts
	//
	mBlockBroadcasts = false;
}


iControlModule::~iControlModule()
{
	int i;

	mControlScript->Delete();
	mComposer->Delete();
	mInteractorObserver->Delete();

	//
	//  Deleting view modules is tricky: we need to delete clones first so that
	//  data reader and limits are not deleted before the view object are.
	//
	for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++) if(mViewModules->GetMember(i)->IsClone()) mViewModules->DeleteMember(i);
	mViewModules->Delete();

	mParallelObserver->Delete();
	mParallelManager->Delete();

	for(i=0; i<mPalList.Size(); i++) delete mPalList[i];
}


bool iControlModule::Execute(const iString &command, bool render, int option, int mod)
{
	if(mInExecute) return true; // Protect from concurrent calls from a multitheaded shell
	mInExecute = true;

	int j;
	mExecuteOk = false;

	if(option == 0)
	{
		//
		//  Default options
		//
		option = _ObjectModeCurrent | _ModuleModeCurrent | _RenderModeAuto;
	}
	//
	//  Is it a ControlModule command?
	//
	iString pre = command.Section(iObjectKey::PrefixSeparator(),0,0);
	if(iControlModule::Type().IsMatch(pre)) 
	{
#ifdef I_CHECK1
		bool saveReportMissingKeys = iObject::ReportMissingKeys;
		iObject::ReportMissingKeys = false; //commands are not complete states
#endif
		this->UnPackState(command);
#ifdef I_CHECK1
		iObject::ReportMissingKeys = saveReportMissingKeys;
#endif
		mExecuteOk = true;
		if(render || mAutoRender)
		{
			for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
			{
				mViewModules->GetMember(j)->Render();
			}
		}
		mInExecute = false;
		return mExecuteOk;
	}
	//
	//  Set modes
	//
	int objMode = option & _ObjectModeMask;
	int modMode = option & _ModuleModeMask;
	int renMode = option & _RenderModeMask;

	if(objMode==_ObjectModeInvalid || modMode==_ModuleModeInvalid || renMode==_RenderModeInvalid)
	{
		IERROR_REPORT_ERROR("Invalid option.");
		mExecuteOk = false;
		mInExecute = false;
		return mExecuteOk;
	}

	render = render || mAutoRender;

	if(mod>=0 && mod<mViewModules->GetMaxMemberIndex())
	{
		renMode = _ModuleModeInvalid;
	}

	//
	//  Loop over view modules
	//
	iDataReader *r = mViewModules->GetCurrentMember()->GetReader();

#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //commands are not complete states
#endif
	//
	//  Execute and Set renMode to prefered value if it is Set to Auto initially
	//
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
	{
		switch(modMode)
		{
		case _ModuleModeCurrent:
			{
				if(j == mViewModules->GetCurrentMemberIndex()) ExecuteForViewModule(mViewModules->GetMember(j),command,render,objMode,renMode);
				break;
			}
		case _ModuleModeAll:
			{
				if(renMode == _RenderModeAuto) renMode = _RenderModeAll;
				this->ExecuteForViewModule(mViewModules->GetMember(j),command,render,objMode,renMode);
				break;
			}
		case _ModuleModeClones:
			{
				if(renMode == _RenderModeAuto) renMode = _RenderModeClones;
				if(r == mViewModules->GetMember(j)->GetReader()) ExecuteForViewModule(mViewModules->GetMember(j),command,render,objMode,renMode);
				break;
			}
		case _ModuleModeInvalid:
			{
				if(j == mod) ExecuteForViewModule(mViewModules->GetMember(j),command,render,objMode,renMode);
				break;
			}
		}
	}

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
	if(renMode == _RenderModeAuto) IERROR_REPORT_ERROR("Improper RenderMode.");
#endif

	//
	//  Execute changes the state of the object. So, if the command is non-empty, we need to
	//  clear the cached state
	//
	if(!command.IsEmpty())
	{
		this->ClearCache();
	}
	
	//
	//  Don't render if not asked explicitly
	//
	if(!render)
	{
		mInExecute = false;
		return mExecuteOk;
	}

	//
	//  Render
	//
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
	{
		switch(renMode)
		{
		case _RenderModeCurrent:
			{
				if(j == mViewModules->GetCurrentMemberIndex()) mViewModules->GetMember(j)->Render();
				break;
			}
		case _RenderModeAll:
			{
				mViewModules->GetMember(j)->Render();
				break;
			}
		case _RenderModeClones:
			{
				if(r == mViewModules->GetMember(j)->GetReader()) mViewModules->GetMember(j)->Render();
				break;
			}
		}
	}

	mInExecute = false;
	return mExecuteOk;
}


int iControlModule::FindTypeFromString(const iString &command) const
{
	//
	//  Find the object type from its type
	//
	iString pre = command.Section(iObjectKey::PrefixSeparator(),0,0);
	
	if(iControlModule::Type().IsMatch(pre)) return _ObjectTypeControlModule;
	if(iViewModule::Type().IsMatch(pre)) return _ObjectTypeViewModule;
	if(iAnimator::Type().IsMatch(pre)) return _ObjectTypeAnimator;
	if(iCamera::Type().IsMatch(pre)) return _ObjectTypeCamera;
	if(iColorBars::Type().IsMatch(pre)) return _ObjectTypeColorBars;
	if(iDataReader::Type().IsMatch(pre)) return _ObjectTypeDataReader;
	if(iImageComposer::Type().IsMatch(pre)) return _ObjectTypeImageComposer;
	if(iMarkerViewObject::Type().IsMatch(pre)) return _ObjectTypeMarker;
	if(iParticleGroup::Type().IsMatch(pre)) return _ObjectTypeParticleGroup;
	if(iParticlesViewObject::Type().IsMatch(pre)) return _ObjectTypeParticles;
	if(iPicker::Type().IsMatch(pre)) return _ObjectTypePicker;
	if(iSurfaceViewObject::Type().IsMatch(pre)) return _ObjectTypeSurface;
	if(iTensorFieldViewObject::Type().IsMatch(pre)) return _ObjectTypeTensorField;
	if(iVectorFieldViewObject::Type().IsMatch(pre)) return _ObjectTypeVectorField;
	if(iVolumeViewObject::Type().IsMatch(pre)) return _ObjectTypeVolume;
	if(iCrossSectionViewObject::Type().IsMatch(pre)) return _ObjectTypeCrossSection;
	
	//
	//  Is it a DataSubject?
	//
	if(iDataSubject::IsTypeMatch(pre))
	{
		mDataSubjectName = iDataSubject::GetDataSubjectName(pre);
		/*if(!mDataSubjectName.IsEmpty())*/ return _ObjectTypeDataSubject;
	}
	
	return _ObjectTypeUndefined;
}


iObject* iControlModule::FindObject(const iObjectType &type, int mod, int obj)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	iViewObject *object = this->FindViewObject(type,mod,obj);
	if(object != 0) return object;

	switch(FindTypeFromString(type.FullName()))
	{
	case _ObjectTypeControlModule: 
		{
			return this;
		}
	case _ObjectTypeViewModule: 
		{
			return module;
		}
	case _ObjectTypeAnimator:
		{
			return module->GetAnimator();
		}
	case _ObjectTypeCamera:
		{
			return module->GetRenderTool()->GetCamera();
		}
	case _ObjectTypeColorBars:
		{
			return module->GetColorBars();
		}
	case _ObjectTypeDataReader:
		{
			return module->GetReader();
		}
	case _ObjectTypeDataSubject:
		{
			return module->GetReader()->GetSubject(iDataType::FindTypeByName(mDataSubjectName));
		}
	case _ObjectTypeImageComposer:
		{
			return mComposer;
		}
	case _ObjectTypePicker:
		{
			return module->GetPicker();
		}
	default: return 0;
	}
}


iViewObject* iControlModule::FindViewObject(const iObjectType &type, int mod, int obj) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	switch(FindTypeFromString(type.FullName()))
	{
	case _ObjectTypeMarker:
		{
			if(obj == -1) obj = module->GetMarkerFamily()->GetCurrentMemberIndex();
			return module->GetMarkerFamily()->GetMember(obj);
		}
	case _ObjectTypeParticleGroup:
		{
			if(obj == -1) obj = module->GetParticlesFamily()->GetCurrentMember()->GetCurrentGroupIndex();
			return module->GetParticlesFamily()->GetCurrentMember()->GetGroup(obj);
		}
	case _ObjectTypeParticles:
		{
			if(obj == -1) obj = module->GetParticlesFamily()->GetCurrentMemberIndex();
			return module->GetParticlesFamily()->GetMember(obj);
		}
	case _ObjectTypeSurface:
		{
			if(obj == -1) obj = module->GetSurfaceFamily()->GetCurrentMemberIndex();
			return module->GetSurfaceFamily()->GetMember(obj);
		}
	case _ObjectTypeTensorField:
		{
			if(obj == -1) obj = module->GetTensorFieldFamily()->GetCurrentMemberIndex();
			return module->GetTensorFieldFamily()->GetMember(obj);
		}
	case _ObjectTypeVectorField:
		{
			if(obj == -1) obj = module->GetVectorFieldFamily()->GetCurrentMemberIndex();
			return module->GetVectorFieldFamily()->GetMember(obj);
		}
	case _ObjectTypeVolume:
		{
			if(obj == -1) obj = module->GetVolumeFamily()->GetCurrentMemberIndex();
			return module->GetVolumeFamily()->GetMember(obj);
		}
	case _ObjectTypeCrossSection:
		{
			if(obj == -1) obj = module->GetCrossSectionFamily()->GetCurrentMemberIndex();
			return module->GetCrossSectionFamily()->GetMember(obj);
		}
	default: return 0;
	}
}


void iControlModule::ExecuteForViewModule(iViewModule *module, const iString &command, bool /*render*/, int objMode, int &renMode)
{
	int j;
	
	switch(FindTypeFromString(command))
	{
	case _ObjectTypeViewModule: 
		{
			module->UnPackState(command);
			mExecuteOk = module->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeAnimator:
		{
			module->GetAnimator()->UnPackState(command);
			mExecuteOk = module->GetAnimator()->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeCamera:
		{
			module->GetRenderTool()->GetCamera()->UnPackState(command);
			mExecuteOk = module->GetRenderTool()->GetCamera()->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeColorBars:
		{
			module->GetColorBars()->UnPackState(command);
			mExecuteOk = module->GetColorBars()->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeDataReader:
		{
			module->GetReader()->UnPackState(command);
			mExecuteOk = module->GetReader()->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeClones;
			break;
		}
	case _ObjectTypeDataSubject:
		{
			iDataSubject *subject = module->GetReader()->GetSubject(iDataType::FindTypeByName(mDataSubjectName));
			if(subject != 0)
			{
				subject->UnPackState(command);
				mExecuteOk = subject->UnPackedSomething();
			}
			else mExecuteOk = false;
			if(renMode == _RenderModeAuto) renMode = _RenderModeClones;
			break;
		}
	case _ObjectTypeImageComposer:
		{
			mComposer->UnPackState(command);
			mExecuteOk = mComposer->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeMarker:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(Marker);
			if(mExecuteOk) module->BuildMarkerLegend(module->GetMarkerFamily()->GetCurrentMemberIndex());
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeParticleGroup:
		{
			iParticlesViewObject *w = module->GetParticlesFamily()->GetCurrentMember();
			ICONTROLMODULE_EXECUTE_FOR_OBJECT1(Group);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeParticles:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(Particles);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypePicker:
		{
			module->GetPicker()->UnPackState(command);
			mExecuteOk = module->GetPicker()->UnPackedSomething();
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeSurface:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(Surface);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeTensorField:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(TensorField);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeVectorField:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(VectorField);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeVolume:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(Volume);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	case _ObjectTypeCrossSection:
		{
			ICONTROLMODULE_EXECUTE_FOR_OBJECT(CrossSection);
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	default:
		{
			if(renMode == _RenderModeAuto) renMode = _RenderModeCurrent;
			break;
		}
	}
}


bool iControlModule::CreateObject(const iObjectType &type, bool render, int mod, int vmtype)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return false;

	render = render || mAutoRender;

	switch(FindTypeFromString(type.FullName()))
	{
	case _ObjectTypeViewModule: 
		{
			int i = mViewModules->CreateMember(); 
			if(i != -1) 
			{
				mViewModules->GetMember(i)->SetImageComposer(mComposer);
				//
				//  Set the type of the instance created
				//
				if(vmtype==_ModuleTypeClone || vmtype==_ModuleTypeCopy)
				{
					mViewModules->GetMember(i)->CopyState(module);
					mViewModules->GetMember(i)->BecomeClone(module);
				}
				//
				//  If a copy, unclone
				//
				if(vmtype == _ModuleTypeCopy) mViewModules->GetMember(i)->BecomeClone(0);
				//
				//  Update composer
				//
				mComposer->UpdateWindowList();
				if(render) mViewModules->GetMember(i)->Render();
				int j;
				for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
				mViewModules->GetCurrentMember()->UpdateIcon();
			}
			return i != -1;
		}
	case _ObjectTypeMarker:
		{
			if(module->GetMarkerFamily()->GetMaxMemberIndex()==0 && !module->GetMarkerFamily()->IsVisible())
			{
				module->GetMarkerFamily()->Show(true);
			}
			else
			{
				ICONTROLMODULE_CREATE_OBJECT(Marker);
			}
			module->BuildMarkerLegend();
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeParticleGroup:
		{
			int i = module->GetParticlesFamily()->GetCurrentMember()->CreateGroup();
			if(i == -1) return false;
			module->ClearCache();
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeSurface:
		{
			ICONTROLMODULE_CREATE_OBJECT(Surface);
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeCrossSection:
		{
			ICONTROLMODULE_CREATE_OBJECT(CrossSection);
			if(render) module->Render();
			return true;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_ERROR("This object cannot be created.");
#endif
			return false;
		}
	}
}


bool iControlModule::DeleteObject(const iObjectType &type, bool render, int mod, int obj)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return false;

	switch(FindTypeFromString(type.FullName()))
	{
	case _ObjectTypeViewModule: 
		{
			int j;
			bool ok;
			//
			//  First check whether it is the only non-cloned one, then it cannot be deleted
			//
			if(!module->IsClone())
			{
				ok = true;
				for(j=0; ok && j<=mViewModules->GetMaxMemberIndex(); j++) if(!mViewModules->GetMember(j)->IsClone()) ok = false;
				if(ok) return false;
			}

			//
			//  If this is not a clone, delete all its clones
			//
			if(!module->IsClone())
			{
				while(module->mClones.Size() > 0)
				{
					if(!mViewModules->DeleteMember(module->mClones[0]->GetWindowNumber()))
					{
						for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
						mComposer->UpdateWindowList();
						return false;
					}
				}
			}
			//
			//  Now delete the window itself
			//
			if(mod == mViewModules->GetCurrentMemberIndex()) mInteractorObserver->SetDriver(0);

			ok = mViewModules->DeleteMember(mod);
			for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
			mComposer->UpdateWindowList();
		
			this->SetCurrentViewModuleIndex(-1);

			return ok;
		}
	case _ObjectTypeMarker:
		{
			if(module->GetMarkerFamily()->GetMaxMemberIndex()==0 && module->GetMarkerFamily()->IsVisible())
			{
				module->GetMarkerFamily()->Show(false);
			}
			else
			{
				ICONTROLMODULE_DELETE_OBJECT(Marker);
			}
			module->BuildMarkerLegend();
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeParticleGroup:
		{
			if(obj == -1) obj = module->GetParticlesFamily()->GetCurrentMember()->GetCurrentGroupIndex();
			if(!module->GetParticlesFamily()->GetCurrentMember()->DeleteGroup(obj)) return false;
			module->ClearCache();
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeSurface:
		{
			ICONTROLMODULE_DELETE_OBJECT(Surface);
			if(render) module->Render();
			return true;
		}
	case _ObjectTypeCrossSection:
		{
			ICONTROLMODULE_DELETE_OBJECT(CrossSection);
			if(render) module->Render();
			return true;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_ERROR("This object cannot be deleted.");
#endif
			return false;
		}
	}
}


bool iControlModule::SetCurrentViewModuleIndex(int mod)
{
	int oldcw = mViewModules->GetCurrentMemberIndex();

	if(mod>=-1 && mod<=mViewModules->GetMaxMemberIndex() && mod!=oldcw)
	{
		mViewModules->SetCurrentMemberIndex(mod);
		if(mSyncInteractors) mInteractorObserver->SetDriver(mViewModules->GetCurrentMember());
		mViewModules->GetMember(oldcw)->UpdateIcon();
		mViewModules->GetCurrentMember()->UpdateIcon();
		return true;
	}
	else return (mod == oldcw);
}


void iControlModule::Show(const iObjectType &type, bool show, int mod)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	switch(FindTypeFromString(type.FullName()))
	{
	case _ObjectTypeMarker:
		{
			module->GetMarkerFamily()->Show(show);
			break;
		}
	case _ObjectTypeParticles:
		{
			module->GetParticlesFamily()->GetCurrentMember()->Show(show); // down't show all members at once
			break;
		}
	case _ObjectTypeSurface:
		{
			module->GetSurfaceFamily()->Show(show);
			break;
		}
	case _ObjectTypeTensorField:
		{
			module->GetTensorFieldFamily()->Show(show);
			break;
		}
	case _ObjectTypeVectorField:
		{
			module->GetVectorFieldFamily()->Show(show);
			break;
		}
	case _ObjectTypeVolume:
		{
			module->GetVolumeFamily()->Show(show);
			break;
		}
	case _ObjectTypeCrossSection:
		{
			module->GetCrossSectionFamily()->Show(show);
			break;
		}
	default: return;  // return without rendering
	}
	module->Render();
}


float iControlModule::GetMemorySize(int mod) const
{
	int i;
	float s = 0.0;
	if(mod == -2)
	{
		for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++) s += mViewModules->GetMember(i)->GetMemorySize();
	}
	else if(mod>-2 && mod<=mViewModules->GetMaxMemberIndex())
	{
		if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
		mViewModules->GetMember(mod)->GetMemorySize();
	}
	return s;
}


float iControlModule::GetUpdateTime(int mod) const
{
	float s = 0.0;
	if(mod>-1 && mod<=mViewModules->GetMaxMemberIndex())
	{
		if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
		mViewModules->GetMember(mod)->GetUpdateTime();
	}
	return s;
}


void iControlModule::FlyTo(const iObjectType &type, float time, int mod, int obj) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	const double *pos;

	iViewObject *object = this->FindViewObject(type,mod,obj);
	if(object != 0)
	{
		pos = object->GetPosition();
	}
	else
	{
		if(type == iPicker::Type()) pos = module->GetPicker()->GetPosition(); else return;
	}

	this->FlyTo(pos,time,mod,obj);
}


void iControlModule::FlyTo(const double *pos, float time, int mod, int /*obj*/) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module==0 || pos==0) return;

	if(time > 1.0e-30)
	{
		//
		//  Gradual fly-to
		//
		int n = round(time/module->GetRenderer()->GetLastRenderTimeInSeconds());
		if(n > 60) n = 60;
		if(n < 10) n = 0;
		module->GetInteractor()->SetNumberOfFlyFrames(n);
		module->GetInteractor()->FlyTo(module->GetRenderer(),pos[0],pos[1],pos[2]);
	}
	else
	{
		//
		//  A jump
		//
		int j;
		vtkCamera *cam = module->GetRenderer()->GetActiveCamera();
		double cf[3], cp[3];

		cam->GetPosition(cp);
		cam->GetFocalPoint(cf);
		for(j=0; j<3; j++) cp[j] += (pos[j]-cf[j]);
		cam->SetFocalPoint(pos);
		cam->SetPosition(cp);
		module->Render();
	}
}


void iControlModule::PackStateBody(iString &s) const
{

	this->PackValue(s,KeyAutoRender(),mAutoRender);
	this->PackValue(s,KeySynchronizeInteractors(),mSyncInteractors);
	this->PackValue(s,KeyOptimizationMode(),this->GetOptimizationMode());
}


void iControlModule::UnPackStateBody(const iString &s)
{
	int i; bool b;

	if(this->UnPackValue(s,KeyAutoRender(),b)) this->SetAutoRender(b);
	if(this->UnPackValue(s,KeySynchronizeInteractors(),b)) this->SetSynchronizeInteractors(b);
	if(this->UnPackValue(s,KeyOptimizationMode(),i)) this->SetOptimizationMode(i);

	//
	//  Action keys
	//
#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //action keys are not part of the states
#endif

	if(this->UnPackValue(s,KeySynchronizeCameras(),b) && b) this->SynchronizeCameras();

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
#endif
}


void iControlModule::QueryState(const iObjectType &type, iString &state, int vis, int obj)
{ 
	iObject *o = this->FindObject(type,vis,obj);
	o->PackState(state);
}


//
//  Give access to piecewise functions directly for higher efficiency
//
bool iControlModule::QueryValue(const iObjectKey &key, iFunctionMapping* &f, int mod, int obj) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	switch(FindTypeFromString(key.Type().FullName()))
	{
	case _ObjectTypeParticleGroup:
		{
			if(obj == -1) obj = module->GetParticlesFamily()->GetCurrentMember()->GetCurrentGroupIndex();
			if(key == iParticleGroup::KeySizeFunction())
			{
				f = module->GetParticlesFamily()->GetCurrentMember()->GetGroup(obj)->GetSizeFunction();
				return true;
			}
		}
	case _ObjectTypeParticles:
		{
			if(obj == -1) obj = module->GetParticlesFamily()->GetCurrentMember()->GetCurrentGroupIndex();
			if(key == iParticlesViewObject::KeySizeFunction())
			{
				f = module->GetParticlesFamily()->GetCurrentMember()->GetGroup(obj)->GetSizeFunction();
				return true;
			}
		}
	case _ObjectTypeVolume:
		{
			if(obj == -1) obj = module->GetVolumeFamily()->GetCurrentMemberIndex();
			if(key == iVolumeViewObject::KeyOpacityFunction())
			{
				f = module->GetVolumeFamily()->GetMember(obj)->GetOpacityFunction();
				return true;
			}
		}
	}
	f = 0;
	return false;
}

//
//  Give access to range collections directly for higher efficiency
//
bool iControlModule::QueryValue(const iObjectKey &key, iRangeMapping* &f, int mod, int /*obj*/) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	switch(FindTypeFromString(key.Type().FullName()))
	{
	case _ObjectTypeParticles:
		{
			if(key == iParticlesViewObject::KeySplitRanges())
			{
				f = module->GetParticlesFamily()->GetCurrentMember()->GetSplitRanges();
				return true;
			}
		}
	}
	f = 0;
	return false;
}
//
//  Query images
//
bool iControlModule::QueryValue(const iObjectKey &key, iImage &im, int mod, int /*obj*/) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	switch(FindTypeFromString(key.Type().FullName()))
	{
	case _ObjectTypeAnimator:
		{
			if(key == iAnimator::KeyTitlePageImage())
			{
				im = module->GetAnimator()->GetTitlePageImage();
				return true;
			}
			if(key == iAnimator::KeyLogoImage())
			{
				im = module->GetAnimator()->GetLogoImage();
				return true;
			}
		}
	}
	return false;
}

//
//  Get any query as a portion of the state string
//
bool iControlModule::QueryValueAsString(const iObjectKey &key, iString &val, int mod, int obj)
{
	iObject *o = this->FindObject(key,mod,obj);
	if(o != 0)
	{
		iString ws;
		o->PackState(ws);
		int index, ind = FindUnPackKey(ws,key,index);
		if(ind >=0)
		{
			val = ws.Part(ind).Section(key.FieldSeparator(),1,1);
			return true;
		}
		else return false;
	}
	else return false;
}

//
//  generic QueryValue functions
//
#define ICONTROLMODULE_QUERYVALUE_DEFINITION(_type_) \
	bool iControlModule::QueryValue(const iObjectKey &key, _type_ *val, int n, int mod, int obj) \
	{ \
		iObject *o = this->FindObject(key,mod,obj); \
		if(o != 0) \
		{ \
			iString ws; \
			o->PackState(ws); \
			return o->UnPackValue(ws,key,val,n); \
		} \
		else return false; \
	} \
	bool iControlModule::QueryValue(const iObjectKey &key, int index, _type_ &val, int n, int mod, int obj) \
	{ \
		if(index<0 || index>=n) return false; \
		iObject *o = this->FindObject(key,mod,obj); \
		if(o != 0) \
		{ \
			iString ws; \
			o->PackState(ws); \
			_type_ *tmp = new _type_[n]; IERROR_ASSERT_NULL_POINTER(tmp); \
			bool ret = o->UnPackValue(ws,key,tmp,n); \
			val = tmp[index]; \
			delete [] tmp; \
			return ret; \
		} \
		else return false; \
	} 

//
//  Packing/unpacking helpers - mostly moved iObject helpers into public interface
//
#define ICONTROLMODULE_PACKCOMMAND_DEFINITION(_type1_,_type2_) \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, _type1_ *val, int n) const \
	{ \
		this->PackValue(s,key,val,n,-1,true); \
	} \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, _type2_ val) const \
	{ \
		this->PackValue(s,key,val,-1,true); \
	} \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, int index, _type2_ val) const \
	{ \
		this->PackValue(s,key,val,index,true); \
	}


ICONTROLMODULE_QUERYVALUE_DEFINITION(int);
ICONTROLMODULE_QUERYVALUE_DEFINITION(bool);
ICONTROLMODULE_QUERYVALUE_DEFINITION(float);
ICONTROLMODULE_QUERYVALUE_DEFINITION(double);
ICONTROLMODULE_QUERYVALUE_DEFINITION(iColor);
ICONTROLMODULE_QUERYVALUE_DEFINITION(iString);


ICONTROLMODULE_PACKCOMMAND_DEFINITION(int,int);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(bool,bool);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(float,float);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(double,double);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(iColor,iColor);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(iString,const iString&);


//
//  Environment querying
//
const iString& iControlModule::GetEnvironment(int key) const
{
	switch(key)
	{
	case _EnvironmentScalarFieldData:
		{
			return mEnvScalarFieldDataDir;
		}
	case _EnvironmentParticleSetData:
		{
			return mEnvParticleSetDataDir;
		}
	case _EnvironmentVectorFieldData:
		{
			return mEnvVectorFieldDataDir;
		}
	case _EnvironmentTensorFieldData:
		{
			return mEnvTensorFieldDataDir;
		}
	default:
		{
			return this->GetShell()->GetEnvironment(key);
		}
	}
}


iString iControlModule::GetFileName(int key, const iString &fname) const
{
	return this->GetEnvironment(key) + fname;
}


void iControlModule::UpdateDataDir()
{
	char *e;

	const iString &d(this->GetShell()->GetEnvironment(_EnvironmentData));

    mEnvScalarFieldDataDir = d;
    mEnvParticleSetDataDir = d;
    mEnvVectorFieldDataDir = d;
    mEnvTensorFieldDataDir = d;

#ifdef BACKWARD_COMPATIBLE
    e = getenv("IFRIT_MESH_DATA_DIR");          if(e != 0) mEnvScalarFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_PARTICLE_DATA_DIR");      if(e != 0) mEnvParticleSetDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_VECTOR_DATA_DIR");        if(e != 0) mEnvVectorFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_TENSOR_DATA_DIR");        if(e != 0) mEnvTensorFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
#endif

    e = getenv("IFRIT_SCALAR_FIELD_DATA_DIR");  if(e != 0) mEnvScalarFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_PARTICLE_SET_DATA_DIR");  if(e != 0) mEnvParticleSetDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_VECTOR_FIELD_DATA_DIR");  if(e != 0) mEnvVectorFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
    e = getenv("IFRIT_TENSOR_FIELD_DATA_DIR");  if(e != 0) mEnvTensorFieldDataDir = iString(e) + this->GetShell()->GetDirSeparator();
}

//
// Palette manipulation
//
void iControlModule::CreateDefaultPalettes()
{
	if(mPalList.Size() > 0) return;

	mPalList.Add(Palette_Rainbow());
	mPalList.Add(Palette_Temperature());
	mPalList.Add(Palette_Bluewhite());
	mPalList.Add(Palette_Prizm());
	mPalList.Add(Palette_Starlight());
	mPalList.Add(Palette_Greenwhite());
	mPalList.Add(Palette_Bluered());
	mPalList.Add(Palette_Haze());
	mPalList.Add(Palette_Stern());
	mPalList.Add(Palette_Greyscale());
	mPalList.Add(Palette_3color());
	mPalList.Add(Palette_4color());
	
	mPalList.Add(Palette_Rainbow());     mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Temperature()); mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Bluewhite());   mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Prizm());       mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Starlight());   mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Greenwhite());  mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Bluered());     mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Haze());        mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Stern());       mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
	mPalList.Add(Palette_Greyscale());   mPalList[mPalList.Last()]->Reverse(); mPalList[mPalList.Last()]->SetName(mPalList[mPalList.Last()]->GetName()+" reversed");
}


iPalette* iControlModule::GetPalette(int n) const
{
	return mPalList[n];
}


int iControlModule::GetNumberOfPalettes() const
{ 
	return mPalList.Size(); 
}


bool iControlModule::AddEmptyPalette()
{
	iPalette *tmp = new iPalette; if(tmp == 0) return false;
	mPalList.Add(tmp);
	return true;
}


//
//  Saving and restoring the state
//
bool iControlModule::SaveStateToFile(const iString &filename) const
{
    int i, k;
	iString prefix = "";
	iString state, ws;
	iFile F(filename);
	
	if(!F.Open(iFile::_Write,iFile::_Text)) return false;
	//
	//  Compatibility mode
	//
	if(!F.WriteLine("CompatibilityMode "+iString::FromNumber(COMPATIBILITY_MODE))) { F.Close(); return false; }
	//
	//  Save the state of this ControlModule
	//
	this->PackCompleteState(state);
	ws = prefix + iControlModule::Type().FullName() + " " + state;
	if(!F.WriteLine(ws)) { F.Close(); return false; }
	//
	//  Save ViewModules with their window geometries, animators, data readers, limits,
	//  particle splitters, and all view objects
	//
	for(k=0; k<=mViewModules->GetMaxMemberIndex(); k++) 
	{
		//
		//  Visual module location
		//
		ws = prefix + iViewModule::Type().FullName() + " " + iString::FromNumber(k) + " " + iString::FromNumber(mViewModules->GetMember(k)->GetRenderWindow()->GetPosition()[0]) + " " + iString::FromNumber(mViewModules->GetMember(k)->GetRenderWindow()->GetPosition()[1]) + " ";
		//
		//  Visual module parameters
		//
		mViewModules->GetMember(k)->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Animator
		//
		ws = prefix + iAnimator::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetAnimator()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Camera
		//
		ws = prefix + iCamera::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetRenderTool()->GetCamera()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Color Bars
		//
		ws = prefix + iColorBars::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetColorBars()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Data reader
		//
		ws = prefix + iDataReader::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetReader()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Picker
		//
		ws = prefix + iPicker::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetPicker()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) { F.Close(); return false; }
		//
		//  Save multiple instances of view objects
		//
		ICONTROLMODULE_SAVE_OBJECT(Particles);
		ICONTROLMODULE_SAVE_OBJECT(Surface);
		ICONTROLMODULE_SAVE_OBJECT(TensorField);
		ICONTROLMODULE_SAVE_OBJECT(Volume);
		ICONTROLMODULE_SAVE_OBJECT(VectorField);
		ICONTROLMODULE_SAVE_OBJECT(CrossSection);
		//
		//  Don't save single invisible marker
		//
		if(mViewModules->GetMember(k)->GetMarkerFamily()->GetMaxMemberIndex()>0 || mViewModules->GetMember(k)->GetMarkerFamily()->GetMember(0)->IsVisible()) ICONTROLMODULE_SAVE_OBJECT(Marker);
	}
	//
	//  Save the state of ImageComposer
	//
	mComposer->PackCompleteState(state);
	ws = prefix + iImageComposer::Type().FullName() + " " + state;
	if(!F.WriteLine(ws)) { F.Close(); return false; }
	//
	//  Save the palettes
	//
	for(i=0; i<mPalList.Size(); i++)
	{
		this->PackValuePalette(ws,mPalList[i]);
		ws = prefix + "Palette " + iString::FromNumber(i) + " " + ws;
		if(!F.WriteLine(ws)) 
		{
			F.Close();
			return false;
		}
	}
	
	if(!this->SaveExtensionStateToFile(prefix,F))
	{
		return false;
	}

	bool ret = this->GetShell()->SaveShellStateToFile(F);

	F.Close();	
	return ret;
}


bool iControlModule::LoadStateFromFile(const iString &filename)
{
	bool ok;
	int k, joff, i, i1, i2;
	int CompatibilityMode = 0;
	iString line, ws;
	iFile F(filename);
	
	if(!F.Open(iFile::_Read,iFile::_Text)) return false;
	
	//
	// ***************** Pre-Pass ****************************
	//
	//  Find the Compatibility parameter and restore this control module
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == "CompatibilityMode")
		{
			i = line.Section(" ",1,1).ToInt(ok);
			if(!ok || i<10 || i>COMPATIBILITY_MODE) { F.Close(); return false; }
			CompatibilityMode = i;
			break;
		}

		if(ws == iControlModule::Type().FullName()) 
		{
			joff = 1; 
			ICONTROLMODULE_UNPACK_STATE(this);
		}
	}
	//
	// ***************** Pass 1 ****************************
	//
	//  Count the number of view modules and create them with default Settings. 
	//
	F.Rewind();
	int nvm = 0, kvmMax = 0;
	while(F.ReadLine(line))
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iViewModule::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k > kvmMax) kvmMax = k;
			if(nvm > 0) mViewModules->CreateMember();
			nvm++;
		}
	}
	
	if(kvmMax+1 != nvm) { F.Close(); return false; }
	//
	//  set all window numbers
	//
	if(nvm > 1)
	{
		for(k=0; k<nvm; k++)
		{
			mViewModules->GetMember(k)->UpdateWindowNumber();
		}
	}
	//
	// ***************** Pass 2 ****************************
	//
	//  Restore view modules
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iViewModule::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				//
				//  Visual module position
				//
				i1 = line.Section(" ",2,2).ToInt(ok);
				if(!ok) { F.Close(); return false; }
				i2 = line.Section(" ",3,3).ToInt(ok);
				if(!ok) { F.Close(); return false; }
				mViewModules->GetMember(k)->GetRenderWindow()->SetPosition(i1,i2);
				//
				//  Visual module parameters
				//
				joff = 4;
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k));
			}
		}
	}
	//	
	// ***************** Pass 3 ****************************
	//
	//  Restore animators, color bars, data reader, and limits
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iAnimator::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				joff = 2; 
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k)->GetAnimator());
			}
		}		
	
		if(ws == iCamera::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				joff = 2; 
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k)->GetRenderTool()->GetCamera());
			}
		}		
		
		if(ws == iColorBars::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				joff = 2; 
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k)->GetColorBars());
			}
		}		
		
		if(ws == iDataReader::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				joff = 2; 
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k)->GetReader());
			}
		}		

		if(ws == iPicker::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			if(k>=0 && k<nvm)
			{		
				joff = 2; 
				ICONTROLMODULE_UNPACK_STATE(mViewModules->GetMember(k)->GetPicker());
			}
		}		
	}
	//	
	// ***************** Pass 4 ****************************
	//
	//  Restore view objects, palettes, and the image composer
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		ICONTROLMODULE_LOAD_OBJECT(Marker,true);
		ICONTROLMODULE_LOAD_OBJECT(Particles,false);
		ICONTROLMODULE_LOAD_OBJECT(Surface,false);
		ICONTROLMODULE_LOAD_OBJECT(TensorField,false);
		ICONTROLMODULE_LOAD_OBJECT(VectorField,false);
		ICONTROLMODULE_LOAD_OBJECT(Volume,false);
		ICONTROLMODULE_LOAD_OBJECT(CrossSection,false);
		
		if(ws == "Palette") 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) { F.Close(); return false; }
			ok = true;
			if(k<0 || k>=100000) ok = false;
			while(k>=mPalList.Size() && ok)
			{
				ok = this->AddEmptyPalette();
			}
			if(ok)
			{
				ws = line.Section(" ",2);
				if(!this->UnPackValuePalette(ws,mPalList[k])) { F.Close(); return false; }
			}
		}		

		if(ws == iImageComposer::Type().FullName()) 
		{
			joff = 1; 
			ICONTROLMODULE_UNPACK_STATE(mComposer);
		}
	}

	for(k=0; k<=mViewModules->GetMaxMemberIndex(); k++) 
	{
		mViewModules->GetMember(k)->GetParticlesFamily()->Show(false);
		mViewModules->GetMember(k)->GetSurfaceFamily()->Show(false);
		mViewModules->GetMember(k)->GetTensorFieldFamily()->Show(false);
		mViewModules->GetMember(k)->GetVectorFieldFamily()->Show(false);
		mViewModules->GetMember(k)->GetVolumeFamily()->Show(false);
		mViewModules->GetMember(k)->GetCrossSectionFamily()->Show(false);
	}

	if(!this->LoadExtensionStateFromFile(F,nvm))
	{
		return false;
	}

	bool ret = this->GetShell()->LoadShellStateFromFile(F);

	F.Close();
	return ret;
}


void iControlModule::PackValuePalette(iString &s, const iPalette *p) const
{
	iViewObject *tmpObject = mViewModules->GetMember(0)->GetSurfaceFamily()->GetMember(0);
	const iObjectKey& keyRed = iObjectKeyRegistry::GetTempKey(Type(),"Red");
	const iObjectKey& keyGreen = iObjectKeyRegistry::GetTempKey(Type(),"Green");
	const iObjectKey& keyBlue = iObjectKeyRegistry::GetTempKey(Type(),"Blue");

	s = p->GetName().Substitute(" ","*") + iObjectKey::FieldSeparator();
	tmpObject->PackValuePiecewiseFunction(s,keyRed,p->GetRedComponent());
	tmpObject->PackValuePiecewiseFunction(s,keyGreen,p->GetGreenComponent());
	tmpObject->PackValuePiecewiseFunction(s,keyBlue,p->GetBlueComponent());

	iObjectKeyRegistry::ReleaseTempKey(keyRed);
	iObjectKeyRegistry::ReleaseTempKey(keyGreen);
	iObjectKeyRegistry::ReleaseTempKey(keyBlue);
}


bool iControlModule::UnPackValuePalette(const iString &s, iPalette *p)
{
	bool ok = true;

	iViewObject *tmpObject = mViewModules->GetMember(0)->GetSurfaceFamily()->GetMember(0);
	const iObjectKey& keyRed = iObjectKeyRegistry::GetTempKey(Type(),"Red");
	const iObjectKey& keyGreen = iObjectKeyRegistry::GetTempKey(Type(),"Green");
	const iObjectKey& keyBlue = iObjectKeyRegistry::GetTempKey(Type(),"Blue");

	iString ws;
	ws = s.Section(iObjectKey::FieldSeparator(),0,0);
	ws.Replace("*"," ");
	if(!ws.IsEmpty()) p->SetName(ws); else ok = false;

	if(ok)
	{
		//
		//  Loop over 3 color functions
		//
		ws = s.Section(iObjectKey::FieldSeparator(),1);
		iPiecewiseFunction *r = p->GetRedComponent();
		iPiecewiseFunction *g = p->GetGreenComponent();
		iPiecewiseFunction *b = p->GetBlueComponent();

		ok = tmpObject->UnPackValuePiecewiseFunction(ws,keyRed,r) && 
		 	tmpObject->UnPackValuePiecewiseFunction(ws,keyGreen,g) &&
			tmpObject->UnPackValuePiecewiseFunction(ws,keyBlue,b);
	}

	iObjectKeyRegistry::ReleaseTempKey(keyRed);
	iObjectKeyRegistry::ReleaseTempKey(keyGreen);
	iObjectKeyRegistry::ReleaseTempKey(keyBlue);

	return ok;
}


bool iControlModule::SaveExtensionStateToFile(const iString&, iFile&) const
{
	return true;
}


bool iControlModule::LoadExtensionStateFromFile(iFile&, int)
{
	return true;
}


int iControlModule::GetNumberOfViewModules() const
{
	return 1 + mViewModules->GetMaxMemberIndex();
}


int iControlModule::GetCurrentViewModuleIndex() const
{
	return mViewModules->GetCurrentMemberIndex();
}


iViewModule* iControlModule::GetViewModule() const
{
	return mViewModules->GetCurrentMember();
}


iViewModule* iControlModule::GetViewModule(int mod) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	return mViewModules->GetMember(mod);
}


iAnimatorScript* iControlModule::GetAnimatorScript(int mod) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0; else return module->GetAnimator()->GetScript();
}


void iControlModule::SynchronizeCameras(int mod)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	int i;
	vtkCamera *c = module->GetRenderer()->GetActiveCamera();
	double *fp = c->GetFocalPoint();
	double *cp = c->GetPosition();
	double *vu = c->GetViewUp();
	double ps = c->GetParallelScale();
	for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++)
	{
		c = mViewModules->GetMember(i)->GetRenderer()->GetActiveCamera();
		c->SetFocalPoint(fp);
		c->SetPosition(cp);
		c->SetViewUp(vu);
		c->SetParallelScale(ps);
		mViewModules->GetMember(i)->Render();
	}
}

//
//  We add/remove observers so that in the non-sync mode no extra callbacks take place.
//
void iControlModule::SetSynchronizeInteractors(bool s)
{
	mSyncInteractors = s; 
	if(mSyncInteractors) 
	{
		mInteractorObserver->SetDriver(mViewModules->GetCurrentMember());
	}
	else
	{
		mInteractorObserver->SetDriver(0);
	}
	this->ClearCache();
}


void iControlModule::SetOptimizationMode(int m)
{
	iDataConsumer::SetGlobalOptimizationMode(m);
	this->ClearCache();
	//
	//  We need to reset everything
	//
	int j;
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->GetReader()->ResetPipeline();
}


int iControlModule::GetOptimizationMode() const
{
	return iDataConsumer::GetGlobalOptimizationMode();
}


//
//  Helper functions hidden in a private namespace
//
namespace iControlModule_Private
{
	iPalette* Palette_Rainbow()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Rainbow");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.1);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.9);
		b->MovePoint(1,1.0,0.9);

		r->AddPoint(1.0/6.0,0.0);
		g->AddPoint(1.0/6.0,0.0);
		b->AddPoint(1.0/6.0,1.0);

		r->AddPoint(2.0/6.0,0.0);
		g->AddPoint(2.0/6.0,1.0);
		b->AddPoint(2.0/6.0,1.0);

		r->AddPoint(3.0/6.0,0.0);
		g->AddPoint(3.0/6.0,1.0);
		b->AddPoint(3.0/6.0,0.0);

		r->AddPoint(4.0/6.0,1.0);
		g->AddPoint(4.0/6.0,1.0);
		b->AddPoint(4.0/6.0,0.0);

		r->AddPoint(5.0/6.0,1.0);
		g->AddPoint(5.0/6.0,0.0);
		b->AddPoint(5.0/6.0,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Temperature()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Temperature");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.3);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.8);
		b->MovePoint(1,1.0,0.8);

		r->AddPoint(0.7,1.0);
		g->AddPoint(0.6,0.0);
		b->AddPoint(0.8,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Greyscale()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Greyscale");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Bluewhite()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Blue-white");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.1);

		r->MovePoint(1,1.0,0.9);
		g->MovePoint(1,1.0,0.9);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.75,0.0);
		g->AddPoint(0.38,0.0);
		b->AddPoint(0.78,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Prizm()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Prizm");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.25,1.0);
		g->AddPoint(0.25,0.0);
		b->AddPoint(0.25,0.0);

		r->AddPoint(0.50,0.0);
		g->AddPoint(0.50,1.0);
		b->AddPoint(0.50,0.0);

		r->AddPoint(0.75,0.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Greenwhite()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Green-white");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.1);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.375,0.0);
		b->AddPoint(0.750,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Bluered()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Blue-red");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.25,0.0);
		g->AddPoint(0.25,1.0);
		b->AddPoint(0.25,1.0);

		r->AddPoint(0.5,0.0);
		g->AddPoint(0.5,0.0);
		b->AddPoint(0.5,1.0);

		r->AddPoint(0.75,1.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Stern()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Stern");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.05,1.0);
		r->AddPoint(0.25,0.0);
		b->AddPoint(0.50,1.0);
		b->AddPoint(0.75,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Haze()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Haze");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.8);
		b->MovePoint(0,0.0,1.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.8);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.5,0.0);
		g->AddPoint(0.5,0.1);
		b->AddPoint(0.5,0.5);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Starlight()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("Starlight");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.5);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,0.7);

		r->AddPoint(0.5,0.9);
		g->AddPoint(0.5,0.7);
		b->AddPoint(0.5,0.2);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_3color()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("3 color");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.3333,1.0);
		g->AddPoint(0.3333,0.0);
		b->AddPoint(0.3333,0.0);
		r->AddPoint(0.3334,0.0);
		g->AddPoint(0.3334,1.0);
		b->AddPoint(0.3334,0.0);

		r->AddPoint(0.6666,0.0);
		g->AddPoint(0.6666,1.0);
		b->AddPoint(0.6666,0.0);
		r->AddPoint(0.6667,0.0);
		g->AddPoint(0.6667,0.0);
		b->AddPoint(0.6667,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_4color()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT_NULL_POINTER(tmp);
		tmp->SetName("4 color");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.24,1.0);
		g->AddPoint(0.24,0.0);
		b->AddPoint(0.24,0.0);
		r->AddPoint(0.25,1.0);
		g->AddPoint(0.25,1.0);
		b->AddPoint(0.25,0.0);

		r->AddPoint(0.49,1.0);
		g->AddPoint(0.49,1.0);
		b->AddPoint(0.49,0.0);
		r->AddPoint(0.50,0.0);
		g->AddPoint(0.50,1.0);
		b->AddPoint(0.50,0.0);

		r->AddPoint(0.74,0.0);
		g->AddPoint(0.74,1.0);
		b->AddPoint(0.74,0.0);
		r->AddPoint(0.75,0.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}
};

