/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ibasicparticlesdatasubject.h"


#include "ibuffer.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ifile.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "isystem.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


IDATASUBJECT_DEFINE_TYPE(iBasicParticlesDataSubject,BasicParticles,bp);

//
//  Inherited keys
//
IDATASUBJECT_DEFINE_INHERITED_KEYS(iBasicParticlesDataSubject);


//
//  Main class
//
iBasicParticlesDataSubject::iBasicParticlesDataSubject(iDataReader *r) : iGenericParticlesDataSubject(r,iDataType::BasicParticles())
{
	mOrderInFileIsAttribute = false;

	mInternalData = mData = vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(mInternalData);
	mExternalData =         vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(mExternalData);
}


iBasicParticlesDataSubject::~iBasicParticlesDataSubject()
{
}


void iBasicParticlesDataSubject::ReadFileBody(const iString &suffix, const iString &fname)
{
	static iBuffer<float> amin;
	static iBuffer<float> amax;
	//
	//  is the suffix valid?
	//
	if(suffix.Lower()!="txt" && suffix.Lower()!="bin")
	{
		this->GetErrorStatus()->Set("Incorrect file suffix.");
		return;
	}

	bool err, isBin = (suffix.Lower() == "bin");

	//
	//  Open the file
	//
	iFile F(fname);
	if(!F.Open(iFile::_Read,isBin?iFile::_Binary:iFile::_Text))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);

	//
	//  Read the header
	//
	int i, natt;
	bool paf;
	vtkIdType ntot, ntot0;
	float ll[3], ur[3];
	iString buffer;
	if(isBin)
	{
		err = !this->ReadBinFileHeader(F,ntot,ll,ur,paf,natt);
	}
	else
	{
		err = !this->ReadTxtFileHeader(F,ntot,ll,ur,paf,natt,buffer);
	}
	if(ntot <= 0) err = true;

	if(err)
	{
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	//
	//  Compute scale and offset
	//
	float offsetF[3], scaleF[3];
	double offsetD[3], scaleD[3];
	if(paf)
	{
		for(i=0; i<3; i++)
		{
			offsetF[i] = ll[i];
			scaleF[i] = 2.0/(ur[i]-ll[i]);
		}
	}
	else
	{
		for(i=0; i<3; i++)
		{
			offsetD[i] = ll[i];
			scaleD[i] = 2.0/(ur[i]-ll[i]);
		}
	}

	//
	//  Determine the number of attributes
	//
	int iattOff = 0;
	if(mOrderInFileIsAttribute)
	{
		iattOff = 1;
		natt++;
	}
	//
	//  Try to set all the records from the file. DataLimits will either expand to accommodate 
	//  all of them or limit the allowed number to the number of listed records.
	//
	this->GetLimits()->AssignVars(natt);
	natt = this->GetLimits()->GetNumVars();
	mObserver->SetProgress(0.01);

	//
	//  Build particle reading mask
	//
	ntot0 = ntot;
	ntot = this->PrepareMask(ntot0,mDownsampleFactor);

	//
	//  Allocate memory; erase the old data if needed.
	//
	vtkPoints *newPoints, *oldPoints = mData->GetPoints();
	vtkCellArray *newVerts, *oldVerts = mData->GetVerts();
	vtkFloatArray *newScalars, *oldScalars = iPointerCast<vtkFloatArray,vtkDataArray>(mData->GetPointData()->GetScalars());

	int type = paf ? VTK_FLOAT : VTK_DOUBLE;

	if(oldPoints==0 || oldPoints->GetNumberOfPoints()!=ntot || oldPoints->GetDataType()!=type)
	{
		mData->SetPoints(0);
		newPoints = vtkPoints::New(type); IERROR_ASSERT_NULL_POINTER(newPoints);
		// Allocates and Sets MaxId
		newPoints->SetNumberOfPoints(ntot);
	}
	else
	{
		newPoints = oldPoints;
		newPoints->Register(0);
	}

	if(oldVerts==0 || oldVerts->GetNumberOfCells()!=ntot)
	{
		mData->SetVerts(0);
		newVerts = vtkCellArray::New(); IERROR_ASSERT_NULL_POINTER(newVerts);
		// This allocates but does not Set Max Id
		newVerts->Allocate(newVerts->EstimateSize(ntot,1));
		vtkIdType l;
		for(l=0; l<ntot; l++)
		{
			newVerts->InsertNextCell(1);
			newVerts->InsertCellPoint(l);
		}
	}
	else
	{
		newVerts = oldVerts;
		newVerts->Register(0);
	}

	if(oldScalars==0 || oldScalars->GetNumberOfTuples()!=ntot || oldScalars->GetNumberOfComponents()!=natt)
	{
		mData->GetPointData()->SetScalars(0);
		newScalars = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(newScalars);
		newScalars->SetNumberOfComponents(natt);
		newScalars->SetNumberOfTuples(ntot);
	}
	else
	{
		newScalars = oldScalars;
		newScalars->Register(0);
	}

	//
	//  Attribute limits.
	//
	for(i=0; i<natt; i++)
	{
		amin[i] =  iMath::_LargeFloat;
		amax[i] = -iMath::_LargeFloat;
	}

	if(isBin)
	{
		err = !this->ReadBinFileContents(F,paf,iattOff,natt,ntot,ntot0,newPoints,newScalars,amin,amax,scaleF,offsetF,scaleD,offsetD);
	}
	else
	{
		err = !this->ReadTxtFileContents(F,paf,iattOff,natt,ntot,ntot0,newPoints,newScalars,amin,amax,scaleF,offsetF,scaleD,offsetD,buffer);
	}

	if(err || mObserver->IsAborted())
	{
		mObserver->Finished();
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		F.Close();
		if(!err) this->GetErrorStatus()->Set("Aborted."); else this->GetErrorStatus()->Set("Corrupted data.");
		return;
	}
	
	if(mOrderInFileIsAttribute)
	{
		amin[0] = 0.0;
		amax[0] = ntot - 1.0;
	}

	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();

	if(mAdjustableLimits)
	{
		for(i=0; i<natt; i++)
		{
			this->GetLimits()->SetMin(i,amin[i]);
			this->GetLimits()->SetMax(i,amax[i]);
		}
	}

	//
	//  Finish updating
	//
	mData->SetPoints(newPoints);
	newPoints->Delete();

	mData->SetVerts(newVerts);
	newVerts->Delete();

	mData->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();
}


bool iBasicParticlesDataSubject::ReadBinFileHeader(iFile &F, vtkIdType &ntot, float *ll, float *ur, bool &paf, int &natt)
{
	//
	//  Read the header
	//
	if(!this->DetectFileStructureFromFirstRecord(F,sizeof(int))) return false;

	int ntot1;
	if(!this->ReadFortranRecord(F,Buffer(ntot1),0.0f,0.0f)) return false;
	ntot = ntot1;

	float bounds[6];
	if(!this->ReadFortranRecord(F,Buffer(bounds,6),0.0f,0.0f)) return false;

	int i;
	for(i=0; i<3; i++)
	{
		ll[i] = bounds[i];
		ur[i] = bounds[3+i];
	}

	//
	//  Auto-detect whether points are float or double
	//
	long nrec = sizeof(float)*ntot;
	int mar = F.SetMarker();
	if(!this->SkipFortranRecord(F,nrec))
	{
		//
		//  not float - try double
		//
		paf = false;
		nrec = sizeof(double)*ntot;
		F.ReturnToMarker(mar);
		if(!this->SkipFortranRecord(F,nrec)) return false;
	}
	else
	{
		paf = true;
	}
	F.ReturnToMarker(mar);
	//
	//  Measure the file size to find out the number of attributes
	//
	for(i=0; i<999 && this->SkipFortranRecord(F,nrec); i++);
	F.ReturnToMarker(mar,true);
	natt = i - 3;

	return true;
}



bool iBasicParticlesDataSubject::ReadTxtFileHeader(iFile &F, vtkIdType &ntot, float *ll, float *ur, bool &paf, int &natt, iString &buffer)
{
	iString s;

	//
	//  TxT file coordinates are float - do we need to check whether they are long enough to be double?
	//
	paf = true;

	//
	// First line
	//
	if(!F.ReadLine(s)) return false;
	int ret = sscanf(s.ToCharPointer(),"%ld",&ntot);
	if(ret != 1) return false;
	if(ntot <= 0) return false;

	//
	//  Second line
	//
	if(!F.ReadLine(s)) return false;

	char *axisName[3];
	int i;
	for(i=0; i<3; i++) 
	{
		axisName[i] = 0;
		axisName[i] = new char[8192]; if(axisName[i] == 0) break;
	}
	if(i < 3 )
	{
		for(i=0; i<3; i++) if(axisName[i] != 0) delete [] axisName[i];
		return false;
	}

	ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %s %s %s",&ll[0],&ll[1],&ll[2],&ur[0],&ur[1],&ur[2],axisName[0],axisName[1],axisName[2]);
	if(ret == 9) this->GetViewModule()->SetAxesBox(axisName[0],axisName[1],axisName[2],ll[0],ur[0],ll[1],ur[1],ll[2],ur[2]);
	for(i=0; i<3; i++) if(axisName[i] != 0) delete [] axisName[i];

	if(ret!=6 && ret!=9) return false;

	//
	//  Find out the number of attributes
	//
	if(!F.ReadLine(s)) return false;

	double xyz[3];
	float f[10];
	ret = sscanf(s.ToCharPointer(),"%lg %lg %lg %g %g %g %g %g %g %g %g %g",&xyz[0],&xyz[1],&xyz[2],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<3 || ret>12) return false;

	natt = ret - 3;
	buffer = s;

	return true;
}


bool iBasicParticlesDataSubject::ReadBinFileContents(iFile &F, bool paf, int iattOff, int natt, vtkIdType ntot, vtkIdType ntot0, vtkPoints *newPoints, vtkFloatArray *newScalars, float *amin, float *amax, float *scaleF, float *offsetF, double *scaleD, double *offsetD)
{
	//
	//  Create tmp arrays
	//
	float *xptrF = 0;
	double *xptrD = 0;

	if(paf)
	{
		xptrF = (float *)newPoints->GetVoidPointer(0);
	}
	else
	{
		xptrD = (double *)newPoints->GetVoidPointer(0);
	}
	float *sptr = (float *)newScalars->GetVoidPointer(0);

	if((xptrF==0&&xptrD==0) || sptr==0)
	{
		this->GetErrorStatus()->Set("Not enough memory to create the data.");
		return false;
	}

	vtkIdType l, idm, idt;
	vtkIdType lpiece1, lpiece = 1000;
	int j, n;
	long lrec1, lrec2;
	float *df = 0;
	double *dd = 0;
	if(paf)
	{
		df = new float[lpiece]; if(df == 0) 
		{ 
			this->GetErrorStatus()->Set("Not enough memory to create the data.");
			return false;
		}
	}
	else
	{
		dd = new double[lpiece]; if(dd == 0) 
		{ 
			this->GetErrorStatus()->Set("Not enough memory to create the data.");
			return false;
		}
	}

	int npieces = (ntot0+lpiece-1)/lpiece;

	//
	//  parameters for the Progress Bar
	//
	float prog1 = 0.99/(natt+3);
	float updateStart, updateDuration = prog1/npieces;

	//
	//  Read coordinates
	//
	bool err = false;
	for(n=0; !err && n<3; n++)
	{
		updateStart = 0.01 + prog1*n;
		//
		//  Read piece by piece
		//
		err = !this->ReadFortranHeaderFooter(F,lrec1);
		if(err || mObserver->IsAborted()) break;
		
		bool addAttr = (n==2 && mOrderInFileIsAttribute);
		idm = idt = 0;
		for(j=0; j<npieces; j++)
		{
			if(j < npieces-1)
			{
				lpiece1 = lpiece;
			}
			else
			{
				//
				//  Correct for the last record
				//
				lpiece1 = ntot0 - j*lpiece;
			}
			if(paf)
			{
				err = !this->ReadBlock(F,df,lpiece1,updateStart+j*updateDuration,updateDuration);
			}
			else
			{
				err = !this->ReadBlock(F,dd,lpiece1,updateStart+j*updateDuration,updateDuration);
			}
			if(err || mObserver->IsAborted()) break;

			for(l=0; l<lpiece1; l++, idt++) if(this->IsMasked(idm,idt))
			{
#ifdef I_CHECK1
				if(idm>=ntot || idt>=ntot0)
				{
					IERROR_REPORT_BUG;
				}
#endif
				if(paf)
				{
					xptrF[n+3*idm] = -1.0 + scaleF[n]*(df[l]-offsetF[n]);
				}
				else
				{
					xptrD[n+3*idm] = -1.0 + scaleD[n]*(dd[l]-offsetD[n]);
				}
				if(addAttr)
				{
					sptr[natt*idm] = idm;
				}
				idm++;
			}
		}

		if(err || mObserver->IsAborted()) break;
	
		err = !this->ReadFortranHeaderFooter(F,lrec2); 
		if(lrec1!=lrec2 || lrec1!=(paf?sizeof(float):sizeof(double))*ntot0) err = true;
	}

	if(dd != 0)
	{
		delete [] dd;
		dd = 0;
	}

	if(err || mObserver->IsAborted())
	{
		if(df != 0) delete [] df;
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return false;
	}
	
	if(df == 0)
	{
		df = new float[lpiece]; if(df == 0) 
		{	 
			this->GetErrorStatus()->Set("Not enough memory to create the data.");
			return false;
		}
	}

	for(n=iattOff; !err && n<natt; n++) 
	{
		updateStart = 0.01 + prog1*(n+3-iattOff);
		//
		//  Read piece by piece
		//
		err = !this->ReadFortranHeaderFooter(F,lrec1); 
		if(err || mObserver->IsAborted()) break;

		idm = idt = 0;
		for(j=0; j<npieces; j++)
		{
			if(j < npieces-1)
			{
				lpiece1 = lpiece;
			}
			else
			{
				//
				//  Correct for the last record
				//
				lpiece1 = ntot0 - j*lpiece;
			}
			err = !this->ReadBlock(F,df,lpiece1,updateStart+j*updateDuration,updateDuration);
			if(err || mObserver->IsAborted()) break;

			for(l=0; l<lpiece1; l++, idt++) if(this->IsMasked(idm,idt))
			{
#ifdef I_CHECK1
				if(idm>=ntot || idt>=ntot0)
				{
					IERROR_REPORT_BUG;
				}
#endif
				sptr[n+natt*idm] = df[l];
				amin[n] = (amin[n]>df[l]) ? df[l] : amin[n];
				amax[n] = (amax[n]<df[l]) ? df[l] : amax[n];
				idm++;
			}
		}

		if(err || mObserver->IsAborted()) break;
	
		err = !this->ReadFortranHeaderFooter(F,lrec2); 
		if(lrec1!=lrec2 || lrec1!=sizeof(float)*ntot0) err = true;
	}
	delete[] df;

	if(err || mObserver->IsAborted())
	{
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return false;
	}

	return true;
}


bool iBasicParticlesDataSubject::ReadTxtFileContents(iFile &F, bool paf, int iattOff, int natt, vtkIdType ntot, vtkIdType ntot0, vtkPoints *newPoints, vtkFloatArray *newScalars, float *amin, float *amax, float *scaleF, float *offsetF, double *scaleD, double *offsetD, iString buffer)
{
	//
	//  Use the buffer
	//
	double xyz[3], xyzD[3];
	float xyzF[3], f[10];
	vtkIdType idm, idt;
	float *fOff = f + 1 - iattOff;

	idm = idt = 0;
	int i, ret;
	if(this->IsMasked(idm,idt))
	{
#ifdef I_CHECK1
		if(idm>=ntot || idt>=ntot0)
		{
			IERROR_REPORT_BUG;
		}
#endif
		ret = sscanf(buffer.ToCharPointer(),"%lg %lg %lg %g %g %g %g %g %g %g %g %g",&xyz[0],&xyz[1],&xyz[2],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret+iattOff < natt+3)
		{
			this->GetErrorStatus()->Set("Corrupted data.");
			return false;
		}
		if(paf)
		{
			for(i=0; i<3; i++) xyzF[i] = -1.0 + scaleF[i]*(xyz[i]-offsetF[i]);
			newPoints->SetPoint(idm,xyzF);
		}
		else
		{
			for(i=0; i<3; i++) xyzD[i] = -1.0 + scaleD[i]*(xyz[i]-offsetD[i]);
			newPoints->SetPoint(idm,xyzD);
		}
		if(mOrderInFileIsAttribute)
		{
			f[0] = 0.0;
		}
		newScalars->SetTuple(idm,fOff);
		idm++;
	}
	idt++;

	vtkIdType l;
	iString s;
	for(l=1; l<ntot0; l++, idt++) 
	{
		if(!F.ReadLine(s))
		{
			this->GetErrorStatus()->Set("Truncated file.");
			return false;
		}
		ret = sscanf(s.ToCharPointer(),"%lg %lg %lg %g %g %g %g %g %g %g %g %g",&xyz[0],&xyz[1],&xyz[2],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret+iattOff < natt+3)
		{
			this->GetErrorStatus()->Set("Corrupted data.");
			return false;
		}
		if((100*l)/ntot0 < (100*(l+1))/ntot0) 
		{
			mObserver->SetProgress(0.01+(float)l/ntot0);
			if(mObserver->IsAborted())
			{
				this->GetErrorStatus()->SetAbort();
				return false;
			}
		}
		if(this->IsMasked(idm,idt))
		{
#ifdef I_CHECK1
			if(idm>=ntot || idt>=ntot0)
			{
				IERROR_REPORT_BUG;
			}
#endif
			if(paf)
			{
				for(i=0; i<3; i++) xyzF[i] = -1.0 + scaleF[i]*(xyz[i]-offsetF[i]);
				newPoints->SetPoint(idm,xyzF);
			}
			else
			{
				for(i=0; i<3; i++) xyzD[i] = -1.0 + scaleD[i]*(xyz[i]-offsetD[i]);
				newPoints->SetPoint(idm,xyzD);
			}
			if(mOrderInFileIsAttribute)
			{
				f[0] = idm;
			}
			newScalars->SetTuple(idm,fOff);
			for(i=iattOff; i<natt; i++)
			{
				amin[i] = (amin[i]>fOff[i]) ? fOff[i] : amin[i];
				amax[i] = (amax[i]<fOff[i]) ? fOff[i] : amax[i];
			}
			idm++;
		}
	}	

	return true;
}


void iBasicParticlesDataSubject::FinalizeBody()
{
	this->FinalizePolyData(mData);
}


void iBasicParticlesDataSubject::ShiftDataBody(int d, double dx)
{
	this->ShiftPolyData(mData,d,dx);
}


void iBasicParticlesDataSubject::SetOrderInFileIsAttribute(bool s)
{ 
	mOrderInFileIsAttribute = s; 
}


iDataLimits* iBasicParticlesDataSubject::CreateLimits() const
{
	return new iDataLimits(this,3,"Attribute");
}


void iBasicParticlesDataSubject::DataSubjectPackStateBody(iString &) const
{
}


void iBasicParticlesDataSubject::DataSubjectUnPackStateBody(const iString &)
{
}

