/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IARRAY_H
#define IARRAY_H

//
//  Automatically-resizable array of type T; the type must have operator =.
//

template<class T>
class iArray
{

public:

	iArray(const T &n, int inc = 10);
	virtual ~iArray();

    T& operator[](int i);
    const T& operator[](int i) const;

	inline int Size() const { return this->mNum; }
	inline int Last() const { return this->mNum - 1; }

	virtual void Add(const T &val);
	T Remove(int n);
	T RemoveLast();  // useful for deleteting all: while(arr.Size() > 0) delete arr.RemoveLast();
	void Clear();

	inline const T* Data() const { return this->mArr; }

protected:

	void Extend(int len);

	T mNull;
	T* mArr;
	int mNum, mLen, mInc;
};


//
//  This version also needs operator ==
//
template<class T>
class iSearchableArray : public iArray<T>
{

public:

	iSearchableArray(const T &n, int inc = 10);

	virtual int Find(const T &val) const;
	void AddUnique(const T &val);
	T Remove(const T &val);
};


//
//  This version also needs operator <
//
template<class T>
class iOrderedArray : public iSearchableArray<T>
{

public:

	iOrderedArray(const T &n, int inc = 10);

	virtual int Find(const T &val) const;
	virtual void Add(const T &val);
	virtual bool Less(const T& v1, const T& v2) const;
};

	
template<class T>
inline bool iOrderedArray<T>::Less(const T& v1, const T& v2) const
{
	return v1 < v2;
}

//
//  Herb Sutter in "Exceptional C++" claims that comparing pointers using < is not
//  portable. So, I declare a special iPointerArray that inherits Less() function 
//  to compare pointers as unsigned longs. This must be fully portable.
//
template<class T>
class iPointerArray : public iOrderedArray<T*>
{

public:

	iPointerArray(int inc = 10) : iOrderedArray<T*>((T*)0,inc){}

	virtual bool Less(const T* &v1, const T* &v2) const;
};


template<class T>
inline bool iPointerArray<T>::Less(const T* &v1, const T* &v2) const
{
	return (unsigned long)v1 < (unsigned long)v2;
}

#endif // IARRAY_H
