/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/****************************************************************************
** ui.h extension file, included from the uic-generated form implementation.
**
** If you wish to add, delete or rename slots use Qt Designer which will
** update this file, preserving your code. Create an init() slot in place of
** a constructor, and a destroy() slot in place of a destructor.
*****************************************************************************/


#include "idialogscripter.h"
#include "idialogerrorlist.h"
#include "idialogscriptvariablesview.h"

#include <qaction.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qdockwindow.h>
#include <qevent.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qfontdialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qobjectlist.h>
#include <qpalette.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qstatusbar.h>
#include <qtabwidget.h>
#include <qtextbrowser.h>
#include <qtextedit.h>
#include <qtooltip.h>
#include <qregexp.h>

#include "iqt.h"
#include "iqt_view.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "ienvironment.h"
#include "iglobals.h"

#include "iqtdefs.h"

#include "ianimator.h"
#include "ianimatorscript.h"
#include "ieditor.h"


QPixmap image(char *s);
void reportNullPointer(int ec);
void iSleep(int);



class iDialogScripterPopup : public QDialog
{

public:

    iDialogScripterPopup(QWidget *parent, QStringList s) : QDialog(parent,"",true,!WType_Dialog | WType_Popup)
	{
		QVBoxLayout *l = new QVBoxLayout(this,1,1,"");
		b = new QListBox(this);
		b->setFrameStyle(QListBox::NoFrame);
		l->add(b);
		b->insertStringList(s);
		connect( b, SIGNAL( clicked(QListBoxItem *) ), this, SLOT( close() ) );
		connect( b, SIGNAL( returnPressed(QListBoxItem *) ), this, SLOT( close() ) );
	    b->setPaletteBackgroundColor(QColor(231,228,179));
		b->setMaximumSize(150,200);
	}

	QString getSelected(){ return b->text(b->currentItem()); }

private:

	QListBox *b;

};



void iDialogScripter::init()
{
	int i, j;
	static char helpTextStatements[] = 
		"# Valid statements:\n"
		"   var <type>[dimension] <variable>, ... \n"
		"#       declare variables of type <type>=[int,float,bool], optionally as an array.\n"
		"   set <variable> [+-*]= <expression>\n"
		"#       sets a value for the variable.\n"
		"   reset \n"
		"#       resets all variables to factory defaults.\n"
		"   render <expression>\n"
		"#       renders the scene in the current window <expression> times.\n"
		"   render-all <expression>\n"
		"#       renders the scene(s) in all windows <expression> times.\n"
		"   render-set <array>\n"
		"#       renders the scene(s) once in all windows listed in <array>.\n"
		"   load <expression>\n"
		"#       loads the data set with the record <expression>.\n"
		"   loop <expression> ... end\n"
		"#       repeats statements between loop and end <expression> times.\n"
		"   for <variable> to <expression> ... end\n"
		"#       loops <expression> times, incrementing <variable> at each iteration.\n"
		"   if <bool expression> then ... else ... endif\n"
		"#       executes statements conditionally.\n";
	static char helpTextParameters[] = 
		"# Pre-defined parameters (constants):\n"
		"   static\n"
		"   rotate-scale\n"
		"   tumble\n"
		"   fly-by\n"
		"   camera-path\n"
		"#       used in <set style> statement.\n"
		"   next\n"
		"#       used in <load> statement.\n"
		"   parallel\n"
		"   perspective\n"
		"#       used in <set projection> statement.\n"
		"   hidden\n"
		"   visible\n"
		"#       used in <set [time-label,color-bars,bounding-box,marker,...]> statements.\n"
		"   current\n"
		"#       used in setting properties of visualization objects.\n"
		"   marker[<expression>]\n"
		"#       used for getting a position of a marker with the number given by <expression>.\n"
		"   upper-left-corner\n"
		"   upper-right-corner\n"
		"   lower-left-corner\n"
		"   lower-right-corner\n"
		"#       used in <set logo-position> statement.\n";
	static char helpTextExpressions[] = 
		"# Valid expression components:\n"
		"#       expression can include scalars and 3-component arrays (vectors).\n"
		"   < > <= >= != ==\n"
		"#       compares two values.\n"
		"   && || !\n"
		"#       boolean operators.\n"
		"   + - * / ^ \n"
		"#       arithmetic operators.\n"
		"   .\n"
		"#       vector scalar product.\n"
		"   abs\n"
		"   exp\n"
		"   ceil\n"
		"   floor\n"
		"   log\n"
		"   sqrt\n"
		"   sin\n"
		"   cos\n"
		"   tan\n"
		"   asin\n"
		"   acos\n"
		"   atan\n"
		"   sinh\n"
		"   cosh\n"
		"   tanh\n"
		"#       scalar functions.\n"
		"   mag     #magnitude of a vector\n"
		"   norm    #normalized vector\n"
		"#       vector functions.\n";

	bool res;
	static QIconSet toolbarLabel;
	//
	//
	//
	runRunAction->setIconSet(QIconSet(image("run.png")));
	runDebugAction->setIconSet(QIconSet(image("debug.png")));
	runCompileAction->setIconSet(QIconSet(image("compile.png")));

	toolbarLabel.setPixmap(image("fontsize.png"),QIconSet::Automatic,QIconSet::Disabled);
	fontsizeDummyAction->setIconSet(toolbarLabel);

	CALL_FUNCTION1(FontSizeBox,setCurrentItem,1);
    //
    //  Initial widget settings
    //
    CurrentFileLabel = new QLineEdit("",statusBar());
	if(CurrentFileLabel == 0) reportNullPointer(9301);
    CurrentFileLabel->setReadOnly(true);
	
	CurrentPos = new QLabel("Line: 1   Column: 1",statusBar());
	if(CurrentPos == 0) reportNullPointer(9306);

	statusBar()->addWidget(CurrentPos,0,true);
    statusBar()->addWidget(CurrentFileLabel,1,true);
    
	errorList = new iDialogErrorlist;
   	if(errorList == 0) reportNullPointer(9307);
	res = connect((QObject *)errorList->ErrorListBox, SIGNAL( selected(const QString &) ) , this, SLOT( highlightErrors(const QString &) ) );

    findStringUsed = false;
	inDebug = inRun = false;
	
	res = connect(TextWindow, SIGNAL( textChanged() ) , this, SLOT( updateChangedText() ) );
	res = connect(TextWindow, SIGNAL( cursorPositionChanged(int,int) ) , this, SLOT( updateChangedCursorPosition(int,int) ) );
	res = connect(TextWindow, SIGNAL( returnPressed() ) , this, SLOT( updateReturnPressed() ) );
	res = connect(TextWindow, SIGNAL( tabPressed() ) , this, SLOT( updateTabPressed() ) );
	//
	//  This is 1 less than the full width of the tab stop - one character goes to the \t
	//
	nTabStopInSpaces = 2;
	TextWindow->setTabStopWidth(nTabStopInSpaces*TextWindow->pointSize());  // this is in pixels, so make it in symbols
	//
    //  Colors
    //
	errorColor = QColor(255,0,0);
	//
	//  Debugging tools
	//
	DebugFrame->hide();

	completionMode = 1;
	debugSceneType = 0;
	debugSpeed = 5;
	CALL_FUNCTION1(RunSpeedSlider,setValue,debugSpeed);

	breakpointColor = QColor(200,200,200);
	breakpoints.clear();

	stopped = true;
	//
    //  Specify indentating and unindentating words
    //
	indentatingWordList += "loop";
	indentatingWordList += "for"; 
	indentatingWordList += "if"; 
	indentatingWordList += "else"; 

	unindentatingWordList += "end";
	unindentatingWordList += "endif";
	unindentatingWordList += "else";

 	//
    //  Set tooltip with all reserved words
    //
	iScript *script = iVTKWindow::getCurrentWindow()->getAnimator()->getScript();
	int reservedWordMax = script->getNumberOfReservedWords();

	QString s1 = "  Variables:\n";
    for(i=0, j=0; i<reservedWordMax; i++)
	{
		if(script->isPrefixWord(script->getReservedWord(i)) || script->isDummyWord(script->getReservedWord(i)) ||
			(script->isCommandWord(script->getReservedWord(i)) && script->getUserTypeForReservedWord(i)==1))
		{
			keywordList += script->getReservedWord(i);
		}
		if(script->isCommandWord(script->getReservedWord(i)) && script->getUserTypeForReservedWord(i)>2)
		{
			variableList += script->getReservedWord(i);
			if(script->getUserTypeForReservedWord(i) != 255) s1 += "     " + script->getReservedWord(i);
			if(script->getUserTypeForReservedWord(i) == 4) s1 += " (array component)";
			if(script->getUserTypeForReservedWord(i) == 5) s1 += " (vector)";
			s1 += "\n";
		}
		if(script->isParameterWord(script->getReservedWord(i)))
		{
			parameterList += script->getReservedWord(i);
		}
	}

	TextWindow->setKeywords(keywordList);
	TextWindow->setVariables(variableList);
	TextWindow->setParameters(parameterList);

	TextBrowserStatements->setText(helpTextStatements);
	TextBrowserVariables->setText("# "+s1);
	TextBrowserParameters->setText(helpTextParameters);
	TextBrowserExpressions->setText(helpTextExpressions);

	TextBrowserStatements->setKeywords(keywordList);
	TextBrowserStatements->setVariables(variableList);
	TextBrowserStatements->setParameters(parameterList);
	TextBrowserVariables->setKeywords(keywordList);
	TextBrowserVariables->setVariables(variableList);
	TextBrowserVariables->setParameters(parameterList);
	TextBrowserParameters->setKeywords(keywordList);
	TextBrowserParameters->setVariables(variableList);
	TextBrowserParameters->setParameters(parameterList);
	TextBrowserExpressions->setKeywords(keywordList);
	TextBrowserExpressions->setVariables(variableList);
	TextBrowserExpressions->setParameters(parameterList);

	TextBrowserStatements->setWordWrap(QTextEdit::NoWrap);
	TextBrowserVariables->setWordWrap(QTextEdit::NoWrap);
	TextBrowserParameters->setWordWrap(QTextEdit::NoWrap);
	TextBrowserExpressions->setWordWrap(QTextEdit::NoWrap);

	TextBrowserStatements->setReadOnly(true);
	TextBrowserVariables->setReadOnly(true);
	TextBrowserParameters->setReadOnly(true);
	TextBrowserExpressions->setReadOnly(true);

	TextWindow->setWordWrap(QTextEdit::NoWrap);
	TextWindow->setModified(false);
	this->fileNew();

	this->setFontSize(FontSizeBox->currentText());

	this->setDockEnabled(Qt::DockLeft,false);
	this->setDockEnabled(Qt::DockRight,false);

	variablesDock = new QDockWindow(QDockWindow::InDock,this,"variablesDock");
	variablesDock->setCaption("IFRIT - Script Variables View");
	variablesDock->setResizeEnabled(true);
	variablesDock->setHorizontallyStretchable(true);
	variablesDock->setOpaqueMoving(true);
	variablesDock->hide();

	variablesView = new iDialogScriptVariablesView(variablesDock);
	variablesDock->boxLayout()->addWidget(variablesView);
	this->addDockWindow(variablesDock,Qt::DockBottom);

	helperDock = new QDockWindow(QDockWindow::InDock,this,"helperDock");
	helperDock->setCaption("IFRIT - Script Syntax Helper");
	helperDock->setResizeEnabled(true);
	helperDock->setHorizontallyStretchable(true);
	helperDock->setOpaqueMoving(true);

	SyntaxHelper->reparent(helperDock,QPoint(0,0));
	helperDock->boxLayout()->addWidget(SyntaxHelper);
	this->addDockWindow(helperDock,Qt::DockBottom);

	savedText = "";

	TextWindow->setFrameShape(QFrame::NoFrame);

	TextBar = new QScrollView(EditorFrame);
	EditorLayout->insertWidget(0,TextBar);
	TextBar->setVScrollBarMode(QScrollView::AlwaysOff);
	TextBar->setHScrollBarMode(QScrollView::AlwaysOff);
	TextBar->setMinimumWidth(20);
	TextBar->setMaximumWidth(20);
	TextBar->viewport()->setBackgroundMode(Qt::PaletteBackground);
	TextBar->viewport()->setPaletteBackgroundColor(white);
	TextBar->setResizePolicy(QScrollView::Manual);
	TextBar->setFrameShape(QFrame::NoFrame);
	TextBar->hide();

	EditorLayout->setSpacing(0);
	EditorLayout->setMargin(2);

	arrow = new QLabel(TextBar->viewport(),"Arrow");
	arrow->setPixmap(QPixmap());
	arrow->setFrameShape(QFrame::NoFrame);
	TextBar->addChild(arrow);	

	res = connect(TextWindow, SIGNAL( contentsMoving(int,int) ) , TextBar, SLOT( setContentsPos(int,int) ) );

}


void iDialogScripter::destroy()
{

	delete variablesDock;
	delete helperDock;
	
    delete CurrentFileLabel;
	delete CurrentPos;
	delete errorList;

}


void iDialogScripter::closeEvent( QCloseEvent *e )
{
    
	if(inDebug) this->debug(false);
	if(TextWindow->isModified())
	{
		switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
			"The current script is not saved.\n\n",
			"Save", "Ignore", 0, 0, 1 ) ) 
		{
		case 0: // The user clicked the Retry again button or pressed Enter
			{ this->fileSave(); break; }
		case 1: // The user clicked the Quit or pressed Escape
			{ break; }
		}
	}
	
	e->accept();
	iQTWindow::getCurrentWindow()->resetCursor();
	
}


void iDialogScripter::show()
{
	if(inDebug) variablesDock->show(); else variablesDock->hide();
	this->move(iQTWindow::getCurrentWindow()->pos().x()+10,iQTWindow::getCurrentWindow()->pos().y()+100);
	QMainWindow::show();
}


void iDialogScripter::fileNew()
{
	
	if(inDebug) this->debug(false);

	if(TextWindow->isModified())
	{
		switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
			"The current script is not saved.\n\n",
			"Save", "Ignore", 0, 0, 1 ) ) 
		{
		case 0: // The user clicked the Retry again button or pressed Enter
			{ this->fileSave(); break; }
		case 1: // The user clicked the Quit or pressed Escape
			{ break; }
		}
    }
	
 	currentFileName = "";
	TextWindow->clear();
	//
	//  Add comments with help on top of the window
	//
	savedText = "";
	CALL_FUNCTION1(TextWindow,setModified,false);
	CALL_FUNCTION2(TextWindow,setCursorPosition,TextWindow->paragraphs()-1,0);
	
}


void iDialogScripter::fileOpen()
{
	QString s1 = currentFileName;
	if(s1.isEmpty()) s1 = iEnvironment::getInstance()->get_IFRIT_SCRIPT_DIR();
    QString s = QFileDialog::getOpenFileName(s1,
		"IFRIT animation script (*.ias)",
		this,
		"open IFRIT animation script",
		"Choose a file" );
	
    if(!s.isEmpty()) 
	{
		
		if(inDebug) this->debug(false);

		if(TextWindow->isModified())
		{
			switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
				"The current script is not saved.\n\n",
				"Save", "Ignore", 0, 0, 1 ) ) 
			{
			case 0: // The user clicked the Retry again button or pressed Enter
				{ this->fileSave(); break; }
			case 1: // The user clicked the Quit or pressed Escape
				{ break; }
			}
		}
		
		QFile f(s);
		
		if(!f.exists() || !f.open(IO_ReadOnly)) 
		{
			switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
				"Could not open the animation script.\n\n",
				"Ignore", "Close Animation Script Editor", 0, 0, 1 ) ) 
			{
			case 0: // The user clicked the Retry again button or pressed Enter
				break;
			case 1: // The user clicked the Quit or pressed Escape
				{ this->close(); break; }
			}
			return;
		}
		
		QTextStream ts( &f );
		CALL_FUNCTION1(TextWindow,setText,ts.read());
		savedText = "";
		CALL_FUNCTION1(TextWindow,setModified,false);
		CALL_FUNCTION1(CurrentFileLabel,setText,s);
		iVTKWindow::getCurrentWindow()->getAnimator()->setScriptFileName(s);
		emit scriptFileSet(s);
		f.close();
		currentFileName = s;
    }
	
}

void iDialogScripter::fileSave()
{
    
	if(inDebug) this->debug(false);

	if(currentFileName.isEmpty()) 
	{
		this->fileSaveAs();
		return;
    }
	
    QFile f(currentFileName);
    
    if(!f.open(IO_WriteOnly)) 
	{
		switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
			"Could not save the animation script.\n\n",
			"Ignore", "Close Animation Script Editor", 0, 0, 1 ) ) 
		{
		case 0: // The user clicked the Retry again button or pressed Enter
			break;
		case 1: // The user clicked the Quit or pressed Escape
			{ this->close(); break; }
		}
		return;
	}
	
    QTextStream ts( &f );
    ts << TextWindow->text();
    TextWindow->setModified(false);
    f.close();
	
}


void iDialogScripter::fileSaveAs()
{
	QString s1 = currentFileName;
	if(s1.isEmpty()) s1 = iEnvironment::getInstance()->get_IFRIT_SCRIPT_DIR();
    QString s = QFileDialog::getSaveFileName(s1,
		"IFRIT animation script (*.ias)",
		this,
		"save IFRIT animation script",
		"Save as" );
	
    if(!s.isEmpty()) 
	{
	    if(s.right(4) != ".ias") s += ".ias";

		if(QFile::exists(s))
		{
			switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
				"File already exists. Would you like to overwrite it?\n\n",
				"Ok", "Cancel", 0, 0, 1 ) ) 
			{
			case 0: // The user clicked the Retry again button or pressed Enter
				break;
			case 1: // The user clicked the Quit or pressed Escape
				{ return; }
			}
		}

		currentFileName = s;
		CurrentFileLabel->setText(s);
		iVTKWindow::getCurrentWindow()->getAnimator()->setScriptFileName(s);
		emit scriptFileSet(s);
		fileSave();
    }
	
}


int iDialogScripter::getDelay()
{
	return debugSpeed*50;
}


void iDialogScripter::fileExit()
{
	this->close();
}


void iDialogScripter::editUndo()
{
	if(inDebug) this->debug(false);
    TextWindow->undo();
}

void iDialogScripter::editRedo()
{
	if(inDebug) this->debug(false);
    TextWindow->redo();
}

void iDialogScripter::editCut()
{
	if(inDebug) this->debug(false);
    TextWindow->cut();
}

void iDialogScripter::editCopy()
{
	if(inDebug) this->debug(false);
    TextWindow->copy();
}

void iDialogScripter::editPaste()
{
	if(inDebug) this->debug(false);
    TextWindow->paste();
}

void iDialogScripter::editFind()
{
    static int paraFrom = 0, indexFrom = 0;
    
    if(FindString->currentText().isEmpty()) return;
    
    if(!findStringUsed)
	{
		findStringUsed = true;
		FindString->insertItem(FindString->currentText(),0);
    }
    
    if(TextWindow->find(FindString->currentText(),false,false,true,&paraFrom,&indexFrom))
	{
		indexFrom++;
    } 
	else 
	{
		if(paraFrom>0 || indexFrom>0) 
		{
			paraFrom = indexFrom = 0;
			this->editFind();
		} 
		else QMessageBox::information(this,"IFRIT Animation Script Editor","String is not found.");
    }
    
}


void iDialogScripter::findStringChanged(const QString & vtkNotUsed(text) )
{
	
    findStringUsed = false;
    
}


void iDialogScripter::compile()
{
    int i, res;
	bool ok = true;
	static QString msg;
	int ds = debugSpeed;
    debugSpeed = 0;

	errorList->clearList();
	savedText = "";

	bool mod = TextWindow->isModified();
	iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->setText(TextWindow->text());

	TextWindow->blockSignals(true);
    for(i=0; i<iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getNumberOfLines(); i++)
	{
		
		if(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->runOneLine(true) > 0)
		{
			res = iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCompileReturnCode();
			ok = false;
			TextWindow->setUndoRedoEnabled(false);
			TextWindow->setSelection(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCurrentLine()-1,res,iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCurrentLine()-1,TextWindow->paragraphLength(i));
			TextWindow->setColor(errorColor);
			TextWindow->removeSelection();
			TextWindow->setUndoRedoEnabled(true);

			msg = iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getErrorMessage();
			errorList->addErrorMessage(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCurrentLine()-1,msg);
		}
		
    }
	//
	//  Check that all ends closed
	//
	if(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->areThereOpenEntryPoints())
	{
		ok = false;
		msg = "Missing end statement(s)";
		errorList->addErrorMessage(i,msg);
	}

	if(ok) QMessageBox::information(this,"IFRIT","Congratulations, no errors found in your script."); 

    if(!ok && !errorList->isVisible()) errorList->show();
	if(ok && errorList->isVisible()) errorList->hide();
    if(!mod) TextWindow->setModified(false);
	TextWindow->blockSignals(false);

	debugSpeed = ds;

}

void iDialogScripter::run()
{
	
	if(inDebug) this->debug(false);

	if(!iVTKWindow::getCurrentWindow()->getAnimator()->getUseScript())
	{
		switch( QMessageBox::warning(this,"IFRIT Animation Script Editor",
			"<b>Use script</b> is not selected.\n What do you want to do?\n\n",
			"Animate using script", "Animate without script", "Cancel", 0, 2 ) ) 
		{
		case 0: // use script: set check button and then do as if no script was selected (no break in this case operator)
			{ 
				CALL_FUNCTION1(iQTWindow::getCurrentWindow()->TabWidgetView->View_Anim_Scripting,setChecked,true); 
				iVTKWindow::getCurrentWindow()->getAnimator()->setUseScript(true);
			}
		case 1: // do not use script
			{ break; }
		case 2: // The user clicked the Quit or pressed Escape
			{ return; }
		}
	} 

	emit scriptRun();
	
}


void iDialogScripter::updateChangedText()
{

	if(inDebug) this->debug(false);

}


void iDialogScripter::updateChangedCursorPosition(int para, int index)
{
	static QString s, s1;

	//
	//  Find all tabs and count them as a number of spaces
	//
	s = TextWindow->text(para);
	s1 = s.left(index);
	index += nTabStopInSpaces*TextWindow->text(para).left(index).contains('\t');

	s = "   Line: ";
	s += s1.setNum(1+para);
	s += "   Column: ";
	s += s1.setNum(1+index);
	CALL_FUNCTION1(CurrentPos,setText,s);
 
}


void iDialogScripter::showScriptLine(int lineInScript, int entryPointInteration, int entryPointIterationCount, bool done)
{
	static int curLine = -1;

	if(!savedText.isEmpty()) 
	{
		//
		//  show line with an "arrow"
		//

		if(TextWindow->paragraphBackgroundColor(lineInScript) == breakpointColor)
		{
			if(done) arrow->setPixmap(image("debug_bpdone.png")); else arrow->setPixmap(image("debug_bpwork.png"));
		}
		else
		{
			if(done) arrow->setPixmap(image("debug_done.png")); else arrow->setPixmap(image("debug_work.png"));
		}
		if(lineInScript != curLine)
		{
//			if(curLine >= 0)
//			{
//				arrow->setPixmap(QPixmap());
//			}

			if(lineInScript<0 || lineInScript>=TextWindow->paragraphs())
			{
				curLine = -1;
			}
			else
			{
				curLine = lineInScript;
				QRect r = TextWindow->paragraphRect(curLine);
				arrow->show();
//				arrow->setBackgroundMode(Qt::NoBackground);
				arrow->resize(arrow->width(),r.height());
				TextBar->moveChild(arrow,0,r.y()+(r.height()-arrow->height())/2);
				TextWindow->ensureVisible(r.x(),r.y());
				TextBar->viewport()->repaint();
			}
		}
		//
		//  Check if we are in a loop. If yes, display the current iteration number
		//
		if(entryPointIterationCount > 1)
		{

			TextWindow->blockSignals(true);

			int i = TextWindow->text(lineInScript).find('{');
			//
			//  A label of the type { number } already exists. Remove it then
			//
			if(i > 2)
			{
				TextWindow->setSelection(lineInScript,i-2,lineInScript,TextWindow->paragraphLength(lineInScript));
				TextWindow->removeSelectedText();
				TextWindow->removeSelection();
			}
			//
			//  Form a label
			//
			if(entryPointInteration > 0)
			{
				QString lab = QString("  { Iteration: ") + QString::number(entryPointInteration) + " }"; 
				TextWindow->setColor(errorColor);
				TextWindow->insertAt(lab,lineInScript,TextWindow->paragraphLength(lineInScript));
			}

			TextWindow->blockSignals(false);

		}

	}
	//
	//  Update the variable display too
	//
	if(variablesView != NULL) variablesView->updateVariable(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getLastChangedVariable());

}


void iDialogScripter::highlightErrors(const QString &s )
{
	
	if(s.left(5) != "Line ") return;
	
	bool ok;
	int line = s.mid(5).section(':',0,0).toInt(&ok) - 1;
	if(!ok) return;

	TextWindow->setSelection(line,0,line,TextWindow->paragraphLength(line));

}


void iDialogScripter::updateReturnPressed()
{
    int i, ind;
    int para, index;
    int paraSave, indexSave;
	int nIndentating = 0;
	QString line;
	
    TextWindow->getCursorPosition(&paraSave,&indexSave);

	if(indexSave !=0) return;
	//
	//  Scan all paragraphs
	//
	for(para=0; para<=paraSave; para++)
	{
		line = TextWindow->text(para);
		//
		//  Compute the indentation
		//
		for(i=0; !indentatingWordList[i].isEmpty(); i++)
		{
			index = line.find(indentatingWordList[i],0,false);
			if(index != -1)
			{
				//
				//  An indentating word found in the paragraph preceeding the current one.
				//  Check that it is not after the comment sign
				//
				ind = line.left(index).find("#",0);
				if(ind==-1 && (index==0 || (!line[index-1].isLetterOrNumber() && line[index-1]!='-') ) && (index+indentatingWordList[i].length()==line.length() || (!line[index+(int)indentatingWordList[i].length()].isLetterOrNumber() && line[index+(int)indentatingWordList[i].length()]!='-') ) ) nIndentating++;
			}
		
	    }
		//
		//  Compute the un-indentation
		//
		for(i=0; !unindentatingWordList[i].isEmpty(); i++)
		{
			index = line.find(unindentatingWordList[i],0,false);
			if(index != -1)
			{
				//
				//  An un-indentating word found in the paragraph preceeding the current one.
				//  Check that it is not after the comment sign
				//
				ind = line.left(index).find("#",0);
				if(ind==-1 && (index==0 || (!line[index-1].isLetterOrNumber() && line[index-1]!='-') ) && (index+unindentatingWordList[i].length()==line.length() || (!line[index+(int)unindentatingWordList[i].length()].isLetterOrNumber() && line[index+(int)unindentatingWordList[i].length()]!='-') ) ) nIndentating--;
			}
		
	    }
	
	}

	line = "";
	for(i=0; i<nIndentating; i++) line += "\t";

	TextWindow->blockSignals(true);
    TextWindow->setCursorPosition(paraSave,indexSave);
	TextWindow->insert(line,false,false,true);
	TextWindow->blockSignals(false);

}


void iDialogScripter::updateTabPressed()
{
    int i, j;
    int paraSave, indexSave;
	QString line, w;
	QStringList m;

	TextWindow->getCursorPosition(&paraSave,&indexSave);
	if(indexSave < 2) return;

	line = TextWindow->text(paraSave);
	if(!line[indexSave-2].isLetter() && line[indexSave-2]!='-') return;

	indexSave--;

	i = indexSave - 1;
	while(i>=0 && (line[i].isLetter() || line[i]=='-')) i--; 
	i++;
	
	w = line.mid(i,indexSave-i);
	
	m = keywordList.grep(QRegExp("^"+w));
	m += variableList.grep(QRegExp("^"+w));
	m += parameterList.grep(QRegExp("^"+w));
	
	if(m.count() > 0)
	{
		TextWindow->doKeyboardAction(QTextEdit::ActionBackspace);
		if(m.count() == 1)
		{
			w = m.first().mid(indexSave-i);
		}
		else
		{
			m.sort();
			switch (completionMode)
			{
			case 0:
				{
					j = w.length();
					while(j<(int)m.first().length() && j<(int)m.last().length() && m.first()[j]==m.last()[j]) j++; 
					w = m.first().left(j).mid(indexSave-i);
					break;
				}
			case 1:
				{
					iDialogScripterPopup *p = new iDialogScripterPopup(0,m);
					p->move(TextWindow->getTextCursorPos());
					p->exec();
					w = p->getSelected().mid(indexSave-i);
					delete p;
					break;
				}
			}
		}
		if(w.length() > 0)
		{
			TextWindow->blockSignals(true);
			TextWindow->insert(w,false,false,true);
			TextWindow->setCursorPosition(paraSave,indexSave+w.length());
			TextWindow->blockSignals(false);
		}
	}
	
}


void iDialogScripter::setFontSize(const QString &s)
{
	bool ok, mod;
    int fs, paraSave, indexSave;
	
	fs = s.toInt(&ok);
	if(!ok) return;

	TextWindow->blockSignals(true);

	TextWindow->setUndoRedoEnabled(false);
	TextWindow->getCursorPosition(&paraSave,&indexSave);
	TextWindow->setPointSize(fs);

	mod = TextWindow->isModified();
	TextWindow->setModified(true);
	TextWindow->setText(TextWindow->text());
	TextWindow->setModified(mod);

    TextWindow->setCursorPosition(paraSave,indexSave);
	TextWindow->setUndoRedoEnabled(true);

	TextWindow->blockSignals(false);

}


void iDialogScripter::helpContents()
{
	static QString s = "vtkscrpt.ihf";
	iQTWindow::getCurrentWindow()->displayHelp(s);
}


void iDialogScripter::helpAbout()
{
	iQTWindow::getCurrentWindow()->helpAbout();
}


void iDialogScripter::showBreakpoint(int para)
{
	TextWindow->setParagraphBackgroundColor(para,breakpointColor);
	QRect r = TextWindow->paragraphRect(para);
	QLabel *bp = new QLabel(TextBar->viewport(),"Breakpoint");
	bp->setPixmap(image("debug_breakpoint.png"));
	TextBar->addChild(bp,0,r.y()+(r.height()-bp->height())/2);
	bp->show();
	arrow->raise();
}


void iDialogScripter::hideBreakpoint(int para)
{
	QRect r;
	int ybp, i;

	if(para >= 0)
	{
		TextWindow->clearParagraphBackground(para);
		r = TextWindow->paragraphRect(para);
	}
	else
	{
		for(i=0; i<TextWindow->paragraphs(); i++) TextWindow->clearParagraphBackground(i);
		r = QRect(0,0,TextBar->contentsWidth(),TextBar->contentsHeight());
	}
	//
	// Remove breakpoint marker(s)
	//
	const QObjectList *l = TextBar->viewport()->children();
	QObjectListIt it(*l); // iterate over the buttons
	QObject *obj;
	
	while((obj=it.current()) != 0) 
	{
		if(strcmp(obj->name(),"Breakpoint") == 0)
		{
			ybp = TextBar->childY((QWidget*)obj) + ((QWidget*)obj)->height()/2;
			if(para<0 || (ybp>=r.y() && ybp<=r.y()+r.height())) 
			{
				delete obj;
			}
		}
		++it;
	}
	
}


void iDialogScripter::debug(bool s)
{
	int i;
	static bool mod = false;
	static bool hds = false;

#ifndef _DEBUG
	if(!iQTWindow::getCurrentWindow()->Button_Animate->isEnabled() && s)
	{
		QMessageBox::information(this,"IFRIT","Animatable data file is not loaded.\n Please, load the data file.");
		CALL_FUNCTION1(runDebugAction,setOn,false);
		return;
	}
#endif

	inDebug = s;
	CALL_FUNCTION1(runDebugAction,setOn,s);
	runRunAction->setEnabled(!s);
	runCompileAction->setEnabled(!s);

	TextWindow->blockSignals(true);

	if(s)
	{

		iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->setText(TextWindow->text());
		if(variablesDock != NULL) variablesDock->show();

		mod = TextWindow->isModified();

		switch (debugSceneType)
		{
		default:
		case 0: { iVTKWindow::getCurrentWindow()->setDebugMode(true); break; }
		case 1: { break; }
		}

		hds = helperDock->isVisible();
		helperDock->hide();

		iQTWindow::getCurrentWindow()->animateStart();
		TextWindow->activateHighlighter(false);
		TextWindow->setColor(QColor(255,0,0));
		DebugFrame->show();

		//
		//  Set breakpoints
		//
		for(QStringList::Iterator is=breakpoints.begin(); is!=breakpoints.end(); ++is)
		{
			for(i=0; i<TextWindow->paragraphs(); i++)
			{
				if(*is == TextWindow->text(i))
				{
					this->showBreakpoint(i);
				}
			}
		}

	}
	else
	{

		breakpoints.clear();
		for(i=0; i<TextWindow->paragraphs(); i++)
		{
			if(TextWindow->paragraphBackgroundColor(i) == breakpointColor) breakpoints += TextWindow->text(i);
		}
		//
		// Remove all old breakpoint markers
		//
		this->hideBreakpoint(-1);

		TextWindow->activateHighlighter(true);
		TextWindow->setColor(QColor(0,0,0));
		iQTWindow::getCurrentWindow()->animateFinish();
		iVTKWindow::getCurrentWindow()->setDebugMode(false);
		DebugFrame->hide();

		if(hds) helperDock->show();

		if(variablesDock != NULL) variablesDock->hide();
		
		TextWindow->setModified(mod);

	}

	TextWindow->blockSignals(false);
	this->updateGeometry();

}


void iDialogScripter::debugRemoveAllBreakpoints()
{
 	TextWindow->blockSignals(true);
	this->hideBreakpoint(-1);
	TextWindow->blockSignals(false);

}


void iDialogScripter::debugRemoveBreakpoint()
{
    int para, index;
	
	TextWindow->blockSignals(true);
    TextWindow->getCursorPosition(&para,&index);
    this->hideBreakpoint(para);
	TextWindow->blockSignals(false);

}


void iDialogScripter::debugSetBreakpoint()
{
    int para, index;
	
	TextWindow->blockSignals(true);
    TextWindow->getCursorPosition(&para,&index);
    this->showBreakpoint(para);
	TextWindow->blockSignals(false);

}


void iDialogScripter::debugContinue()
{
	int i, ret = 0;
	bool bp = false;

	stopped = false;
    while(!stopped && ret==0 && !bp)
	{

		ret = iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->runOneLine(false);
		i = iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCurrentLine();
		
		if(TextWindow->paragraphBackgroundColor(i) == breakpointColor)
		{
			this->showScriptLine(i,-1,0,false);
			bp = true;
		}
		qApp->flush(); qApp->processEvents(); 
    }
	if(ret == -1) 
	{
		this->showScriptLine(-1,-1,0,true);
		QMessageBox::information(this,"IFRIT","Script is completed."); 
	}
	if(ret > 0) iQTWindow::getCurrentWindow()->animateReportError(ret,0); 
	stopped = true;

}


void iDialogScripter::debugStep()
{
	int ds = debugSpeed;

	debugSpeed = 0;
	int ret = iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->runOneLine(false);
	if(ret == -1) 
	{
		this->showScriptLine(-1,-1,0,true);
		QMessageBox::information(this,"IFRIT","Script is completed."); 
	}
	if(ret > 0) iQTWindow::getCurrentWindow()->animateReportError(ret,0); 
	debugSpeed = ds;
}


void iDialogScripter::debugStop()
{
	if(!stopped)
	{
		stopped = true;
		CALL_FUNCTION0(StopButton,animateClick);
		qApp->processEvents();
	}
}


void iDialogScripter::debugRestart()
{

	this->debug(false);
	this->debug(true);
}


void iDialogScripter::runStart()
{
	if(inRun)
	{
		QMessageBox::information(this,"IFRIT","Bug in iDialogScripter::runStart. Please report.");
	}
	inRun = true;

	modBeforeRun = TextWindow->isModified();

	TextWindow->blockSignals(true);
	TextWindow->setUndoRedoEnabled(false);
	savedText = TextWindow->text();
	TextWindow->blockSignals(false);

	TextBar->resizeContents(TextBar->maximumWidth(),TextWindow->contentsHeight()+TextWindow->horizontalScrollBar()->height());
	TextBar->moveChild(arrow,0,0);
	TextBar->show();

}


void iDialogScripter::runFinish()
{

	if(!inRun)
	{
		QMessageBox::information(this,"IFRIT","Bug in iDialogScripter::runFinish. Please report.");
	}
	inRun = false;

	TextWindow->blockSignals(true);
	TextWindow->setText(savedText);
	TextWindow->setModified(modBeforeRun);
	TextWindow->setUndoRedoEnabled(true);
	savedText = "";
	TextWindow->blockSignals(false);

	arrow->hide();
	TextBar->hide();

}


void iDialogScripter::debugRunSpeedSlider(int v)
{
	debugSpeed = v;
}


void iDialogScripter::debugScene(int v)
{

	if(v != debugSceneType)
	{
		debugSceneType = v;
		switch (debugSceneType)
		{
		default:
		case 0: { iVTKWindow::getCurrentWindow()->setDebugMode(true); break; }
		case 1: { iVTKWindow::getCurrentWindow()->setDebugMode(false); break; }
		}
		iQTWindow::getCurrentWindow()->render(true);	
	}
	
}


void iDialogScripter::setCompletionMode(QAction *a)
{
	if(a == completionModeTextAction) completionMode = 0;
	if(a == completionModeWindowAction) completionMode = 1;
}



