/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of idataconverter.h
  
=========================================================================*/


#include "iglobals.h"
#include "ivolumedataconverter.h"

#include "ilimits.h"
#include "ivisualobject.h"
#include "imath.h"
#include "iuniformmeshdata.h"


//------------------------------------------------------------------------------
iVolumeDataConverter* iVolumeDataConverter::New(iVisualObject *m)
{
	return new iVolumeDataConverter(m);
}


iVolumeDataConverter::iVolumeDataConverter(iVisualObject *m) : iVisualObjectSource(m)
{
	
	curvar = 1;
	
}


void iVolumeDataConverter::setCurrentVar(int n)
{ 
	if(n < 0) return;
	
	if(myObject->getLimits() != 0)
	{
		if(n <= myObject->getLimits()->getNvar())
		{
			curvar = n;
			this->Modified();
		}
	}
	else
	{
		curvar = n;
		this->Modified();
	}

}


void iVolumeDataConverter::ExecuteData(vtkDataObject *)
{
	iUniformMeshData *input = (iUniformMeshData *)this->GetInput();
	iUniformMeshData *output = (iUniformMeshData *)this->GetOutput();
	int dims[3];
	vtkFloat pos[3];
	
	if(input == 0) return;

	output->Initialize();

	output->ShallowCopy(input);

	int n = input->GetNumberOfScalarComponents();
	if(n == 0) return;
	if (n != myObject->getLimits()->getNvar())
	{
		vtkErrorMacro("Wrong number of components in the input data");
		return;
    }
	if(curvar<1 || curvar>n){
		vtkErrorMacro("Incorrect variable number");
		return;
    }
	
	vtkDebugMacro(<< "Converting StrucuturedPoints to char with limits");

	input->GetDimensions(dims);
	output->SetDimensions(dims);
	
	vtkIdType size = (vtkIdType)dims[0]*dims[1]*dims[2];
	
	input->GetSpacing(pos);
	output->SetSpacing(pos);
	
	input->GetOrigin(pos);
	output->SetOrigin(pos);
	
	output->SetScalarTypeToUnsignedChar();
	output->SetNumberOfScalarComponents(1);
	output->AllocateScalars();
	
	float *ptrin = (float *) input->GetScalarPointer();
	unsigned char *ptrout = (unsigned char *) output->GetScalarPointer();
	
	vtkIdType loff = size*((vtkIdType)curvar - 1);
	float fl = myObject->getLimits()->getVarLow(curvar);
	float fh = myObject->getLimits()->getVarHigh(curvar);
	float fnorm = 255.0/(fh-fl+1.0e-10);
	for(vtkIdType l=0; l<size; l++)
	{
		if(l%1000 == 0) 
		{
			this->UpdateProgress((float)l/size);
			if(this->GetAbortExecute()) break;
		}

		float f = *(ptrin+loff+l);
		f = (f > fl) ? f : fl;
		f = (f < fh) ? f : fh;
		*(ptrout+l) = (char)round(fnorm*(f-fl));
	}
	output->Modified();
	this->Modified();
	
}


float iVolumeDataConverter::getMemorySize()
{
	return this->GetOutput()->GetActualMemorySize();
}
