/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "imeasuringboxinteractorstyle.h"
#include "imeasuringbox.h"

#include <vtkCommand.h>
#include <vtkMath.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkRenderer.h>
#include <vtkTransform.h>
#include <vtkMatrix4x4.h>
#include <vtkCamera.h>

void reportNullPointer(int);


iMeasuringBoxInteractorStyle* iMeasuringBoxInteractorStyle::New()
{

	return new iMeasuringBoxInteractorStyle;

}


void iMeasuringBoxInteractorStyle::FindPickedActor(int x, int y)
{ 
	this->InteractionProp = mBox; x = y = 0; 
}


#ifndef I_VTK_VERSION_40

//
//  Panning in XYZ - a copy of vtkInteractorStyleTrackballActor::Pan() with 
//  one change
//
void iMeasuringBoxInteractorStyle::UniformScale()
{
	this->InteractionProp = mBox;
	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// Use initial center as the origin from which to pan
	
	vtkFloat *obj_center = this->InteractionProp->GetCenter();
	
	vtkFloat disp_obj_center[3], new_pick_point[4], old_pick_point[4], motion_vector[3];
	
	this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
		disp_obj_center);
	
	this->ComputeDisplayToWorld((double)rwi->GetEventPosition()[0], 
		(double)rwi->GetEventPosition()[1], 
		disp_obj_center[2],
		new_pick_point);
	
	this->ComputeDisplayToWorld(double(rwi->GetLastEventPosition()[0]), 
		double(rwi->GetLastEventPosition()[1]), 
		disp_obj_center[2],
		old_pick_point);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	float x = rwi->GetEventPosition()[0];
	float y = rwi->GetEventPosition()[1];
	float oldX = rwi->GetLastEventPosition()[0];
	float oldY = rwi->GetLastEventPosition()[1];
	
	float d = sqrt(vtkMath::Distance2BetweenPoints(new_pick_point,old_pick_point));
	
	vtkFloat wc[4], wx[4], wy[4], wz[4];
	vtkFloat ex[3], ey[3], ez[3];
	vtkFloat dx[2];
	int i;
	
	dx[0] = x - oldX;
	dx[1] = y - oldY;
	vtkMath::Normalize2D(dx);
	
	this->ComputeWorldToDisplay(0.0,0.0,0.0,wc);
	this->ComputeWorldToDisplay(1.0,0.0,0.0,wx);
	this->ComputeWorldToDisplay(0.0,1.0,0.0,wy);
	this->ComputeWorldToDisplay(0.0,0.0,1.0,wz);
	
	for(i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
		ez[i] = wz[i] - wc[i];
	}
	
	vtkMath::Normalize2D(ex);
	vtkMath::Normalize2D(ey);
	vtkMath::Normalize2D(ez);
	
	vtkFloat crossProduct[3];
	crossProduct[0] = dx[0]*ex[0] + dx[1]*ex[1];
	crossProduct[1] = dx[0]*ey[0] + dx[1]*ey[1];
	crossProduct[2] = dx[0]*ez[0] + dx[1]*ez[1];
	
	for(i=0; i<3; i++) crossProduct[i] = pow((double)crossProduct[i],9.0);
	
	vtkMath::Normalize(crossProduct);
	
	motion_vector[0] = d*crossProduct[0];
	motion_vector[1] = d*crossProduct[1];
	motion_vector[2] = d*crossProduct[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != NULL)
    {
		vtkTransform *t = vtkTransform::New();
		if(t == 0) reportNullPointer(8401);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(motion_vector[0], motion_vector[1], motion_vector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(motion_vector);
    }
	
	if (this->AutoAdjustCameraClippingRange)
    {
		this->CurrentRenderer->ResetCameraClippingRange();
    }
	
	rwi->Render();
}
//
//  Panning in screen - a copy of vtkInteractorStyleTrackballActor::PanXY() with 
//  one change
//
void iMeasuringBoxInteractorStyle::Pan()
{
	this->InteractionProp = mBox;
	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// Use initial center as the origin from which to pan
	
	vtkFloat *obj_center = this->InteractionProp->GetCenter();
	
	vtkFloat disp_obj_center[3], new_pick_point[4], old_pick_point[4], motion_vector[3];
	
	this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
		disp_obj_center);
	
	this->ComputeDisplayToWorld((double)rwi->GetEventPosition()[0], 
		(double)rwi->GetEventPosition()[1], 
		disp_obj_center[2],
		new_pick_point);
	
	this->ComputeDisplayToWorld(double(rwi->GetLastEventPosition()[0]), 
		double(rwi->GetLastEventPosition()[1]), 
		disp_obj_center[2],
		old_pick_point);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	// float x = rwi->GetEventPosition()[0];
	float y = rwi->GetEventPosition()[1];
	// float oldX = rwi->GetLastEventPosition()[0];
	float oldY = rwi->GetLastEventPosition()[1];
	
	float d = sqrt(vtkMath::Distance2BetweenPoints(new_pick_point,old_pick_point));
	if(y > oldY) d = -d;
	
	vtkFloat wc[4], wx[4], wy[4];
	vtkFloat ex[3], ey[3], ez[3];
	
	this->ComputeDisplayToWorld(0.0,0.0,disp_obj_center[2],wc);
	this->ComputeDisplayToWorld(1.0,0.0,disp_obj_center[2],wx);
	this->ComputeDisplayToWorld(0.0,1.0,disp_obj_center[2],wy);
	
	for(int i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
	}
	
	vtkMath::Cross(ex,ey,ez);
	vtkMath::Normalize(ez);
	
	motion_vector[0] = d*ez[0];
	motion_vector[1] = d*ez[1];
	motion_vector[2] = d*ez[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != NULL)
    {
		vtkTransform *t = vtkTransform::New();
		if(t == 0) reportNullPointer(8402);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(motion_vector[0], motion_vector[1], motion_vector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(motion_vector);
    }
	
	if (this->AutoAdjustCameraClippingRange)
    {
		this->CurrentRenderer->ResetCameraClippingRange();
    }
	
	rwi->Render();
}
//
//  Rotating - a copy of vtkInteractorStyleTrackballActor::Rotate() with 
//  no changes
//
void iMeasuringBoxInteractorStyle::Rotate()
{
  this->InteractionProp = mBox;
  if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
    return;
    }
  
  vtkRenderWindowInteractor *rwi = this->Interactor;
  vtkCamera *cam = this->CurrentRenderer->GetActiveCamera();

  // First get the origin of the assembly

  vtkFloat *obj_center = this->InteractionProp->GetCenter();
  
  // GetLength gets the length of the diagonal of the bounding box

  double boundRadius = mBox->GetLength() * 0.5;
//  double boundRadius = this->InteractionProp->GetLength() * 0.5;

  // Get the view up and view right vectors

  double view_up[3], view_look[3], view_right[3];

  cam->OrthogonalizeViewUp();
  cam->ComputeViewPlaneNormal();
  cam->GetViewUp(view_up);
  vtkMath::Normalize(view_up);
  cam->GetViewPlaneNormal(view_look);
  vtkMath::Cross(view_up, view_look, view_right);
  vtkMath::Normalize(view_right);
  
  // Get the furtherest point from object position+origin

  vtkFloat outsidept[3];

  outsidept[0] = obj_center[0] + view_right[0] * boundRadius;
  outsidept[1] = obj_center[1] + view_right[1] * boundRadius;
  outsidept[2] = obj_center[2] + view_right[2] * boundRadius;
  
  // Convert them to display coord

  vtkFloat disp_obj_center[3];

  this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
                              disp_obj_center);

  this->ComputeWorldToDisplay(outsidept[0], outsidept[1], outsidept[2], 
                              outsidept);
  
  double radius = sqrt(vtkMath::Distance2BetweenPoints(disp_obj_center,
                                                       outsidept));
  double nxf = 
    ((double)rwi->GetEventPosition()[0] - (double)disp_obj_center[0]) / radius;

  double nyf = 
    ((double)rwi->GetEventPosition()[1] - (double)disp_obj_center[1]) / radius;

  double oxf = 
    ((double)rwi->GetLastEventPosition()[0] - (double)disp_obj_center[0]) / radius;

  double oyf = 
    ((double)rwi->GetLastEventPosition()[1] - (double)disp_obj_center[1]) / radius;

  if (((nxf * nxf + nyf * nyf) <= 1.0) &&
      ((oxf * oxf + oyf * oyf) <= 1.0))
    {
    double newXAngle = asin(nxf) * vtkMath::RadiansToDegrees();
    double newYAngle = asin(nyf) * vtkMath::RadiansToDegrees();
    double oldXAngle = asin(oxf) * vtkMath::RadiansToDegrees();
    double oldYAngle = asin(oyf) * vtkMath::RadiansToDegrees();
    
    double scale[3];
    scale[0] = scale[1] = scale[2] = 1.0;

    double **rotate = new double*[2];

    rotate[0] = new double[4];
    rotate[1] = new double[4];
    
    rotate[0][0] = newXAngle - oldXAngle;
    rotate[0][1] = view_up[0];
    rotate[0][2] = view_up[1];
    rotate[0][3] = view_up[2];
    
    rotate[1][0] = oldYAngle - newYAngle;
    rotate[1][1] = view_right[0];
    rotate[1][2] = view_right[1];
    rotate[1][3] = view_right[2];
    
    
    this->Prop3DTransform(this->InteractionProp,
                          obj_center,
                          2, 
                          rotate, 
                          scale);
    
    delete [] rotate[0];
    delete [] rotate[1];
    delete [] rotate;
    
    if (this->AutoAdjustCameraClippingRange)
      {
      this->CurrentRenderer->ResetCameraClippingRange();
      }

    rwi->Render();
    }
}
  

void iMeasuringBoxInteractorStyle::OnChar() 
{

	this->FindPokedRenderer(0, 0);
	this->FindPickedActor(0, 0);

	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	float s = mBox->getSize();
	float o = mBox->getOpacity();

	// catch keycodes
	switch (rwi->GetKeyCode())
	{
    case 'a':
    case 'A': { mBox->setSize(s*0.95); break; }
    case 'z':
    case 'Z': { mBox->setSize(s/0.95); break; }
    case 's':
    case 'S': { mBox->setOpacity(o*0.8); break; }
    case 'x':
    case 'X': { mBox->setOpacity(o/0.8); break; }
	case 'f':
	case 'F': { this->CurrentRenderer->GetActiveCamera()->SetFocalPoint(this->mBox->GetPosition()); break; }
    case 'u':
    case 'U': { rwi->UserCallback(); break; }
	}

	rwi->Render();

}


#else


//----------------------------------------------------------------------------
//  Replace scale with panning along Z axis - need to copy the whole function
//  because vtkInteractorStyleTrackballActor function are not declared virtual - 
//  BAD DESIGN!!!
//----------------------------------------------------------------------------
void iMeasuringBoxInteractorStyle::OnMouseMove(int vtkNotUsed(ctrl), 
                                                   int vtkNotUsed(shift),
                                                   int x, int y) 
{
	if (this->State == VTK_INTERACTOR_STYLE_ACTOR_ROTATE)
    {
		this->FindPokedCamera(x, y);
		this->RotateXY(x, y, this->LastPos[0], this->LastPos[1]);
    }
	else if (this->State == VTK_INTERACTOR_STYLE_ACTOR_PAN)
    {
		this->FindPokedCamera(x, y);
		//
		//  Here is 1 line change
		//
		this->PanInScreen(x, y, this->LastPos[0], this->LastPos[1]);
    }
	else if (this->State == VTK_INTERACTOR_STYLE_ACTOR_ZOOM)
    {
		this->FindPokedCamera(x, y);
		this->DollyXY(this->LastPos[0] - x, this->LastPos[1] - y);
    }
	else if (this->State == VTK_INTERACTOR_STYLE_ACTOR_SPIN)
    {
		this->FindPokedCamera(x, y);
		this->SpinXY(x, y, this->LastPos[0], this->LastPos[1]);
    }
	else if (this->State == VTK_INTERACTOR_STYLE_ACTOR_SCALE)
    {
		this->FindPokedCamera(x, y);
		//
		//  Here is 1 line change
		//
		this->PanXYZ(x, y, this->LastPos[0], this->LastPos[1]);
    }
	
	this->LastPos[0] = x;
	this->LastPos[1] = y;
}
//
//  Next three functions are copied identically from vtkInteractorStyleTrackballActor 
//  for this->FindPickedActor to work - again, bad design of the VTK class
//
void iMeasuringBoxInteractorStyle::OnLeftButtonDown(int ctrl, int shift, 
													int x, int y) 
{
	if (this->HasObserver(vtkCommand::LeftButtonPressEvent)) 
    {
		this->InvokeEvent(vtkCommand::LeftButtonPressEvent,NULL);
		return;
    }
	this->FindPokedRenderer(x, y);
	this->FindPickedActor(x, y);
	
	if (this->CurrentRenderer == NULL)
    {
		return;
    }
	
	this->UpdateInternalState(ctrl, shift, x, y);
	if (shift)
    {
		this->State = VTK_INTERACTOR_STYLE_ACTOR_PAN;
    }
	else if (this->CtrlKey)
    {
		this->State = VTK_INTERACTOR_STYLE_ACTOR_SPIN;
    }
	else
    {
		this->State = VTK_INTERACTOR_STYLE_ACTOR_ROTATE;
    }
}


void iMeasuringBoxInteractorStyle::OnMiddleButtonDown(int ctrl, int shift, 
													  int x, int y) 
{
	if (this->HasObserver(vtkCommand::MiddleButtonPressEvent)) 
    {
		this->InvokeEvent(vtkCommand::MiddleButtonPressEvent,NULL);
		return;
    }
	this->FindPokedRenderer(x, y);
	this->FindPickedActor(x, y);
	
	if (this->CurrentRenderer == NULL)
    {
		return;
    }
	
	this->UpdateInternalState(ctrl, shift, x, y);
	if (this->CtrlKey)
    {
		this->State = VTK_INTERACTOR_STYLE_ACTOR_ZOOM;
    }
	else
    {
		this->State = VTK_INTERACTOR_STYLE_ACTOR_PAN;
    }
}


void iMeasuringBoxInteractorStyle::OnRightButtonDown(int vtkNotUsed(ctrl),
													 int vtkNotUsed(shift), 
													 int x, int y) 
{
	if (this->HasObserver(vtkCommand::RightButtonPressEvent)) 
    {
		this->InvokeEvent(vtkCommand::RightButtonPressEvent,NULL);
		return;
    }
	this->FindPokedRenderer(x, y);
	this->FindPickedActor(x, y);
	
	if (this->CurrentRenderer == NULL)
    {
		return;
    }
	
	this->State = VTK_INTERACTOR_STYLE_ACTOR_SCALE;
}
//
//  Panning in XYZ - a copy of vtkInteractorStyleTrackballActor::PanXY() with 
//  one change
//
void iMeasuringBoxInteractorStyle::PanXYZ(int x, int y, int oldX, int oldY)
{
	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// use initial center as the origin from which to pan
	float *center = this->InteractionProp->GetCenter();
	this->ObjCenter[0] = center[0];
	this->ObjCenter[1] = center[1];
	this->ObjCenter[2] = center[2];
	this->ComputeWorldToDisplay(this->ObjCenter[0], this->ObjCenter[1],
		this->ObjCenter[2], this->DispObjCenter);
	this->FocalDepth = this->DispObjCenter[2];
	
	this->HighlightProp3D(NULL);
	
	this->ComputeDisplayToWorld(double(x), double(y),
		this->FocalDepth,
		this->NewPickPoint);
	
	this->ComputeDisplayToWorld(double(oldX), double(oldY),
		this->FocalDepth, this->OldPickPoint);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	float d = sqrt(vtkMath::Distance2BetweenPoints(this->NewPickPoint,this->OldPickPoint));
	
	float wc[4], wx[4], wy[4], wz[4];
	float ex[3], ey[3], ez[3];
	float dx[2];
	int i;
	
	dx[0] = x - oldX;
	dx[1] = y - oldY;
	vtkMath::Normalize2D(dx);

	this->ComputeWorldToDisplay(0.0,0.0,0.0,wc);
	this->ComputeWorldToDisplay(1.0,0.0,0.0,wx);
	this->ComputeWorldToDisplay(0.0,1.0,0.0,wy);
	this->ComputeWorldToDisplay(0.0,0.0,1.0,wz);

	for(i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
		ez[i] = wz[i] - wc[i];
	}
	
	vtkMath::Normalize2D(ex);
	vtkMath::Normalize2D(ey);
	vtkMath::Normalize2D(ez);
	
	float crossProduct[3];
	crossProduct[0] = dx[0]*ex[0] + dx[1]*ex[1];
	crossProduct[1] = dx[0]*ey[0] + dx[1]*ey[1];
	crossProduct[2] = dx[0]*ez[0] + dx[1]*ez[1];

	for(i=0; i<3; i++) crossProduct[i] = pow(crossProduct[i],9.0);

	vtkMath::Normalize(crossProduct);

	this->MotionVector[0] = d*crossProduct[0];
	this->MotionVector[1] = d*crossProduct[1];
	this->MotionVector[2] = d*crossProduct[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != NULL)
    {
		vtkTransform *t = vtkTransform::New();
		if(t == 0) reportNullPointer(8403);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(this->MotionVector[0], this->MotionVector[1], 
			this->MotionVector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(this->MotionVector);
    }
	
	rwi->Render();
}
//
//  Panning in screen - a copy of vtkInteractorStyleTrackballActor::PanXY() with 
//  one change
//
void iMeasuringBoxInteractorStyle::PanInScreen(int vtkNotUsed(x), int y, int vtkNotUsed(oldX), int oldY)
{
	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// use initial center as the origin from which to pan
	float *center = this->InteractionProp->GetCenter();
	this->ObjCenter[0] = center[0];
	this->ObjCenter[1] = center[1];
	this->ObjCenter[2] = center[2];
	this->ComputeWorldToDisplay(this->ObjCenter[0], this->ObjCenter[1],
		this->ObjCenter[2], this->DispObjCenter);
	this->FocalDepth = this->DispObjCenter[2];
	
	this->HighlightProp3D(NULL);
	
	this->ComputeDisplayToWorld(0.0, double(y),
		this->FocalDepth, this->NewPickPoint);
	
	this->ComputeDisplayToWorld(0.0, double(oldY),
		this->FocalDepth, this->OldPickPoint);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	float d = sqrt(vtkMath::Distance2BetweenPoints(this->NewPickPoint,this->OldPickPoint));
	if(y > oldY) d = -d;
	
	float wc[4], wx[4], wy[4];
	float ex[3], ey[3], ez[3];

	this->ComputeDisplayToWorld(0.0,0.0,this->FocalDepth,wc);
	this->ComputeDisplayToWorld(1.0,0.0,this->FocalDepth,wx);
	this->ComputeDisplayToWorld(0.0,1.0,this->FocalDepth,wy);

	for(int i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
	}

	vtkMath::Cross(ex,ey,ez);
	vtkMath::Normalize(ez);

	this->MotionVector[0] = d*ez[0];
	this->MotionVector[1] = d*ez[1];
	this->MotionVector[2] = d*ez[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != NULL)
    {
		vtkTransform *t = vtkTransform::New();
		if(t == 0) reportNullPointer(8404);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(this->MotionVector[0], this->MotionVector[1], 
			this->MotionVector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(this->MotionVector);
    }
	
	rwi->Render();
}


void iMeasuringBoxInteractorStyle::OnChar(int vtkNotUsed(ctrl), int vtkNotUsed(shift), 
                                         char keycode, int vtkNotUsed(repeatcount)) 
{

	this->FindPokedRenderer(0, 0);
	this->FindPickedActor(0, 0);

	if (this->CurrentRenderer == NULL || this->InteractionProp == NULL)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	float s = mBox->getSize();
	float o = mBox->getOpacity();

	// catch keycodes
	switch (keycode)
	{
    case 'a':
    case 'A': { mBox->setSize(s*0.95); break; }
    case 'z':
    case 'Z': { mBox->setSize(s/0.95); break; }
    case 's':
    case 'S': { mBox->setOpacity(o*0.95); break; }
    case 'x':
    case 'X': { mBox->setOpacity(o/0.95); break; }
	case 'f':
	case 'F': { this->CurrentRenderer->GetActiveCamera()->SetFocalPoint(this->mBox->GetPosition()); break; }
    case 'u':
    case 'U': { rwi->UserCallback(); break; }
	}

	rwi->Render();
}

#endif
