/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of idataconverter.h
  
=========================================================================*/


#include "iglobals.h"
#include "imeshfromparticlesfilter.h"

#include <vtkImageFFT.h>
#include <vtkImageRFFT.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkStructuredPoints.h>
#include <vtkPointData.h>


void reportNullPointer(int);


//------------------------------------------------------------------------------
iMeshFromParticlesFilter* iMeshFromParticlesFilter::New()
{
	return new iMeshFromParticlesFilter;
}


iMeshFromParticlesFilter::iMeshFromParticlesFilter()
{
	method = 0;
	kernel = 0;
}


void iMeshFromParticlesFilter::ExecuteData(vtkDataObject *)
{

	if(this->GetInput() == 0) return;

	vtkImageData *output = (vtkImageData *)this->GetOutput();

	switch (method) 
	{
	case IMESHFROMPARTICLESFILTER_METHOD_SHEPARD: { this->ExecuteShepardMethod(); break; }
	default: { vtkErrorMacro("Invalid mode in iMeshFromParticlesFilter"); }
	}
	//
	//  Reformat the mesh data - turn the vertical array (1 nvar-D array) into
	//  a horizontal one ( nvar 1-D arrays).
	//
	int ncom = output->GetPointData()->GetScalars()->GetNumberOfComponents();
	if(ncom == 1) return;
	
	vtkIdType size = (vtkIdType)this->SampleDimensions[0]*this->SampleDimensions[1]*this->SampleDimensions[2];
	float *data = (float *)output->GetPointData()->GetScalars()->GetVoidPointer(0);

	float *hold = new float[ncom*size];
	if(hold == 0) reportNullPointer(8501);

	memcpy(hold,data,4*ncom*size);

	int i,j,k,m;
	vtkIdType loff;
	for(k=0; k<this->SampleDimensions[2]; k++)
	{
		for(j=0; j<this->SampleDimensions[1]; j++)
		{
			loff = this->SampleDimensions[0]*(j+this->SampleDimensions[1]*k);
			for(i=0; i<this->SampleDimensions[0]; i++)
			{
				for(m=0; m<ncom; m++)
				{
					data[i+loff+m*size] = hold[m+ncom*(i+loff)];
				}
			}
		}
	}

	delete[] hold;

}


float iMeshFromParticlesFilter::getMemorySize()
{
	return this->GetOutput()->GetActualMemorySize();
}


void iMeshFromParticlesFilter::ExecuteShepardMethod()
{
	vtkIdType ptId, i;
	int j, k, l, numComponents;
	vtkFloat px[3], x[3], spacing[3], origin[3];
	float s, *sum;
	float kernelValue, maxDistance2;
	
	float maxDistance, distance2, *inScalar;
	vtkDataArray *inScalars;
	vtkFloatArray *newScalars;
	vtkIdType numPts, numNewPts, idx;
	int min[3], max[3];
	int jkFactor;
	vtkDataSet *input = this->GetInput();
	vtkImageData *output = this->GetOutput();
	
	vtkDebugMacro(<< "Executing Shepard method");
	
	// Check input
	//
	if ( (numPts=input->GetNumberOfPoints()) < 1 )
    {
		vtkErrorMacro(<<"Points must be defined!");
		return;
    }
	
	if ( (inScalars = input->GetPointData()->GetScalars()) == NULL )
    {
		vtkErrorMacro(<<"Scalars must be defined!");
		return;
    }
	
	//
	//  The first component is the particle number, it will be ignored.
	//
	numComponents = inScalars->GetNumberOfComponents() - 1;
	if ( numComponents == 0 )
    {
		vtkErrorMacro(<<"Input data contain no components!");
		return;
    }
	
	//
	// Allocate
	//
	numNewPts = this->SampleDimensions[0] * this->SampleDimensions[1] 
		* this->SampleDimensions[2];
	
	newScalars = vtkFloatArray::New();
	if(newScalars == 0) reportNullPointer(8502);
	newScalars->SetNumberOfComponents(numComponents);
	newScalars->SetNumberOfTuples(numNewPts);
	
	sum = new float[numNewPts];
	if(sum == 0) reportNullPointer(8503);
	inScalar = new float[numComponents];
	if(inScalar == 0) reportNullPointer(8504);

	for (i=0; i<numNewPts; i++) 
    {
		for(l=0; l<numComponents; l++) newScalars->SetComponent(i,l,0.0);
		sum[i] = 0.0;
    }
	
	output->SetDimensions(this->GetSampleDimensions());
	maxDistance = this->ComputeModelBounds(origin,spacing);
	maxDistance2 = maxDistance*maxDistance;
	
	// Traverse all input points. 
	// Each input point affects voxels within maxDistance.
	//
	int abortExecute=0;
	for (ptId=0; ptId < numPts && !abortExecute; ptId++)
    {
		if ( ! (ptId % 1000) )
		{
			this->UpdateProgress (ptId/numPts);
			if (this->GetAbortExecute())
			{
				abortExecute = 1;
				break;
			}
		}
		
		input->GetPoint(ptId,px);
		for(l=0; l<numComponents; l++)
		inScalar[l] = inScalars->GetComponent(ptId,l+1); // the 0-th component is particle number
		
		for (i=0; i<3; i++) //compute dimensional bounds in data set
		{
			float amin = (float)((px[i] - maxDistance) - origin[i]) / spacing[i];
			float amax = (float)((px[i] + maxDistance) - origin[i]) / spacing[i];
			min[i] = (int) amin;
			max[i] = (int) amax;
			
			if (min[i] < amin)
			{
				min[i]++; // round upward to nearest integer to get min[i]
			}
			if (max[i] > amax)
			{
				max[i]--; // round downward to nearest integer to get max[i]
			}
			
			if (min[i] < 0)
			{
				min[i] = 0; // valid range check
			}
			if (max[i] >= this->SampleDimensions[i]) 
			{
				max[i] = this->SampleDimensions[i] - 1;
			}
		}
		
		jkFactor = this->SampleDimensions[0]*this->SampleDimensions[1];
		for (k = min[2]; k <= max[2]; k++) 
		{
			x[2] = spacing[2] * k + origin[2];
			for (j = min[1]; j <= max[1]; j++)
			{
				x[1] = spacing[1] * j + origin[1];
				for (i = min[0]; i <= max[0]; i++) 
				{
					x[0] = spacing[0] * i + origin[0];
					idx = jkFactor*k + this->SampleDimensions[0]*j + i;
					
					distance2 = vtkMath::Distance2BetweenPoints(x,px);
					//
					//  Compute the smoothing kernel
					//
					switch (kernel) 
					{
					case IMESHFROMPARTICLESFILTER_KERNEL_SHEPARD: 
						{
							if(distance2==0.0) kernelValue = VTK_LARGE_FLOAT; else kernelValue = 1.0 / distance2;
							break; 
						}
					case IMESHFROMPARTICLESFILTER_KERNEL_LINEAR: 
						{
							kernelValue = 1.0 - distance2/maxDistance2;
							break; 
						}
					case IMESHFROMPARTICLESFILTER_KERNEL_TOPHAT: 
					default: 
						{
							kernelValue = 1.0;
						}
					}
					if(distance2 > maxDistance2) kernelValue = 0.0;
					
					sum[idx] += kernelValue;
					for(l=0; l<numComponents; l++)
					{
						s = newScalars->GetComponent(idx,l);
						newScalars->SetComponent(idx,l,s+(inScalar[l]*kernelValue));
					}

				}
			}
		}
    }
	
	// Run through scalars and compute final values
	//
	for (ptId=0; ptId<numNewPts; ptId++)
    {
		for(l=0; l<numComponents; l++)
		{
			s = newScalars->GetComponent(ptId,l);
			if ( sum[ptId] != 0.0 )
			{
				newScalars->SetComponent(ptId,l,s/sum[ptId]);
			}
			else
			{
				newScalars->SetComponent(ptId,l,this->NullValue);
			}
		}
    }
	
	// Update self
	//
	delete [] inScalar;
	delete [] sum;
	output->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

}

