/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of idatareader.h

=========================================================================*/

#include <stdio.h>
#include <string.h>

#include "iglobals.h"
#include "idatareader.h"

#include "imath.h"
#include "ienvironment.h"
#include "ilimits.h"
#include "ipalette.h"
#include "ivtk.h"
#include "iuniformmeshdata.h"
#include "ipolygonaldata.h"
#include "imeshfromparticlesfilter.h"
#include "ifile.h"
#include "ivtkcallback.h"
#include "idatainfo.h"
#include "iexpressionparser.h"
#include "iobjectfactory.h"
#include "iprogressbar.h"

#include <vtkCommand.h>
#include <vtkObjectFactory.h>
#include <vtkArrayCalculator.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkCellArray.h>
#include <vtkStructuredPoints.h>


void reportNullPointer(int ec);

//
//  Define static members
//
bool iDataReader::endinessOfMachine;


//------------------------------------------------------------------------------
iDataReader* iDataReader::New(iVTK* v)
{
	return (iDataReader *)iObjectFactory::createObject("DataReader",v);
}


iDataReader::iDataReader(iVTK *v) : iObject(v)
{
	int i;

	modeCellToPoint = 0;
	pointsAreFloat = true;

	adjustableBounds = true;
	endinessOfData = true;

	aborted = false;

	maxDataChannel = dataChannel = 0;
	for(i=0; i<=MAXDATACHANNEL; i++) 
	{
		dataChannelActive[i] = false;
		maxDataStream[i] = -1;
		dataStream[i] = 0;
	}
	maxDataStream[0] = 0;  // standart IFRIT data stream
	dataChannelActive[0] = true;

	limits[0] = new iLimits*[1];
	if(limits[0] == 0) reportNullPointer(8209);
	limits[0][0] = new iLimits(myVTK);
	if(limits[0][0] == 0) reportNullPointer(8209);
	for(i=1; i<=MAXDATACHANNEL; i++)
	{
		limits[i] = 0;
	}

	meshData = 0;
	vectData = 0;
	tensData = 0;
	for(i=0; i<3; i++)
	{
		meshDims[i] = 0;
		vectDims[i] = 0;
		tensDims[i] = 0;
		newShift[i] = oldShift[i] = 0.0;
	}
	meshSize = vectSize = tensSize = 0L;
	meshSet = partSet = vectSet = tensSet = false;
	currec = -1;
	endrec = 9999;
	periodicityLeader = 'z';
	setLeader = ' ';

	for(i=0; i<=NVARMAX; i++)
	{
		levMin[i] = 0.0;
		levMax[i] = 1.0;
	}

	meshArrayOutput = 1;
	meshArrayFunction = "";

	meshDataAssigned = false;

	meshOutput = iUniformMeshData::New();
	if(meshOutput == 0) reportNullPointer(8201);
	meshArray = vtkFloatArray::New();
	if(meshArray == 0) reportNullPointer(8202);
	meshArray->SetName("IfritMesh");
	meshOutput->GetPointData()->SetScalars(meshArray);
	meshOutput->SetNumberOfScalarComponents(0);

	partOutput = iPolygonalData::New();
	if(partOutput == 0) reportNullPointer(8203);

	partBasicScale = 1.0;

	vectOutput = iUniformMeshData::New();
	if(vectOutput == 0) reportNullPointer(8204);
	vectArray = vtkFloatArray::New();
	if(vectArray == 0) reportNullPointer(8205);
	vectArray->SetName("IfritVect");
	vectArray->SetNumberOfComponents(3);
	vectOutput->GetPointData()->SetVectors(vectArray);

	tensOutput = iUniformMeshData::New();
	if(tensOutput == 0) reportNullPointer(8206);
	tensArray = vtkFloatArray::New();
	if(tensArray == 0) reportNullPointer(8207);
	tensArray->SetName("IfritTens");
	tensArray->SetNumberOfComponents(9);
	tensOutput->GetPointData()->SetTensors(tensArray);
	
	passedMeshOutput = iUniformMeshData::New();
	if(passedMeshOutput == 0) reportNullPointer(8201);
	passedPartOutput = iPolygonalData::New();
	if(passedPartOutput == 0) reportNullPointer(8203);
	passedVectOutput = iUniformMeshData::New();
	if(passedVectOutput == 0) reportNullPointer(8204);
	passedTensOutput = iUniformMeshData::New();
	if(passedTensOutput == 0) reportNullPointer(8206);

	part2meshFilter = iMeshFromParticlesFilter::New();
	if(part2meshFilter == 0) reportNullPointer(8208);
	part2meshFilter->SetMaximumDistance(1.0);
	part2meshFilter->SetNullValue(0.0);
	part2meshFilter->SetModelBounds(-1.0,1.0,-1.0,1.0,-1.0,1.0);
	part2meshFilter->SetSampleDimensions(16,16,16);
	part2meshFilter->setMethod(0);
	part2meshFilter->setKernel(0);
	part2meshOn = false;

	meshShifted = partShifted = vectShifted = tensShifted = false;
	meshUpdated = partUpdated = vectUpdated = tensUpdated = false;
	meshAnimatable = partAnimatable = vectAnimatable = tensAnimatable = false;
	//
	//  Check the endiness of the machine - do not trust VTK
	//
	unsigned char c[4];
	c[0] = c[1] = c[2] = (unsigned char)0;
	c[3] = (unsigned char)1;
	endinessOfMachine = (*((int *)c) == 1);

	scaledDim = -1; //max dim

	lastFileName = lastAttemptedFileName = iString("");

    lastMeshFileName = iString("");
    lastPartFileName = iString("");
    lastVectFileName = iString("");
    lastTensFileName = iString("");
	
	partDownsampleMode = 0;
	partDownsampleFactor = 1;

	meshProgressBar = partProgressBar = vectProgressBar = tensProgressBar = 0;

}


iDataReader::~iDataReader()
{

	part2meshFilter->Delete();

	if(meshData && !meshDataAssigned) delete [] meshData;
	if(vectData) delete [] vectData;
	if(tensData) delete [] tensData;

	meshOutput->Delete();
	meshArray->Delete();

	partOutput->Delete();

	vectOutput->Delete();
	vectArray->Delete();

	tensOutput->Delete();
	tensArray->Delete();

	passedMeshOutput->Delete();
	passedPartOutput->Delete();
	passedVectOutput->Delete();
	passedTensOutput->Delete();

	int i, j;
	for(i=0; i<=MAXDATACHANNEL; i++) if(limits[i] != NULL) 
	{
		for(j=0; j<=maxDataStream[i]; j++) if(limits[i][j] != NULL) delete limits[i][j];
		delete [] limits[i];
	}

}


void iDataReader::eraseData(int, int m)
{
	this->setProgressBar(m,0);
	if(m & IDATAREADER_MESH)
	{
		if(!meshDataAssigned )
		{
			meshArray->Initialize();
			if(meshData != 0) delete[] meshData;
			meshData = 0;
			meshSize = 0L; meshDims[0] = meshDims[1] = meshDims[2] = 0;
			meshSet = meshUpdated = meshAnimatable = false;
		}
	}
	if(m & IDATAREADER_PART)
	{
		partOutput->Initialize();
		partSize = 0L;
		partSet = partUpdated = partAnimatable = false;
		if(meshDataAssigned )
		{
			meshArray->Initialize();
			meshSize = 0L; meshDims[0] = meshDims[1] = meshDims[2] = 0;
			meshSet = meshUpdated = meshAnimatable = false;
		}
	}
	if(m & IDATAREADER_VECT)
	{
		vectArray->Initialize();
		if(vectData != 0) delete[] vectData;
		vectData = 0;
		vectSize = 0L; vectDims[0] = vectDims[1] = vectDims[2] = 0;
		vectSet = vectUpdated = vectAnimatable = false;
	}
	if(m & IDATAREADER_TENS)
	{
		tensArray->Initialize();
		if(tensData != 0) delete[] tensData;
		tensData = 0;
		tensSize = 0L; tensDims[0] = tensDims[1] = tensDims[2] = 0;
		tensSet = tensUpdated = tensAnimatable = false;
	}
}


void iDataReader::detachData(int, int m)
{
	if(m & IDATAREADER_MESH)
	{
		passedMeshOutput->Initialize();
	}
	if(m & IDATAREADER_PART)
	{
		passedPartOutput->Initialize();
	}
	if(m & IDATAREADER_VECT)
	{
		passedVectOutput->Initialize();
	}
	if(m & IDATAREADER_TENS)
	{
		passedTensOutput->Initialize();
	}
}


iString iDataReader::getCurrentMeshFileDir()
{ 
	if(currentMeshFileDir.isEmpty())
	{
		currentMeshFileDir = (iEnvironment::getInstance()->get_IFRIT_MESH_DATA_DIR().isEmpty()) ? iEnvironment::getInstance()->get_IFRIT_DATA_DIR() : iEnvironment::getInstance()->get_IFRIT_MESH_DATA_DIR();
	}
	return currentMeshFileDir; 
}


iString iDataReader::getCurrentPartFileDir()
{ 
	if(currentPartFileDir.isEmpty())
	{
		currentPartFileDir = (iEnvironment::getInstance()->get_IFRIT_PART_DATA_DIR().isEmpty()) ? iEnvironment::getInstance()->get_IFRIT_DATA_DIR() : iEnvironment::getInstance()->get_IFRIT_PART_DATA_DIR();
	}
	return currentPartFileDir; 
}


iString iDataReader::getCurrentVectFileDir()
{ 
	if(currentVectFileDir.isEmpty())
	{
		currentVectFileDir = (iEnvironment::getInstance()->get_IFRIT_VECT_DATA_DIR().isEmpty()) ? iEnvironment::getInstance()->get_IFRIT_DATA_DIR() : iEnvironment::getInstance()->get_IFRIT_VECT_DATA_DIR();
	}
	return currentVectFileDir; 
}


iString iDataReader::getCurrentTensFileDir()
{ 
	if(currentTensFileDir.isEmpty())
	{
		currentTensFileDir = (iEnvironment::getInstance()->get_IFRIT_TENS_DATA_DIR().isEmpty()) ? iEnvironment::getInstance()->get_IFRIT_DATA_DIR() : iEnvironment::getInstance()->get_IFRIT_TENS_DATA_DIR();
	}
	return currentTensFileDir; 
}


bool iDataReader::isThereMeshData(int dc, int) 
{ 
	if(dc == 0) return (meshSize > 0); else return false;
}


bool iDataReader::isTherePartData(int dc, int) 
{ 
	if(dc == 0) return (partSize > 0); else return false;
}


bool iDataReader::isThereVectData(int dc, int) 
{ 
	if(dc == 0) return (vectSize > 0); else return false;
}


bool iDataReader::isThereTensData(int dc, int) 
{ 
	if(dc == 0) return (tensSize > 0); else return false;
}


void iDataReader::setParticlesDownsampleMode(int v)
{
	if(v>=0 && v<=2 && v!=partDownsampleMode)
	{
		partDownsampleMode = v;
	}
}


void iDataReader::setParticlesDownsampleFactor(float v)
{
	if(v>0.99999 && v<1.1e6)
	{
		partDownsampleFactor = v;
	}
}


int iDataReader::loadFileSet(const iString &fname, bool saveInTheList)
{
	//
	//  Extra error codes:
	//		11:	no set specified.
	//		12:	filename does not belong to the set.
	//		13:	file type is not correct
	//		14:	set is empty
	//		1 to 4 mean: mesh, part, vect, tens files
	//
	iString suf, root, record;
	int newrec;

	if(currec < 0) return 11;

	record = (fname.right(8)).left(4);													 
	bool ok;																			 
	int i = record.toInt(&ok,10);														 
	iString q;																			 
	q.sprintf("%04d",i);																 
	if(ok && q==record) newrec = i; else return 12;
	
	root = fname.left(fname.length()-8);												 
	suf = fname.right(4);																 

	switch(setLeader)
	{
	case 'a': { if(root!=meshFroot || suf!=meshFsuf) return 13; break; }
	case 'b': { if(root!=vectFroot || suf!=vectFsuf) return 13; break; }
	case 'c': { if(root!=tensFroot || suf!=tensFsuf) return 13; break; }
	case 'd': { if(root!=partFroot || suf!=partFsuf) return 13; break; }
	default: return 13;
	}
	
	return this->loadFileSet(newrec,false,saveInTheList); 
	
}


int iDataReader::loadFileSet(int irec, bool justCheckIfExists, bool saveInTheList)
{
	iString record, fname;
	int ret;

	record.sprintf("%04d",irec);

	aborted = false;

#define LOADFILESET_TWOSTEP(label,Label,retoff)													 \
	if(!aborted && label##Set)																				 \
	{																							 \
		fname = label##Froot + record + label##Fsuf;											 \
		ret = this->load##Label##File(fname,true,justCheckIfExists,saveInTheList);				 \
		if(ret !=0 ) return retoff+ret;															 \
	}

	LOADFILESET_TWOSTEP(mesh,Mesh,100)
	LOADFILESET_TWOSTEP(part,Part,200)
	LOADFILESET_TWOSTEP(vect,Vect,300)
	LOADFILESET_TWOSTEP(tens,Tens,400)

	currec = irec;
	return 0;

}

	
int iDataReader::loadRecord(int rec, bool justCheckIfExists, bool saveInTheList)
{

	iString q;

	q.sprintf("%04d",rec);

	switch (setLeader) 
	{
	case 'a': { q = meshFroot + q + meshFsuf; break; }
	case 'd': { q = partFroot + q + partFsuf; break; }
	case 'b': { q = vectFroot + q + vectFsuf; break; }
	case 'c': { q = tensFroot + q + tensFsuf; break; }
	}
	if(iFile::exists(q)) return this->loadFileSet(rec,justCheckIfExists,saveInTheList);

	return -9;

}


int iDataReader::loadNextRecord(int skip, bool justCheckIfExists, bool saveInTheList)
{

	iString q;
	int rec = currec;

	rec += skip;
	rec++;
	if(rec < 0) rec = 0;

	for(; rec<endrec; rec++) 
	{
		
		q.sprintf("%04d",rec);
		
		switch (setLeader) 
		{
		case 'a': { q = meshFroot + q + meshFsuf; break; }
		case 'd': { q = partFroot + q + partFsuf; break; }
		case 'b': { q = vectFroot + q + vectFsuf; break; }
		case 'c': { q = tensFroot + q + tensFsuf; break; }
		}
		if(iFile::exists(q)) return this->loadFileSet(rec,justCheckIfExists,saveInTheList);

	}

	return -9;

}


int iDataReader::reloadCurrentSet(int mode)
{
	int ret = 0;

	if(mode&IDATAREADER_MESH && !lastMeshFileName.isEmpty())
	{ 
		ret += this->loadMeshFile(lastMeshFileName,this->isSet(),false,false); 
	}
	if(mode&IDATAREADER_PART && !lastPartFileName.isEmpty())
	{
		ret += 100*this->loadPartFile(lastPartFileName,this->isSet(),false,false); 
	}
	if(mode&IDATAREADER_VECT && !lastVectFileName.isEmpty())
	{ 
		ret += 10000*this->loadVectFile(lastVectFileName,this->isSet(),false,false); 
	}
	if(mode&IDATAREADER_TENS && !lastTensFileName.isEmpty())
	{
		ret += 1000000*this->loadTensFile(lastTensFileName,this->isSet(),false,false); 
	}
	meshProgressBar = partProgressBar = vectProgressBar = tensProgressBar = 0;
	return ret;
}


iString iDataReader::getLastFileSetName()
{
	switch (setLeader) 
	{ 
	case 'a': return lastMeshFileName; 
	case 'd': return lastPartFileName; 
	case 'b': return lastVectFileName; 
	case 'c': return lastTensFileName; 
	default: return ""; 
	}
}


#define IDATAREADER_LOADFILE_DECLARATION(label,Label,LABEL,modeChar) \
int iDataReader::load##Label##File(const iString &fname, bool inSet, bool justCheckIfExists, bool saveInTheList) \
{ \
	iString suf; int ret, rec; suf = fname.right(4); aborted = false; \
	lastAttemptedFileName = fname; \
	dataChannel = 0; \
	if(suf.lower() == iString(".txt")) \
	{ \
	    if(justCheckIfExists) ret = (int)(!iFile::exists(fname)); else ret = this->load##Label##TxtFile(fname); \
		mode = iString("t") + modeChar;	\
	} \
    else if(suf.lower() == iString(".bin")) \
	{ \
	    if(justCheckIfExists) ret = (int)(!iFile::exists(fname)); else ret = this->load##Label##BinFile(fname); \
		mode = iString("b") + modeChar;	\
	} \
	else ret = -1; \
	if(!inSet) label##Set = false; \
	if(aborted) { this->eraseData(0,IDATAREADER_##LABEL); ret = -99; } \
	if(ret == 0) \
	{ \
		if(modeChar < periodicityLeader) periodicityLeader = modeChar; \
		last##Label##FileName = fname; \
		current##Label##FileDir = fname; \
		lastFileName = fname; \
		label##Fsuf = suf; \
		suf = (fname.right(8)).left(4); \
		bool ok; \
		rec = suf.toInt(&ok,10); \
		iString q; q.sprintf("%04d",rec);	\
		if(ok && rec>=0 && q==suf) \
		{ \
			label##Animatable = true; \
			label##Froot = fname.left(fname.length()-8); \
			if(currec < 0) \
			{ \
				label##Set = true; \
				setLeader = modeChar; \
				currec = rec; \
				if(saveInTheList) visitedFilesList.insert(visitedFilesList.begin(),(char)(modeChar-32)+fname); \
			} \
			else \
			{ \
				if(rec == currec) \
				{ \
					label##Set = true; \
				} \
				else \
				{ \
					currec = rec; \
					if(!inSet) \
					{ \
						meshSet = partSet = vectSet = tensSet = false; \
						setLeader = modeChar; \
					} \
					label##Set = true; \
					if(saveInTheList) visitedFilesList.insert(visitedFilesList.begin(),(char)(modeChar-32)+fname); \
				} \
			} \
		} \
		else \
		{ \
			label##Set = false; \
			label##Animatable = false; \
			label##Froot = iString("");	\
			if(saveInTheList) visitedFilesList.insert(visitedFilesList.begin(),modeChar+fname); \
		} \
		if(!justCheckIfExists) \
        { \
			label##Shifted = false;	\
			label##Updated = false;	\
			ExecuteData(0); \
			if(aborted) { this->eraseData(0,IDATAREADER_##LABEL); ret = -99; } \
			doDataShift(newShift,true); \
			if(aborted) { this->eraseData(0,IDATAREADER_##LABEL); ret = -99; } \
		} \
	} \
	else label##Size = 0; \
	return ret;	\
}



IDATAREADER_LOADFILE_DECLARATION(mesh,Mesh,MESH,'a')
IDATAREADER_LOADFILE_DECLARATION(part,Part,PART,'d')
IDATAREADER_LOADFILE_DECLARATION(vect,Vect,VECT,'b')
IDATAREADER_LOADFILE_DECLARATION(tens,Tens,TENS,'c')



int iDataReader::loadMeshBinFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int n, n1, n2, n3, ntemp;
	vtkIdType ntot, noff, l;
	vtkIdType nline, nstep, nlast, ret, i;
	float *d;
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_MESH,-0.1);
	this->updateProgress(IDATAREADER_MESH,0.0);
//
//  Read the header
//
	err = false;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(F.readBlock((char *)&n3,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n3);
	if(F.readBlock((char *)&n2,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n2);
	if(F.readBlock((char *)&n1,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n1);
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	
    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
	if(n1 == nmax) limits[0][0]->periodic[2] = true; 
	if(n2 == nmax) limits[0][0]->periodic[1] = true; 
	if(n3 == nmax) limits[0][0]->periodic[0] = true; 

	ntot = (vtkIdType)n1*n2*n3;
//
//  Measure the file size to find out the number of variables
//
	long lp = F.size();
	long nrec = 4L*(ntot+2L);
	int nvar = (lp-20L)/nrec;
//
//  If the file is not a whole number of records, set warno to 1
//
	if((lp-20L) != nvar*nrec) warno = 1;
//
//  If no records, exit
//
	if(nvar < 1) { F.close(); return 3; }
//
//  If the number of records is greater than NVARMAX, set it to NVARMAX
//	and set warno to 2
//
	if(nvar > NVARMAX)
	{
		nvar = NVARMAX;
		warno = 2;
	}
	this->updateProgress(IDATAREADER_MESH,0.01);
//
//  parameters for the Progress Bar
//
    float prog1 = 0.99/NVARMAX;
	nstep = 1 + (int)floor(100.0*prog1);
	nline = ntot/nstep;
	nlast = ntot - nline*nstep;
	float prog2 = prog1/nstep;
//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=meshDims[0] || n2!=meshDims[1] || n3!=meshDims[2] || nvar!=limits[0][0]->getNvar())
	{
		if(meshData!=0 && !meshDataAssigned) delete[] meshData;
		meshData = new float[ntot*nvar];
		if(meshData == 0) reportNullPointer(8210);
		meshDataAssigned = false;
		if(meshData == 0) return 10;
	}
//
//  Read the data from the file.
//
	int ndone = 0;
	d = new float[ntot];
	if(d == 0) reportNullPointer(8211);
    for(n=1; !err && n<=nvar; n++) 
	{
		noff = ntot*ndone;
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		for(i=0; !err && i<nstep; i++) 
		{
			ret = F.readBlock((char *)(d+i*nline),4*nline);
			if(ret != 4*nline) err = true;
			this->updateProgress(IDATAREADER_MESH,0.01+prog1*(n-1)+prog2*(i+1.0));
			if(aborted) { F.close(); return -99; }
		}
		if(nlast != 0) if(F.readBlock((char *)(d+nstep*nline),4*nlast) != 4*nlast) err = true;
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		if(!err)
		{
			if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(d,ntot);
			memcpy((void *)(meshData+noff),(void *)d,ntot*sizeof(float));
			ndone++;
		}
	}
	delete[] d;

	this->updateProgress(IDATAREADER_MESH,1.0);
	F.close();

	if(ndone==0) { if(!meshDataAssigned) delete[] meshData; meshData = 0; return 3; }

	limits[0][0]->setNvar(ndone);
	meshDims[0] = n3;
	meshDims[1] = n2;
	meshDims[2] = n1;
	meshSize = ntot;

	if(ndone < nvar)
	{
		for(l=0; l<meshSize; l++) 
		{
			for(int j=ndone; j<nvar; j++)
			{
				meshData[l+meshSize*j] = -30.0;
			}
		}
	}

	int ii;
	float dv, dmin1, dmin2, dmax1, dmax2;
	for(n=0; n<nvar; n++)
	{
		noff = n*ntot;
		dmin1 = dmin2 = dmax1 = dmax2 = meshData[noff];
		for(ii=1; ii<ntot; ii++)
		{
			dv = meshData[ii+noff];
			if(dv > dmax1) dmax1 = dv; else if(dv > dmax2) dmax2 = dv;
			if(dv < dmin1) dmin1 = dv; else if(dv < dmin2) dmin2 = dv;
		}
		levMin[n+1] = dmin1 + 0.1*(dmin2-dmin1);
		levMax[n+1] = dmax1 + 0.1*(dmax2-dmax1);
	}

	return 0;

}


int iDataReader::loadMeshTxtFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int n, n1, n2, n3, ret;
	iString s;
	long ntot, noff, l;
	float f[999];
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_MESH,-0.1);
	this->updateProgress(IDATAREADER_MESH,0.0);
//
//  Read the header
//
	err = false;
	if(F.readLine(s,1024) == -1) err = true;

	ret = sscanf(s.latin1(),"%d %d %d",&n3,&n2,&n1);
	if(ret != 3) err = true;

    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
	if(n1 == nmax) limits[0][0]->periodic[2] = true;
	if(n2 == nmax) limits[0][0]->periodic[1] = true;
	if(n3 == nmax) limits[0][0]->periodic[0] = true;

	ntot = (vtkIdType)n1*n2*n3;
//
//  Find out the number of variables
//
	if(F.readLine(s,1024) == -1) err = true;
	ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<1 || ret>10) err = true;

	int nvar = ret;
//
//  If no records or an error, exit
//
	if(nvar<1 || err) { F.close(); return 3; }
//
//  If the number of records is greater than NVARMAX, set it to NVARMAX
//	and set warno to 2
//
	if(nvar > NVARMAX)
	{
		nvar = NVARMAX;
		warno = 2;
	}
	this->updateProgress(IDATAREADER_MESH,0.01);

//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=meshDims[0] || n2!=meshDims[1] || n3!=meshDims[2] || nvar!=limits[0][0]->getNvar())
	{
		if(meshData!=0 && !meshDataAssigned) delete[] meshData;
		meshData = new float[ntot*nvar];
		if(meshData == 0) reportNullPointer(8212);
		meshDataAssigned = false;
		if(meshData == 0) return 10;
	}
	for(n=0; n<nvar; n++) *(meshData+n*ntot) = f[n];
//
//  Read the data from the file.
//
	for(l=1; !err && l<ntot; l++) 
	{
		if(F.readLine(s,1024) == -1) err = true;
		ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret != nvar) err = true;
		if((100*l)/ntot < (100*(l+1))/ntot) 
		{
			this->updateProgress(IDATAREADER_MESH,0.01+(float)l/ntot);
			if(aborted) { F.close(); return -99; }
		}
		for(n=0; n<nvar; n++) *(meshData+l+n*ntot) = f[n];
	}	
	this->updateProgress(IDATAREADER_MESH,1.0);
	F.close();

	if(err) return 3;

	limits[0][0]->setNvar(nvar);
	meshDims[0] = n3;
	meshDims[1] = n2;
	meshDims[2] = n1;
	meshSize = ntot;

	int ii;
	float dv, dmin1, dmin2, dmax1, dmax2;
	for(n=0; n<nvar; n++)
	{
		noff = n*ntot;
		dmin1 = dmin2 = dmax1 = dmax2 = meshData[noff];
		for(ii=1; ii<ntot; ii++)
		{
			dv = meshData[ii+noff];
			if(dv > dmax1) dmax1 = dv; else if(dv > dmax2) dmax2 = dv;
			if(dv < dmin1) dmin1 = dv; else if(dv < dmin2) dmin2 = dv;
		}
		levMin[n+1] = dmin1 + 0.1*(dmin2-dmin1);
		levMax[n+1] = dmax1 + 0.1*(dmax2-dmax1);
	}

	return 0;

}


int iDataReader::loadPartBinFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int i, ret;
	iString s;
	vtkIdType ntot = 0, l, l0, loff;
	int ntemp;
	iFile F;
	bool err;
	float ll[3], ur[3];

	this->warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_PART,-0.1);
	this->updateProgress(IDATAREADER_PART,0.0);
//
//  Read the header
//
	err = false;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true; else 
	{
		if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&ntemp);
		ntot = ntemp;
	}
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
    if(ntot<=0 || ntot>1000000000) err = true;
	if(err) { F.close(); return 2; }

	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(F.readBlock((char *)ll,3*sizeof(float)) != 3*sizeof(float)) err = true;
	if(F.readBlock((char *)ur,3*sizeof(float)) != 3*sizeof(float)) err = true;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(err) { F.close(); return 2; }
	if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(ll,3);
	if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(ur,3);
//
//  Measure the file size to find out the number of variables
//
	long lp = F.size();
	long nrec = 4L*(ntot+2L);
	ret = (lp-44L)/nrec;
	if(ret<3 || ret>10) err = true;

	int natt = ret - 3;
//
//  Build particle reading mask
//
	vtkIdType ntot0 = ntot;
	bool *mask = new bool[ntot0];
	if(mask == 0) { F.close(); return 10; }

	ntot = this->buildParticlesMask(ntot0,mask,partDownsampleFactor);
//
//  If no records or an error, exit
//
	if(err) { F.close(); delete [] mask; return 3; }
//
//  If the number of attributes is greater than NATTMAX, set it to NATTMAX
//	and set warno to 2
//
	if(natt > NATTMAX)
	{
		natt = NATTMAX;
		warno = 2;
	}
	this->updateProgress(IDATAREADER_PART,0.01);
//
//  Allocate memory; erase the old data if needed.
//
	vtkPoints *newPoints;
	if(pointsAreFloat)
	{
		newPoints = vtkPoints::New(VTK_FLOAT);
	}
	else
	{
		newPoints = vtkPoints::New(VTK_DOUBLE);
	}
	if(newPoints == 0) reportNullPointer(8213);
	vtkCellArray *newVerts = vtkCellArray::New();
	if(newVerts == 0) reportNullPointer(8214);
	vtkFloatArray *newScalars = 0;

// Allocates and sets MaxId
	newPoints->SetNumberOfPoints(ntot);
// This allocates but does not set MaxId
	newVerts->Allocate(newVerts->EstimateSize(ntot,1));

	newScalars = vtkFloatArray::New();
	if(newScalars == 0) reportNullPointer(8215);
	newScalars->SetNumberOfComponents(natt+1);
// Allocates and sets MaxId
	newScalars->SetNumberOfTuples(ntot);

	float *xptrF = 0;
	double *xptrD = 0;

	if(pointsAreFloat)
	{
		xptrF = (float *)newPoints->GetVoidPointer(0);
	}
	else
	{
		xptrD = (double *)newPoints->GetVoidPointer(0);
	}
	float *sptr = (float *)newScalars->GetVoidPointer(0);

	if((xptrF==0&&xptrD==0) || sptr==0 || newVerts->GetPointer()==0)
	{
		F.close();
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		return 10;
	}
//
//  parameters for the Progress Bar
//
    float prog1 = 0.99/(natt+3);
	int nstep = 1 + (int)floor(100.0*prog1);
	int nline = ntot0/nstep;
	int nlast = ntot0 - nline*nstep;
	float prog2 = prog1/nstep;
//
//  Read the data from the file.
//
	float amin[NATTMAX+1];
	float amax[NATTMAX+1];
	for(i=1; i<=natt; i++)
	{
		amin[i] = 1.0e35;
		amax[i] = -1.0e35;
	}

	int n, ndone = 0;
	float *d = new float[ntot0];
	if(d == 0) 
	{ 
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		delete [] mask; 
		return 10; 
	}
    for(n=1; !err && n<=natt+3; n++) 
	{
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		for(i=0; !err && i<nstep; i++) 
		{
			ret = F.readBlock((char *)(d+i*nline),4*nline);
			if(ret != 4*nline) err = true;
			this->updateProgress(IDATAREADER_PART,0.01+prog1*(n-1)+prog2*(i+1.0));
			if(aborted) { F.close(); return -99; }
		}
		if(nlast != 0) if(F.readBlock((char *)(d+nstep*nline),4*nlast) != 4*nlast) err = true;
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		if(!err)
		{
			if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(d,ntot0);
			if(n < 4) 
			{
				l = 0;
				if(pointsAreFloat)
				{
					for(l0=0; l0<ntot0; l0++) if(mask[l0]) 
					{
						xptrF[n-1+3*l] = -1.0 + 2.0*(d[l0]-ll[n-1])/(ur[n-1]-ll[n-1]);
						if(n == 3) sptr[(natt+1)*l] = l; l++;
					}
				}
				else
				{
					for(l0=0; l0<ntot0; l0++) if(mask[l0]) 
					{
						xptrD[n-1+3*l] = -1.0 + 2.0*(d[l0]-ll[n-1])/(ur[n-1]-ll[n-1]);
						if(n == 3) sptr[(natt+1)*l] = l; l++;
					}
				}
			} 
			else 
			{
				l = 0;
				for(l0=0; l0<ntot0; l0++) if(mask[l0])
				{
					sptr[n-3+(natt+1)*l] = d[l0];
					amin[n-3] = (amin[n-3]>d[l0]) ? d[l0] : amin[n-3];
					amax[n-3] = (amax[n-3]<d[l0]) ? d[l0] : amax[n-3]; l++;
				}
			}
			ndone++;
		}
	}
	delete[] d;

	if(ndone > 2) 
	{
		for(l=0; l<ntot; l++) 
		{
			newVerts->InsertNextCell(1);
			newVerts->InsertCellPoint(l);
		}
	} else err = true;

	this->updateProgress(IDATAREADER_PART,1.0);
	F.close();

	delete [] mask; 

	if(err) 
	{
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		return 3; 
	}
//
//  Boundary condition
//
	if(periodicityLeader >= 'd') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = limits[0][0]->periodicBC;
	}

	if(limits[0][0]->periodicBC)
	{
		if(pointsAreFloat)
		{
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<3; i++)
				{
					loff = i + 3*l;
					if(limits[0][0]->periodic[i])
					{
						xptrF[loff] -= 2.0*floor(0.5*(1.0+xptrF[loff]));
					}
				}
			}
		}
		else
		{
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<3; i++)
				{
					loff = i + 3*l;
					if(limits[0][0]->periodic[i])
					{
						xptrD[loff] -= 2.0*floor(0.5*(1.0+xptrD[loff]));
					}
				}
			}
		}
	}
//
//  Finish updating
//
	limits[0][0]->setNatt(natt);
	partSize = ntot;

	for(i=1; i<=natt; i++)
	{
		limits[0][0]->setAttMin(i,amin[i]);
		limits[0][0]->setAttMax(i,amax[i]);
		limits[0][0]->setAttLow(i,amin[i]);
		limits[0][0]->setAttHigh(i,amax[i]);
		limits[0][0]->setAttLow(i,amin[i]);
	}

	partOutput->Initialize();

	partBasicScale = ur[0] - ll[0];

	partOutput->SetPoints(newPoints);
	newPoints->Delete();

	partOutput->SetVerts(newVerts);
	newVerts->Delete();

 	partOutput->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

	vtkFloatArray *newNormals = vtkFloatArray::New();
	newNormals->SetNumberOfComponents(3);
// Allocates and sets MaxId
    newNormals->SetNumberOfTuples(ntot);
	if(newNormals->GetVoidPointer(0) == 0)
	{
		newNormals->Delete();
		return 10;
	}
	for(l=0; l<ntot; l++) newNormals->SetTuple3(l,0.0,0.0,1.0);
    partOutput->GetPointData()->SetNormals(newNormals);
	newNormals->Delete();
//    partOutput->GetPointData()->SetNormals(0);
	
	if(part2meshOn) this->createMeshFromParticles();

	return 0;

}


int iDataReader::loadPartTxtFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int i, ret;
	iString s;
	long ntot;
	vtkIdType l, l0;
	iFile F;
	bool err;
	double llD[3], urD[3], xyzD[3], fD[99];
	float llF[3], urF[3], xyzF[3], fF[99];
	char *axisName[3];

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_PART,-0.1);
	this->updateProgress(IDATAREADER_PART,0.0);
//
//  Read the header
//
	err = false;

	vtkPoints *newPoints;
	if(pointsAreFloat)
	{
		newPoints = vtkPoints::New(VTK_FLOAT);
	}
	else
	{
		newPoints = vtkPoints::New(VTK_DOUBLE);
	}
	if(newPoints == 0) reportNullPointer(8220);

	if(F.readLine(s,1024) == -1) err = true;
	ret = sscanf(s.latin1(),"%ld",&ntot);
	if(ret != 1) err = true;
    if(ntot<=0 || ntot>1000000000) err = true;
	if(err) { F.close(); return 2; }

	for(i=0; i<3; i++) axisName[i] = new char[8192];

	if(F.readLine(s,1024) == -1) err = true;
	if(pointsAreFloat)
	{
		ret = sscanf(s.latin1(),"%g %g %g %g %g %g %s %s %s",&llF[0],&llF[1],&llF[2],&urF[0],&urF[1],&urF[2],axisName[0],axisName[1],axisName[2]);
		if(ret == 9) myVTK->setBox3Axes(axisName[0],axisName[1],axisName[2],llF[0],urF[0],llF[1],urF[1],llF[2],urF[2]);
	}
	else
	{
		ret = sscanf(s.latin1(),"%lg %lg %lg %lg %lg %lg %s %s %s",&llD[0],&llD[1],&llD[2],&urD[0],&urD[1],&urD[2],axisName[0],axisName[1],axisName[2]);
		if(ret == 9) myVTK->setBox3Axes(axisName[0],axisName[1],axisName[2],llD[0],urD[0],llD[1],urD[1],llD[2],urD[2]);
	}
	if(ret!=6 && ret!=9) err = true;
	if(err) { F.close(); return 2; }

	for(i=0; i<3; i++) delete axisName[i];
//
//  Find out the number of variables
//
	if(F.readLine(s,1024) == -1) err = true;
	if(pointsAreFloat)
	{
		ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&fF[0],&fF[1],&fF[2],&fF[3],&fF[4],&fF[5],&fF[6],&fF[7],&fF[8],&fF[9]);
	}
	else
	{
		ret = sscanf(s.latin1(),"%lg %lg %lg %g %g %g %g %g %g %g",&fD[0],&fD[1],&fD[2],&fF[3],&fF[4],&fF[5],&fF[6],&fF[7],&fF[8],&fF[9]);
	}
	if(ret<3 || ret>10) err = true;

	int natt = ret - 3;
//
//  Build particle reading mask
//
	vtkIdType ntot0 = ntot;
	bool *mask = new bool[ntot0];
	if(mask == 0) { F.close(); return 10; }

	ntot = this->buildParticlesMask(ntot0,mask,partDownsampleFactor);
//
//  If no records or an error, exit
//
	if(err) { F.close(); delete [] mask; return 3; }
//
//  If the number of records is greater than NVARMAX, set it to NVARMAX
//	and set warno to 2
//
	if(natt > NATTMAX)
	{
		natt = NATTMAX;
		warno = 2;
	}
	this->updateProgress(IDATAREADER_PART,0.01);
//
//  Allocate memory; erase the old data if needed.
//
	vtkCellArray *newVerts = vtkCellArray::New();
	if(newVerts == 0) reportNullPointer(8221);
	vtkFloatArray *newScalars = 0;

// Allocates and sets MaxId
	newPoints->SetNumberOfPoints(ntot);
// This allocates but does not set Max Id
	newVerts->Allocate(newVerts->EstimateSize(ntot,1));

	newScalars = vtkFloatArray::New();
	if(newScalars == 0) reportNullPointer(8222);
	newScalars->SetNumberOfComponents(natt+1);
// Allocates and sets MaxId
	newScalars->SetNumberOfTuples(ntot);

	if(newPoints->GetVoidPointer(0)==0 || newScalars->GetVoidPointer(0)==0 || newVerts->GetPointer()==0)
	{
		F.close();
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		return 10;
	}

	l = 0;
	if(mask[0])
	{
		newVerts->InsertNextCell(1);
		newVerts->InsertCellPoint(0);
		if(pointsAreFloat)
		{
			for(i=0; i<3; i++) xyzF[i] = -1.0 + 2.0*(fF[i]-llF[i])/(urF[i]-llF[i]);
			newPoints->SetPoint(l,xyzF);
		}
		else
		{
			for(i=0; i<3; i++) xyzD[i] = -1.0 + 2.0*(fD[i]-llD[i])/(urD[i]-llD[i]);
			newPoints->SetPoint(l,xyzD);
		}
		fF[2] = 0;
		newScalars->SetTuple(l,fF+2);
		l++;
	}
//
//  Read the data from the file.
//
	float amin[NATTMAX+1];
	float amax[NATTMAX+1];
	for(i=1; i<=natt; i++)
	{
		amin[i] = fF[2+i];
		amax[i] = fF[2+i];
	}

    for(l0=1; !err && l0<ntot0; l0++) 
	{
		if(F.readLine(s,1024) == -1) err = true;
		if(pointsAreFloat)
		{
			ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&fF[0],&fF[1],&fF[2],&fF[3],&fF[4],&fF[5],&fF[6],&fF[7],&fF[8],&fF[9]);
		}
		else
		{
			ret = sscanf(s.latin1(),"%lg %lg %lg %g %g %g %g %g %g %g",&fD[0],&fD[1],&fD[2],&fF[3],&fF[4],&fF[5],&fF[6],&fF[7],&fF[8],&fF[9]);
		}
		if(ret != natt+3) err = true;
		if((100*l0)/ntot0 < (100*(l0+1))/ntot0) 
		{
			this->updateProgress(IDATAREADER_PART,0.01+(float)l0/ntot0);
			if(aborted) { F.close(); return -99; }
		}
		if(mask[l0])
		{
			newVerts->InsertNextCell(1);
			newVerts->InsertCellPoint(l);
			if(pointsAreFloat)
			{
				for(i=0; i<3; i++) xyzF[i] = -1.0 + 2.0*(fF[i]-llF[i])/(urF[i]-llF[i]);
				newPoints->SetPoint(l,xyzF);
			}
			else
			{
				for(i=0; i<3; i++) xyzD[i] = -1.0 + 2.0*(fD[i]-llD[i])/(urD[i]-llD[i]);
				newPoints->SetPoint(l,xyzD);
			}
			fF[2] = l;
			newScalars->SetTuple(l,fF+2);
			for(i=1; i<=natt; i++)
			{
				amin[i] = (amin[i]>fF[2+i]) ? fF[2+i] : amin[i];
				amax[i] = (amax[i]<fF[2+i]) ? fF[2+i] : amax[i];
			}
			l++;
		}
	}	
	this->updateProgress(IDATAREADER_PART,1.0);
	F.close();

	delete [] mask;

	if(err) 
	{
		newPoints->Delete();
		newVerts->Delete();
		newScalars->Delete();
		return 3; 
	}
//
//  Boundary condition
//
	if(periodicityLeader >= 'd') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = limits[0][0]->periodicBC;
	}

	if(limits[0][0]->periodicBC)
	{
		vtkIdType loff;
		if(pointsAreFloat)
		{
			float *xptrF = (float *)newPoints->GetVoidPointer(0);
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<3; i++)
				{
					loff = i + 3*l;
					if(limits[0][0]->periodic[i])
					{
						xptrF[loff] -= 2.0*floor(0.5*(1.0+xptrF[loff]));
					}
				}
			}
		}
		else
		{
			double *xptrD = (double *)newPoints->GetVoidPointer(0);
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<3; i++)
				{
					loff = i + 3*l;
					if(limits[0][0]->periodic[i])
					{
						xptrD[loff] -= 2.0*floor(0.5*(1.0+xptrD[loff]));
					}
				}
			}
		}
	}
//
//  Finish updating
//
	limits[0][0]->setNatt(natt);
	partSize = ntot;

	for(i=1; i<=natt; i++)
	{
		limits[0][0]->setAttMin(i,amin[i]);
		limits[0][0]->setAttMax(i,amax[i]);
		limits[0][0]->setAttLow(i,amin[i]);
		limits[0][0]->setAttHigh(i,amax[i]);
		limits[0][0]->setAttLow(i,amin[i]);
	}

	partOutput->Initialize();

	if(pointsAreFloat)
	{
		partBasicScale = urF[0] - llF[0];
	}
	else
	{
		partBasicScale = urD[0] - llD[0];
	}

	partOutput->SetPoints(newPoints);
	newPoints->Delete();

	partOutput->SetVerts(newVerts);
	newVerts->Delete();

	partOutput->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

	vtkFloatArray *newNormals = vtkFloatArray::New();
	newNormals->SetNumberOfComponents(3);
// Allocates and sets MaxId
    newNormals->SetNumberOfTuples(ntot);
	if(newNormals->GetVoidPointer(0) == 0)
	{
		newNormals->Delete();
		return 10;
	}
	for(l=0; l<ntot; l++) newNormals->SetTuple3(l,0.0,0.0,1.0);
    partOutput->GetPointData()->SetNormals(newNormals);
	newNormals->Delete();
//    partOutput->GetPointData()->SetNormals(0);
	
	if(part2meshOn) this->createMeshFromParticles();

	return 0;

}


int iDataReader::loadVectBinFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int n, n1, n2, n3, ntemp;
	vtkIdType ntot;
	vtkIdType nline, nstep, nlast, ret, i;
	float *d;
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_VECT,-0.1);
	this->updateProgress(IDATAREADER_VECT,0.0);
//
//  Read the header
//
	err = false;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(F.readBlock((char *)&n3,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n3);
	if(F.readBlock((char *)&n2,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n2);
	if(F.readBlock((char *)&n1,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n1);
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	
    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	if(periodicityLeader >= 'b') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
		if(n1 == nmax) limits[0][0]->periodic[2] = true;
		if(n2 == nmax) limits[0][0]->periodic[1] = true;
		if(n3 == nmax) limits[0][0]->periodic[0] = true;
	} 

	ntot = (vtkIdType)n1*n2*n3;
//
//  Measure the file size to find out the number of variables
//
	long lp = F.size();
	long nrec = 4L*(ntot+2L);
	int nvar = (lp-20L)/nrec;
//
//  If the file is not a whole number of records, set warno to 1
//
	if((lp-20L) != nvar*nrec) warno = 1;
//
//  If number of records is not 3, exit
//
	if(nvar != 3) { F.close(); return 3; }

	this->updateProgress(IDATAREADER_VECT,0.01);
//
//  parameters for the Progress Bar
//
    float prog1 = 0.33;
	nstep = 1 + (int)floor(100.0*prog1);
	nline = ntot/nstep;
	nlast = ntot - nline*nstep;
	float prog2 = prog1/nstep;
//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=vectDims[0] || n2!=vectDims[1] || n3!=vectDims[2])
	{
		if(vectData != 0) delete[] vectData;
		vectData = new float[ntot*3];
		if(vectData == 0) reportNullPointer(8224);
		if(vectData == 0) return 10;
	}
//
//  Read the data from the file.
//
	d = new float[ntot];
	if(d == 0) reportNullPointer(8225);
    for(n=1; !err && n<=3; n++) 
	{
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		for(i=0; !err && i<nstep; i++) 
		{
			ret = F.readBlock((char *)(d+i*nline),4*nline);
			if(ret != 4*nline) err = true;
			this->updateProgress(IDATAREADER_VECT,0.01+prog1*(n-1)+prog2*(i+1.0));
			if(aborted) { F.close(); return -99; }
		}
		if(nlast != 0) if(F.readBlock((char *)(d+nstep*nline),4*nlast) != 4*nlast) err = true;
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		if(!err)
		{
			if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(d,ntot);
			for(i=0; i<ntot; i++) vectData[n-1+3*i] = d[i];
		}
	}
	delete[] d;

	this->updateProgress(IDATAREADER_VECT,1.0);
	F.close();

	if(err) { delete[] vectData; vectData = 0; return 3; }

	vectDims[0] = n3;
	vectDims[1] = n2;
	vectDims[2] = n1;
	vectSize = ntot;

	float vmax = 0.0;
	for(vtkIdType l=0; l<vectSize; l++) 
	{
		float *f = vectData+3*l;
		float v = f[0]*f[0] + f[1]*f[1] + f[2]*f[2];
		if(v > vmax) vmax = v;
	}
	limits[0][0]->setVecMax(1,sqrt(vmax));
	limits[0][0]->setVecHigh(1,sqrt(vmax));

	return 0;

}


int iDataReader::loadVectTxtFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//    10:   not enough memory for the data
//
//
	int n, n1, n2, n3, ret;
	iString s;
	vtkIdType ntot, l;
	float f[999];
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_VECT,-0.1);
	this->updateProgress(IDATAREADER_VECT,0.0);
//
//  Read the header
//
	err = false;
	if(F.readLine(s,1024) == -1) err = true;

	ret = sscanf(s.latin1(),"%d %d %d",&n3,&n2,&n1);
	if(ret != 3) err = true;

    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	if(periodicityLeader >= 'b') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
		if(n1 == nmax) limits[0][0]->periodic[2] = true;
		if(n2 == nmax) limits[0][0]->periodic[1] = true;
		if(n3 == nmax) limits[0][0]->periodic[0] = true;
	}

	ntot = (vtkIdType)n1*n2*n3;
//
//  Find out the number of variables
//
	if(F.readLine(s,1024) == -1) err = true;
	ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<1 || ret>10) err = true;

	int nvar = ret;
//
//  If number of records is not 3, exit
//
	if(nvar != 3) { F.close(); return 3; }

	this->updateProgress(IDATAREADER_VECT,0.01);
//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=vectDims[0] || n2!=vectDims[1] || n3!=vectDims[2])
	{
		if(vectData != 0) delete[] vectData;
		vectData = new float[ntot*3];
		if(vectData == 0) reportNullPointer(8226);
		if(vectData == 0) return 10;
	}
	for(n=0; n<3; n++) vectData[n] = f[n];
//
//  Read the data from the file.
//
	for(l=1; !err && l<ntot; l++) 
	{
		if(F.readLine(s,1024) == -1) err = true;
		ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret != nvar) err = true;
		if((100*l)/ntot < (100*(l+1))/ntot) 
		{
			this->updateProgress(IDATAREADER_VECT,0.01+(float)l/ntot);
			if(aborted) { F.close(); return -99; }
		}
		for(n=0; n<3; n++) vectData[n+3*l] = f[n];

	}	
	this->updateProgress(IDATAREADER_VECT,1.0);
	F.close();

	if(err) { delete[] vectData; vectData = 0; return 3; }

	vectDims[0] = n3;
	vectDims[1] = n2;
	vectDims[2] = n1;
	vectSize = ntot;

	float vmax = 0.0;
	for(l=0; l<vectSize; l++) 
	{
		float *f = vectData+3*l;
		float v = f[0]*f[0] + f[1]*f[1] + f[2]*f[2];
		if(v > vmax) vmax = v;
	}
	limits[0][0]->setVecMax(1,sqrt(vmax));
	limits[0][0]->setVecHigh(1,sqrt(vmax));

	return 0;

}

	
int iDataReader::loadTensBinFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//     4:	corrupted file, some data read
//    10:   not enough memory for the data
//
//
	int n, n1, n2, n3, ntemp;
	vtkIdType ntot;
	vtkIdType nline, nstep, nlast, ret, i;
	float *d;
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_TENS,-0.1);
	this->updateProgress(IDATAREADER_TENS,0.0);
//
//  Read the header
//
	err = false;
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	if(F.readBlock((char *)&n3,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n3);
	if(F.readBlock((char *)&n2,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n2);
	if(F.readBlock((char *)&n1,4) != 4) err = true;
	if(endinessOfData != endinessOfMachine) this->Swap4Bytes(&n1);
	if(F.readBlock((char *)&ntemp,4) != 4) err = true;
	
    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	if(periodicityLeader >= 'c') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
		if(n1 == nmax) limits[0][0]->periodic[2] = true;
		if(n2 == nmax) limits[0][0]->periodic[1] = true;
		if(n3 == nmax) limits[0][0]->periodic[0] = true;
	}

	ntot = (vtkIdType)n1*n2*n3;
//
//  Measure the file size to find out the number of variables
//
	long lp = F.size();
	long nrec = 4L*(ntot+2L);
	int nvar = (lp-20L)/nrec;
//
//  If the file is not a whole number of records, set warno to 1
//
	if((lp-20L) != nvar*nrec) warno = 1;
//
//  If number of records is not 6 exit
//
	if(nvar != 6) { F.close(); return 3; }

	this->updateProgress(IDATAREADER_TENS,0.01);
//
//  parameters for the Progress Bar
//
    float prog1 = 0.167;
	nstep = 1 + (int)floor(100.0*prog1);
	nline = ntot/nstep;
	nlast = ntot - nline*nstep;
	float prog2 = prog1/nstep;
//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=tensDims[0] || n2!=tensDims[1] || n3!=tensDims[2])
	{
		if(tensData != 0) delete[] tensData;
		tensData = new float[ntot*9];
		if(tensData == 0) reportNullPointer(8230);
		if(tensData == 0) return 10;
	}
//
//  Read the data from the file.
//
	d = new float[ntot];
	if(d == 0) reportNullPointer(8231);
    for(n=1; !err && n<=6; n++) 
	{
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		for(i=0; !err && i<nstep; i++) 
		{
			ret = F.readBlock((char *)(d+i*nline),4*nline);
			if(ret != 4*nline) err = true;
			this->updateProgress(IDATAREADER_TENS,0.01+prog1*(n-1)+prog2*(i+1.0));
			if(aborted) { F.close(); return -99; }
		}
		if(nlast != 0) if(F.readBlock((char *)(d+nstep*nline),4*nlast) != 4*nlast) err = true;
		if(F.readBlock((char *)&ntemp,4) != 4) err = true;
		if(!err)
		{
			if(endinessOfData != endinessOfMachine) this->Swap4BytesRange(d,ntot);
			int noff1 = 0, noff2 = -1;
			switch (n)
			{
			case 1: { noff1 = 0; break; }
			case 2: { noff1 = 1; noff2 = 3; break; }
			case 3: { noff1 = 2; noff2 = 6; break; }
			case 4: { noff1 = 4; break; }
			case 5: { noff1 = 5; noff2 = 7; break; }
			case 6: { noff1 = 8; break; }
			}
			
			for(i=0; i<ntot; i++) tensData[noff1+9*i] = d[i];
			if(noff2 > 0) for(i=0; i<ntot; i++) tensData[noff2+9*i] = d[i];
		}
	}
	delete[] d;

	this->updateProgress(IDATAREADER_TENS,1.0);
	F.close();

	if(err) { delete[] tensData; tensData = 0; return 3; }

	tensDims[0] = n3;
	tensDims[1] = n2;
	tensDims[2] = n1;
	tensSize = ntot;

	//
	// Compute the maximum eigenvalue
	//
	float eigmax = 0.0;
	float *mat[3], eig[3], *eiv[3];
	float mat0[3], mat1[3], mat2[3];
	float eiv0[3], eiv1[3], eiv2[3];
	// set up working matrices
	mat[0] = mat0; mat[1] = mat1; mat[2] = mat2; 
	eiv[0] = eiv0; eiv[1] = eiv1; eiv[2] = eiv2; 
	for(vtkIdType l=0; l<tensSize; l++) 
	{
		float *f = tensData+9*l;
		mat[0][0] = *(f+0); mat[0][1] = *(f+1); mat[0][2] = *(f+2); 
		mat[1][0] = *(f+3); mat[1][1] = *(f+4); mat[1][2] = *(f+5); 
		mat[2][0] = *(f+6); mat[2][1] = *(f+7); mat[2][2] = *(f+8); 
		vtkMath::Jacobi(mat,eig,eiv);
		if(fabs(eig[0]) > eigmax) eigmax = fabs(eig[0]);
		if(fabs(eig[1]) > eigmax) eigmax = fabs(eig[1]);
		if(fabs(eig[2]) > eigmax) eigmax = fabs(eig[2]);
	}
	limits[0][0]->setTenMax(1,eigmax);
	limits[0][0]->setTenHigh(1,eigmax);

	return 0;

}


int iDataReader::loadTensTxtFile(const iString &fname)
{
//
//  Error codes:
//	   0:	no error
//    -2:	system error
//	   1:	file does not exist	
//     2:	no header
//	   3:	corrupted file, no data read
//    10:   not enough memory for the data
//
//
	int n1, n2, n3, ret;
	iString s;
	vtkIdType ntot, l;
	float f[999];
	iFile F;
	bool err;

	warno = 0;
	F.setName(fname);

	if(!F.open(IO_ReadOnly)) return 1;

	this->updateProgress(IDATAREADER_TENS,-0.1);
	this->updateProgress(IDATAREADER_TENS,0.0);
//
//  Read the header
//
	err = false;
	if(F.readLine(s,1024) == -1) err = true;

	ret = sscanf(s.latin1(),"%d %d %d",&n3,&n2,&n1);
	if(ret != 3) err = true;

    if(n1<1 || n1>1600 || n2<1 || n2>1600 || n3<1 || n3>1600) err = true;

	if(err) { F.close(); return 2; }
//
//  Boundary conditions
//
	int nmax = (n1 > n2) ? n1 : n2;
	nmax = (n3 > nmax) ? n3 : nmax;

	if(periodicityLeader >= 'c') 
	{ 
		limits[0][0]->periodic[0] = limits[0][0]->periodic[1] = limits[0][0]->periodic[2] = false;
		if(n1 == nmax) limits[0][0]->periodic[2] = true;
		if(n2 == nmax) limits[0][0]->periodic[1] = true;
		if(n3 == nmax) limits[0][0]->periodic[0] = true;
	}

	ntot = (vtkIdType)n1*n2*n3;
//
//  Find out the number of variables
//
	if(F.readLine(s,1024) == -1) err = true;
	ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<1 || ret>10) err = true;

	int nvar = ret;
//
//  If number of records is not 6, exit
//
	if(nvar != 6) { F.close(); return 3; }

	this->updateProgress(IDATAREADER_TENS,0.01);
//
//  Allocate memory; erase the old data if needed.
//
	if(n1!=tensDims[0] || n2!=tensDims[1] || n3!=tensDims[2])
	{
		if(tensData != 0) delete[] tensData;
		tensData = new float[ntot*9];
		if(tensData == 0) reportNullPointer(8232);
		if(tensData == 0) return 10;
	}
//
//  Fill in 3x3 tensor with 6 components
//
	tensData[0] = f[0];
	tensData[1] = f[1];
	tensData[2] = f[2];
	tensData[3] = f[1];
	tensData[4] = f[3];
	tensData[5] = f[4];
	tensData[6] = f[2];
	tensData[7] = f[4];
	tensData[8] = f[5];
//
//  Read the data from the file.
//
	for(l=1; !err && l<ntot; l++) 
	{
		if(F.readLine(s,1024) == -1) err = true;
		ret = sscanf(s.latin1(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret != nvar) err = true;
		if((100*l)/ntot < (100*(l+1))/ntot) 
		{
			this->updateProgress(IDATAREADER_TENS,0.01+(float)l/ntot);
			if(aborted) { F.close(); return -99; }
		}
		//
		//  Fill in 3x3 tensor with 6 components
		//
		tensData[0+9*l] = f[0];
		tensData[1+9*l] = f[1];
		tensData[2+9*l] = f[2];
		tensData[3+9*l] = f[1];
		tensData[4+9*l] = f[3];
		tensData[5+9*l] = f[4];
		tensData[6+9*l] = f[2];
		tensData[7+9*l] = f[4];
		tensData[8+9*l] = f[5];

	}	
	this->updateProgress(IDATAREADER_TENS,1.0);
	F.close();

	if(err) { delete[] tensData; tensData = 0; return 3; }

	tensDims[0] = n3;
	tensDims[1] = n2;
	tensDims[2] = n1;
	tensSize = ntot;

	//
	// Compute the maximum eigenvalue
	//
	float eigmax = 0.0;
	float *mat[3], eig[3], *eiv[3];
	float mat0[3], mat1[3], mat2[3];
	float eiv0[3], eiv1[3], eiv2[3];
	// set up working matrices
	mat[0] = mat0; mat[1] = mat1; mat[2] = mat2; 
	eiv[0] = eiv0; eiv[1] = eiv1; eiv[2] = eiv2; 
	for(l=0; l<tensSize; l++) 
	{
		float *f = tensData+9*l;
		mat[0][0] = *(f+0); mat[0][1] = *(f+1); mat[0][2] = *(f+2); 
		mat[1][0] = *(f+3); mat[1][1] = *(f+4); mat[1][2] = *(f+5); 
		mat[2][0] = *(f+6); mat[2][1] = *(f+7); mat[2][2] = *(f+8); 
		vtkMath::Jacobi(mat,eig,eiv);
		if(fabs(eig[0]) > eigmax) eigmax = fabs(eig[0]);
		if(fabs(eig[1]) > eigmax) eigmax = fabs(eig[1]);
		if(fabs(eig[2]) > eigmax) eigmax = fabs(eig[2]);
	}
	limits[0][0]->setTenMax(1,eigmax);
	limits[0][0]->setTenHigh(1,eigmax);

	return 0;

}

	
void iDataReader::ExecuteData(vtkDataObject *vtkNotUsed(out))
{
	//
	//  Now we need to decide whether the data are CellData or PointData.
	//  We use voxelLocation for that: if it is 0, we have point data and we
	//  don't need to do anything. If it is 1, we have cell data and we need to
	//  interpolate to points with the appropriate boundary conditions.
	//
	if(!meshUpdated && meshData!=0 && meshSize>0)
	{

		meshUpdated = true;
		meshOutput->Modified();
		
		this->transformToPointData(IDATAREADER_MESH,false,limits[0][0]->getNvar(),meshData,meshDims,meshSize);

		meshOutput->SetDimensions(meshDims);
		meshOutput->SetScalarTypeToFloat();
		meshOutput->SetNumberOfScalarComponents(limits[0][0]->getNvar());
//		meshOutput->GetPointData()->GetScalars()->SetNumberOfComponents(limits[0][0]->getNvar());
		
		this->computeSpacing(meshDims,meshOutput);
		meshArray->SetArray(meshData,limits[0][0]->getNvar()*meshSize,1);

		this->operateOnMeshData();

		passedMeshOutput->ShallowCopy(meshOutput);
		meshProgressBar = 0;

	}

	if(!partUpdated && partOutput!=0 && partSize>0)
	{

		partUpdated = true;
		partOutput->Modified();
		this->operateOnPartData();

		passedPartOutput->ShallowCopy(partOutput);
		partProgressBar = 0;

	}
	
	if(!vectUpdated && vectData!=0 && vectSize>0)
	{

		vectUpdated = true;
		vectOutput->Modified();
		
		this->transformToPointData(IDATAREADER_VECT,true,3,vectData,vectDims,vectSize);

		vectOutput->SetDimensions(vectDims);
		vectOutput->SetScalarTypeToFloat();
		vectOutput->SetNumberOfScalarComponents(3);
//		vectOutput->GetPointData()->GetScalars()->SetNumberOfComponents(3);
		
		this->computeSpacing(vectDims,vectOutput);
		vectArray->SetArray(vectData,3*vectSize,1);

		this->operateOnVectData();

		passedVectOutput->ShallowCopy(vectOutput);
		vectProgressBar = 0;

	}

	if(!tensUpdated && tensData!=0 && tensSize>0)
	{

		tensUpdated = true;
		tensOutput->Modified();
		
		this->transformToPointData(IDATAREADER_TENS,true,9,tensData,tensDims,tensSize);

		tensOutput->SetDimensions(tensDims);
		tensOutput->SetScalarTypeToFloat();
		tensOutput->SetNumberOfScalarComponents(9);
//		tensOutput->GetPointData()->GetScalars()->SetNumberOfComponents(9);
		
		this->computeSpacing(tensDims,tensOutput);
		tensArray->SetArray(tensData,9*tensSize,1);

		this->operateOnTensData();

		passedTensOutput->ShallowCopy(tensOutput);
		tensProgressBar = 0;

	}

	this->Modified();

}


void iDataReader::getDataInfo(iDataInfo *i)
{

	int nvar = limits[0][0]->getNvar();
	int natt = limits[0][0]->getNatt();

	i->animatable = (currec != -1);
	i->mode = mode;
	i->isSet = (currec >= 0);
//
//  Mesh data
//	
	int n;
	for(n=0; n<nvar; n++) 
	{
		vtkIdType noff = meshSize*n;
		float fl = 1.0e30;
		float fu = -1.0e30;
		for(vtkIdType l=0; l<meshSize; l++) 
		{
			float f = *(meshData+l+noff);
			if(fl > f) fl = f;
			if(fu < f) fu = f;
		}
		i->varMin[n] = fl;
		i->varMax[n] = fu;
	}
	for(n=0; n<3; n++) i->meshDims[n] = meshDims[n];
//
//  Particle data
//
	i->npart = partSize;
	for(n=0; n<natt; n++)
	{
		i->attMin[n] = limits[0][0]->getAttMin(n+1);
		i->attMax[n] = limits[0][0]->getAttMax(n+1);
	}
//
//  Vector data
//
	i->vecMax = limits[0][0]->getVecHigh();
	for(n=0; n<3; n++) i->vectDims[n] = vectDims[n];
//
//  Tensor data
//
	i->tenMax = limits[0][0]->getTenHigh();
	for(n=0; n<3; n++) i->tensDims[n] = tensDims[n];

}


void iDataReader::setDataShift(int d, float f)
{
	if(d>=0 && d<=2) newShift[d] = f;
}


bool iDataReader::doDataShift(float *dr0, bool resetAll)
{
	int d;
	bool ret = true;
	float dr;

	for(d=0; d<3; d++)
	{	
		if(dr0 == 0) dr = newShift[d] - oldShift[d]; else dr = dr0[d];
		if(fabs(dr) > 1.0e-5)
		{
			ret = ret & shiftData(d,dr,resetAll);
		}
		oldShift[d] = newShift[d];
	}
	
	return ret;

}


bool iDataReader::shiftData(int d, float dr, bool resetAll)
{

	if(d<0 || d>2) return false;
	
	if(meshDims[0]>0 && meshDims[1]>0 && meshDims[2]>0 && (resetAll||!meshShifted)) 
	{
		
		if(limits[0][0]->periodicBC && limits[0][0]->periodic[d])
		{

			int dimt[3];
			this->getCellDims(meshDims,dimt);

			int dn = round(dr*dimt[d]);
			if(dn==0 || abs(dn)>=dimt[d]) return false;

			dr = (float)dn/dimt[d];

			int l1, l2, l, lold, n;
			int d1 = 0, d2 = 0;

			switch(d) 
			{
			case 0: { d1 = 1; d2 = 2; break; }
			case 1: { d1 = 0; d2 = 2; break; }
			case 2: { d1 = 0; d2 = 1; break; }
			}

			int ddim = meshDims[d];
			int nvar = limits[0][0]->getNvar();
			vtkIdType off0 = 0L, off1 = 0L;
			float *buf = new float[ddim];
			if(buf == 0) reportNullPointer(8227);

			for(n=0; n<nvar; n++)
			{
				vtkIdType noff = meshSize*n;
				for(l1=0; l1<meshDims[d1]; l1++)
				{
					for(l2=0; l2<meshDims[d2]; l2++)
					{

						switch(d) 
						{
						case 0: { off0 = meshDims[0]*(l1+meshDims[1]*l2); off1 = 1;	break; }
						case 1: { off0 = l1+meshDims[0]*meshDims[1]*l2; off1 = meshDims[0];	break; }
						case 2: { off0 = l1+meshDims[0]*l2; off1 = meshDims[0]*meshDims[1];	break; }
						}

						off0 += noff;
						for(l=0; l<ddim-1; l++) 
						{
							lold = l - dn;
							lold = (lold<0) ? lold+ddim : lold;
							lold = (lold>=ddim) ? lold-ddim : lold;
							buf[l] = *(meshData+off0+off1*lold);
						}
						buf[ddim-1] = buf[0]; 

						for(l=0; l<ddim; l++) 
						{
							*(meshData+off0+off1*l) = buf[l];
						}

					}
				}
			}
	
			delete[] buf;

		}
		else
		{

			vtkFloat sp[3];
			meshOutput->GetOrigin(sp);
			sp[d] += 2.0*dr;
			meshOutput->SetOrigin(sp);

		}

		passedMeshOutput->Modified();
		meshOutput->Modified();
		meshShifted = true;
		
	} 

	vtkPoints *p;
	if((p=partOutput->GetPoints())!=0 && (resetAll||!partShifted))
	{
		vtkIdType n = p->GetNumberOfPoints();
		float *x = (float *)p->GetVoidPointer(0);
		//	d = 2 - d;
		for(vtkIdType l=0; l<n; l++)
		{
			x[d] += 2.0*dr;
			if(limits[0][0]->periodicBC && limits[0][0]->periodic[d])
			{
				if(x[d] > 1.0) x[d] -= 2.0;
				if(x[d] < -1.0) x[d] += 2.0;
			}
			x += 3;
		}

		passedPartOutput->Modified();
		partOutput->Modified();
		partShifted = true;

	}

	if(vectDims[0]>0 && vectDims[1]>0 && vectDims[2]>0 && (resetAll||!vectShifted))
	{
		
		if(limits[0][0]->periodicBC && limits[0][0]->periodic[d])
		{

			int dimt[3];
			this->getCellDims(vectDims,dimt);

			int dn = round(dr*dimt[d]);
			if(dn==0 || abs(dn)>=dimt[d]) return false;

			dr = (float)dn/dimt[d];

			int l1, l2, l, lold;
			int d1 = 0, d2 = 0;

			switch(d) 
			{
			case 0: { d1 = 1; d2 = 2; break; }
			case 1: { d1 = 0; d2 = 2; break; }
			case 2: { d1 = 0; d2 = 1; break; }
			}

			int ddim = vectDims[d];
			vtkIdType off0 = 0L, off1 = 0L, add1, add2;
			float *buf = new float[3*ddim];
			if(buf == 0) reportNullPointer(8228);

			for(l1=0; l1<vectDims[d1]; l1++)
			{
				for(l2=0; l2<vectDims[d2]; l2++)
				{

					switch(d) 
					{
					case 0: { off0 = 3*vectDims[0]*(l1+vectDims[1]*l2); off1 = 3;	break; }
					case 1: { off0 = 3*(l1+vectDims[0]*vectDims[1]*l2); off1 = 3*vectDims[0];	break; }
					case 2: { off0 = 3*(l1+vectDims[0]*l2); off1 = 3*vectDims[0]*vectDims[1];	break; }
					}

					for(l=0; l<ddim-1; l++) 
					{
						lold = l - dn;
						lold = (lold<0) ? lold+ddim : lold;
						lold = (lold>=ddim) ? lold-ddim : lold;

						add1 = 3*l;
						add2 = off0+off1*lold;

						buf[0+add1] = *(vectData+0+add2);
						buf[1+add1] = *(vectData+1+add2);
						buf[2+add1] = *(vectData+2+add2);
					}
					add1 = 3*(ddim-1);
					buf[0+add1] = buf[0];
					buf[1+add1] = buf[1];
					buf[2+add1] = buf[2];

					for(l=0; l<ddim; l++) 
					{
						add1 = 3*l;
						add2 = off0+off1*l;

						*(vectData+0+add2) = buf[0+add1];
						*(vectData+1+add2) = buf[1+add1];
						*(vectData+2+add2) = buf[2+add1];
					}

				}
			}

			delete[] buf;

		}
		else
		{

			vtkFloat sp[3];
			vectOutput->GetOrigin(sp);
			sp[d] += 2.0*dr;
			vectOutput->SetOrigin(sp);

		}

		passedVectOutput->Modified();
		vectOutput->Modified();
		vectShifted = true;
				
	} 

	if(tensDims[0]>0 && tensDims[1]>0 && tensDims[2]>0 && (resetAll||!tensShifted))
	{
		
		if(limits[0][0]->periodicBC && limits[0][0]->periodic[d])
		{

			int dimt[3];
			this->getCellDims(tensDims,dimt);

			int dn = round(dr*dimt[d]);
			if(dn==0 || abs(dn)>=dimt[d]) return false;

			dr = (float)dn/dimt[d];

			int l1, l2, l, lold;
			int d1 = 0, d2 = 0;

			switch(d) 
			{
			case 0: { d1 = 1; d2 = 2; break; }
			case 1: { d1 = 0; d2 = 2; break; }
			case 2: { d1 = 0; d2 = 1; break; }
			}

			int ddim = tensDims[d];
			vtkIdType off0 = 0L, off1 = 0L, add1, add2;
			float *buf = new float[9*ddim];
			if(buf == 0) reportNullPointer(8233);

			for(l1=0; l1<tensDims[d1]; l1++)
			{
				for(l2=0; l2<tensDims[d2]; l2++)
				{

					switch(d) 
					{
					case 0: { off0 = 9*tensDims[0]*(l1+tensDims[1]*l2); off1 = 9;	break; }
					case 1: { off0 = 9*(l1+tensDims[0]*tensDims[1]*l2); off1 = 9*tensDims[0];	break; }
					case 2: { off0 = 9*(l1+tensDims[0]*l2); off1 = 9*tensDims[0]*tensDims[1];	break; }
					}

					for(l=0; l<ddim-1; l++) 
					{
						lold = l - dn;
						lold = (lold<0) ? lold+ddim : lold;
						lold = (lold>=ddim) ? lold-ddim : lold;

						add1 = 9*l;
						add2 = off0+off1*lold;

						buf[0+add1] = *(tensData+0+add2);
						buf[1+add1] = *(tensData+1+add2);
						buf[2+add1] = *(tensData+2+add2);
						buf[3+add1] = *(tensData+3+add2);
						buf[4+add1] = *(tensData+4+add2);
						buf[5+add1] = *(tensData+5+add2);
						buf[6+add1] = *(tensData+6+add2);
						buf[7+add1] = *(tensData+7+add2);
						buf[8+add1] = *(tensData+8+add2);
					}
					add1 = 9*(ddim-1);
					buf[0+add1] = buf[0];
					buf[1+add1] = buf[1];
					buf[2+add1] = buf[2];
					buf[3+add1] = buf[3];
					buf[4+add1] = buf[4];
					buf[5+add1] = buf[5];
					buf[6+add1] = buf[6];
					buf[7+add1] = buf[7];
					buf[8+add1] = buf[8];

					for(l=0; l<ddim; l++) 
					{
						add1 = 9*l;
						add2 = off0+off1*l;

						*(tensData+0+add2) = buf[0+add1];
						*(tensData+1+add2) = buf[1+add1];
						*(tensData+2+add2) = buf[2+add1];
						*(tensData+3+add2) = buf[3+add1];
						*(tensData+4+add2) = buf[4+add1];
						*(tensData+5+add2) = buf[5+add1];
						*(tensData+6+add2) = buf[6+add1];
						*(tensData+7+add2) = buf[7+add1];
						*(tensData+8+add2) = buf[8+add1];
					}

				}
			}

			delete[] buf;

		}
		else
		{

			vtkFloat sp[3];
			tensOutput->GetOrigin(sp);
			sp[d] += 2.0*dr;
			tensOutput->SetOrigin(sp);

		}

		passedTensOutput->Modified();
		tensOutput->Modified();
		tensShifted = true;
				
	} 

	return true;

}


float iDataReader::getMemorySize()
{

	float s = 0.0;
	if(meshOutput > 0) s += meshOutput->GetActualMemorySize();
	if(partOutput > 0) s += partOutput->GetActualMemorySize();
	if(vectOutput > 0) s += vectOutput->GetActualMemorySize();
	if(tensOutput > 0) s += tensOutput->GetActualMemorySize();
	return s;

}


void iDataReader::setMeshArrayOutput(int n)
{
	if(n>0 && n<=limits[0][0]->getNvar()) meshArrayOutput = n;
}


void iDataReader::operateOnMeshData()
{
	int n;
	vtkIdType loff, l;
	float fMin = 1.0e30, fMax = -1.0e30, f = 0.0;

	int nvar = limits[0][0]->getNvar();
	//
	//  Do the array function first
	//
	if(meshArrayOutput>0 && meshArrayOutput<=nvar && !meshArrayFunction.isEmpty())
	{

		iExpressionParser *arCalc = new iExpressionParser;
		if(arCalc == 0) reportNullPointer(8229);
		char arName[5];
		bool doVector = false;
		double res;

		if(meshDims[0]==vectDims[0] && meshDims[1]==vectDims[1] && meshDims[2]==vectDims[2] && meshDims[0]!=0 && meshDims[1]!=0 && meshDims[2]!=0) doVector = true;

		arName[0] = 'V';
		arName[1] = 'a';
		arName[2] = 'r';
		arName[4] = 0;

		arCalc->SetFunction(meshArrayFunction.latin1());

		loff = (meshArrayOutput-1)*meshSize;

		for(l=0; l<meshSize; l++) 
		{

			for(n=0; n<nvar; n++)
			{
				arName[3] = (char)((int)'1'+n);
				arCalc->SetScalarVariableValue(arName,*(meshData+n*meshSize+l));
			}
			if(doVector)
			{
				arCalc->SetVectorVariableValue("Vector",*(vectData+0+3*l),*(vectData+1+3*l),*(vectData+2+3*l));
			} 
			else 
			{
				arCalc->SetVectorVariableValue("Vector",0.0,0.0,0.0);
			}
		
			if(arCalc->GetScalarResult(res)) *(meshData+loff+l) = res;
		}
		//
		//  Recompute limits
		//
		int nCases = round(pow(2.0,1.0+nvar));

		int i, j, k;
		float f;
		for(k=0; k<nCases; k++)
		{
			for(n=0; n<nvar; n++)
			{
				arName[3] = (char)((int)'1'+n);
				i = round(pow(2.0,n+1.0));
				j = (k/i) % 2;
				if(j == 0) 
				{
					f = limits[0][0]->unstretchVar(limits[0][0]->getVarMin(n+1),n+1);
					arCalc->SetScalarVariableValue(arName,f); 
				}
				else 
				{
					f = limits[0][0]->unstretchVar(limits[0][0]->getVarMax(n+1),n+1);
					arCalc->SetScalarVariableValue(arName,f);
				}
			}
			if(doVector)
			{
				if(k%2 == 0)
					arCalc->SetVectorVariableValue("Vector",limits[0][0]->getVecMax(1),0.0,0.0);
				else
					arCalc->SetVectorVariableValue("Vector",limits[0][0]->getVecMin(1),0.0,0.0);
			} 
			else 
			{
					arCalc->SetVectorVariableValue("Vector",0.0,0.0,0.0);
			}

			if(arCalc->GetScalarResult(res)) f = res;
			if(f > fMax) fMax = f;
			if(f < fMin) fMin = f;
		}

		fMin = limits[0][0]->stretchVar(fMin,meshArrayOutput);
		fMin = limits[0][0]->stretchVar(fMax,meshArrayOutput);

		if(adjustableBounds)
		{
			limits[0][0]->setVarMin(meshArrayOutput,fMin);
			limits[0][0]->setVarMax(meshArrayOutput,fMax);
		}

		limits[0][0]->setVarLow(meshArrayOutput,fMin);
		limits[0][0]->setVarHigh(meshArrayOutput,fMax);
	
		delete arCalc;

	}	
	//
	//  Format everything
	//
	for(n=0; n<nvar; n++)
	{
		fMin = 1.0e30;
		fMax = -1.0e30;
		loff = n*meshSize;
		for(l=0; l<meshSize; l++)
		{
			f = *(meshData+loff+l) = limits[0][0]->stretchVar(*(meshData+loff+l),n+1);
			if(f > fMax) fMax = f;
			if(f < fMin) fMin = f;
		}
		
		if(limits[0][0]->getVarStretch(n+1) == 1)
		{
			fMin = 0.1*floor(10.0*fMin);
			fMax = 0.1*floor(1+10.0*fMax);
		}

		if(adjustableBounds)
		{
			limits[0][0]->setVarMin(n+1,fMin);
			limits[0][0]->setVarMax(n+1,fMax);
			if(fabs(limits[0][0]->getVarHigh(n+1)-limits[0][0]->getVarLow(n+1))/(1.0e-30+fabs(limits[0][0]->getVarHigh(n+1))+fabs(limits[0][0]->getVarLow(n+1))) < 1.0e-5)
			{
				limits[0][0]->setVarLow(n+1,limits[0][0]->getVarMin(n+1));
				limits[0][0]->setVarHigh(n+1,limits[0][0]->getVarMax(n+1));
			}
		} 
	}

	if(!adjustableBounds) limits[0][0]->restoreVarLimits();

	for(n=0; n<nvar; n++)
	{
		limits[0][0]->setVarLow(n+1,limits[0][0]->getVarLow(n+1));
		limits[0][0]->setVarHigh(n+1,limits[0][0]->getVarHigh(n+1));
	}

}


void iDataReader::operateOnPartData()
{

}


void iDataReader::operateOnVectData()
{

}


void iDataReader::operateOnTensData()
{

}


void iDataReader::setMeshFromParticles(bool s)
{ 
	part2meshOn = s; 
	if(!s)
	{
		meshData = 0;
		meshDataAssigned = false;
		meshSize = 0;
		meshDims[0] = meshDims[1] = meshDims[2] = 0;
		meshUpdated = false;
		meshOutput->SetDimensions(meshDims);
		meshOutput->Modified();
		passedMeshOutput->ShallowCopy(meshOutput);
	}

}


void iDataReader::createMeshFromParticles()
{

	int nvar = limits[0][0]->getNatt();

	if(partOutput!=0 && partSize>0 && part2meshOn && nvar>0)
	{
		//
		//  Use VTK Shepard method - cannot deal with periodic BC
		//
		limits[0][0]->periodic[2] = limits[0][0]->periodic[1] = limits[0][0]->periodic[0] = false;

		limits[0][0]->setNvar(nvar);
		float meanInterpaticleSeparation = 2.0/pow(partSize,0.33333);
		float maxDist = part2meshFilter->GetMaximumDistance();
		part2meshFilter->SetMaximumDistance(maxDist*meanInterpaticleSeparation);

		part2meshFilter->SetNullValue(limits[0][0]->getAttMin(1)*0.1);

		iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
		part2meshFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		obsAbortRender->start();
		part2meshFilter->SetInput(partOutput);
		part2meshFilter->Update();
		part2meshFilter->GetOutput()->GetDimensions(meshDims);
		obsAbortRender->finish();
		part2meshFilter->RemoveObserver(vtkCommand::ProgressEvent);

		part2meshFilter->SetMaximumDistance(maxDist);

		if(meshData!=0 && !meshDataAssigned) delete[] meshData;
		meshData = (float *)part2meshFilter->GetOutput()->GetScalarPointer();
		meshDataAssigned = true;
		meshSize = meshDims[0]*meshDims[1]*meshDims[2];
		//
		//  Repack the data array to be horizontal rather than vertical
		//  (nvar 1D-arrays one after another rather than 1 nvar-D array).
		//
		meshOutput->Modified();
		passedMeshOutput->ShallowCopy(meshOutput);
		meshUpdated = false;
		this->ExecuteData(0);
	}
		
}


void iDataReader::getLevelLimits(int v, float &lmin, float &lmax)
{
	if(meshUpdated && meshData!=0 && meshSize>0 && v>=1 && v<=NVARMAX)
	{
		lmin = limits[0][0]->stretchVar(levMin[v],v);
		lmax = limits[0][0]->stretchVar(levMax[v],v,true);
	} 
	else 
	{
		lmin = -30.0;
		lmax = 30.0;
	}
}


void iDataReader::setScaledDimension(int v)
{
	if(v>=-2 && v<=2)
	{
		scaledDim = v;
		if(!limits[0][0]->periodicBC)
		{
			meshShifted = false;
			vectShifted = false;
			tensShifted = false;
			this->computeSpacing(meshDims,meshOutput);
			this->computeSpacing(vectDims,vectOutput);
			this->computeSpacing(tensDims,tensOutput);
			doDataShift(newShift,false); 
		}
	}
}


extern int rffti1_(int *n, float *wa, int *ifac);
extern int rfftf1_(int *n, float *c__, float *ch, float *wa, int *ifac);
extern int rfftb1_(int *n, float *c__, float *ch, float *wa, int *ifac);
extern int cffti1_(int *n, float *wa, int *ifac);
extern int cfftf1_(int *n, float *c__, float *ch, float *wa, int *ifac);
extern int cfftb1_(int *n, float *c__, float *ch, float *wa, int *ifac);

void iDataReader::computeSpacing(int dims[3], iUniformMeshData *output)
{
	
	if(dims[0]<=0 || dims[1]<=0 || dims[2]<=0 || output==0) return;

	int nmax, nmin;
	int i, imax = 0, imin = 0;
	nmax = nmin = dims[0];
	for(i=1; i<3; i++)
	{
		if(nmax < dims[i]) 
		{
			nmax = dims[i];
			imax = i;
		}
		if(nmin > dims[i]) 
		{
			nmin = dims[i];
			imin = i;
		}
	
	}

	float s;
	if(limits[0][0]->periodicBC && limits[0][0]->periodic[imax]) 
	{
		s = 2.0/(nmax-1);
	}
	else
	{
		switch (scaledDim)
		{
		case -1: { s = 2.0/(nmin-1); break; }
		case  0: { s = 2.0/(dims[0]-1); break; }
		case  1: { s = 2.0/(dims[1]-1); break; }
		case  2: { s = 2.0/(dims[2]-1); break; }
		case -2: 
		default: { s = 2.0/(nmax-1); break; }
		}
	}

	vtkFloat org[3], spa[3];
	for(i=0; i<3; i++) 
	{	
		org[i] = -1.0;
		spa[i] = s;
	}
//	if(limits[0][0]->getVoxelLocation()==1 && modeCellToPoint==0) for(i=0; i<3; i++) newShift[i] += 0.5*spa[i];

	output->SetOrigin(org);
	output->SetSpacing(spa);

}


void iDataReader::getCellDims(int dim[3], int dimCell[3])
{
	int n;
	for(n=0; n<3; n++) if(limits[0][0]->getVoxelLocation() == 0) dimCell[n] = dim[n] - 1; else dimCell[n] = dim[n];
}


void iDataReader::setCellToPointMode(int m)
{ 
	if(m>=0 && m<=5) modeCellToPoint = m;
}


void iDataReader::transformToPointData(int mode, bool arrayIsVertical, int numComponents, float *&cellData, int cellDims[3], long &cellSize)
{
	if(limits[0][0]->getVoxelLocation() == 0) return;  // already point data
	
	int i, j, k, n, ii, jj, kk;
	vtkIdType pointLoc, cellOffset[3], pointOffset[3], cellComponentOffset, pointComponentOffset;
	
	int pointDims[3];
	vtkIdType pointSize;
	float *pointData;

	for(i=0; i<3; i++) pointDims[i] = cellDims[i] + 1; 
	pointSize = (vtkIdType)pointDims[0]*pointDims[1]*pointDims[2];

	vtkIdType arraySize = pointSize;
	if(modeCellToPoint == 0)
	{
		//
		//  Point data array will be used as auxilary storage for FFT.
		//  Make sure it has enough size (at least (n1+2)*n2*n3)
		//
		vtkIdType fftSize = ((vtkIdType)cellDims[0]+2L)*cellDims[1]*cellDims[2];
		if(fftSize > arraySize) arraySize = fftSize;
	}

	pointData = new float[numComponents*arraySize];
	if(pointData == 0) return;

	cellOffset[0] = 1;
	cellOffset[1] = cellDims[0];
	cellOffset[2] = cellDims[0]*cellDims[1];
	pointOffset[0] = 1;
	pointOffset[1] = pointDims[0];
	pointOffset[2] = pointDims[0]*pointDims[1];
	cellComponentOffset = cellSize;
	pointComponentOffset = pointSize;

	if(arrayIsVertical)
	{
		for(i=0; i<3; i++) 
		{
			cellOffset[i] *= numComponents;
			pointOffset[i] *= numComponents;
		}
		cellComponentOffset = pointComponentOffset = 1;
	}

	this->updateProgress(mode,-0.11);
	float progMax = cellDims[2]*numComponents;
	float prog = 0.0;
	this->updateProgress(mode,0.0);
	//
	//  Fourier transform method(s)
	//
	if(modeCellToPoint == 0)
	{
		//
		//  Use point data array as auxilary storage, and then put everything back into cellData
		//
		float *data = pointData;

		int n1 = cellDims[0];
		int n2 = cellDims[1];
		int n3 = cellDims[2];
		float *work1 = new float[1*n1+15];
		float *work2 = new float[2*n2+15];
		float *work3 = new float[2*n3+15];

		int n12;
		if(n1%2 == 0) n12 = n1/2 + 1; else n12 = (n1+1)/2;

		int nmin = n1;
		if(nmin > n2) nmin = n2;
		if(nmin > n3) nmin = n3;
		int nmax = n1;
		if(nmax < n2) nmax = n2;
		if(nmax < n3) nmax = n3;

		float *z1 = new float[2*cellSize/nmin];
		float *w1 = new float[2*cellSize/nmin];  // workspace for parallelization

		float *w, *z;
		vtkIdType lf, lc;
		int i,j;

		rffti1_(&n1,work1,(int *)(work1+1*n1));
		cffti1_(&n2,work2,(int *)(work2+2*n2));
		cffti1_(&n3,work3,(int *)(work3+2*n3));

		this->updateProgress(mode,0.01);

		progMax += numComponents*(2*n2+7*n3);

		for(n=0; n<numComponents; n++)
		{
			//
			//  X - direction
			//
			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n1*k;
				z = z1 + n1*k;
				for(j=0; j<n2; j++) 
				{
					
					lf = (n1+2)*(j+n2*k);
					lc = cellOffset[1]*j + cellOffset[2]*k + n*cellComponentOffset;
					
//					for(i=0; i<n1; i++) data[i+lf] = cellData[cellOffset[0]*i+lc];
					for(i=0; i<n1; i++) z[i] = cellData[cellOffset[0]*i+lc];
					rfftf1_(&n1,z,w,work1,(int *)(work1+n1));
					
					data[0+lf] = z[0];
					data[1+lf] = 0.0;
					for(i=1; i<n1; i++) data[i+1+lf] = z[i];
					data[n1+1+lf] = 0.0;
						
				}
			}
			//
			//  Y & Z directions
			//
			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n2*k;
				z = z1 + 2*n2*k;
				for(i=0; i<n12; i++) 
				{
					
					lf = 2*i + (n1+2)*n2*k;
					for(j=0; j<n2; j++)
					{
						z[2*j+0] = data[(n1+2)*j+lf+0];
						z[2*j+1] = data[(n1+2)*j+lf+1];
					}
					
					cfftf1_(&n2,z,w,work2,(int *)(work2+2*n2));
						
					for(j=0; j<n2; j++)
					{
						data[(n1+2)*j+lf+0] = z[2*j+0];
						data[(n1+2)*j+lf+1] = z[2*j+1];
					}
						
				}
			}

			for(j=0; j<n2; j++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n3*j;
				z = z1 + 2*n3*j;
				for(i=0; i<n12; i++) 
				{
					
					lf = 2*i + (n1+2)*j;
					for(k=0; k<n3; k++)
					{
						z[2*k+0] = data[(n1+2)*n2*k+lf+0];
						z[2*k+1] = data[(n1+2)*n2*k+lf+1];
					}
					
					cfftf1_(&n3,z,w,work3,(int *)(work3+2*n3));
						
					for(k=0; k<n3; k++)
					{
						data[(n1+2)*n2*k+lf+0] = z[2*k+0];
						data[(n1+2)*n2*k+lf+1] = z[2*k+1];
					}
						
				}
			}

			float akz;
			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				kk = k; if(k >= n3/2) kk -= n3;
				akz = 6.2830/n3*kk;
				float aky, akx, cc, ss, dr, di;
				int jj;
				for(j=0; j<n2; j++)
				{
					jj = j; if(j >= n2/2) jj -= n2;
					aky = 6.2830/n2*jj;
					lf = (n1+2)*(j+n2*k);
					for(i=0; i<n12; i++)
					{
						akx = 6.2830/n1*i;

//						cc = 1.0;
//						ss = 0.0;
						cc = cos(0.5*(akx+aky+akz));
						ss = sin(0.5*(akx+aky+akz));
						dr = data[2*i+0+lf];
						di = data[2*i+1+lf]; 
						data[2*i+0+lf] = dr*cc + di*ss;
						data[2*i+1+lf] = di*cc - dr*ss; 

					}
				}
			}
			//
			//  Y & Z directions
			//
			for(j=0; j<n2; j++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n3*j;
				z = z1 + 2*n3*j;
				for(i=0; i<n12; i++) 
				{
					
					lf = 2*i + (n1+2)*j;
					for(k=0; k<n3; k++)
					{
						z[2*k+0] = data[(n1+2)*n2*k+lf+0];
						z[2*k+1] = data[(n1+2)*n2*k+lf+1];
					}
					
					cfftb1_(&n3,z,w,work3,(int *)(work3+2*n3));
						
					for(k=0; k<n3; k++)
					{
						data[(n1+2)*n2*k+lf+0] = z[2*k+0];
						data[(n1+2)*n2*k+lf+1] = z[2*k+1];
					}
						
				}
			}

			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n2*k;
				z = z1 + 2*n2*k;
				for(i=0; i<n12; i++) 
				{
					
					lf = 2*i + (n1+2)*n2*k;
					for(j=0; j<n2; j++)
					{
						z[2*j+0] = data[(n1+2)*j+lf+0];
						z[2*j+1] = data[(n1+2)*j+lf+1];
					}
					
					cfftb1_(&n2,z,w,work2,(int *)(work2+2*n2));
						
					for(j=0; j<n2; j++)
					{
						data[(n1+2)*j+lf+0] = z[2*j+0];
						data[(n1+2)*j+lf+1] = z[2*j+1];
					}
						
				}
			}
			//
			//  X - direction
			//
			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				w = w1 + n1*k;
				for(j=0; j<n2; j++) 
				{
					
					lf = (n1+2)*(j+n2*k);
					lc = cellOffset[1]*j + cellOffset[2]*k + n*cellComponentOffset;

					for(i=1; i<n1; i++) data[i+lf] = data[i+1+lf];

					rfftb1_(&n1,data+lf,w,work1,(int *)(work1+n1));
					
				}
			}
			//
			//  Limit with the cell data
			//
			for(k=0; k<n3; k++) // this loop can be parallelized
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				vtkIdType lim, ljm, lkm, lip, ljp, lkp;  // cell offsets
				float vmin, vmax, vv, v[8];
				int m;
				lkm = k - 1;
				lkp = k;
				if(lkm < 0) 
				{
					if(limits[0][0]->periodic[2]) lkm += cellDims[2]; else lkm = lkp;
				}
				if(lkp >= cellDims[2]) 
				{
					if(limits[0][0]->periodic[2]) lkp -= cellDims[2]; else lkp = lkm;
				}
				lkm *= cellOffset[2];
				lkp *= cellOffset[2];
				
				for(j=0; j<n2; j++)
				{
					ljm = j - 1;
					ljp = j;
					if(ljm < 0) 
					{
						if(limits[0][0]->periodic[1]) ljm += cellDims[1]; else ljm = ljp;
					}
					if(ljp >= cellDims[1]) 
					{
						if(limits[0][0]->periodic[1]) ljp -= cellDims[1]; else ljp = ljm;
					}
					ljm *= cellOffset[1];
					ljp *= cellOffset[1];
					
					lf = (n1+2)*(j+n2*k);
					
					for(i=0; i<n1; i++)
					{
						lim = i - 1;
						lip = i;
						if(lim < 0) 
						{
							if(limits[0][0]->periodic[0]) lim += cellDims[0]; else lim = lip;
						}
						if(lip >= cellDims[0]) 
						{
							if(limits[0][0]->periodic[0]) lip -= cellDims[2]; else lip = lim;
						}
						lim *= cellOffset[0];
						lip *= cellOffset[0];
						//
						//  8 neighbors
						//
						v[0] = *(cellData+lim+ljm+lkm+n*cellComponentOffset);
						v[1] = *(cellData+lip+ljm+lkm+n*cellComponentOffset);
						v[2] = *(cellData+lim+ljp+lkm+n*cellComponentOffset);
						v[3] = *(cellData+lip+ljp+lkm+n*cellComponentOffset);
						v[4] = *(cellData+lim+ljm+lkp+n*cellComponentOffset);
						v[5] = *(cellData+lip+ljm+lkp+n*cellComponentOffset);
						v[6] = *(cellData+lim+ljp+lkp+n*cellComponentOffset);
						v[7] = *(cellData+lip+ljp+lkp+n*cellComponentOffset);
						
						vmin = vmax = v[0];
						for(m=1; m<8; m++)
						{
							if(vmin > v[m]) vmin = v[m];
							if(vmax < v[m]) vmax = v[m];
						}

						vv = data[i+lf]/=cellSize;
						if(vv < vmin) vv = vmin;
						if(vv > vmax) vv = vmax;
						data[i+lf] = vv; 

					}
				}
			}
			//
			//  Put back into cell data
			//
			for(k=0; k<n3; k++)
			{
				prog ++; this->updateProgress(mode,prog/progMax);
				if(aborted) return;
				for(j=0; j<n2; j++)
				{
					lf = (n1+2)*(j+n2*k);
					lc = cellOffset[1]*j + cellOffset[2]*k + n*cellComponentOffset;
					for(i=0; i<n1; i++)
					{
						*(cellData+cellOffset[0]*i+lc) = data[i+lf]; 
					}
				}
			}
		}

		delete [] work1;
		delete [] work2;
		delete [] work3;
		delete [] z1;
		delete [] w1; 

	}
	//
	//  Projection
	//
	vtkIdType lim, ljm, lkm, lip, ljp, lkp;  // cell offsets
	float v[8], vv;
	for(k=0; k<pointDims[2]; k++)
	{
		prog ++; this->updateProgress(mode,prog/progMax);
		if(aborted) return;
		lkm = k - 1;
		lkp = k;
		if(lkm < 0) 
		{
			if(limits[0][0]->periodic[2]) lkm += cellDims[2]; else lkm = lkp;
		}
		if(lkp >= cellDims[2]) 
		{
			if(limits[0][0]->periodic[2]) lkp -= cellDims[2]; else lkp = lkm;
		}
		lkm *= cellOffset[2];
		lkp *= cellOffset[2];

		for(j=0; j<pointDims[1]; j++)
		{
			ljm = j - 1;
			ljp = j;
			if(ljm < 0) 
			{
				if(limits[0][0]->periodic[1]) ljm += cellDims[1]; else ljm = ljp;
			}
			if(ljp >= cellDims[1]) 
			{
				if(limits[0][0]->periodic[1]) ljp -= cellDims[1]; else ljp = ljm;
			}
			ljm *= cellOffset[1];
			ljp *= cellOffset[1];
			
			pointLoc = (vtkIdType)pointOffset[1]*j + pointOffset[2]*k;

			for(i=0; i<pointDims[0]; i++)
			{
				lim = i - 1;
				lip = i;
				if(lim < 0) 
				{
					if(limits[0][0]->periodic[0]) lim += cellDims[0]; else lim = lip;
				}
				if(lip >= cellDims[0]) 
				{
					if(limits[0][0]->periodic[0]) lip -= cellDims[2]; else lip = lim;
				}
				lim *= cellOffset[0];
				lip *= cellOffset[0];

				for(n=0; n<numComponents; n++) 
				{
					//
					//  8 neighbors
					//
					v[0] = *(cellData+lim+ljm+lkm+n*cellComponentOffset);
					v[1] = *(cellData+lip+ljm+lkm+n*cellComponentOffset);
					v[2] = *(cellData+lim+ljp+lkm+n*cellComponentOffset);
					v[3] = *(cellData+lip+ljp+lkm+n*cellComponentOffset);
					v[4] = *(cellData+lim+ljm+lkp+n*cellComponentOffset);
					v[5] = *(cellData+lip+ljm+lkp+n*cellComponentOffset);
					v[6] = *(cellData+lim+ljp+lkp+n*cellComponentOffset);
					v[7] = *(cellData+lip+ljp+lkp+n*cellComponentOffset);
					//
					//  Order data for some of modes
					//
					if(modeCellToPoint>1 && modeCellToPoint<5)
					{
						for(jj=1; jj<8; jj++) 
						{ 
							vv = v[jj];
							ii = jj - 1;
							while(ii>0 && v[ii] > vv) 
							{ 
								v[ii+1] = v[ii];
								ii--;
							}
							v[ii+1] = vv; 
						}
					}
					//
					//  Do projection
					//
					switch (modeCellToPoint) 
					{
					case 0:  // use Fourier transform - at this moment the data are already shifted.
						{
							vv = v[7];
							break;
						}
					case 2:  // median
						{
							vv = v[3];
							break;
						}
					case 3:  // min
						{
							vv = v[0];
							break;
						}
					case 4:  // max
						{
							vv = v[7];
							break;
						}
					case 5:  // corner projection
						{
							vv = v[0];
							break;
						}
						
					case 1:  // 8-point average
					default: // is the default one
						{
							vv = 0.125*(v[0]+v[1]+v[2]+v[3]+v[4]+v[5]+v[6]+v[7]);
							break;
						}
					}
					
					*(pointData+pointLoc+i*pointOffset[0]+n*pointComponentOffset) = vv;
					
				}
			}
		}
	}
	//
	//  Replace cell data with point data
	//
	cellSize = pointSize;
	for(i=0; i<3; i++) cellDims[i] = pointDims[i];
	delete [] cellData;
	cellData = pointData;

	this->updateProgress(mode,1.0);

}


void iDataReader::Swap4BytesRange(char *data, long count)
{
	char *pos;
	int i;
	
	pos = data;
	for(i = 0; i<count; i++)
    {
		Swap4Bytes(pos);
		pos += 4;
    }
}


void iDataReader::Swap8BytesRange(char *data, long count)
{
	char *pos;
	int i;
	
	pos = data;
	for(i = 0; i<count; i++)
    {
		Swap8Bytes(pos);
		pos += 8;
    }
}


long iDataReader::buildParticlesMask(long ntot0, bool*mask, float pdf)
{
	int ntot1d = (int)floor(pow((double)ntot0,0.3333333));
	vtkIdType ntot1 = (vtkIdType)ntot1d*ntot1d*ntot1d;
	vtkIdType ntot = 0;
	vtkIdType l, loff;
	int i, j, k;

	if(pdf <1.0+1.0e-5)
	{
		for(l=0; l<ntot0; l++) mask[l] = true;
		return ntot0;
	}

	switch(partDownsampleMode)
	{
	case 0:
		{
			int partIntDownsampleFactor = round(pow(pdf,0.33333f));
			for(k=0; k<ntot1d; k++) 
			{
				for(j=0; j<ntot1d; j++) 
				{
					loff = ntot1d*(j+ntot1d*k);
					for(i=0; i<ntot1d; i++) 
					{
						mask[i+loff] = k%partIntDownsampleFactor==0 && j%partIntDownsampleFactor==0 && i%partIntDownsampleFactor==0; 
					}
				}
			}
			for(l=ntot1; l<ntot0; l++) mask[l] = false;
			break;
		}
	case 1:
		{
			float threshold = 1.0/pdf;
			for(l=0; l<ntot0; l++)
			{
				mask[l] = (vtkMath::Random() < threshold);
			}
			break;
		}
	case 2:
		{
			ntot = round(ntot0/pdf);
			for(l=0; l<ntot; l++)
			{
				mask[l] = true;
			}
			for(l=ntot; l<ntot0; l++)
			{
				mask[l] = false;
			}
			break;
		}
	}

	if(ntot == 0) for(l=0; l<ntot0; l++) if(mask[l]) ntot++; 

	if(ntot == 0)
	{
		ntot = 1;
		mask[0] = true;
	}

	return ntot;

}


void iDataReader::setVarStretch(int v, int s)
{
	limits[0][0]->setVarStretch(v,s);
}


void iDataReader::packState(iString &s)
{
	iString s1;

	s = "";

	this->packValue(s,"iDataReader::dataChannel",dataChannel);
	this->packValue(s,"iDataReader::dataStream",dataStream);

	this->packValue(s,"iDataReader::adjustableBounds",adjustableBounds);
	this->packValue(s,"iDataReader::endinessOfData",endinessOfData);
	this->packValue(s,"iDataReader::part2meshOn",part2meshOn);
	this->packValue(s,"iDataReader::pointsAreFloat",pointsAreFloat);

	this->packValue(s,"iDataReader::modeCellToPoint",modeCellToPoint);
	this->packValue(s,"iDataReader::scaledDim",scaledDim);
	this->packValue(s,"iDataReader::meshArrayOutput",meshArrayOutput);
	this->packValue(s,"iDataReader::partDownsampleMode",partDownsampleMode);

	this->packValue(s,"iDataReader::partDownsampleFactor",partDownsampleFactor);
	this->packValue(s,"iDataReader::levMin",levMin,1+NVARMAX);
	this->packValue(s,"iDataReader::levMax",levMax,1+NVARMAX);
	
	this->packValue(s,"iDataReader::meshArrayFunction",meshArrayFunction);
	this->packValue(s,"iDataReader::partBasicScale",partBasicScale);

//	this->packValue(s,"iDataReader::maxDataStream",maxDataStream,MAXDATACHANNEL+1);

	int i, j;
	for(i=0; i<=MAXDATACHANNEL; i++) if(limits[i] != NULL) 
	{
		for(j=0; j<=maxDataStream[i]; j++) if(limits[i][j] != NULL)
		{
			limits[i][j]->packState(s1);
			s += "##" + s1;
		}
	}

}


void iDataReader::unpackState(iString s)
{
	int i, j, k; bool b; float f; iString s1; float v[1+NVARMAX];

	if(this->unpackValue(s,"iDataReader::adjustableBounds",b)) this->setAdjustableBounds(b);
	if(this->unpackValue(s,"iDataReader::endinessOfData",b)) this->setEndinessOfData(b);
	if(this->unpackValue(s,"iDataReader::part2meshOn",b)) this->setMeshFromParticles(b);
	if(this->unpackValue(s,"iDataReader::pointsAreFloat",b)) this->setPointsAreFloat(b);

	if(this->unpackValue(s,"iDataReader::modeCellToPoint",i)) this->setCellToPointMode(i);
	if(this->unpackValue(s,"iDataReader::scaledDim",i)) this->setScaledDimension(i);
	if(this->unpackValue(s,"iDataReader::meshArrayOutput",i)) this->setMeshArrayOutput(i);
	if(this->unpackValue(s,"iDataReader::partDownsampleMode",i)) this->setParticlesDownsampleMode(i);

	if(this->unpackValue(s,"iDataReader::partDownsampleFactor",f)) this->setParticlesDownsampleFactor(f);
	if(this->unpackValue(s,"iDataReader::levMin",v,1+NVARMAX)) memcpy(levMin,v,(1+NVARMAX)*sizeof(float));
	if(this->unpackValue(s,"iDataReader::levMax",v,1+NVARMAX)) memcpy(levMax,v,(1+NVARMAX)*sizeof(float));
	
	if(this->unpackValue(s,"iDataReader::meshArrayFunction",s1)) this->setMeshArrayFunction(s1);
	if(this->unpackValue(s,"iDataReader::partBasicScale",f)) partBasicScale = f;

//  why is this? It would crash when extended version reads a state file from the non-extented one
//	if(this->unpackValue(s,"iDataReader::maxDataStream",d,MAXDATACHANNEL+1)) memcpy(maxDataStream,d,(MAXDATACHANNEL+1)*sizeof(int));

	k = 0;
	for(i=0; i<=MAXDATACHANNEL; i++) if(limits[i] != NULL) 
	{
		for(j=0; j<=maxDataStream[i]; j++) if(limits[i][j] != NULL)
		{
			k++;
			s1 = s.section("##",k,k);
			limits[i][j]->unpackState(s1);
		}
	}

//	if(this->unpackValue(s,"iDataReader::dataChannel",i) && i>=0 && i<=MAXDATACHANNEL) dataChannel = i;
//	if(this->unpackValue(s,"iDataReader::dataStream",i) && i>=0 && i<=maxDataStream[dataChannel]) dataStream[dataChannel] = i;

}


void iDataReader::updateProgress(int m, float p)
{
	if(meshProgressBar!=NULL && (m&IDATAREADER_MESH))
	{
		meshProgressBar->setProgress(round(100.0*p));
	}
	if(partProgressBar!=NULL && (m&IDATAREADER_PART))
	{
		partProgressBar->setProgress(round(100.0*p));
	}
	if(vectProgressBar!=NULL && (m&IDATAREADER_VECT))
	{
		vectProgressBar->setProgress(round(100.0*p));
	}
	if(tensProgressBar!=NULL && (m&IDATAREADER_TENS))
	{
		tensProgressBar->setProgress(round(100.0*p));
	}
}


void iDataReader::setProgressBar(int m, iProgressBar *pb)
{
	if(m & IDATAREADER_MESH) meshProgressBar = pb;
	if(m & IDATAREADER_PART) partProgressBar = pb;
	if(m & IDATAREADER_VECT) vectProgressBar = pb;
	if(m & IDATAREADER_TENS) tensProgressBar = pb;
}


bool iDataReader::isProgressBarActive(int m)
{
	if(meshProgressBar!=NULL && (m&IDATAREADER_MESH)) return true;
	if(partProgressBar!=NULL && (m&IDATAREADER_PART)) return true;
	if(vectProgressBar!=NULL && (m&IDATAREADER_VECT)) return true;
	if(tensProgressBar!=NULL && (m&IDATAREADER_TENS)) return true;
	return false;
}







